#include <limits.h>
#include <stdio.h>
#include <allegro.h>
#include "item.h"
#include "obstacle.h"
#include "area.h"
#include "bomb.h"
#include "items.h"
#include "xpm.h"
#include "gfx/grounds/ground.xpm"
#include "anim.h"
#include "message.h"
#include "eggs.h"
#include "color.h"
#include "sfx/explode.inc"
#include "sounds.h"

static inline int pitch_variance()
{
  //return rand()%200-100;
  return -rand()%100;
}

area::area(int width,int height,int bomb_freq,int bomb_lifetime):
  width(width),
  height(height),
  bomb_freq(bomb_freq),
  bomb_lifetime(bomb_lifetime),
  pickable_count(0),
  amb(0),
  explo(new anim(explosion_egg,64,64,32)),
  time(0),
  next_cry_time(0)
{
}

area::~area()
{
  while (!items.empty()) {
    item *i = items.front();
    items.pop_front();
    delete i;
  }
  delete explo;
  delete amb;
}

int area::get_random_cry_delay() const
{
  static const int min_delay=TICK_SPEED*50;
  static const int max_delay=TICK_SPEED*300;
  return (min_delay+rand()%(max_delay-min_delay))/(1+get_pickable_count());
}

int area::compute_volume(int x,int y) const
{
  if (!amb) return 0;
  int dist=amb->get_squared_distance(x,y);
  int max_dist=get_width()*get_width()+get_height()+get_height();
  return 255*(max_dist-dist)/max_dist;
}

int area::compute_pan(int x,int y) const
{
  if (!amb) return 128;
  int dist=x-amb->get_x();
  int max_dist=get_width();
  return 128+127*dist/max_dist;
}

void area::add(item *i)
{
  if (i->is_pickable()) {
    pickable_count++;
  }
  items.push_back(i);
}

void area::remove(item *i)
{
  if (i->is_pickable()) {
    pickable_count--;
  }
  items.remove(i);
}

void area::destroy(item *i)
{
  remove(i);
  delete i;
}

/* note: to_destroy must not be items */
void area::destroy(list<item*> &to_destroy)
{
  list<item*>::iterator last = to_destroy.end();
  for (list<item*>::iterator i=to_destroy.begin();i!=last;++i) {
    destroy(*i);
  }
}

void area::add_bomb(int radius,int lifetime)
{
  int x=rand()%get_width();
  int y=rand()%get_height();
  bombs.push_back(new bomb(x,y,radius,lifetime));
}

void area::set_ambulance(ambulance *a)
{
  if (amb) delete amb;
  if (a) {
    /* Make sure there is nothing where the ambulance is put */
    crater *c = new crater(a->get_x(), a->get_y());
    blast(c);
    destroy(c);
  }
  amb=a;
}

void area::add_anim(animator *a)
{
  anims.push_back(a);
}

void area::add_message(const char *text,int color,int lifetime)
{
  char *s=strdup(text);
  const char *token=strtok(s,"\n");
  while (token) {
    msgs.push_back(new message(token,color,lifetime));
    token=strtok(NULL,"\n");
  }
}

int area::get_roughness_at(int x, int y) const
{
  int roughness = 0;
  for (list<item*>::const_iterator i=items.begin();i!=items.end();++i) {
    const obstacle *o = (*i)->get_obstacle();
    if (o && o->is_in(x, y)) {
      int r = o->get_roughness();
      if (r>roughness) roughness=r;
    }
  }
  return roughness;
}

bool area::is_blocked_at(int x, int y) const
{
  return get_roughness_at(x, y) == INT_MAX;
}

bool area::collide_with_any(const item *it) const
{
  for (list<item*>::const_iterator i=items.begin();i!=items.end();++i) {
    if (it->collide_with(*i)) {
      return true;
    }
  }
  return false;
}

bool area::collide_with_blocked(const item *it) const
{
  for (list<item*>::const_iterator i=items.begin();i!=items.end();++i) {
    const obstacle *o = (*i)->get_obstacle();
    if (o && o->get_roughness()==INT_MAX && it->collide_with(*i)) {
      return true;
    }
  }
  return false;
}

void area::draw(BITMAP *dest)
{
  const int irrelevant = 0;
  static BITMAP *xpm_ground = load_xpm(ground_magick);

  drawing_mode(DRAW_MODE_COPY_PATTERN, xpm_ground, 0, 0);
  rectfill(dest, 0, 0, get_width()-1, get_height()-1, irrelevant);
  solid_mode();
  for (list<item*>::const_iterator i=items.begin();i!=items.end();++i) {
    (*i)->draw(dest);
  }
  if (amb) amb->draw(dest);
  for (list<bomb*>::const_iterator i=bombs.begin();i!=bombs.end();++i) {
    (*i)->draw(dest);
  }
  for (list<animator*>::const_iterator i=anims.begin();i!=anims.end();++i) {
    (*i)->draw(dest);
  }
  const int x=get_width()/2;
  const int dy=(3*text_height(font))/2;
  int y=get_height()/2-dy*msgs.size();
  for (list<message*>::const_iterator i=msgs.begin();i!=msgs.end();++i) {
    (*i)->draw(dest,x,y);
    y+=dy;
  }
}

/* crater must not be already in the area */
void area::blast(item *crater,bool in_game)
{
  list<item*> to_remove;
  int pickable_count=0;
  for (list<item*>::const_iterator i=items.begin();i!=items.end();++i) {
    if ((*i)->is_blastable() && crater->collide_with(*i)) {
      to_remove.push_back(*i);
      if ((*i)->is_pickable()) ++pickable_count;
    }
  }

  if (pickable_count) {
    char msg[256];
#ifdef ALLEGRO_LINUX
    snprintf(msg,sizeof(msg),"%d died",pickable_count);
#else
    sprintf(msg,"%d died",pickable_count);
#endif
    add_message(msg,DIED_COLOR,TICK_SPEED*4);
  }

  destroy(to_remove);
  add(crater);
  if (amb && amb->is_blastable() && crater->collide_with(amb)) {
    set_ambulance(0);
  }

  if (in_game) {
    add_anim(new animator(crater->get_x(),crater->get_y(),explo));
    const int vol=2*compute_volume(crater->get_x(),crater->get_y())/3;
    const int pan=compute_pan(crater->get_x(),crater->get_y());
    play_sample(&sfx_explode,vol,pan,1000,0);
  }
}

int area::pickup(position &position,int radius)
{
  static const SAMPLE *thanks[] = {
    &sfx_f_ar, &sfx_f_de, &sfx_f_en, &sfx_f_es, &sfx_f_fr, &sfx_f_it,
    &sfx_m_ar, &sfx_m_de1, &sfx_m_en1, &sfx_m_fr, &sfx_m_fr, &sfx_m_it,
    &sfx_m_de2, &sfx_m_en2
    //TODO: sfx_m_es
  };
  static const int thanks_count = sizeof(thanks)/sizeof(thanks[0]);
  int count=0;
  list<item*>::iterator it=items.begin();
  while (it!=items.end()) {
    list<item*>::iterator i=it;
    ++it;
    if ((*i)->is_pickable()) {
      int dx=position.get_x()-(*i)->get_x();
      int dy=position.get_y()-(*i)->get_y();
      if (dx*dx+dy*dy<radius*radius) {
        destroy(*i);
        ++count;
      }
    }
  }

  if (count) {
    char msg[256];
#ifdef ALLEGRO_LINUX
    snprintf(msg,sizeof(msg),"%d picked up",count);
#else
    sprintf(msg,"%d picked up",count);
#endif
    add_message(msg,PICKED_COLOR,TICK_SPEED*4);
    int sfx_id = rand()%thanks_count;
    play_sample(thanks[sfx_id], 128, 128, 1000+pitch_variance(), 0);
  }

  return count;
}

struct line_info {
  item *mover;
  int blocked;
  line_info(item *i): mover(i), blocked(0) {}
};

static void advance_step(BITMAP *a,int x,int y,int d)
{
  line_info *info=(line_info*)d;
  if (info->blocked) return; // do_line does not have a shortcut
  int prevx=info->mover->get_x();
  int prevy=info->mover->get_y();
  info->mover->set_position(x,y);
  if (((area*)a)->collide_with_blocked(info->mover)) {
    info->mover->set_position(prevx,prevy);
    info->blocked=1;
  }
}

int area::advance(item *i,int x,int y) const
{
  line_info info(i);
  do_line((BITMAP*)this,i->get_x(),i->get_y(),x,y,(int)&info,&advance_step);
  return !info.blocked;
}

template<class c>
void tick_and_destroy(area *map,std::list<c*> &l)
{
  for (list<c*>::iterator i=l.begin();i!=l.end();) {
    list<c*>::iterator it=i;
    ++i;
    c *m=*it;
    if (!m->tick()) {
      delete *it;
      l.erase(it);
    }
  }
}

int area::tick(int dx,int dy)
{
  ++time;

  static const SAMPLE *cries[] = {
    &sfx_f_cry0, /*&sfx_f_cry1,*/ &sfx_f_cry2,
    &sfx_m_cry0, /*&sfx_m_cry1,*/ &sfx_m_cry2, &sfx_m_cry3, &sfx_m_cry4,
  };
  static const int cries_count = sizeof(cries)/sizeof(cries[0]);

  if (get_pickable_count() && time>=next_cry_time) {
    int sfx_id = rand()%cries_count;
    if (time!=1) play_sample(cries[sfx_id], 128, 128, 1000+pitch_variance(), 0);
    next_cry_time=time+get_random_cry_delay();
  }

  int count=0;

  if (amb) count=amb->tick(this,dx,dy);

  for (list<bomb*>::iterator i=bombs.begin();i!=bombs.end();) {
    list<bomb*>::iterator it=i;
    ++i;
    bomb *b=*it;
    if (!b->tick()) {
      blast(new crater(b->get_x(),b->get_y()),1);
      delete *it;
      bombs.erase(it);
    }
  }
  tick_and_destroy(this,anims);
  tick_and_destroy(this,msgs);

  if (get_bomb_freq()>0 && (rand()%get_bomb_freq())==0) {
    int radius=32;
    int lifetime=get_bomb_lifetime();
    add_bomb(radius,lifetime);
  }

  return count;
}

