/*
 * Allegro DIALOG Editor
 * by Julien Cugniere
 *
 * strparse.c : String parsing functions
 */

#include "strparse.h"
#include <allegro.h>
#include <stdio.h>



static const char *flag_strings[] =
{
	"D_EXIT",
	"D_SELECTED",
	"D_GOTFOCUS",
	"D_GOTMOUSE",
	"D_HIDDEN",
	"D_DISABLED",
	"D_DIRTY",
	"D_INTERNAL"
};



/***************************************************************************/
/********************************* Helpers *********************************/
/***************************************************************************/



/* get a copy of a string, allocating the memory */
char *get_copy(char *str)
{
	char *res = malloc(NCHAR);
	if (res)
		ustrcpy(res, str);
	return res;
}



/* advances a string iterator past any whitespaces. whitespaces include
 * comments.
 */
void str_skip_ws(char **in)
{
	int c = ugetx(in);
	while (c) {
		if (!uisspace(c)) {
			if (c == '/') {
				if (ugetc(*in) == '/') {
					do c = ugetx(in);
					while (c && c != '\n');
				}
				else if (ugetc(*in) == '*') {
					ugetx(in);
					do c = ugetx(in);
					while (c && !(c == '*' && ugetc(*in) == '/'));
					ugetx(in);
				}
				else break;
			}
			else break;
		}
		c = ugetx(in);
	}
	(*in) -= ucwidth(c);
}



/* converts the text of a C string literal to a null terminated C string. Can
 * handle literals broken in multiple parts separated by plain whitespaces.
 * Returns a pointer to where it stopped parsing the input on succes, and NULL
 * on error.
 */
char *littostr(char *buff, char *lit)
{
	char *ibuff = buff;
	char *ilit = lit;
	int c, i, n;

	if (!lit)
		return NULL;

	do c = ugetx(&ilit);
	while (c && uisspace(c));
	if (c != '\"')
		return NULL;

	while (c == '\"') {
		c = ugetx(&ilit);
		while (c && c != '\"' && c != '\n') {
			switch (c) {
			case '\\':
				c = ugetx(&ilit);
				switch (c) {
				case '\'':
				case '\"':
				case '?':
				case '\\': ibuff += usetc(ibuff, c); break;

				case 'a': ibuff += usetc(ibuff, '\a'); break;
				case 'b': ibuff += usetc(ibuff, '\b'); break;
				case 'f': ibuff += usetc(ibuff, '\f'); break;
				case 'n': ibuff += usetc(ibuff, '\n'); break;
				case 'r': ibuff += usetc(ibuff, '\r'); break;
				case 't': ibuff += usetc(ibuff, '\t'); break;
				case 'v': ibuff += usetc(ibuff, '\v'); break;

				case '\n':
					break;

				case 0:
					ilit -= ucwidth(c);
					break;

				case 'x':
				case 'X':
					c = ugetc(ilit);
					if (ustrchr("0123456789abcdefABCDEF", c)) {
						ibuff += usetc(ibuff, ustrtol(ilit, NULL, 16));
						while (ustrchr("0123456789abcdefABCDEF", c)) {
							ilit += ucwidth(c);
							c = ugetc(ilit);
						}
					}
					break;

				case '0':   case '1':   case '2':   case '3':
				case '4':   case '5':   case '6':   case '7':
					ilit -= ucwidth(c);
					n = 0;
					for (i=0; i<3 && '0'<=c && c<='7'; i++) {
						n = (n*8) + (c-'0');
						ilit += ucwidth(c);
						c = ugetc(ilit);
					}
					if (i)
						ibuff += usetc(ibuff, n);
					break;

				default:
					ibuff += usetc(ibuff, c);
					break;
				}
				break; /* case '\\': */
		
			default:
				ibuff += usetc(ibuff, c);
				break;
			}
			c = ugetx(&ilit);
		}

		if (c) {
			do c = ugetx(&ilit);
			while(uisspace(c));
		}
	}
	ilit -= ucwidth(c);

	usetc(ibuff, 0);
	return ilit;
}



/***************************************************************************/
/********************************* Parsers *********************************/
/***************************************************************************/



/* converts a c string-literal into a run-time c string. Handles casting,
 * comments, and other niceties like that. Returns buff on succes, and NULL
 * on error.
 */
char *get_string(char *buff, char *lit)
{
	char *ibuff = buff;
	char *ilit = lit;
	int c, found_something = FALSE;

	usetc(buff, 0);
	do {
		do {
			str_skip_ws(&ilit);
			c = ugetx(&ilit);
		}
		while (c && c != '\"');
		ilit -= ucwidth(c);
		if ((ilit = littostr(ibuff, ilit)) != 0)
			found_something = TRUE;
		ibuff += ustrsize(ibuff);
	}
	while (ilit);

	if (!found_something)
		return NULL;

	usetc(ibuff, 0);

	return buff;
}



/* reads a number, skipping any leading irrelevant data */
int get_number(char *s)
{
	str_skip_ws(&s);
	return ustrtol(s, NULL, 0);
}



/* simple flags reader. it understands expressions like:
 *     D_EXIT | D_DISABLED | ... | 512
 *     D_HIDDEN + D_SELECTED ...
 */
int get_flags(char *s)
{
	int flags, i;
	char *tok, *str;

	str_skip_ws(&s);
	flags = 0;
	str = ustrdup(s);
	tok = ustrtok(str, "+| \t\n");
	while (tok) {
		if (uisdigit(ugetc(tok)))
			flags |= ustrtol(tok, NULL, 0);
		else {
			for (i=0; i<8; i++)
				if (ustrcmp(flag_strings[i], tok) == 0)
					flags |= 1<<i;
		}
		tok = ustrtok(NULL, "+| \t\n");
	}
	free(str);

	return flags;
}



/* translate a character constant into a character. Doesn't support multibyte
 * characters. Can understand any macro beginning with a C, and considers it
 * was defined as "#define CTRL(c) (c-'a'+1)", which correspond to Control-c.
 */
int get_key(char *str)
{
	int c, ctrl = 0;
	char *s = str;

	str_skip_ws(&s);
	c = ugetx(&s);

	if (c == 0)
		return 0;

	if (c == 'C') {
		ctrl = 1 - 'a';
		while (c != '(' && c)
			c = ugetx(&s);

		if (c == '\0')
			return 0;

		c = ugetx(&s);
	}

	if (c == '\'') {
		c = ugetx(&s);
		if (c == '\\') {
			c = ugetc(s);
			switch (c) {
				case '\0':
				case '\'':
				case '"':
				case '?':
				case '\\': return c;

				case 'a': return '\a';
				case 'b': return '\b';
				case 'f': return '\f';
				case 'n': return '\n';
				case 'r': return '\r';
				case 't': return '\t';
				case 'v': return '\v';

				case 'x': return ustrtol(s+ucwidth(c), NULL, 16);

				default:  return ustrtol(s, NULL, 8);
			}
		}
		else return (c + ctrl);
	}

	return ustrtol(str, NULL, 0);
}



/***************************************************************************/
/********************************* Makers **********************************/
/***************************************************************************/



/* flags maker. the result is in the form:
 *      D_EXIT | D_SELECTED | ... | 512
 * the number at the end stands for the flags above D_USER (if any)
 */
char *make_flags(char *buff, int flags)
{
	int i;
	char *ibuff = buff;

	for (i=0; i<8; i++) {
		if (flags & (1<<i)) {
			flags &= ~(1<<i);
			ibuff += usprintf(ibuff, "%s|", flag_strings[i]);
		}
	}

	if (flags || ustrlen(buff) == 0)
		usprintf(ibuff, "%d", flags);
	else
		usetat(buff, -1, 0); /* erase the last '|' char */

	return buff;
}



