/*
 * Allegro DIALOG Editor
 * by Julien Cugniere
 *
 * dedit.c : The core of the editor
 */

#include "dedit.h"
#include "guiprocs.h"
#include "strparse.h"
#include "fileops.h"
#include <allegro.h>
#include <allegro/internal/aintern.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#define XOR_COLOR   makecol(170, 170, 170)

#define SCARED(x)   do { scare_mouse(); x; unscare_mouse(); } while(0)
#define FREE(x)     do { free(x); x = NULL; } while(0)
#define C(c)        (c - 'a' + 1)



/* editor functions */
extern void init_plugins(void);
void init_editor(void);
void deinit_editor(void);
int change_gfx_mode(int, int, int, int);
void set_menu_new(MENU*);
void set_pointers(void);
void set_editor_gui_state(void);
void set_dialog_gui_state(void);


/* menu callbacks */
int about(void);
int closer(void);
int mover(void);
int snaper(void);
int object_newer(void);
int grid_resizer(void);
int grid_checker(void);
int grid_hider(void);
int properties(void);
int select_all(void);
int select_none(void);
int invert_selection(void);
int list_editer(void);
int cutter(void);
int copier(void);
int paster(void);
int deleter(void);
int newer(void);
int opener(void);
int saver(void);
int save_as(void);
int gui_setter(void);
int preferences(void);
int palette_loader(void);
int graphic_mode_setter(void);
int color_setter(void);
int source_viewer(void);
int tester(void);
int shift_sel_up(void);
int shift_sel_down(void);
int shift_sel_home(void);
int shift_sel_end(void);


/* dialog procs */
int workspace_proc(int, DIALOG*, int);
int d_unknown_proc(int, DIALOG*, int);
int d_close_proc(int, DIALOG*, int);


/* tdialog handling */
static TDIALOG *new_tdialog(void);
void destroy_tdialog(TDIALOG*);
void delete_tdialog(int);
void copy_tdialog(TDIALOG*, TDIALOG*);
void update_tdialog(TDIALOG*);
void tdialog_get_bounds(TDIALOG*, int, int*, int*, int*, int*);
void position_tdialog(int, int);


/* loading functions */
void load_dialog(char*, char*);
static SCAN_INFO *scan_info;  /* filled by load_dialog, read by dlg_getter */


/* tdialog containers */
static char *selected = NULL;
static TDIALOG *tdialog = NULL;
static int tdlg_size = 0;
static TDIALOG *clipboard = NULL;
static int clipboard_size = 0;


/* info about the current dialog */
static char save_path[1024] = EMPTY_STRING;
static char save_name[1024] = "dlg[] ";
static int save_offset = 0;

static char grid_str[3][20] = { { 0 } };
static int grid_w[4] = { 1, 4, 8, 12 };
static int grid_h[4] = { 1, 4, 8, 12 };
static int grid = 2;

static void (*reinit_callbacks[32])(void) = { NULL };
static void (*shutdown_callbacks[32])(void) = { NULL };
static PROC *proc_list = NULL;
static PROC *new_object = NULL;
static DIALOG_PLAYER *editor_player = NULL;
static BITMAP *default_pointer   = NULL;
static BITMAP *resize_pointer_h  = NULL;
static BITMAP *resize_pointer_v  = NULL;
static BITMAP *resize_pointer_d1 = NULL;
static BITMAP *resize_pointer_d2 = NULL;
static DATAFILE *datafile = NULL;
static FONT *tfont = NULL;
static int close_requested = FALSE;

/* gui settings of the edited dialog (not those of the editor) */
static FONT *back_font;
static int back_fg, back_mg, back_bg, back_baseline, back_mouse_focus;
static DIALOG_PROC back_gui_shadow_box_proc;
static DIALOG_PROC back_gui_ctext_proc;
static DIALOG_PROC back_gui_button_proc;
static DIALOG_PROC back_gui_edit_proc;
static DIALOG_PROC back_gui_list_proc;
static DIALOG_PROC back_gui_text_list_proc;
static void (*back_gui_menu_draw_menu)(int, int, int, int);
static void (*back_gui_menu_draw_menu_item)(MENU*, int, int, int, int, int, int);



/******************************************/
/**************** Helpers *****************/
/******************************************/



/* some inlined stuff */
INLINE int snapx(int x)
{
	x += grid_w[grid]/2;
	x -= x % grid_w[grid];
	return x;
}

INLINE int snapy(int y)
{
	y += grid_h[grid]/2;
	y -= y % grid_h[grid];
	return y;
}



/* converts a color from one color depth to another */
INLINE int convcol(int from, int to, int c)
{
	return makecol_depth(to, getr_depth(from, c), getg_depth(from, c), getb_depth(from, c));
}

#define convcol_from(from, c) convcol(from, bitmap_color_depth(screen), c)
#define convcol_to(to, c)     convcol(bitmap_color_depth(screen), to, c)



/* compares two strings, ignoring '&' characters */
int gui_stricmp(const char *a, const char *b)
{
	for (; *a && *b; a++, b++) {
		if (*a == '&')
			a++;
		if (*b == '&')
			b++;
		if (tolower(*a) != tolower(*b))
			return tolower(*a) - tolower(*b);
	}
	return 0;
}



/* finds an entry in a menu array
 *  Its better to use this than an index, because when you add or remove
 *  entries, indices are invalidated. Any criterion can be NULL if you don't
 *  care about it. Text is *not* case sensitive.
 */
MENU *find_menu(MENU *menu, const char *text, int (*const proc)(void), const MENU *child, void *dp)
{
	if (menu) {
		MENU *entry;
		for (entry = menu; entry->text; entry++)
			if ((!text || gui_stricmp(text, entry->text) == 0)
			 && (!proc || proc == entry->proc)
			 && (!child || child == entry->child)
			 && (!dp || dp == entry->dp))
				return entry;
	}

	return NULL;
}



/* returns the number of entries in a menu, not counting the terminator */
int get_menu_length(const MENU *m)
{
	int n = 0;
	if (m)
		while (m[n].text)
			n++;
	return n;
}



/* returns non-zero if a file is cpp. when in doubt (ie *.h, *) returns the
 * value from the configuration variable "cpp_output".
 */
int path_is_cpp(const char *path)
{
	const char *extension = get_extension(path);

	if (ustricmp(extension, "cpp") == 0
	 || ustricmp(extension, "cxx") == 0
	 || ustricmp(extension, "hpp") == 0)
		return 1;
	else if (ustricmp(extension, "c") == 0)
		return 0;
	else
		return get_config_int("[dlg]", "cpp_output", 0);
}



int is_directory(const char *path)
{
	struct al_ffblk dummy;
	int indeed = (al_findfirst(path, &dummy, FA_DIREC) == 0);
	al_findclose(&dummy);
	return indeed;
}









/******************************************/
/************** Dummy Params **************/
/******************************************/



BITMAP *dummy_bmp = NULL;

MENU dummy_child_menu[] =
{
	{ "&Open\tCtrl-O", NULL, NULL, 0, NULL },
	{ "&Save\tCtrl-S", NULL, NULL, 0, NULL },
	{ "",              NULL, NULL, 0, NULL },
	{ "E&xit",         NULL, NULL, 0, NULL },
	{ 0 }
};

MENU dummy_menu[] =
{
	{ "&File", NULL,       dummy_child_menu, 0, NULL },
	{ "&Help", dummy_proc, NULL,             0, NULL },
	{ 0 }
};

int dummy_proc(void)
{
	alert(" dummy_proc ", NULL, NULL, " Ok ", NULL, 0, 0);
	return D_O_K;
}

char *dummy_getter(int index, int *size)
{
	static const char *items[] = {
		"Allegro",
		"Low",
		"Level",
		"E",
		"Game",
		"Routines",
		"O"
	};
	if (index<0) {
		*size = sizeof(items) / sizeof(items[0]);
		return NULL;
	}

	return (char*)items[index];
}








/******************************************/
/********** The menus & dialogs ***********/
/******************************************/

static MENU menu_dialog[] =
{
/*	(text)             (proc)   (child) (flags) (dp) */
	{ "&New\tCtrl-N",  newer,   NULL,   0,      NULL },
	{ "&Open\tCtrl-O", opener,  NULL,   0,      NULL },
	{ "",              NULL,    NULL,   0,      NULL },
	{ "&Save\tCtrl-S", saver,   NULL,   0,      NULL },
	{ "Save &as...",   save_as, NULL,   0,      NULL },
	{ "",              NULL,    NULL,   0,      NULL },
	{ "&Quit\tCtrl-Q", closer,  NULL,   0,      NULL },
	{ 0 }
};

static MENU menu_selection[] =
{
/*	(text)                   (proc)            (child) (flags) (dp) */
	{ "Select &none",        select_none,      NULL,   0,      NULL },
	{ "Select &all\tCtrl-A", select_all,       NULL,   0,      NULL },
	{ "&Invert\tCtrl-I",     invert_selection, NULL,   0,      NULL },
	{ "",                    NULL,             NULL,   0,      NULL },
	{ "C&ut\tCtrl-X",        cutter,           NULL,   0,      NULL },
	{ "C&opy\tCtrl-C",       copier,           NULL,   0,      NULL },
	{ "Pa&ste\tCtrl-V",      paster,           NULL,   0,      NULL },
	{ "&Delete\tSuppr.",     deleter,          NULL,   0,      NULL },
	{ "",                    NULL,             NULL,   0,      NULL },
	{ "&List Edit\tCtrl-L",  list_editer,      NULL,   0,      NULL },
	{ "Set &colors",         color_setter,     NULL,   0,      NULL },
	{ "&Properties",         properties,       NULL,   0,      NULL },
	{ 0 }
};

static MENU menu_grid[] =
{
/*	(text)          (proc)        (child) (flags) (dp) */
	{ "&No grid",   grid_checker, NULL,   0,      NULL },
	{ grid_str[0],  grid_checker, NULL,   0,      NULL },
	{ grid_str[1],  grid_checker, NULL,   0,      NULL },
	{ grid_str[2],  grid_checker, NULL,   0,      NULL },
	{ "",           NULL,         NULL,   0,      NULL },
	{ "&Hide",      grid_hider,   NULL,   0,      NULL },
	{ "&Resize",    grid_resizer, NULL,   0,      NULL },
	{ "&Snap now!", snaper,       NULL,   0,      NULL },
	{ 0 }
};

static MENU menu_misc[] =
{
/*	(text)                         (proc)               (child) (flags) (dp) */
	{ "&Preferences",              preferences,         NULL,   0,      NULL },
	{ "",                          NULL,                NULL,   0,      NULL },
	{ "&Load palette",             palette_loader,      NULL,   0,      NULL },
	{ "&Graphic mode",             graphic_mode_setter, NULL,   0,      NULL },
	{ "GUI &settings",             gui_setter,          NULL,   0,      NULL },
	{ "",                          NULL,                NULL,   0,      NULL },
	{ "&View source",              source_viewer,       NULL,   0,      NULL },
	{ "&Test the dialog (unsafe)", tester,              NULL,   0,      NULL },
	{ "",                          NULL,                NULL,   0,      NULL },
	{ "&About...",                 about,               NULL,   0,      NULL },
	{ 0 }
};

static MENU *menu_new = NULL;
static MENU menu_all_procs[32] = { { 0 } };
static MENU menu_sets[32] = { { 0 } };
static MENU menu_hooks[32] = { { 0 } };

static MENU menu[] =
{
/*	(text)          (proc) (child)         (flags) (dp) */
	{ "&Dialog",    NULL,  menu_dialog,    0,      NULL },
	{ "&Selection", NULL,  menu_selection, 0,      NULL },
	{ "&Grid",      NULL,  menu_grid,      0,      NULL },
	{ "S&ets",      NULL,  menu_sets,      0,      NULL },
	{ "&New",       NULL,  NULL,           0,      NULL },
	{ "&Hooks",     NULL,  menu_hooks,     0,      NULL },
	{ "Move&!",     mover, NULL,           0,      NULL },
	{ "&Misc.",     NULL,  menu_misc,      0,      NULL },
	{ 0 }
};

static MENU menu_right_click[] =
{
/*	(text)           (proc)        (child)         (flags) (dp) */
	{ "&New",        NULL,         NULL,           0,      NULL },
	{ "&All procs",  NULL,         menu_all_procs, 0,      NULL },
	{ "",            NULL,         NULL,           0,      NULL },
	{ "C&ut",        cutter,       NULL,           0,      NULL },
	{ "C&opy",       copier,       NULL,           0,      NULL },
	{ "Pa&ste",      paster,       NULL,           0,      NULL },
	{ "",            NULL,         NULL,           0,      NULL },
	{ "Set &colors", color_setter, NULL,           0,      NULL },
	{ "&Properties", properties,   NULL,           0,      NULL },
	{ 0 }
};

static DIALOG editor[] =
{
/*	(dialog proc)      (x) (y) (w) (h) (fg) (bg) (key)   (flags) (d1)      (d2) (dp)              (dp2) (dp3) */
	{ workspace_proc,   0,  0,  0,  0,  0,   0,   0,      0,      0,        0,   NULL,             NULL, NULL },
	{ d_menu_proc,      1,  1,  0,  0,  0,   0,   0,      0,      0,        0,   menu,             NULL, NULL },
	{ d_keyboard_proc,  0,  0,  0,  0,  0,   0,   C('q'), 0,      0,        0,   closer,           NULL, NULL },
	{ d_keyboard_proc,  0,  0,  0,  0,  0,   0,   C('a'), 0,      0,        0,   select_all,       NULL, NULL },
	{ d_keyboard_proc,  0,  0,  0,  0,  0,   0,   C('i'), 0,      0,        0,   invert_selection, NULL, NULL },
	{ d_keyboard_proc,  0,  0,  0,  0,  0,   0,   C('l'), 0,      0,        0,   list_editer,      NULL, NULL },
	{ d_keyboard_proc,  0,  0,  0,  0,  0,   0,   C('x'), 0,      0,        0,   cutter,           NULL, NULL },
	{ d_keyboard_proc,  0,  0,  0,  0,  0,   0,   C('c'), 0,      0,        0,   copier,           NULL, NULL },
	{ d_keyboard_proc,  0,  0,  0,  0,  0,   0,   C('v'), 0,      0,        0,   paster,           NULL, NULL },
	{ d_keyboard_proc,  0,  0,  0,  0,  0,   0,   C('n'), 0,      0,        0,   newer,            NULL, NULL },
	{ d_keyboard_proc,  0,  0,  0,  0,  0,   0,   C('o'), 0,      0,        0,   opener,           NULL, NULL },
	{ d_keyboard_proc,  0,  0,  0,  0,  0,   0,   C('s'), 0,      0,        0,   saver,            NULL, NULL },
	{ d_keyboard_proc,  0,  0,  0,  0,  0,   0,   0,      0,      KEY_DEL,  0,   deleter,          NULL, NULL },
	{ d_keyboard_proc,  0,  0,  0,  0,  0,   0,   0,      0,      KEY_PGUP, 0,   shift_sel_up,     NULL, NULL },
	{ d_keyboard_proc,  0,  0,  0,  0,  0,   0,   0,      0,      KEY_PGDN, 0,   shift_sel_down,   NULL, NULL },
	{ d_keyboard_proc,  0,  0,  0,  0,  0,   0,   0,      0,      KEY_HOME, 0,   shift_sel_home,   NULL, NULL },
	{ d_keyboard_proc,  0,  0,  0,  0,  0,   0,   0,      0,      KEY_END,  0,   shift_sel_end,    NULL, NULL },
	{ d_keyboard_proc,  0,  0,  0,  0,  0,   0,   0,      0,      KEY_ESC,  0,   closer,           NULL, NULL },
	{ d_yield_proc,     0,  0,  0,  0,  0,   0,   0,      0,      0,        0,   NULL,             NULL, NULL },
	{ 0 }
};

static const char resize_pointer_data1[] =
	"......    "
	".xxx.     "
	".xx.      "
	".x.x.     "
	".. .x.   ."
	".   .x. .."
	"     .x.x."
	"      .xx."
	"     .xxx."
	"    ......";

static const char resize_pointer_data2[] =
	"   ..    ..   "
	"  .x.    .x.  "
	" .xx......xx. "
	".xxxxxxxxxxxx."
	" .xx......xx. "
	"  .x.    .x.  "
	"   ..    ..   ";

static const char default_pointer_data[] =
	"xx        "
	"x.x       "
	"x..x      "
	"x...x     "
	"x....x    "
	"x.....x   "
	"x......x  "
	"x.......x "
	"x........x"
	"x.....xxx "
	"x..x..x   "
	"x.x x..x  "
	" x  x..x  "
	"     x..x "
	"     x..x "
	"      xx  ";









/******************************************/
/********* Some dialog procedures *********/
/******************************************/



/* let the user drag a rectangle on the screen (the mouse button must already
 * be pressed), returning it thru the pointers. You can safely assume (x1,y1)
 * is the upper-left corner and (x2,y2) the bottom-right one.
 * Grid snaping is ignored while ALT is pressed.
 */
void drag_rectangle(int *x1, int *y1, int *x2, int *y2, int snap)
{
	int ox, oy, x, y, back_grid = grid;

	if (!snap || (key_shifts & KB_ALT_FLAG))
		grid = 0;

	ox = x = snapx(mouse_x);
	oy = y = snapy(mouse_y);

	set_mouse_sprite(default_pointer);
	xor_mode(TRUE);
	SCARED( rect(screen, ox, oy, x, y, XOR_COLOR) );
	while (mouse_b) {
		grid = (!snap || (key_shifts & KB_ALT_FLAG)) ? 0 : back_grid;

		if (snapx(mouse_x) != x || snapy(mouse_y) != y) {
			scare_mouse();
			rect(screen, ox, oy, x, y, XOR_COLOR);
			x = snapx(mouse_x);
			y = snapy(mouse_y);
			rect(screen, ox, oy, x, y, XOR_COLOR);
			unscare_mouse();
		}
		broadcast_dialog_message(MSG_IDLE, 0);
	}
	SCARED( rect(screen, ox, oy, x, y, XOR_COLOR) );
	xor_mode(FALSE);
	grid = back_grid;

	*x1 = MIN(ox, x);
	*y1 = MIN(oy, y);
	*x2 = MAX(ox, x);
	*y2 = MAX(oy, y);
}



/* helpers for deciding if an object should be selected. Useful because CTRL
 * adds object to the selection, while SHIFT removes them.
 */
void selection_in(int i)
{
	selected[i] = !(key_shifts & KB_SHIFT_FLAG);
}

void selection_out(int i)
{
	if (!(key_shifts & KB_CTRL_FLAG || key_shifts & KB_SHIFT_FLAG))
		selected[i] = FALSE;
}



/* select a particular object */
void selection(int i)
{
	int c;

	for (c=0; c<tdlg_size; c++) {
		if(c == i)
			selection_in(c);
		else
			selection_out(c);
	}

	/* I known i should have used a D_REDRAW where appropriate instead
	 * of playing with flags, but it somehow messed up double clicks...
	 */
	editor[0].flags |= D_DIRTY;
	editor[1].flags |= D_DIRTY;
}



/* let the user drag a selection rectangle with the mouse and select/unselect
 * objects accordingly (CTRL to add objects to an existing selection, SHIFT
 * to remove them).
 */
int drag_select(void)
{
	int x1, y1, x2, y2, i;

	drag_rectangle(&x1, &y1, &x2, &y2, FALSE);

	for (i=0; i<tdlg_size; i++) {
		if ((x1 <= tdialog[i].x && x2 >= tdialog[i].x+tdialog[i].w-1 &&
		     y1 <= tdialog[i].y && y2 >= tdialog[i].y+tdialog[i].h-1))
			selection_in(i);
		else
			selection_out(i);
	}

	return D_REDRAW;
}



/* let the user drag the shape of a new object, and then create it */
int drag_new_object(void)
{
	int x1, y1, x2, y2;
	drag_rectangle(&x1, &y1, &x2, &y2, TRUE);

	if (x1!=x2 && y1!=y2) {
		TDIALOG *td = new_tdialog();

		td->sx = malloc(NCHAR);
		td->sy = malloc(NCHAR);
		td->sw = malloc(NCHAR);
		td->sh = malloc(NCHAR);

		usprintf(td->sx, "%d", x1);
		usprintf(td->sy, "%d", y1);
		usprintf(td->sw, "%d", x2-x1);
		usprintf(td->sh, "%d", y2-y1);

		td->sproc  = get_copy(new_object->sproc );
		td->sfg    = get_copy(new_object->sfg   );
		td->sbg    = get_copy(new_object->sbg   );
		td->skey   = get_copy(new_object->skey  );
		td->sflags = get_copy(new_object->sflags);
		td->sd1    = get_copy(new_object->sd1   );
		td->sd2    = get_copy(new_object->sd2   );
		td->sdp    = get_copy(new_object->sdp   );
		td->sdp2   = get_copy(new_object->sdp2  );
		td->sdp3   = get_copy(new_object->sdp3  );

		td->d = malloc(sizeof(DIALOG));
		update_tdialog(td);

		selection(tdlg_size-1);
	}

	return D_REDRAW;
}



/* xor-draw rectangles around selected objects */
void draw_selection_rectangles(void)
{
	int i;
	scare_mouse();
	xor_mode(TRUE);

	for (i=0; i<tdlg_size; i++)
		if (selected[i])
			rect(
				screen, tdialog[i].x, tdialog[i].y,
				tdialog[i].x + tdialog[i].w-1, tdialog[i].y + tdialog[i].h-1,
				XOR_COLOR
			);

	xor_mode(FALSE);
	unscare_mouse();
}



/* move selected objects by draging the mouse. td is the object that was
 * clicked, and thus will be snaped to the grid, the others keeping their
 * position relative to it.
 */
int drag_move(TDIALOG *td)
{
	int x, y, ox, oy, i, dx, dy, back_grid = grid;

	x = ox = mouse_x;
	y = oy = mouse_y;

	/* If the user release the mouse button without moving, just select the object */
	while (!(ox-mouse_x || oy-mouse_y)) {
		if (!mouse_b) {
			selection(td-tdialog);
			return D_O_K;
		}
	}

	/* Move selection rectangles */
	set_mouse_sprite(default_pointer);
	draw_selection_rectangles();
	while (mouse_b) {
		grid = (key_shifts & KB_ALT_FLAG) ? 0 : back_grid;

		if (snapx(mouse_x) != x || snapy(mouse_y) != y) {
			x = snapx(mouse_x);
			y = snapy(mouse_y);
			dx = snapx(td->x + x-ox) - td->x;
			dy = snapy(td->y + y-oy) - td->y;

			draw_selection_rectangles();
			for (i=0; i<tdlg_size; i++) {
				if (selected[i]) {
					tdialog[i].x += dx;
					tdialog[i].y += dy;
				}
			}
			draw_selection_rectangles();

			ox = x;
			oy = y;
		}
		broadcast_dialog_message(MSG_IDLE, 0);
	}
	draw_selection_rectangles();
	grid = back_grid;

	/* Then update the sx and sy parameters of the objects (if needed) */
	if (td->d->x != td->x || td->d->y != td->y) {
		for (i=0; i<tdlg_size; i++) {
			if (selected[i]) {
				usprintf(tdialog[i].sx, "%d", tdialog[i].x);
				usprintf(tdialog[i].sy, "%d", tdialog[i].y);
				update_tdialog(&tdialog[i]);
			}
		}
	}

	return D_REDRAW;
}



/* resize a single object by dragging the mouse */
int drag_resize(TDIALOG *td)
{
	int x, y, ox, oy, dx, dy, sx, sy, back_grid = grid;

	x = ox = mouse_x;
	y = oy = mouse_y;

	/* If the user release the mouse button without moving, just select the object */
	while (!(ox-mouse_x || oy-mouse_y)) {
		if (!mouse_b) {
			selection(td-tdialog);
			return D_O_K;
		}
	}

	/* Resize the object */
	sx = sy = 0;
	if (ox <= td->x+3)
		sx = -1;
	else if (ox >= td->x+td->w-3)
		sx = 1;

	if (oy <= td->y+3)
		sy = -1;
	else if (oy >= td->y+td->h-3)
		sy = 1;

	draw_selection_rectangles();
	while (mouse_b) {
		grid = (key_shifts & KB_ALT_FLAG) ? 0 : back_grid;

		if (snapx(mouse_x) != x || snapy(mouse_y) != y) {
			x = snapx( mouse_x );
			y = snapy( mouse_y );

			if (sx<0)      dx = td->x - snapx(td->x + x-ox);
			else if (sx>0) dx = snapx(td->x+td->w + x-ox) - (td->x+td->w);
			else           dx = 0;

			if (sy<0)      dy = td->y - snapy(td->y + y-oy);
			else if (sy>0) dy = snapy(td->y+td->h + y-oy) - (td->y+td->h);
			else           dy = 0;

			draw_selection_rectangles();

			if (td->w+dx < 8) dx = 0;
			if (td->h+dy < 8) dy = 0;
			if (sx<0)         td->x -= dx;
			if (sy<0)         td->y -= dy;
			td->w += dx;
			td->h += dy;

			draw_selection_rectangles();

			ox = x;
			oy = y;
		}
		broadcast_dialog_message(MSG_IDLE, 0);
	}
	draw_selection_rectangles();
	grid = back_grid;

	/* And update the size and pos of the object (if needed) */
	usprintf(td->sx, "%d", td->x);
	usprintf(td->sy, "%d", td->y);
	usprintf(td->sw, "%d", td->w);
	usprintf(td->sh, "%d", td->h);
	update_tdialog(td);

	return D_REDRAW;
}



/* find the object under the mouse */
int find_mouse_object(void)
{
	int i, mx, my;
	mx = mouse_x;
	my = mouse_y;

	for (i=tdlg_size-1; i>=0; i--)
		if ( mx >= tdialog[i].x && mx < tdialog[i].x + tdialog[i].w &&
		     my >= tdialog[i].y && my < tdialog[i].y + tdialog[i].h )
			return i;

	return -1;
}



/* workspace_proc */
int workspace_proc(int msg, DIALOG *d, int c)
{
	int x, y, i, ret = D_O_K;
	BITMAP *bmp, *back_screen;

	switch (msg) {
	case MSG_START:

		d->w = SCREEN_W;
		d->h = SCREEN_H;
		d->dp = create_bitmap(d->w, d->h);
		for (i=0; i<tdlg_size; i++)
			update_tdialog(&tdialog[i]);
		break;

	case MSG_END:

		destroy_bitmap(d->dp);
		break;

	case MSG_CLICK:

		i = find_mouse_object();
		if (mouse_b & 1) {
			if (new_object) {
				ret |= drag_new_object();
				new_object = NULL;
			}
			else if (i>=0 && !(key_shifts & KB_ALT_FLAG)) {
				if(!selected[i])
					selection(i);

				if (mouse_x >= tdialog[i].x+3 && mouse_x < tdialog[i].x+tdialog[i].w-3
				&& mouse_y >= tdialog[i].y+3 && mouse_y < tdialog[i].y+tdialog[i].h-3 )
					ret |= drag_move(&tdialog[i]);
				else
					ret |= drag_resize(&tdialog[i]);
			}
			else {
				ret |= drag_select();
			}
		}
		else if (mouse_b & 2) {
			if (new_object) {
				new_object = NULL;
				while (mouse_b)
					broadcast_dialog_message(MSG_IDLE, 0);
			}
			else {
				set_mouse_sprite(default_pointer);
				if (i >= 0) {
					if (!selected[i])
						selection(i);
				}
				do_menu(menu_right_click, mouse_x, mouse_y);
				ret |= D_REDRAW;
			}
		}
		else if (mouse_b & 4) {
			set_mouse_sprite(default_pointer);
			do_menu(menu_new, mouse_x, mouse_y);
		}
		break;

	case MSG_DCLICK:

		set_mouse_sprite(default_pointer);
		while (mouse_b)
			broadcast_dialog_message(MSG_IDLE, 0);
		ret |= properties();
		break;

	case MSG_LOSTMOUSE:

		if (mouse_sprite != default_pointer)
			set_mouse_sprite(default_pointer);
		break;

	case MSG_DRAW:

		bmp = d->dp;
		clear_to_color(bmp, makecol(0, 0, 127));
		if (grid && !(menu_grid[5].flags & D_SELECTED)) {
			c = makecol(0, 0, 40);
			for (x=0; x<bmp->w; x += grid_w[grid])
				for (y=0; y<bmp->h; y += grid_h[grid])
					putpixel(bmp, x, y, c);
		}

		back_screen = screen;
		screen = bmp;
		set_dialog_gui_state();
		for (i=0; i<tdlg_size; i++)
			ret |= tdialog[i].d->proc(MSG_DRAW, tdialog[i].d, 0);
		set_editor_gui_state();
		screen = back_screen;

		for (i=0; i<tdlg_size; i++) {
			if (selected[i])
				dotted_rect(
					bmp, tdialog[i].x, tdialog[i].y,
					tdialog[i].x + tdialog[i].w-1, tdialog[i].y + tdialog[i].h-1,
					gui_text_color, gui_back_color
				);
		}

		blit(bmp, screen, 0, 0, 0, 0, bmp->w, bmp->h);
		break;
	}

	return ret;
}



/* an unknown proc. dp contains the name of the proc */
int d_unknown_proc(int msg, DIALOG *d, int c)
{
	if (msg == MSG_DRAW) {
		int ct, cb, cl, cr;
		FONT *f = datafile ? (FONT*)datafile[0].dat : font;
		ct = screen->ct;
		cb = screen->cb;
		cl = screen->cl;
		cr = screen->cr;
		rect(screen, d->x, d->y, d->x+d->w-1, d->y+d->h-1, d->fg);
		rectfill(screen, d->x+1, d->y+1, d->x+d->w-2, d->y+d->h-2, d->bg);
		text_mode(d->bg);
		set_clip(screen, d->x+1, d->y+1, d->x+d->w-2, d->y+d->h-2);
		textout(screen, f, d->dp, d->x+2, d->y+2, d->fg);
		set_clip(screen, cl, ct, cr, cb);
	}

	return D_O_K;
}



/* returns D_EXIT when the global close_requested is true */
int d_close_proc(int msg, DIALOG *d, int c)
{
	if (close_requested && msg == MSG_IDLE)
		return D_CLOSE;
	
	return D_O_K;
}








/******************************************/
/********** The menu functions ************/
/******************************************/


/* let the user test the dialog (can be dangerous, and still buggy) */
int tester(void)
{
	DIALOG *d;
	TDIALOG *td;
	DIALOG_PLAYER *player;
	int i;

	d = malloc(sizeof(DIALOG)*(tdlg_size+1));
	td = malloc(sizeof(TDIALOG)*tdlg_size);
	if (!d || !td) {
		free(d);
		free(td);
		return D_O_K;
	}

	for (i=0; i<tdlg_size; i++) {
		memcpy(&td[i], &tdialog[i], sizeof(TDIALOG));
		td[i].d = &d[i];
		td[i].flags = 0;
		update_tdialog(&td[i]);
		set_dialog_gui_state();
		object_message(td[i].d, MSG_END, 0);
	}
	d[tdlg_size].proc = NULL;
	SCARED( clear(screen) );
	player = init_dialog(d, -1);
	while (!key[KEY_ESC] && update_dialog(player)) {
	}
	shutdown_dialog(player);
	set_editor_gui_state();

	for (i=0; i<tdlg_size; i++) {
		if (td[i].flags & FREE_dp)  FREE(td[i].d->dp);
		if (td[i].flags & FREE_dp2) FREE(td[i].d->dp2);
		if (td[i].flags & FREE_dp3) FREE(td[i].d->dp3);
	}

	free(d);
	free(td);
	return D_REDRAW;
}



/* callback that the font of the textbox in the font viewer */
int source_font_changer(void)
{
	DIALOG *d = active_dialog;
	while (d->proc) {
		if (d->proc == xtextbox_proc) {
			if (d->dp2 == tfont)
				d->dp2 = font;
			else
				d->dp2 = tfont;
			object_message(d, MSG_DRAW, 0);
			return D_O_K;
		}
		d++;
	}
	return D_O_K; /* something went terribly wrong, but can we do ? :) */
}



/* display the c source of the dialog */
int source_viewer(void)
{
	char *s;
	DIALOG src_dlg[] =
	{
	   /* (proc)        (x) (y) (w)  (h)  (fg) (bg) (key) (flags) (d1) (d2) (dp)                 (dp2)                (dp3) */
	   { xbox_proc,     0,  0,  160, 112, 0,   0,   0,    0,      0,   0,   NULL,                NULL,                NULL },
	   { xtext_proc,    8,  8,  40,  8,   0,   0,   0,    0,      0,   0,   "File:",             NULL,                NULL },
	   { xtext_proc,    56, 8,  96,  8,   0,   0,   0,    0,      0,   0,   NULL,                NULL,                NULL },
	   { xtextbox_proc, 8,  40, 144, 40,  0,   0,   0,    0,      0,   0,   NULL,                NULL,                NULL },
	   { xbutton_proc,  56, 88, 48,  16,  0,   0,   13,   D_EXIT, 0,   0,   "OK",                NULL,                NULL },
	   { xcheck_proc,   8,  24, 144, 8,   0,   0,   'u',  D_EXIT, 0,   0,   "&Use default font", source_font_changer, NULL },
	   { NULL,          0,  0,  0,   0,   0,   0,   0,    0,      0,   0,   NULL,                NULL,                NULL }
	};

	s = make_source(tdialog, tdlg_size, save_name, path_is_cpp(save_path));
	if (!s) {
		alert("Woops...", " Not enough memory to make the source... ", NULL, " OK ", NULL, 13, 0);
		return D_O_K;
	}

	if (!datafile)
		src_dlg[5].flags |= D_SELECTED | D_DISABLED;

	src_dlg[0].w   = SCREEN_W;
	src_dlg[0].h   = SCREEN_H;
	src_dlg[1].dp2 = tfont;
	src_dlg[2].dp  = save_path;
	src_dlg[3].w   = SCREEN_W-16;
	src_dlg[3].h   = SCREEN_H - (src_dlg[3].y + 32);
	src_dlg[3].dp  = s;
	src_dlg[4].x   = (SCREEN_W - src_dlg[4].w)/2;
	src_dlg[4].y   = SCREEN_H-24;

	set_dialog_color(src_dlg, gui_text_color, gui_back_color);
	popup_dialog(src_dlg, 3);
	free(s);

	return D_O_K;
}



/* getter for the list of dialogs found in a source file */
char *dlg_getter(int index, int *size)
{
	if (index<0) {
		*size = (scan_info) ? scan_info->n : 0;
		return NULL;
	}
	return scan_info->name[index];
}



/* sets the path and name of the current dialog */
void set_dialog_save_info(const char *path, const char *name)
{
	char buf[256];

	if (path && ugetc(path))
		uszprintf(buf, 256, "DLG - %s - %s", get_filename(path), name);
	else
		uszprintf(buf, 256, "DLG - %s", name);
	set_window_title(buf);

	ustrcpy(save_path, path);
	ustrcpy(save_name, name);
}



/* load a dialog from a file. if no name is given, first ask for one */
void load_dialog(char *path, char *name)
{
	FILE *f = NULL;
	TDIALOG *td;
	char buff[1024];
	int i, res;

	DIALOG open_dlg[] =
	{
	/*	(dialog proc)   (x)   (y)   (w)   (h)   (fg) (bg) (key) (flags)  (d1) (d2) (dp)                         (dp2) (dp3) */
		{ xbox_proc,    0,    0,    320,  200,  0,   0,   0,    0,       0,   0,   NULL,                        NULL, NULL },
		{ xtext_proc,   160,  8,    0,    0,    0,   0,   0,    0,       1,   0,   "Select the dialog to load", NULL, NULL },
		{ xlist_proc,   8,    24,   304,  120,  0,   0,   0,    D_EXIT,  0,   0,   dlg_getter,                  NULL, NULL },
		{ xcheck_proc,  8,    152,  304,  8,    0,   0,   'c',  0,       1,   0,   "&Centre dialog on screen",  NULL, NULL },
		{ xbutton_proc, 100,  176,  40,   16,   0,   0,   13,   D_EXIT,  0,   0,   "OK",                        NULL, NULL },
		{ xbutton_proc, 148,  176,  72,   16,   0,   0,   27,   D_EXIT,  0,   0,   "Cancel",                    NULL, NULL },
		{ 0 }
	};

	open_dlg[1].dp2 = tfont;

	/* open and scan the file */
	if (!exists(path)) {
		alert(path, NULL, " File not found ", " &Ok ", NULL, 'o', 0);
		goto finished;
	}
	scan_info = fscan_for_dialogs(path);

	/* check if dialogs were found */
	if (!scan_info || scan_info->n == 0) {
		alert(path, NULL, " Could not find any dialog in the file! ", " &Ok ", NULL, 'o', 0);
		goto finished;
	}

	/* find/ask which dialog to load */
	if (name) {
		for (i=0; i<scan_info->n; i++) {
			ustrcpy(buff, scan_info->name[i]);
			ustrtok(buff, " [");
			if (ustrcmp(buff, name) == 0)
				break;
		}
		if (i >= scan_info->n) {
			alert(path, name, " Could not find the dialog in the file! ", " &Ok ", NULL, 'o', 0);
			goto finished;
		}
	}
	else if(scan_info->n > 1) {
		centre_dialog(open_dlg);
		set_dialog_color(open_dlg, gui_text_color, gui_back_color);
		res = popup_dialog(open_dlg, 2);
		if (res == 2 || res == 4)
			i = open_dlg[2].d1;
		else
			goto finished;
	}
	else {
		i = 0;
	}

	/* load the dialog */
	newer();
	f = fopen(path, "rb");
	fseek(f, scan_info->offset[i], SEEK_SET);
	fcheck(f, "*{");
	do {
		td = new_tdialog();
		if (fread_dialog_struct(f, td) == -1)
			delete_tdialog(tdlg_size-1);
	}
	while (fcheck(f, " } , "));
	fclose(f);

	if (tdlg_size == 0)
		alert(scan_info->name[i], NULL, " Nothing could be read from the dialog ", " &Ok ", NULL, 'o', 0);
	else {
		int x1, y1, x2, y2;

		set_dialog_save_info(path, scan_info->name[i]);
		save_offset = scan_info->offset[i];
		for (i=0; i<tdlg_size; i++)
			update_tdialog(&tdialog[i]);

		tdialog_get_bounds(tdialog, tdlg_size, &x1, &y1, &x2, &y2);
		if (x2 > SCREEN_W || y2 > SCREEN_H) {
			/* the dialog is bigger than the current screen resolution */
			char *message1 = "The dialog is bigger than the screen!";
			char *message2 = "The following resolution would probably be better:";
			int card = GFX_AUTODETECT;
			int bpp = bitmap_color_depth(screen);
			int w, h, bx, by, bw, bh, len;
			int resolutions[][2] = {
				{512,384}, {640,400}, {640,480}, {800,600},
				{1024,768}, {1280,1024}, {1600,1200}, {0,0}
			};

			/* let's try to find the smallest big enough resolution */
			i = 0;
			while (resolutions[i][0] && (resolutions[i][0] < x2 || resolutions[i][1] < y2))
				i++;
			w = resolutions[i][0];
			h = resolutions[i][1];

			len = MAX(text_length(font, message1), text_length(font, message2));
			bx = (SCREEN_W - (24+len)) / 2;
			by = (64 * SCREEN_H) / 480;
			bw = 24 + len;
			bh = 24 + 4 + 2*text_height(font);
			gui_rect(screen, bx, by, bw, bh, 0);
			rectfill(screen, bx+1, by+1, bx+bw-2, by+bh-2, gui_back_color);
			text_mode(gui_back_color);
			textout_centre(screen, font, message1, SCREEN_W/2, by+12, gui_text_color);
			textout_centre(screen, font, message2, SCREEN_W/2, by+12+text_height(font)+4, gui_text_color);

			if (gfx_mode_select_ex(&card, &w, &h, &bpp))
				change_gfx_mode(card, w, h, bpp);
		}

		if (open_dlg[3].flags & D_SELECTED)
			position_tdialog(-1, -1);
	}

finished:
	destroy_scan_info(scan_info);
	scan_info = NULL;
}



/* open a file */
int opener(void)
{
	int i;
	char path[1024];

	ustrcpy(path, save_path);

	if (!file_select_ex("Select a source file (c, cc, cpp, h)", path, "c;cc;cpp;cxx;h;hpp", 1024, 310, 200))
		return D_O_K;

	load_dialog(path, NULL);

	for (i=0; i<tdlg_size; i++)
		update_tdialog(&tdialog[i]);

	return D_REDRAW;
}



/* 'save as' comand */
int save_as(void)
{
	DIALOG_PLAYER *player;
	char path[1024];
	static char name[NCHAR];
	int i;

	DIALOG save_dlg[] =
	{
	   /* (proc)       (x)  (y)  (w)  (h)  (fg) (bg) (key) (flags) (d1) (d2) (dp)                        (dp2) (dp3) */
	   { xbox_proc,    0,   0,   224, 240, 0,   0,   0,    0,      0,   0,   NULL,                       NULL, NULL },
	   { xtext_proc,   112, 8,   8,   8,   0,   0,   0,    0,      1,   0,   "Select a name",            NULL, NULL },
	   { xedit_proc,   8,   32,  208, 15,  0,   0,   0,    0,      128, 0,   name,                       NULL, NULL },
	   { xlist_proc,   8,   48,  208, 96,  0,   0,   0,    D_EXIT, 0,   0,   dlg_getter,                 NULL, NULL },
	   { xcheck_proc,  8,   152, 208, 8,   0,   0,   0,    0,      1,   0,   "Position dialog on (0,0)", NULL, NULL },
	   { xcheck_proc,  8,   168, 208, 8,   0,   0,   0,    0,      1,   0,   "Backup",                   NULL, NULL },
	   { xcheck_proc,  8,   184, 208, 8,   0,   0,   0,    0,      1,   0,   "C++ friendly output",      NULL, NULL },
	   { xbutton_proc, 48,  216, 48,  16,  0,   0,   13,   D_EXIT, 0,   0,   "OK",                       NULL, NULL },
	   { xbutton_proc, 104, 216, 80,  16,  0,   0,   27,   D_EXIT, 0,   0,   "Cancel",                   NULL, NULL },
	   { NULL,         0,   0,   0,   0,   0,   0,   0,    0,      0,   0,   NULL,                       NULL, NULL }
	};

	save_dlg[1].dp2 = tfont;
	if (get_config_int("[dlg]", "do_backups", 0))
		save_dlg[5].flags |= D_SELECTED;

	ustrcpy(path, save_path);
	if (file_select_ex("Save as...", path, "c;cc;cpp;cxx;h;hpp", 1024, 310, 200)) {
		if (is_directory(path)) {
			alert(path, NULL, "is a directory", " &Ok ", NULL, 'o', 0);
			return D_O_K;
		}
		if (exists(path)) {
			scan_info = fscan_for_dialogs(path);
			if (!scan_info) {
				alert(path, NULL, " Error while reading the file ", " &Ok ", NULL, 'o', 0);
				return D_O_K;
			}
		}

		/* check wether the file is a cpp source file. An *.h can be either C
		 * or C++, so we use the value from the configuration file.
		 */
		if (path_is_cpp(path))
			save_dlg[6].flags |= D_SELECTED;
		else
			save_dlg[6].flags &= ~D_SELECTED;

		/* do the dialog */
		ustrcpy(name, save_name);
		set_dialog_color(save_dlg, gui_text_color, gui_back_color);
		centre_dialog(save_dlg);
		player = init_dialog(save_dlg, -1);

		i = save_dlg[3].d1;
		while (update_dialog(player)) {
			if (i != save_dlg[3].d1) {
				i = save_dlg[3].d1;
				ustrcpy(name, dlg_getter(i, NULL));
				object_message(&save_dlg[2], MSG_DRAW, 0);
			}
		}
		i = shutdown_dialog(player);

		/* handle the result */
		if (i==3 || i==7) {
			if (save_dlg[4].flags & D_SELECTED)
				position_tdialog(0, 0);

			set_dialog_save_info(path, name);

			save_offset = 0;
			if (scan_info && scan_info->n) {
				if (ustrcmp(name, dlg_getter(save_dlg[3].d1, NULL)) == 0)
					save_offset = scan_info->offset[save_dlg[3].d1];
				else {
					for (i=0; i<scan_info->n; i++)
						if (ustrcmp(name, scan_info->name[i]) == 0)
							save_offset = scan_info->offset[i];
				}
			}
			write_dialog(
				tdialog, tdlg_size,
				save_path, save_name, save_offset,
				save_dlg[5].flags & D_SELECTED,
				save_dlg[6].flags & D_SELECTED
			);
		}
		destroy_scan_info(scan_info);
		scan_info = NULL;
	}

	return D_REDRAW;
}



/* 'save' command */
int saver(void)
{
	if (ustrlen(save_path) == 0 || ustrlen(save_name) == 0)
		return save_as();

	write_dialog(
		tdialog, tdlg_size,
		save_path, save_name, save_offset,
		get_config_int("[dlg]", "do_backups", 0),
		path_is_cpp(save_path)
	);
	return D_O_K;
}



/* start a new dialog from scratch */
int newer(void)
{
	while (tdlg_size)
		delete_tdialog(0);

	set_dialog_save_info("", "dlg[] ");

	return D_REDRAW;
}



/* delete (but don't put in the clipboard) the selection */
int deleter(void)
{
	int i;
	for (i=0; i<tdlg_size; i++)
		if (selected[i])
			delete_tdialog(i--);

	return D_REDRAW;
}



/* copy the selection into the clipboard */
int copier(void)
{
	int i, n=0;

	for (i=0; i<tdlg_size; i++)
		if (selected[i])
			n++;

	for (i=0; i<clipboard_size; i++)
		destroy_tdialog(&clipboard[i]);
	clipboard_size = n;
	clipboard = realloc(clipboard, sizeof(TDIALOG)*n);

	n = 0;
	for (i=0; i<tdlg_size; i++)
		if (selected[i])
			copy_tdialog(&clipboard[n++], &tdialog[i]);

	return D_O_K;
}



/* cut the selection to the clipboard */
int cutter(void)
{
	copier();
	deleter();
	return D_REDRAW;
}



/* past the clipboard */
int paster(void)
{
	int i;
	TDIALOG *td;

	if (clipboard) {
		memset(selected, 0, sizeof(char)*tdlg_size);

		for (i=0; i<clipboard_size; i++) {
			clipboard[i].x = (clipboard[i].x+grid_w[grid])%SCREEN_W;
			clipboard[i].y = (clipboard[i].y+grid_h[grid])%SCREEN_H;

			td = new_tdialog();
			copy_tdialog(td, &clipboard[i]);

			selected[tdlg_size-1] = TRUE;
			usprintf(td->sx, "%d", clipboard[i].x);
			usprintf(td->sy, "%d", clipboard[i].y);
			update_tdialog(td);
		}
	}

	return D_REDRAW;
}



/* (MENU proc) set the "create new object on next click" state... */
int object_newer(void)
{
	int i;
	for (i=0; proc_list[i].proc; i++) {
		if (active_menu->dp == proc_list[i].proc) {
			new_object = &proc_list[i];
			return D_O_K;
		}
	}

	return D_O_K;
}



/* (MENU proc) select the current proc set */
int set_selecter(void)
{
	set_menu_new(active_menu->dp);
	return D_O_K;
}



/* obvious enough */
int select_all(void)
{
	memset(selected, 1, tdlg_size);
	return D_REDRAW;
}



/* obvious enough */
int select_none(void)
{
	memset(selected, 0, tdlg_size);
	return D_REDRAW;
}



/* obvious enough */
int invert_selection(void)
{
	int i;
	for (i=0; i<tdlg_size; i++)
		selected[i] = !selected[i];

	return D_REDRAW;
}



/* move the selected items up in the tdialog array */
int shift_sel_up(void)
{
	int i;
	TDIALOG swap;

	for (i=1; i<tdlg_size; i++) {
		if(selected[i]) {
			memcpy(&swap, tdialog+i, sizeof(TDIALOG));
			memcpy(tdialog+i, tdialog+i-1, sizeof(TDIALOG));
			memcpy(tdialog+i-1, &swap, sizeof(TDIALOG));
			selected[i] = selected[i-1];
			selected[i-1] = TRUE;
		}
	}

	return D_REDRAW;
}



/* move the selected items down in the tdialog array */
int shift_sel_down(void)
{
	int i;
	TDIALOG swap;

	for (i=tdlg_size-2; i>=0; i--) {
		if (selected[i]) {
			memcpy(&swap, tdialog+i, sizeof(TDIALOG));
			memcpy(tdialog+i, tdialog+i+1, sizeof(TDIALOG));
			memcpy(tdialog+i+1, &swap, sizeof(TDIALOG));
			selected[i] = selected[i+1];
			selected[i+1] = TRUE;
		}
	}

	return D_REDRAW;
}



/* move the selected items to the beginning of the tdialog array */
int shift_sel_home(void)
{
	int i, j;

	for (i=0; i<tdlg_size; i++)
		if (selected[i])
			break;

	if (i >= tdlg_size)
		return D_O_K;

	for (j=0; j<i; j++)
		shift_sel_up();

	return D_REDRAW;
}



/* move the selected items to the end of the tdialog array */
int shift_sel_end(void)
{
	int i, j;

	for (i=0; i<tdlg_size; i++)
		if (selected[tdlg_size-1 - i])
			break;

	if (i >= tdlg_size)
		return D_O_K;

	for (j=0; j<i; j++)
		shift_sel_down();

	return D_REDRAW;
}



/* getter for the list of the objects of the dialog
 *  this formats a string with the sproc and the id_params
 */
char *select_getter(int index, int *size)
{
	static int pw;
	static char str[NCHAR];
	char *istr;
	int i, n;

	if (index<0) {
		pw = 0;
		for (i=0; proc_list[i].proc; i++) {
			if (text_length(font, proc_list[i].sproc) > pw)
				pw = text_length(font, proc_list[i].sproc);
		}

		*size = tdlg_size;
		return NULL;
	}

	for (i=0; proc_list[i].proc; i++) {
		if (ustrcmp(proc_list[i].sproc, tdialog[index].sproc)==0) {
			n = (pw - text_length(font, proc_list[i].sproc)) / text_length(font, " ");
			istr = str;
			istr += usprintf(istr, "%-*s", ustrlen(proc_list[i].sproc)+n, proc_list[i].sproc);

			if (proc_list[i].id_param & POS)
				istr += usprintf(istr, " (%s, %s),", tdialog[index].sx, tdialog[index].sy);

			if (proc_list[i].id_param & KEY)
				istr += usprintf(istr, " %s,", tdialog[index].skey);

			if (proc_list[i].id_param & D1)
				istr += usprintf(istr, " %s,", tdialog[index].sd1);

			if (proc_list[i].id_param & D2)
				istr += usprintf(istr, " %s,", tdialog[index].sd2);

			if (proc_list[i].id_param & DP)
				istr += usprintf(istr, " %s,", tdialog[index].sdp);

			if (proc_list[i].id_param & DP2)
				istr += usprintf(istr, " %s,", tdialog[index].sdp2);

			if (proc_list[i].id_param & DP3)
				istr += usprintf(istr, " %s,", tdialog[index].sdp3);

			usetat(str, -1, 0);
			return str;
		}
	}

	usprintf(str, "%s", tdialog[index].sproc);
	return str;
}



/* edit the list of tdialog objects */
int list_editer(void)
{
	int res;
	DIALOG_PLAYER *pl;

	DIALOG sel_dlg[] =
	{
	/*	(dialog proc)      (x)  (y)  (w)  (h)  (fg) (bg) (key) (flags)  (d1)     (d2) (dp)           (dp2)             (dp3) */
		{ xbox_proc,       0,   0,   320, 240, 0,   0,   0,    0,       0,       0,   NULL,          NULL,             NULL },
		{ xtext_proc,      160, 8,   0,   0,   0,   0,   0,    0,       1,       0,   "List Edit",   NULL,             NULL },
		{ xlist_proc,      8,   24,  304, 164, 0,   0,   0,    D_EXIT,  0,       0,   select_getter, NULL,             NULL },
		{ xbutton_proc,    8,   196, 40,  16,  0,   0,   'a',  D_EXIT,  0,       0,   "&All",        select_all,       NULL },
		{ xbutton_proc,    56,  196, 48,  16,  0,   0,   'n',  D_EXIT,  0,       0,   "&None",       select_none,      NULL },
		{ xbutton_proc,    112, 196, 64,  16,  0,   0,   'i',  D_EXIT,  0,       0,   "&Invert",     invert_selection, NULL },
		{ xbutton_proc,    228, 196, 48,  16,  0,   0,   'h',  D_EXIT,  0,       0,   "&Home",       shift_sel_home,   NULL },
		{ xbutton_proc,    228, 216, 48,  16,  0,   0,   'e',  D_EXIT,  0,       0,   "&End",        shift_sel_end,    NULL },
		{ xbutton_proc,    280, 196, 32,  16,  0,   0,   'u',  D_EXIT,  0,       0,   "/\\",         shift_sel_up,     NULL },
		{ xbutton_proc,    280, 216, 32,  16,  0,   0,   'd',  D_EXIT,  0,       0,   "\\/",         shift_sel_down,   NULL },
		{ d_keyboard_proc, 0,   0,   0,   0,   0,   0,   0,    0,       KEY_DEL, 0,   deleter,       NULL,             NULL },
		{ xbutton_proc,    8,   216, 41,  16,  0,   0,   13,   D_EXIT,  0,       0,   "OK",          NULL,             NULL },
		{ 0 }
	};

	sel_dlg[1].dp2 = tfont;
	sel_dlg[2].dp2 = selected;
	centre_dialog(sel_dlg);
	set_dialog_color(sel_dlg, gui_text_color, gui_back_color);
	pl = init_dialog(sel_dlg, 1);

	do {
		res = update_dialog(pl);
		if (res == FALSE && pl->obj == 2) {
			memset(selected, 0, sizeof(char)*tdlg_size);
			selected[sel_dlg[2].d1] = TRUE;
			pl->res |= properties();
			res = TRUE;
		}
	}
	while(res);

	shutdown_dialog(pl);
	return D_REDRAW;
}



/* snap the objects of the selection to the grid */
int snaper(void)
{
	int i;
	for (i=0; i<tdlg_size; i++) {
		if (selected[i]) {
			usprintf(tdialog[i].sx, "%d", snapx(tdialog[i].x));
			usprintf(tdialog[i].sy, "%d", snapy(tdialog[i].y));

			if (tdialog[i].d->w && tdialog[i].d->h) {
				usprintf(tdialog[i].sw, "%d", snapx(tdialog[i].w));
				usprintf(tdialog[i].sh, "%d", snapy(tdialog[i].h));
			}

			update_tdialog(&tdialog[i]);
		}
	}
	return D_REDRAW;
}



/* a simple 'about' box */
int about(void)
{
	alert(
		" Allegro DIALOG editor v" DLG_VERSION_STR " (" ALLEGRO_PLATFORM_STR "), Julien Cugniere (" DLG_DATE_STR ") ",
		NULL,
		"Report bugs to <uos@free.fr>",
		" &Ok ", NULL, 'o', 0
	);
	return D_O_K;
}



/* 'close' menu function */
int closer(void)
{
	close_requested = TRUE;
	return D_O_K;
}



/* move the menu bar around so that it doesn't get in the way */
int mover(void)
{
	if (editor[1].y == 1)
		editor[1].y = SCREEN_H/2;
	else
		editor[1].y = 1;

	return D_REDRAW;
}



/* checks/unchecks the Grid->Hide menu item */
int grid_hider(void)
{
	active_menu->flags ^= D_SELECTED;
	return D_REDRAW;
}



/* select the 'Grid*' menu item that was clicked, and unselect the others */
int grid_checker(void)
{
	menu_grid[grid].flags = 0;
	active_menu->flags = D_SELECTED;
	grid = active_menu - menu_grid;

	if (grid == 0)
		find_menu(menu_grid, NULL, snaper, NULL, NULL)->flags |= D_DISABLED;
	else
		find_menu(menu_grid, NULL, snaper, NULL, NULL)->flags &= ~D_DISABLED;

	return D_REDRAW;
}



/* pop up a dialog to edit the different grid sizes */
int grid_resizer(void)
{
	int i;
	static char w[3][20], h[3][20];

	DIALOG grid_dlg[] =
	{
	/*	(proc)          (x)  (y) (w)  (h)  (fg) (bg) (key) (flags) (d1) (d2) (dp)           (dp2) (dp3) */
		{ xbox_proc,    0,   0,  208, 116, 0,   0,   0,    0,      0,   0,   NULL,          NULL, NULL },
		{ xtext_proc,   104, 8,  0,   0,   0,   0,   0,    0,      1,   0,   "Grids sizes", NULL, NULL },
		{ xtext_proc,   8,   48, 48,  8,   0,   0,   0,    0,      0,   0,   "Width",       NULL, NULL },
		{ xtext_proc,   8,   68, 48,  8,   0,   0,   0,    0,      0,   0,   "Height",      NULL, NULL },
		{ xtext_proc,   60,  28, 40,  8,   0,   0,   0,    0,      0,   0,   "Grid1",       NULL, NULL },
		{ xedit_proc,   60,  44, 40,  16,  0,   0,   0,    0,      4,   0,   w[0],          NULL, NULL },
		{ xedit_proc,   60,  64, 40,  16,  0,   0,   0,    0,      4,   0,   h[0],          NULL, NULL },
		{ xtext_proc,   108, 28, 40,  8,   0,   0,   0,    0,      0,   0,   "Grid2",       NULL, NULL },
		{ xedit_proc,   108, 44, 40,  16,  0,   0,   0,    0,      4,   0,   w[1],          NULL, NULL },
		{ xedit_proc,   108, 64, 40,  16,  0,   0,   0,    0,      4,   0,   h[1],          NULL, NULL },
		{ xtext_proc,   156, 28, 40,  8,   0,   0,   0,    0,      0,   0,   "Grid3",       NULL, NULL },
		{ xedit_proc,   156, 44, 40,  16,  0,   0,   0,    0,      4,   0,   w[2],          NULL, NULL },
		{ xedit_proc,   156, 64, 40,  16,  0,   0,   0,    0,      4,   0,   h[2],          NULL, NULL },
		{ xbutton_proc, 44,  92, 48,  16,  0,   0,   13,   D_EXIT, 0,   0,   "OK",          NULL, NULL },
		{ xbutton_proc, 100, 92, 64,  16,  0,   0,   27,   D_EXIT, 0,   0,   "Cancel",      NULL, NULL },
		{ 0 }
	};

	grid_dlg[1].dp2 = tfont;

	/* write the values in the d_edit_proc's */
	for (i=0; i<3; i++) {
		usprintf(w[i], "%d", grid_w[i+1]);
		usprintf(h[i], "%d", grid_h[i+1]);
	}

	/* do the dialog */
	set_dialog_color(grid_dlg, gui_text_color, gui_back_color);
	centre_dialog(grid_dlg);
	i = popup_dialog(grid_dlg, 4);

	/* read the modified values back, if 'Ok' was clicked */
	if (i == 13) {
		for(i=1; i<4; i++) {
			grid_w[i] = MAX(1, ustrtol(w[i-1], NULL, 0));
			grid_h[i] = MAX(1, ustrtol(h[i-1], NULL, 0));
			usprintf(grid_str[i-1], "Grid&%d %dx%d", i, grid_w[i], grid_h[i]);
		}
	}

	return D_REDRAW;
}



/* procedure selecter for the 'object properties' dialog */
int proc_selecter(DIALOG *d)
{
	new_object = NULL;
	do_menu(menu_all_procs, d->x, d->y + d->h + 1);

	if (new_object) {
		ustrcpy(d->dp, new_object->sproc);
		new_object = NULL;
		return D_REDRAWME;
	}

	return D_O_K;
}



/* color selecter for the 'object properties' dialog */
int color_selecter(DIALOG *d)
{
	int c, res, r, g ,b;
	static char red[20], green[20], blue[20], color[50];
	BITMAP *bkg;
	DIALOG_PLAYER *dlg_player;

	DIALOG pal_dlg[] =
	{
	   /* (proc)         (x)  (y)  (w)  (h)  (fg) (bg) (key) (flags) (d1) (d2) (dp)              (dp2) (dp3) */
	   { xbox_proc,      0,   0,   280, 144, 0,   0,   0,    0,      0,   0,   NULL,             NULL, NULL },
	   { xpalette_proc,  143, 7,   130, 130, 0,   0,   0,    0,      -1,  0,   NULL,             NULL, NULL },
	   { xtext_proc,     8,   8,   120, 8,   0,   0,   0,    0,      0,   0,   "Select a color", NULL, NULL },
	   { xtext_proc,     56,  32,  40,  8,   0,   0,   0,    0,      0,   0,   "Red",            NULL, NULL },
	   { xedit_proc,     104, 30,  32,  12,  0,   0,   0,    0,      3,   0,   red,              NULL, NULL },
	   { xtext_proc,     56,  48,  40,  8,   0,   0,   0,    0,      0,   0,   "Green",          NULL, NULL },
	   { xedit_proc,     104, 46,  32,  12,  0,   0,   0,    0,      3,   0,   green,            NULL, NULL },
	   { xtext_proc,     56,  64,  40,  8,   0,   0,   0,    0,      0,   0,   "Blue",           NULL, NULL },
	   { xedit_proc,     104, 62,  32,  12,  0,   0,   0,    0,      3,   0,   blue,             NULL, NULL },
	   { xtext_proc,     8,   80,  120, 8,   0,   0,   0,    0,      0,   0,   "Color number:",  NULL, NULL },
	   { xedit_proc,     8,   94,  128, 12,  0,   0,   0,    0,      10,  0,   color,            NULL, NULL },
	   { xcolorbox_proc, 8,   32,  40,  40,  0,   0,   0,    0,      0,   0,   NULL,             NULL, NULL },
	   { xbutton_proc,   8,   120, 56,  16,  0,   0,   13,   D_EXIT, 0,   0,   "OK",             NULL, NULL },
	   { xbutton_proc,   72,  120, 64,  16,  0,   0,   27,   D_EXIT, 0,   0,   "Cancel",         NULL, NULL },
	   { 0 }
	};

	/* Initialize the dialog */
	pal_dlg[2].dp2 = tfont;
	set_dialog_color(pal_dlg, gui_text_color, gui_back_color);
	centre_dialog(pal_dlg);
	dlg_player = init_dialog(pal_dlg, 10);

	bkg = create_bitmap(pal_dlg[0].w+1, pal_dlg[0].h+1);
	SCARED( blit(screen, bkg, pal_dlg[0].x, pal_dlg[0].y, 0, 0, pal_dlg[0].w+1, pal_dlg[0].h+1) );

	c = ustrtol(d->dp, NULL, 0);
	usprintf(red,   "%d", getr(c));
	usprintf(green, "%d", getg(c));
	usprintf(blue,  "%d", getb(c));
	usprintf(color, "%d", c);
	pal_dlg[11].bg = c;
	pal_dlg[1].d1 = makecol8(getr(c), getg(c), getb(c));

	/* Do the dialog */
	do {
		res = update_dialog(dlg_player);
		if (dlg_player->res) {
			switch (dlg_player->focus_obj) {
			case 1: /* the palette */
				c = palette_color[pal_dlg[1].d1];
				usprintf(red,   "%d", getr(c));
				usprintf(green, "%d", getg(c));
				usprintf(blue,  "%d", getb(c));
				usprintf(color, "%d", c);
				break;

			case 4:
			case 6: /* the red, green, or blue edit */
			case 8:
				r = ustrtol(red,   NULL, 0);
				g = ustrtol(green, NULL, 0);
				b = ustrtol(blue,  NULL, 0);
				c = makecol(r, g, b);
				usprintf(color, "%d", c);
				break;

			case 10: /* the color number */
				c = ustrtol(color, NULL, 0);
				if (_color_depth == 8 && (c>255 || c<0))
					c = 0;
				usprintf(red,   "%d", getr(c));
				usprintf(green, "%d", getg(c));
				usprintf(blue,  "%d", getb(c));
				break;
			}

			pal_dlg[11].bg = c;
			pal_dlg[ 4].flags |= D_DIRTY;
			pal_dlg[ 6].flags |= D_DIRTY;
			pal_dlg[ 8].flags |= D_DIRTY;
			pal_dlg[10].flags |= D_DIRTY;
			pal_dlg[11].flags |= D_DIRTY;
		}
	}
	while (res);

	res = shutdown_dialog(dlg_player);

	SCARED( blit(bkg, screen, 0, 0, pal_dlg[0].x, pal_dlg[0].y, pal_dlg[0].w+1, pal_dlg[0].h+1) );
	destroy_bitmap(bkg);

	/* Get data back from the dialog */
	if (res == 12 || res == 1)
		ustrcpy(d->dp, color);

	return D_REDRAWME;
}



/* a flags selecter */
int flags_selecter(DIALOG *d)
{
	int i, flags;
	static char user[60];

	DIALOG flag_dlg[] =
	{
	   /* (dialog proc) (x)  (y)  (w)  (h)  (fg) (bg) (key) (flags) (d1) (d2) (dp)                  (dp2) (dp3) */
	   { xbox_proc,     0,   0,   168, 196, 0,   0,   0,    0,      0,   0,   NULL,                 NULL, NULL },
	   { xtext_proc,    84,  8,   20,  8,   0,   0,   0,    0,      1,   0,   "Select the flags",   NULL, NULL },
	   { xcheck_proc,   8,   32,  96,  8,   0,   0,   0,    0,      1,   0,   "D_EXIT",             NULL, NULL },
	   { xcheck_proc,   8,   44,  96,  8,   0,   0,   0,    0,      1,   0,   "D_SELECTED",         NULL, NULL },
	   { xcheck_proc,   8,   56,  96,  8,   0,   0,   0,    0,      1,   0,   "D_GOTFOCUS",         NULL, NULL },
	   { xcheck_proc,   8,   68,  96,  8,   0,   0,   0,    0,      1,   0,   "D_GOTMOUSE",         NULL, NULL },
	   { xcheck_proc,   8,   80,  96,  8,   0,   0,   0,    0,      1,   0,   "D_HIDDEN",           NULL, NULL },
	   { xcheck_proc,   8,   92,  96,  8,   0,   0,   0,    0,      1,   0,   "D_DISABLED",         NULL, NULL },
	   { xcheck_proc,   8,   104, 96,  8,   0,   0,   0,    0,      1,   0,   "D_DIRTY",            NULL, NULL },
	   { xcheck_proc,   8,   116, 96,  8,   0,   0,   0,    0,      1,   0,   "D_INTERNAL",         NULL, NULL },
	   { xtext_proc,    8,   132, 48,  8,   0,   0,   0,    0,      0,   0,   "D_USER",             NULL, NULL },
	   { xedit_proc,    72,  130, 88,  12,  0,   0,   0,    0,      13,  0,   user,                 NULL, NULL },
	   { xcheck_proc,   8,   156, 152, 8,   0,   0,   0,    0,      1,   0,   "Output as a number", NULL, NULL },
	   { xbutton_proc,  8,   172, 56,  16,  0,   0,   13,   D_EXIT, 0,   0,   "OK",                 NULL, NULL },
	   { xbutton_proc,  72,  172, 88,  16,  0,   0,   27,   D_EXIT, 0,   0,   "Cancel",             NULL, NULL },
	   { 0 }
	};

	/* write data to the dialog */
	flag_dlg[1].dp2 = tfont;
	flags = get_flags(d->dp);
	for (i=0; i<8; i++) {
		if (flags & (1<<i))
			flag_dlg[2+i].flags |= D_SELECTED;
		else
			flag_dlg[2+i].flags &= ~D_SELECTED;

		flags &= ~(1<<i);
	}
	usprintf(user, "%d", flags);

	/* do the dialog */
	set_dialog_color(flag_dlg, gui_text_color, gui_back_color);
	centre_dialog(flag_dlg);
	i = popup_dialog(flag_dlg, -1);

	/* read data back */
	if (i == 13) {
		flags = 0;
		for (i=0; i<8; i++) {
			if (flag_dlg[2+i].flags & D_SELECTED)
			flags |= (1<<i);
		}
		flags |= ustrtol(user, NULL, 0);

		if (flag_dlg[12].flags & D_SELECTED)
			usprintf(d->dp, "%d", flags);
		else
			make_flags(d->dp, flags);
	}

	return D_REDRAWME;
}



/* pops up a dialog to edit the properties of a DIALOG object */
int properties(void)
{
	int i, res;
	int maxc = NCHAR/uwidth_max(U_CURRENT) -1;

	static char sproc[NCHAR];
	static char sx[NCHAR], sy[NCHAR], sw[NCHAR], sh[NCHAR];
	static char sfg[NCHAR], sbg[NCHAR], skey[NCHAR], sflags[NCHAR];
	static char sd1[NCHAR], sd2[NCHAR];
	static char sdp[NCHAR], sdp2[NCHAR], sdp3[NCHAR];

	DIALOG prop_dlg[] =
	{
	/* (dialog proc)   (x)  (y)  (w)  (h)  (fg) (bg) (key) (flags) (d1) (d2) (dp)                 (dp2) (dp3) */
	   { xbox_proc,    0,   0,   288, 336, 0,   0,   0,    0,      0,   0,   NULL,                NULL, NULL },
	   { xtext_proc,   144, 8,   8,   8,   0,   0,   0,    0,      1,   0,   "Object Properties", NULL, NULL },

	   { xtext_proc,   8,   32,  56,  8,   0,   0,   0,    0,      0,   0,   "proc",   NULL,           NULL },
	   { xedit_proc,   68,  28,  212, 15,  0,   0,   0,    0,      0,   0,   sproc,    proc_selecter,  NULL },
	   { xtext_proc,   8,   52,  56,  8,   0,   0,   0,    0,      0,   0,   "x",      NULL,           NULL },
	   { xedit_proc,   68,  48,  212, 15,  0,   0,   0,    0,      0,   0,   sx,       NULL,           NULL },
	   { xtext_proc,   8,   72,  56,  8,   0,   0,   0,    0,      0,   0,   "y",      NULL,           NULL },
	   { xedit_proc,   68,  68,  212, 15,  0,   0,   0,    0,      0,   0,   sy,       NULL,           NULL },
	   { xtext_proc,   8,   92,  56,  8,   0,   0,   0,    0,      0,   0,   "w",      NULL,           NULL },
	   { xedit_proc,   68,  88,  212, 15,  0,   0,   0,    0,      0,   0,   sw,       NULL,           NULL },
	   { xtext_proc,   8,   112, 56,  8,   0,   0,   0,    0,      0,   0,   "h",      NULL,           NULL },
	   { xedit_proc,   68,  108, 212, 15,  0,   0,   0,    0,      0,   0,   sh,       NULL,           NULL },
	   { xtext_proc,   8,   132, 56,  8,   0,   0,   0,    0,      0,   0,   "fg",     NULL,           NULL },
	   { xedit_proc,   68,  128, 212, 15,  0,   0,   0,    0,      0,   0,   sfg,      color_selecter, NULL },
	   { xtext_proc,   8,   152, 56,  8,   0,   0,   0,    0,      0,   0,   "bg",     NULL,           NULL },
	   { xedit_proc,   68,  148, 212, 15,  0,   0,   0,    0,      0,   0,   sbg,      color_selecter, NULL },
	   { xtext_proc,   8,   172, 56,  8,   0,   0,   0,    0,      0,   0,   "key",    NULL,           NULL },
	   { xedit_proc,   68,  168, 212, 15,  0,   0,   0,    0,      0,   0,   skey,     NULL,           NULL },
	   { xtext_proc,   8,   192, 56,  8,   0,   0,   0,    0,      0,   0,   "flags",  NULL,           NULL },
	   { xedit_proc,   68,  188, 212, 15,  0,   0,   0,    0,      0,   0,   sflags,   flags_selecter, NULL },
	   { xtext_proc,   8,   212, 56,  8,   0,   0,   0,    0,      0,   0,   "d1",     NULL,           NULL },
	   { xedit_proc,   68,  208, 212, 15,  0,   0,   0,    0,      0,   0,   sd1,      NULL,           NULL },
	   { xtext_proc,   8,   232, 56,  8,   0,   0,   0,    0,      0,   0,   "d2",     NULL,           NULL },
	   { xedit_proc,   68,  228, 212, 15,  0,   0,   0,    0,      0,   0,   sd2,      NULL,           NULL },
	   { xtext_proc,   8,   252, 56,  8,   0,   0,   0,    0,      0,   0,   "dp",     NULL,           NULL },
	   { xedit_proc,   68,  248, 212, 15,  0,   0,   0,    0,      0,   0,   sdp,      NULL,           NULL },
	   { xtext_proc,   8,   272, 56,  8,   0,   0,   0,    0,      0,   0,   "dp2",    NULL,           NULL },
	   { xedit_proc,   68,  268, 212, 15,  0,   0,   0,    0,      0,   0,   sdp2,     NULL,           NULL },
	   { xtext_proc,   8,   292, 56,  8,   0,   0,   0,    0,      0,   0,   "dp3",    NULL,           NULL },
	   { xedit_proc,   68,  288, 212, 15,  0,   0,   0,    0,      0,   0,   sdp3,     NULL,           NULL },

	   { xbutton_proc, 116, 312, 60,  16,  0,   0,   13,   D_EXIT, 0,   0,   "OK",     NULL, NULL },
	   { xbutton_proc, 184, 312, 96,  16,  0,   0,   27,   D_EXIT, 0,   0,   "Cancel", NULL, NULL },
	   { 0 }
	};

	prop_dlg[1].dp2 = tfont;
	for (i=3; i<30; i+=2)
		prop_dlg[i].d1 = maxc;

	/* Find the selected object */
	for (i=tdlg_size-1; i>=0; i--)
		if (selected[i])
			break;
	if (i < 0)
		return D_O_K;

	/* Fill the dialog with the object data */
	ustrcpy(sproc,  tdialog[i].sproc );
	ustrcpy(sx,     tdialog[i].sx    );
	ustrcpy(sy,     tdialog[i].sy    );
	ustrcpy(sw,     tdialog[i].sw    );
	ustrcpy(sh,     tdialog[i].sh    );
	ustrcpy(sfg,    tdialog[i].sfg   );
	ustrcpy(sbg,    tdialog[i].sbg   );
	ustrcpy(skey,   tdialog[i].skey  );
	ustrcpy(sflags, tdialog[i].sflags);
	ustrcpy(sd1,    tdialog[i].sd1   );
	ustrcpy(sd2,    tdialog[i].sd2   );
	ustrcpy(sdp,    tdialog[i].sdp   );
	ustrcpy(sdp2,   tdialog[i].sdp2  );
	ustrcpy(sdp3,   tdialog[i].sdp3  );

	/* do the dialog */
	set_dialog_color(prop_dlg, gui_text_color, gui_back_color);
	centre_dialog(prop_dlg);
	res = popup_dialog(prop_dlg, 3);

	/* Get data back */
	if (res == 30) {
		ustrcpy(tdialog[i].sproc,  sproc );
		ustrcpy(tdialog[i].sx,     sx    );
		ustrcpy(tdialog[i].sy,     sy    );
		ustrcpy(tdialog[i].sw,     sw    );
		ustrcpy(tdialog[i].sh,     sh    );
		ustrcpy(tdialog[i].sfg,    sfg   );
		ustrcpy(tdialog[i].sbg,    sbg   );
		ustrcpy(tdialog[i].skey,   skey  );
		ustrcpy(tdialog[i].sflags, sflags);
		ustrcpy(tdialog[i].sd1,    sd1   );
		ustrcpy(tdialog[i].sd2,    sd2   );
		ustrcpy(tdialog[i].sdp,    sdp   );
		ustrcpy(tdialog[i].sdp2,   sdp2  );
		ustrcpy(tdialog[i].sdp3,   sdp3  );
		update_tdialog(&tdialog[i]);
	}

	return D_REDRAW;
}



/* changes the GUI global settings */
int gui_setter(void)
{
	static char fg[30], mg[30], bg[30], bl[30];
	int i;

	DIALOG gui_dlg[] =
	{
	   /* (proc)       (x)  (y)  (w)  (h)  (fg) (bg) (key) (flags) (d1) (d2) (dp)                   (dp2)           (dp3) */
	   { xbox_proc,    0,   0,   248, 176, 0,   0,   0,    0,      0,   0,   NULL,                  NULL,           NULL },
	   { xtext_proc,   124, 12,  80,  8,   0,   0,   0,    0,      1,   0,   "GUI settings",        NULL,           NULL },
	   { xtext_proc,   8,   40,  144, 8,   0,   0,   0,    0,      0,   0,   "Foreground color",    NULL,           NULL },
	   { xedit_proc,   160, 36,  80,  16,  0,   0,   0,    0,      9,   0,   fg,                    color_selecter, NULL },
	   { xtext_proc,   8,   60,  144, 8,   0,   0,   0,    0,      0,   0,   "Middleground color",  NULL,           NULL },
	   { xedit_proc,   160, 56,  80,  16,  0,   0,   0,    0,      9,   0,   mg,                    color_selecter, NULL },
	   { xtext_proc,   8,   80,  144, 8,   0,   0,   0,    0,      0,   0,   "Background color",    NULL,           NULL },
	   { xedit_proc,   160, 76,  80,  16,  0,   0,   0,    0,      9,   0,   bg,                    color_selecter, NULL },
	   { xtext_proc,   8,   100, 144, 8,   0,   0,   0,    0,      0,   0,   "Font baseline",       NULL,           NULL },
	   { xedit_proc,   160, 96,  80,  16,  0,   0,   0,    0,      9,   0,   bl,                    NULL,           NULL },
	   { xcheck_proc,  8,   120, 164, 8,   0,   0,   0,    0,      1,   0,   "Focus follows mouse", NULL,           NULL },
	   { xbutton_proc, 60,  152, 48,  16,  0,   0,   13,   D_EXIT, 0,   0,   "OK",                  NULL,           NULL },
	   { xbutton_proc, 112, 152, 72,  16,  0,   0,   27,   D_EXIT, 0,   0,   "Cancel",              NULL,           NULL },
	   { 0 }
	};

	gui_dlg[1].dp2 = tfont;

	/* Write data in the dialog */
	usprintf(fg, "%d", back_fg);
	usprintf(mg, "%d", back_mg);
	usprintf(bg, "%d", back_bg);
	usprintf(bl, "%d", back_baseline);

	if (back_mouse_focus)
		gui_dlg[10].flags = D_SELECTED;
	else
		gui_dlg[10].flags = 0;

	/* Do the dialog */
	set_dialog_color(gui_dlg, gui_text_color, gui_back_color);
	centre_dialog(gui_dlg);
	i = popup_dialog(gui_dlg, 3);

	/* Read data back */
	if (i == 11) {
		back_fg  = ustrtol(fg, NULL, 0);
		back_mg  = ustrtol(mg, NULL, 0);
		back_bg  = ustrtol(bg, NULL, 0);
		back_baseline = ustrtol(bl, NULL, 0);
		back_mouse_focus = gui_dlg[10].flags & D_SELECTED;

		for (i=0; i<tdlg_size; i++) {
			if (!get_number(tdialog[i].sfg) && !get_number(tdialog[i].sbg)) {
				tdialog[i].d->fg = back_fg;
				tdialog[i].d->bg = back_bg;
			}
		}
	}

	return D_REDRAW;
}



/* loads a new palette */
int palette_loader(void)
{
	char path[1024];
	int i;
	ustrcpy(path, save_path);
	if (file_select_ex("Load a palette from an image", path, "bmp;pcx;tga;lbm", 1024, 310, 200)) {
		PALETTE pal;
		BITMAP *bmp = load_bitmap(path, pal);
		if (bmp) {
			destroy_bitmap(bmp);
			back_fg = convcol_to(24, back_fg);
			back_mg = convcol_to(24, back_mg);
			back_bg = convcol_to(24, back_bg);
			set_palette(pal);
			back_fg = convcol_from(24, back_fg);
			back_mg = convcol_from(24, back_mg);
			back_bg = convcol_from(24, back_bg);
			set_pointers();
			for (i=0; i<tdlg_size; i++)
				update_tdialog(&tdialog[i]);
			return D_REDRAW;
		}
	}
	return D_O_K;
}



/* changes the graphic mode */
int graphic_mode_setter(void)
{
	int card = GFX_AUTODETECT;
	int w = SCREEN_W;
	int h = SCREEN_H;
	int bpp = bitmap_color_depth(screen);

	if (gfx_mode_select_ex(&card, &w, &h, &bpp)) {
		change_gfx_mode(card, w, h, bpp);
		return D_REDRAW;
	}
	return D_O_K;
}



/* set the colors of the selection */
int color_setter(void)
{
	int i, res, cfg, cbg;
	static char fg[30], bg[30];

	DIALOG col_dlg[] =
	{
	   /* (proc)       (x)  (y) (w)  (h)  (fg) (bg) (key) (flags) (d1) (d2) (dp)                (dp2)           (dp3) */
	   { xbox_proc,    0,   0,  232, 112, 16,  15,  0,    0,      0,   0,   NULL,               NULL,           NULL },
	   { xtext_proc,   116, 12, 60,  8,   16,  15,  0,    0,      1,   0,   "Set colors",       NULL,           NULL },
	   { xtext_proc,   8,   40, 120, 8,   16,  15,  0,    0,      0,   0,   "Foreground color", NULL,           NULL },
	   { xedit_proc,   144, 36, 80,  16,  16,  15,  0,    0,      9,   0,   fg,                 color_selecter, NULL },
	   { xtext_proc,   8,   60, 120, 8,   16,  15,  0,    0,      0,   0,   "Background color", NULL,           NULL },
	   { xedit_proc,   144, 56, 80,  16,  16,  15,  0,    0,      9,   0,   bg,                 color_selecter, NULL },
	   { xbutton_proc, 64,  88, 40,  16,  16,  15,  13,   D_EXIT, 0,   0,   "OK",               NULL,           NULL },
	   { xbutton_proc, 108, 88, 62,  16,  16,  15,  27,   D_EXIT, 0,   0,   "Cancel",           NULL,           NULL },
	   { 0 }
	};

	col_dlg[1].dp2 = tfont;

	/* Write data in the dialog */
	usprintf(fg, "%d", back_fg);
	usprintf(bg, "%d", back_bg);

	/* Do the dialog */
	set_dialog_color(col_dlg, gui_text_color, gui_back_color);
	centre_dialog(col_dlg);
	res = popup_dialog(col_dlg, -1);

	/* Read data back */
	if (res == 6) {
		cfg  = ustrtol(fg, NULL, 0);
		cbg  = ustrtol(bg, NULL, 0);

		for (i=0; i<tdlg_size; i++) {
			if (selected[i]) {
				usprintf(tdialog[i].sfg, "%d", cfg);
				usprintf(tdialog[i].sbg, "%d", cbg);
				tdialog[i].d->fg = cfg;
				tdialog[i].d->bg = cbg;
			}
		}
	}

	return D_REDRAW;
}



int preferences(void)
{
	int i;

	DIALOG pref_dlg[] =
	{
	   /* (proc)       (x)  (y)  (w)  (h)  (fg) (bg) (key) (flags) (d1) (d2) (dp)                         (dp2) (dp3) */
	   { xbox_proc,    0,   0,   224, 156, 0,   0,   0,    0,      0,   0,   NULL,                        NULL, NULL },
	   { xtext_proc,   8,   12,  208, 8,   0,   0,   0,    0,      2,   0,   "Preferences",               NULL, NULL },
	   { xcheck_proc,  8,   40,  144, 8,   0,   0,   'u',  0,      0,   0,   "&Use default font",         NULL, NULL },
	   { xcheck_proc,  8,   60,  208, 8,   0,   0,   'm',  0,      0,   0,   "&Make backups when saving", NULL, NULL },
	   { xcheck_proc,  8,   80,  208, 8,   0,   0,   'o',  0,      0,   0,   "&Output C++ when in doubt", NULL, NULL },
	   { xbutton_proc, 48,  132, 48,  16,  0,   0,   13,   D_EXIT, 0,   0,   "OK",                        NULL, NULL },
	   { xbutton_proc, 104, 132, 72,  16,  0,   0,   27,   D_EXIT, 0,   0,   "Cancel",                    NULL, NULL },
	   { xcheck_proc,  8,   100, 124, 8,   0,   0,   's',  0,      0,   0,   "&Start windowed",           NULL, NULL },
	   { NULL,         0,   0,   0,   0,   0,   0,   0,    0,      0,   0,   NULL,                        NULL, NULL }
	};

	/* fill the dialog */
	pref_dlg[1].dp2 = tfont;
	if (get_config_int("[dlg]", "use_default_font", 0))
		pref_dlg[2].flags |= D_SELECTED;
	if (get_config_int("[dlg]", "do_backups", 0))
		pref_dlg[3].flags |= D_SELECTED;
	if (get_config_int("[dlg]", "cpp_output", 0))
		pref_dlg[4].flags |= D_SELECTED;
	if (get_config_int("[dlg]", "windowed", 0))
		pref_dlg[7].flags |= D_SELECTED;

	if (!datafile)
		pref_dlg[2].flags = D_DISABLED | D_SELECTED;

	/* do the dialog */
	centre_dialog(pref_dlg);
	i = do_dialog(pref_dlg, -1);

	/* get values back */
	if (i == 5) {
		set_config_int("[dlg]", "use_default_font", (pref_dlg[2].flags & D_SELECTED) ? 1 : 0);
		set_config_int("[dlg]", "do_backups", (pref_dlg[3].flags & D_SELECTED) ? 1 : 0);
		set_config_int("[dlg]", "cpp_output", (pref_dlg[4].flags & D_SELECTED) ? 1 : 0);
		set_config_int("[dlg]", "windowed", (pref_dlg[7].flags & D_SELECTED) ? 1 : 0);

		if (datafile && !get_config_int("[dlg]", "use_default_font", 0))
			font = datafile[0].dat;
		else
			font = &_default_font;

		/* make the menubar resize itself */
		object_message(&editor[1], MSG_END, 0);
		editor[1].w = 0;
		object_message(&editor[1], MSG_START, 0);
	}

	return D_REDRAW;
}







/******************************************/
/******* The main editor functions ********/
/******************************************/



/* set the gui_* variables to the editor settings */
static int editor_gui_state = FALSE;
void set_editor_gui_state(void)
{
	if (!editor_gui_state) {
		editor_gui_state = TRUE;

		/* save the current state of Allegro's GUI */
		back_font                    = font;
		back_fg                      = gui_fg_color;
		back_mg                      = gui_mg_color;
		back_bg                      = gui_bg_color;
		back_baseline                = gui_font_baseline;
		back_mouse_focus             = gui_mouse_focus;
		back_gui_shadow_box_proc     = gui_shadow_box_proc;
		back_gui_ctext_proc          = gui_ctext_proc;
		back_gui_button_proc         = gui_button_proc;
		back_gui_edit_proc           = gui_edit_proc;
		back_gui_list_proc           = gui_list_proc;
		back_gui_text_list_proc      = gui_text_list_proc;
		back_gui_menu_draw_menu      = gui_menu_draw_menu;
		back_gui_menu_draw_menu_item = gui_menu_draw_menu_item;

		/* set the state of the editor */
		if (datafile && !get_config_int("[dlg]", "use_default_font", 0))
			font = datafile[0].dat;
		else
			font = &_default_font;

		xset_gui_colors();
		gui_fg_color            = makecol(0, 0, 0);
		gui_mg_color            = makecol(127, 127, 127);
		gui_bg_color            = makecol(255, 255, 255);
		gui_font_baseline       = 0;
		gui_mouse_focus         = 1;
		gui_shadow_box_proc     = xbox_proc;
		gui_ctext_proc          = xctext_proc;
		gui_button_proc         = xbutton_proc;
		gui_edit_proc           = xedit_proc;
		gui_list_proc           = xlist_proc;
		gui_text_list_proc      = xtext_list_proc;
		gui_menu_draw_menu      = xdraw_menu;
		gui_menu_draw_menu_item = xdraw_menu_item;
	}
}



/* restore the gui_* variables to their former value (before the editor was
 * started). Used before displaying the edited dialog for example.
 */
void set_dialog_gui_state(void)
{
	if (editor_gui_state) {
		editor_gui_state = FALSE;
		font                    = back_font;
		gui_fg_color            = back_fg;
		gui_mg_color            = back_mg;
		gui_bg_color            = back_bg;
		gui_font_baseline       = back_baseline;
		gui_mouse_focus         = back_mouse_focus;
		gui_shadow_box_proc     = back_gui_shadow_box_proc;
		gui_ctext_proc          = back_gui_ctext_proc;
		gui_button_proc         = back_gui_button_proc;
		gui_edit_proc           = back_gui_edit_proc;
		gui_list_proc           = back_gui_list_proc;
		gui_text_list_proc      = back_gui_text_list_proc;
		gui_menu_draw_menu      = back_gui_menu_draw_menu;
		gui_menu_draw_menu_item = back_gui_menu_draw_menu_item;
	}
}



/* close hook */
void close_hook(void)
{
	close_requested = TRUE;
}



/* create a mouse bitmap from an array */
BITMAP *create_pointer(const char *data, int w, int h, int orientation)
{
	int x, y, c0, c1, c2, pix;
	BITMAP *b = create_bitmap(w ,h);

	c0 = bitmap_mask_color(screen);
	c1 = makecol(0, 0, 0);
	c2 = makecol(255, 255, 255);

	for (x=0; x<w; x++) {
		for (y=0; y<h; y++) {
			switch (orientation) {
			case 1:  pix = data[y + x*h];     break;
			case 2:  pix = data[w-x-1 + y*w]; break;
			default: pix = data[x + y*w];     break;
			}

			switch (pix) {
			case '.': putpixel(b ,x ,y, c2); break;
			case 'x': putpixel(b ,x ,y, c1); break;
			default:  putpixel(b ,x ,y, c0); break;
			}
		}
	}

	return b;
}



/* (re)creates the mouse pointers */
void set_pointers(void)
{
	destroy_bitmap(resize_pointer_d1);
	destroy_bitmap(resize_pointer_d2);
	destroy_bitmap(resize_pointer_h);
	destroy_bitmap(resize_pointer_v);

	resize_pointer_d1 = create_pointer(resize_pointer_data1, 10, 10, 0);
	resize_pointer_d2 = create_pointer(resize_pointer_data1, 10, 10, 2);
	resize_pointer_h  = create_pointer(resize_pointer_data2, 14, 7,  0);
	resize_pointer_v  = create_pointer(resize_pointer_data2, 7,  14, 1);
	default_pointer   = create_pointer(default_pointer_data, 10, 16, 0);

	set_mouse_sprite(default_pointer);
	show_mouse(screen);
}



/* register an array of PROC structures terminated by one with a null proc */
void register_proc_set(PROC *set, char *name)
{
	int n, setn, i;
	MENU *child;

	if (!set || !name)
		return;

	/* add the procs to proc_list */
	setn = 0;
	while (set[setn].proc)
		setn++;
	n = 0;
	if (proc_list)
		while (proc_list[n].proc)
			n++;
	proc_list = realloc(proc_list, sizeof(PROC)*(n+setn+1));
	memcpy(proc_list+n, set, sizeof(PROC)*(setn+1));

	/* creates a new menu with the procs */
	child = malloc(sizeof(MENU)*(setn+1));
	memset(child, 0, sizeof(MENU)*(setn+1));
	for (i=0; i<setn; i++) {
		child[i].text = set[i].sproc;
		child[i].proc = object_newer;
		child[i].dp   = set[i].proc;
	}

	/* insert this menu in menu_sets and menu_all_procs */
	i = get_menu_length(menu_sets);
	menu_sets[i].text = name;
	menu_sets[i].proc = set_selecter;
	menu_sets[i].dp   = child;

	i = get_menu_length(menu_all_procs);
	menu_all_procs[i].text  = name;
	menu_all_procs[i].child = child;
}



/* inserts a single menu structure into the editor */
void register_menu_hook(MENU *menu)
{
	if (menu) {
		int i = get_menu_length(menu_hooks);
		memcpy(&menu_hooks[i], menu, sizeof(MENU));
	}
}



/* registers a callback that will be called on graphic mode changes */
void register_reinit_callback(void (*callback)(void))
{
	if (callback) {
		int i = 0;
		int max = sizeof(reinit_callbacks) / sizeof(*reinit_callbacks);
		while (reinit_callbacks[i])
			i++;
		if (i < max-1)
			reinit_callbacks[i] = callback;
	}
}



/* registers a callback that will be called before the editor shuts down */
void register_shutdown_callback(void (*callback)(void))
{
	if (callback) {
		int i = 0;
		int max = sizeof(shutdown_callbacks) / sizeof(*shutdown_callbacks);
		while (shutdown_callbacks[i])
			i++;
		if (i < max-1)
			shutdown_callbacks[i] = callback;
	}
}



/* select the current proc set */
void set_menu_new(MENU *set)
{
	int i;
	MENU *m = find_menu(menu, "new", NULL, NULL, NULL);
	menu_new = set;
	m->child = set;
	menu_right_click[0].child = set;
	if (menu_new && menu_new[0].text) {
		m->flags &= ~D_DISABLED;
		menu_right_click[0].flags &= ~D_DISABLED;
	}
	else {
		m->flags |= D_DISABLED;
		menu_right_click[0].flags |= D_DISABLED;
	}

	for (i=0; menu_sets[i].text; i++) {
		if (menu_sets[i].dp == set)
			menu_sets[i].flags |= D_SELECTED;
		else
			menu_sets[i].flags &= ~D_SELECTED;
	}
}



/* editor initialization */
static char initialized = FALSE;
void init_editor(void)
{
	if (!initialized) {
		int i;
		MENU *m;

		initialized = TRUE;
		close_requested = FALSE;
		set_window_close_hook(close_hook);
		set_pointers();

		/* take care of some proc params that need run-time initialization */
		dummy_bmp = create_bitmap(SCREEN_W, SCREEN_H);
		clear_to_color(dummy_bmp, 0);
		for (i=0; i<dummy_bmp->w + dummy_bmp->h; i++)
			line(dummy_bmp, i, 0, 0, i, i%255);

		/* the font */
		datafile = load_datafile("font.dat");
		tfont = &_default_font;

		/* the grids */
		for (i=1; i<4; i++) {
			char name[20];
			sprintf(name, "grid%d", i);
			sscanf(get_config_string("[dlg]", name, ""), "%d x %d", &grid_w[i], &grid_h[i]);
		}
		grid = get_config_int("[dlg]", "grid", 1);

		menu_grid[grid].flags = D_SELECTED;
		for (i=1; i<4; i++)
			usprintf(grid_str[i-1], "Grid&%d %dx%d", i, grid_w[i], grid_h[i]);

		m = find_menu(menu_grid, NULL, grid_hider, NULL, NULL);
		if (get_config_int("[dlg]", "hide_grid", 0))
			m->flags |= D_SELECTED;
		else
			m->flags &= ~D_SELECTED;

		/* the plugins */
		proc_list = malloc(sizeof(PROC));
		memset(proc_list, 0, sizeof(PROC));
		set_dialog_gui_state();
		init_plugins();
		set_editor_gui_state();

		m = find_menu(menu_sets, "allegro", NULL, NULL, NULL);
		if (m)
			set_menu_new(m->dp);
		else
			set_menu_new(menu_sets[0].dp);

		m = find_menu(menu, NULL, NULL, menu_hooks, NULL);
		if (menu_hooks[0].text)
			m->flags &= ~D_DISABLED;
		else
			m->flags |= D_DISABLED;

		m = find_menu(menu, NULL, NULL, menu_sets, NULL);
		if (menu_sets[0].text)
			m->flags &= ~D_DISABLED;
		else
			m->flags |= D_DISABLED;
	}
}



/* editor cleanup */
void deinit_editor(void)
{
	int i;
	MENU *m;

	if (!initialized)
		return;
	initialized = FALSE;
	close_requested = FALSE;

	/* the grids */
	for (i=1; i<4; i++) {
		char name[20], value[100];
		sprintf(name, "grid%d", i);
		sprintf(value, "%dx%d", grid_w[i], grid_h[i]);
		set_config_string("[dlg]", name, value);
	}
	set_config_int("[dlg]", "grid", grid);
	
	m = find_menu(menu_grid, NULL, grid_hider, NULL, NULL);
	if (m)
		set_config_int("[dlg]", "hide_grid", (m->flags & D_SELECTED) ? 1 : 0);

	/* shuts down the plugins */
	for (i=0; shutdown_callbacks[i]; i++)
		shutdown_callbacks[i]();

	for (i=0; menu_all_procs[i].text; i++)
		FREE(menu_all_procs[i].child);

	tdlg_size = 0;
	clipboard_size = 0;
	FREE(tdialog);
	FREE(clipboard);
	FREE(proc_list);
	FREE(selected);
	destroy_bitmap(dummy_bmp);
	destroy_bitmap(resize_pointer_v);
	destroy_bitmap(resize_pointer_h);
	destroy_bitmap(resize_pointer_d1);
	destroy_bitmap(resize_pointer_d2);
	destroy_bitmap(default_pointer);
	unload_datafile(datafile);
}



/* changes the gfx mode, updating the editor as needed.
 * returns 0 on success.
 */
int change_gfx_mode(int card, int w, int h, int bpp)
{
	int old_w   = SCREEN_W;
	int old_h   = SCREEN_H;
	int old_bpp = bitmap_color_depth(screen);
	int i;

	if (w<512 || h<384) {
		alert(" The minimal resolution you can set is 512x384 ", NULL, NULL, " Ok ", NULL, 0, 0);
		return D_O_K;
	}

	set_color_depth(bpp);
	if (set_gfx_mode(card, w, h, 0, 0) != 0) {
		set_color_depth(old_bpp);
		set_gfx_mode(GFX_AUTODETECT, old_w, old_h, 0, 0);
		xset_gui_colors();
		show_mouse(screen);
		broadcast_dialog_message(MSG_DRAW, 0);
		alert(" Could not set the selected graphic mode: ", allegro_error, NULL, " Ok ", NULL, 0, 0);
		return -1;
	}
	back_fg = convcol(old_bpp, bpp, back_fg);
	back_mg = convcol(old_bpp, bpp, back_mg);
	back_bg = convcol(old_bpp, bpp, back_bg);
	xset_gui_colors();
	object_message(&editor[0], MSG_END, 0);

	set_dialog_gui_state();
	for (i=0; reinit_callbacks[i]; i++)
		reinit_callbacks[i]();
	set_editor_gui_state();

	object_message(&editor[0], MSG_START, 0);
	set_pointers();
	return 0;
}



/* set the position of all the objects of the tdialog array.
 * Pass (-1, -1) to center the dialog.
 */
void position_tdialog(int x, int y)
{
	int xc, yc, i;
	int x1, y1, x2 ,y2;

	/* how much to move by? */
	tdialog_get_bounds(tdialog, tdlg_size, &x1, &y1, &x2, &y2);
	if (x<0) {
		xc = (SCREEN_W - (x2 - x1)) / 2 - x1;
		yc = (SCREEN_H - (y2 - y1)) / 2 - y1;
	}
	else {
		xc = x - x1;
		yc = y - y1;
	}

	/* move the dialog */
	for (i=0; i<tdlg_size; i++) {
		tdialog[i].x += xc;
		tdialog[i].y += yc;
		usprintf(tdialog[i].sx, "%d", tdialog[i].x);
		usprintf(tdialog[i].sy, "%d", tdialog[i].y);
		update_tdialog(&tdialog[i]);
	}
}



/* reallocates the tdialog array, adding a new empty tdialog object */
TDIALOG *new_tdialog(void)
{
	TDIALOG *td;
	tdlg_size++;

	selected = realloc(selected, sizeof(char)*tdlg_size);
	selected[tdlg_size-1] = 0;

	tdialog = realloc(tdialog, sizeof(TDIALOG)*tdlg_size);

	td = tdialog + tdlg_size - 1;
	td->x = td->y = td->w = td->h = 0;
	td->flags  = 0;
	td->d      = NULL;
	td->sproc  = NULL;
	td->sx     = NULL;
	td->sy     = NULL;
	td->sw     = NULL;
	td->sh     = NULL;
	td->sfg    = NULL;
	td->sbg    = NULL;
	td->skey   = NULL;
	td->sflags = NULL;
	td->sd1    = NULL;
	td->sd2    = NULL;
	td->sdp    = NULL;
	td->sdp2   = NULL;
	td->sdp3   = NULL;

	return tdialog+tdlg_size-1;
}



/* releases the memory used by a tdialog */
void destroy_tdialog(TDIALOG *td)
{
	if (td) {
		if ((td->flags & GOT_MSG_START) && td->d)
			object_message(td->d, MSG_END, 0);
		if (td->flags & FREE_dp)  FREE(td->d->dp);
		if (td->flags & FREE_dp2) FREE(td->d->dp2);
		if (td->flags & FREE_dp3) FREE(td->d->dp3);
		FREE(td->d);
		FREE(td->sproc);
		FREE(td->sx);
		FREE(td->sy);
		FREE(td->sw);
		FREE(td->sh);
		FREE(td->sfg);
		FREE(td->sbg);
		FREE(td->skey);
		FREE(td->sflags);
		FREE(td->sd1);
		FREE(td->sd2);
		FREE(td->sdp);
		FREE(td->sdp2);
		FREE(td->sdp3);
	}
}



/* copy a tdialog structure */
void copy_tdialog(TDIALOG *dest, TDIALOG *src)
{
	dest->x = src->x;
	dest->y = src->y;
	dest->w = src->w;
	dest->h = src->h;
	dest->sproc  = get_copy(src->sproc );
	dest->sx     = get_copy(src->sx    );
	dest->sy     = get_copy(src->sy    );
	dest->sw     = get_copy(src->sw    );
	dest->sh     = get_copy(src->sh    );
	dest->sfg    = get_copy(src->sfg   );
	dest->sbg    = get_copy(src->sbg   );
	dest->skey   = get_copy(src->skey  );
	dest->sflags = get_copy(src->sflags);
	dest->sd1    = get_copy(src->sd1   );
	dest->sd2    = get_copy(src->sd2   );
	dest->sdp    = get_copy(src->sdp   );
	dest->sdp2   = get_copy(src->sdp2  );
	dest->sdp3   = get_copy(src->sdp3  );
	dest->d      = malloc(sizeof(DIALOG));
	dest->flags  = 0;
	update_tdialog(dest);
}



/* reallocate the tdialog array, deleting a tdialog object */
void delete_tdialog(int i)
{
	destroy_tdialog(&tdialog[i]);

	memmove(tdialog+i, tdialog+i+1, sizeof(TDIALOG)*(tdlg_size-i-1));
	memmove(selected+i, selected+i+1, sizeof(char)*(tdlg_size-i-1));

	tdlg_size--;
	selected = realloc(selected, sizeof(char)*tdlg_size);
	tdialog = realloc(tdialog, sizeof(TDIALOG)*tdlg_size);
}



/* update the contained DIALOG in a TDIALOG */
void update_tdialog(TDIALOG *td)
{
	int p;
	char str[NCHAR];

	/* let the DIALOG clean up its internals before we update it */
	if (td->flags & GOT_MSG_START) {
		set_dialog_gui_state();
		object_message(td->d, MSG_END, 0);
		set_editor_gui_state();
	}

	td->x = get_number(td->sx);
	td->y = get_number(td->sy);
	td->w = get_number(td->sw);
	td->h = get_number(td->sh);

	td->d->x     = td->x;
	td->d->y     = td->y;
	td->d->w     = td->w;
	td->d->h     = td->h;
	td->d->fg    = get_number(td->sfg);
	td->d->bg    = get_number(td->sbg);
	td->d->d1    = get_number(td->sd1);
	td->d->d2    = get_number(td->sd2);
	td->d->key   = get_key(td->skey);
	td->d->flags = get_flags(td->sflags);

	if (td->d->fg == 0 && td->d->bg == 0) {
		/* assume "set_dialog_color(dlg, gui_fg_color, gui_bg_color)" */
		td->d->fg = back_fg;
		td->d->bg = back_bg;
	}

	td->w = MAX(8, td->w);
	td->h = MAX(8, td->h);

	if (td->flags & FREE_dp)  FREE(td->d->dp);
	if (td->flags & FREE_dp2) FREE(td->d->dp2);
	if (td->flags & FREE_dp3) FREE(td->d->dp3);

	/* NOTE: it's quite unsafe to set a dp parameter to NULL, so here I try
	 *       some alternatives.
	 */
	#define UPDATE_DP(XX)               \
	{                                   \
	  if (get_string(str, td->s##XX)) { \
	    td->d->XX = get_copy(str);      \
	    td->flags |= FREE_##XX;         \
	  }                                 \
	  else if ((ustrcmp(td->s##XX, "NULL")==0 || td->s##XX[0]=='0') \
	           && ugetc(proc_list[p].s##XX) == '\"') { \
	    td->d->XX = empty_string;       \
	  }                                 \
	  else {                            \
	    td->d->XX = proc_list[p].XX;    \
	  }                                 \
	}

	for (p = 0; proc_list[p].proc; p++)
		if (ustrcmp(proc_list[p].sproc, td->sproc) == 0)
			break;

	if (proc_list[p].proc) {
		td->d->proc = proc_list[p].proc;
		td->flags &= ~(FREE_dp | FREE_dp2 | FREE_dp3);
		UPDATE_DP(dp);
		UPDATE_DP(dp2);
		UPDATE_DP(dp3);
	}
	else {
		td->d->proc   = d_unknown_proc;
		td->d->dp     = td->sproc;
		td->flags &= ~(FREE_dp | FREE_dp2 | FREE_dp3);
	}

	/* let the DIALOG initialize its internals */
	set_dialog_gui_state();
	object_message(td->d, MSG_START, 0);
	set_editor_gui_state();
	td->flags |= GOT_MSG_START;
}



/* returns the bounds of a tdialog array */
void tdialog_get_bounds(TDIALOG *td, int n, int *x1, int *y1, int *x2, int *y2)
{
	int i;
	*x1 = *y1 = INT_MAX;
	*x2 = *y2 = INT_MIN;

	for (i=0; i<n; i++) {
		if (td[i].x < *x1)
			*x1 = td[i].x;

		if (td[i].y < *y1)
			*y1 = td[i].y;

		if (td[i].x + td[i].w > *x2)
			*x2 = td[i].x + td[i].w;

		if (td[i].y + td[i].h > *y2)
			*y2 = td[i].y + td[i].h;
	}
}



/* execute the dialog editor */
void dialog_editor(char *path, char *name)
{
	int closed = FALSE;

	init_editor();
	show_mouse(screen);
	clear_keybuf();
	editor_player = init_dialog(editor, -1);
	broadcast_dialog_message(MSG_DRAW, 0);

	if (path)
		load_dialog(path, name);
	else
		set_dialog_save_info("", "dlg[] ");

	while (update_dialog(editor_player) && !closed) {
		/* update the mouse cursor */
		if (editor_player->mouse_obj == 0) {
			BITMAP *new_pointer = default_pointer;
			int mx = mouse_x;
			int my = mouse_y;
			int i = find_mouse_object();

			if (i >= 0) {
				if (mx < tdialog[i].x+3) {
					if (my < tdialog[i].y+3)
						new_pointer = resize_pointer_d1;
					else if (my >= tdialog[i].y+tdialog[i].h-3)
						new_pointer = resize_pointer_d2;
					else
						new_pointer = resize_pointer_h;
				}
				else if (mx >= tdialog[i].x+tdialog[i].w-3) {
					if (my < tdialog[i].y+3)
						new_pointer = resize_pointer_d2;
					else if (my >= tdialog[i].y+tdialog[i].h-3)
						new_pointer = resize_pointer_d1;
					else
						new_pointer = resize_pointer_h;
				}
				else if (my < tdialog[i].y+3 || my >= tdialog[i].y+tdialog[i].h-3)
					new_pointer = resize_pointer_v;
			}

			if (mouse_sprite != new_pointer) {
				scare_mouse();
				set_mouse_sprite(new_pointer);
				if (new_pointer != default_pointer)
					set_mouse_sprite_focus(new_pointer->w/2, new_pointer->h/2);
				unscare_mouse();
			}
		}

		/* did the user request a close ? */
		if (close_requested) {
			close_requested = FALSE;
			set_mouse_sprite(default_pointer);
			if (alert(" Do you really want to quit? ", NULL, NULL, " Sure ", " No ", 13, 27) == 1)
				closed = TRUE;
		}
	}

	shutdown_dialog(editor_player);
	set_dialog_gui_state();
	editor_player = NULL;
	deinit_editor();
}


