/*
 * Allegro DIALOG Editor
 * by Julien Cugniere
 *
 * dlg.c : This file deals with setting up and launching the editor.
 */

#include "dedit.h"
#include <allegro.h>
#include <string.h>



/* show some information about command line arguments */
void help(void)
{
	allegro_message(
		"\n Allegro DIALOG Editor v" DLG_VERSION_STR " - Julien Cugniere (" DLG_DATE_STR ")\n\n"
		" Usage:    dlg [options] [file [dialog]]\n"
		" Options:\n"
		"\t-BPP\t\tset the color depth to BPP\n"
		"\t-WxH[xBPP]\tset the resolution to WxH, with an optional color depth\n"
		"\t-pal FILE  \tuse the palette from image file FILE (pcx/bmp/lbm/tga)\n"
		"\t-w\t\trun in windowed mode if possible\n"
		"\t-df\t\tuse the default allegro font for the editor's GUI\n"
		"\t-bak\t\twrite backup files by default when saving\n"
		"\t-cpp\t\toutput cpp friendly code when in doubt\n"
		"\t-h, -?\t\tdisplay this help\n\n"
		" Bugs, comments, questions or anything else: <uos@free.fr>.\n\n"
	);
}



/* initialize the environment
 *  You can change a lot of stuff here, like any of allegro gui_* variables,
 *  or the font pointer. This will affect the dialog you're editing, but not
 *  the editor. You can also set your own graphic mode, palette, etc...
 *  Or you have the possibility to changes these values inside a plugin (but
 *  beware of conflicts between plugins!).
 */
int init(int w, int h, int bpp, int windowed)
{
	int res;

	install_keyboard();
	install_mouse();
	install_timer();

	if (!bpp)
		bpp = (windowed && desktop_color_depth()) ? desktop_color_depth() : 8;

	set_color_depth(bpp);
	res = -1;

	if (windowed)
		res = set_gfx_mode(GFX_AUTODETECT_WINDOWED, w, h, 0, 0);

	if (res != 0)
		res = set_gfx_mode(GFX_AUTODETECT, w, h, 0, 0);

	if (res != 0) {
		allegro_message("Error setting video mode %dx%dx%d:\n%s\n", w, h, bpp, allegro_error);
		exit(1);
	}

	gui_fg_color = makecol(0, 0, 0);
	gui_mg_color = makecol(127, 127, 127);
	gui_bg_color = makecol(255, 255, 255);

	return 0;
}



/* main (nooooo, really?) */
int main(int argc, char *argv[])
{
	int i;
	int screen_w, screen_h, bpp;
	int windowed;
	char allegro_cfg_path[1024], *allegro_cfg;
	char *s, tmp[256];
	char *path = NULL;
	char *name = NULL;
	PALETTE pal;
	BITMAP *pal_bmp = NULL;

	set_uformat(U_ASCII);
	allegro_init();

	/* set the config file to dlg.cfg, keeping allegro.cfg as an override */
	set_config_file("dlg.cfg");
	allegro_cfg = uconvert_ascii("allegro.cfg", tmp);
	if (find_allegro_resource(allegro_cfg_path, allegro_cfg, NULL, NULL, NULL, NULL, NULL, sizeof(allegro_cfg_path)) == 0)
		override_config_file(allegro_cfg_path);

	/* read config data */
	screen_w = get_config_int("[dlg]", "screen_w", 640);
	screen_h = get_config_int("[dlg]", "screen_h", 480);
	bpp = get_config_int("[dlg]", "bpp", 8);
	windowed = get_config_int("[dlg]", "windowed", 0);

	/* process the command line arguments */
	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-' && argv[i][1]) {
			if ((strchr(argv[i], 'x')) || (strchr(argv[i], 'X'))) {
				strcpy(tmp, argv[i] + 1);
				s = strtok(tmp, "xX");
				screen_w = (s) ? atoi(s) : 0;
				s = strtok(NULL, "xX");
				screen_h = (s) ? atoi(s) : 0;
				s = strtok(NULL, "xX");
				if (s)
					bpp = atoi(s);
				if ((screen_w < 512) || (screen_h < 384)) {
					allegro_message("Display resolution too low: `%s'\n", argv[i] + 1);
					return 1;
				}
			}
			else if (argv[i][1] >= '0' && argv[i][1] <= '9') {
				bpp = atoi(argv[i] + 1);
				if ((bpp != 8) && (bpp != 15) && (bpp != 16) && (bpp != 24) && (bpp != 32)) {
					allegro_message("Invalid color depth `%s'\n", argv[i] + 1);
					return 1;
				}
			}
			else if (strcmp(argv[i] + 1, "w") == 0) {
				windowed = TRUE;
			}
			else if (strcmp(argv[i] + 1, "df") == 0) {
				set_config_int("[dlg]", "use_default_font", 1);
			}
			else if (strcmp(argv[i] + 1, "bak") == 0) {
				set_config_int("[dlg]", "do_backups", 1);
			}
			else if (strcmp(argv[i] + 1, "cpp") == 0) {
				set_config_int("[dlg]", "cpp_output", 1);
			}
			else if (strcmp(argv[i] + 1, "pal") == 0) {
				if (i + 1 < argc) {
					i++;
					pal_bmp = load_bitmap(argv[i], pal);
					if (pal_bmp == NULL) {
						allegro_message("Could not read bitmap file `%s'\n", argv[i]);
						return 1;
					}
				}
				else {
					allegro_message("Missing parameter to command line argument %s\n", argv[i]);
					return 1;
				}
			}
			else if (!argv[i][2] && (argv[i][1] == 'h' || argv[i][1] == '?')) {
				help();
				return 0;
			}
			else {
				allegro_message("Unrecognized option: `%s'\n", argv[i]);
				return 1;
			}
		}
		else {
			if (!path)
				path = argv[i];
			else
				name = argv[i];
		}
	}

	/* launch the editor */
	init(screen_w, screen_h, bpp, windowed);
	if (pal_bmp) {
		set_palette(pal);
		destroy_bitmap(pal_bmp);
	}
	dialog_editor(path, name);

	return 0;
}
END_OF_MAIN();
