/*
 * Allegro DIALOG Editor
 * by Julien Cugniere
 *
 * fileops.c : File handling functions
 */

#include "fileops.h"
#include "strparse.h"
#include <ctype.h>
#include <string.h>



/******************************************/
/**************** Helpers *****************/
/******************************************/



/* read the string representing a parameter from a file, working around
 * the comments, string literals, characters constants etc... (hope i forgot
 * nothing too important =)
 */
char *fread_parameter(FILE *f, char *buff, int buffsize)
{
	int i, c, c2;

	do c = fgetc(f);
	while (uisspace(c) && !feof(f));

	i = 0;
	while (!(c==',' || c=='}') && !feof(f) && i<buffsize-1) {
		buff[i++] = c;

		switch(c) {
		case '"':
		case '\'':
			buff[i++] = (c2 = fgetc(f));
			while(c2 != c && !feof(f)) {
				if(c2 == '\\')
					buff[i++] = fgetc(f);
				buff[i++] = (c2 = fgetc(f));
			}
			break;

		case '/':
			buff[i++] = (c = fgetc(f));
			if (c == '/') {
				do buff[i++] = (c = fgetc(f));
				while (c != '\n' && !feof(f));
			}
			else if (c == '*') {
				while (!feof(f)) {
					buff[i++] = (c = fgetc(f));
					if (c == '*') {
						buff[i++] = (c = fgetc(f));
						if (c == '/')
							break;
						else {
							i--;
							ungetc(c, f);
						}
					}
				}
			}
			break;
		}
		c = fgetc(f);
	}

	if (c == '}')
		ungetc(c, f);

	if (i) {
		buff[i] = '\0';

		// trim trailing whitespaces
		while (ugetc(buff) != 0) {
			if (uisspace(ugetat(buff, -1)))
				usetat(buff, -1, '\0');
			else
				break;
		}
	}
	else usprintf(buff, "0");

	return buff;
}



/* this reads characters from the file and return TRUE if they follow the
 * given format. the format is quite limited:
 * - a blank ' ' in the format represent any number of white spaces in the
 *   file (including none). Comments (single or multiline) are white spaces.
 * - a '*' in the format allow to scan all characters until the one just
 *   after the '*'. for example, fcheck("[*]") will return true for a string
 *   like "[foo...]" as well as for "[]". Characters found in string or
 *   character constants are ignored.
 * - use two '*' to read an asterisk from the file.
 * - any other character is read as himself.
 * - this routine is subject to many bugs... =)
 */
int fcheck(FILE *f, char *fmt)
{
	int i, c, c2, cstop, pos;

	for (i=0; i<ustrlen(fmt); i++) {
		c = ugetat(fmt, i);
		switch (c) {
		case ' ':
			while (!feof(f)) {
				pos = ftell(f);
				c = fgetc(f);
				if (c == '/') {
					c = fgetc(f);
					if (c == '*') {
						while (!feof(f)) {
							if (fgetc(f) == '*') {
								pos = ftell(f);
								if ((c = fgetc(f)) == '/')
									break;
								else fseek(f, pos, SEEK_SET);
							}
						}
					}
					else if (c == '/') {
						while (fgetc(f) != '\n' && !feof(f));
					}
					else {
						fseek(f, pos, SEEK_SET);
						break;
					}
				}
				else if (!uisspace(c)) {
					fseek(f, pos, SEEK_SET);
					break;
				}
			}
			break;

		case '*':
			cstop = ugetat(fmt, ++i);
			c = fgetc(f);
			while (c != cstop && !feof(f)) {
				if (c == '/') {
					c2 = fgetc(f);
					if (c2 == '*') {
						while (!feof(f))
						if (fgetc(f) == '*') {
							if ((c2 = fgetc(f)) == '/')
								break;
							else ungetc(c2, f);
						}
					}
					else if (c2 == '/')
						while (fgetc(f) != '\n' && !feof(f));
					else
						ungetc(c2, f);
				}
				else if (c == '"' || c == '\'') {
					c2 = fgetc(f);
					while (!feof && c2 != c) {
						if (c2=='\\')
							fgetc(f);
						c2 = fgetc(f);
					}
				}
				c = fgetc(f);
			}

			if (c != cstop)
				return FALSE;
			break;

		default:
			c2 = fgetc(f);
			if (c2 != c) {
				ungetc(c2, f);
				return FALSE;
			}
			break;
		}
	}

	return TRUE;
}



/******************************************/
/*********** Reading routines *************/
/******************************************/



/* allocates a structure to hold information about a source file */
SCAN_INFO *create_scan_info(void)
{
	SCAN_INFO *sc = malloc(sizeof(SCAN_INFO));
	sc->n = 0;
	sc->name = NULL;
	sc->offset = NULL;
	return sc;
}



/* clears the list of dialog found in a source file */
void destroy_scan_info(SCAN_INFO *s)
{
	if (s) {
		while (s->n)
			free(s->name[--s->n]);
		free(s->name);
		free(s->offset);
		free(s);
	}
}



/* scan a file for DIALOG arrays and store their name and offset */
SCAN_INFO *fscan_for_dialogs(char *path)
{
	SCAN_INFO *sc;
	char buff[1024];
	long pos;
	int c;

	FILE *f = fopen(path, "rb");
	if (!f)
		return NULL;

	/* some stupid implementations (linux) let you open directories without
	 * triggering either ferror() or feof(). grumpf...
	 */
	c = fgetc(f);
	if (c == EOF) {
		fclose(f);
		return NULL;
	}
	ungetc(c, f);

	sc = create_scan_info();
	while (!feof(f)) {
		do {
			fscanf(f, "%1023[^_A-Za-z0-9]", buff);
			fscanf(f, "%1023[_A-Za-z0-9]", buff);
		}
		while (ustrcmp(buff, "DIALOG")!=0 && !feof(f));

		if (feof(f))
			break;

		fcheck(f, " ");
		pos = ftell(f);
		if (fscanf(f, "%1023[^=\n]", buff)) {
			if (ustrchr(buff, '[') && fcheck(f, " = { ")) {
				sc->n++;

				sc->name = realloc(sc->name, sizeof(char*)*sc->n);
				sc->name[sc->n-1] = malloc(sizeof(char)*ustrsizez(buff));
				ustrcpy(sc->name[sc->n-1], buff);

				sc->offset = realloc(sc->offset, sizeof(long)*sc->n);
				sc->offset[sc->n-1] = pos;

				fcheck(f, "*;");
			}
		}
	}

	fclose(f);
	return sc;
}



/* read a DIALOG structure from the file */
int fread_dialog_struct(FILE *f, TDIALOG *td)
{
	char buff[4096];

	if (!fcheck(f, " {"))
		return -1;

	fread_parameter(f, buff, 4096);

	if (ustrcmp(buff, "NULL")==0 || ustrcmp(buff, "0")==0) {
		fcheck(f, "*}");
		return -1;
	}

	td->sproc  = get_copy(buff);
	td->sx     = get_copy(fread_parameter(f, buff, 4096));
	td->sy     = get_copy(fread_parameter(f, buff, 4096));
	td->sw     = get_copy(fread_parameter(f, buff, 4096));
	td->sh     = get_copy(fread_parameter(f, buff, 4096));
	td->sfg    = get_copy(fread_parameter(f, buff, 4096));
	td->sbg    = get_copy(fread_parameter(f, buff, 4096));
	td->skey   = get_copy(fread_parameter(f, buff, 4096));
	td->sflags = get_copy(fread_parameter(f, buff, 4096));
	td->sd1    = get_copy(fread_parameter(f, buff, 4096));
	td->sd2    = get_copy(fread_parameter(f, buff, 4096));
	td->sdp    = get_copy(fread_parameter(f, buff, 4096));
	td->sdp2   = get_copy(fread_parameter(f, buff, 4096));
	td->sdp3   = get_copy(fread_parameter(f, buff, 4096));
	td->d = malloc(sizeof(DIALOG));

	return 0;
}



/******************************************/
/************ Saving routines *************/
/******************************************/



/* checks wether the string represent a string literal, and if it is, makes
 * sure there is (void*) at its beginning. There better be enough room in the
 * string :-/
 */
void ensure_void_pointer_cast(char *s)
{
	const char *prefix = "(void*)";
	char *i = s;
	str_skip_ws(&i);
	if (*i == '"') {
		memmove(i+strlen(prefix), i, strlen(i)+1);
		memcpy(i, prefix, strlen(prefix));
	}
}



/* generates the c/c++ code representing the dialog, in ASCII format.
 * this routine allocates the memory needed, but you must free() it yourself.
 */
char *make_source(TDIALOG *td, int tdsize, char *name, int cpp)
{
	char *s, *is;
	int i, n, nalloc, w[14], d;
	char *param_name[] = {
		"(proc) ", "(x)", "(y)", "(w)", "(h)", "(fg)", "(bg)",
		"(key)", "(flags)", "(d1)", "(d2)", "(dp) ", "(dp2)", "(dp3)"
	};

	/* fix the content of the tdialog to allow compilation as c++ code */
	if (cpp) {
		for (i=0; i<tdsize; i++) {
			ensure_void_pointer_cast(td[i].sdp);
			ensure_void_pointer_cast(td[i].sdp2);
			ensure_void_pointer_cast(td[i].sdp3);
		}
	}

	/* calculate maximum parameter length for text justification */
	for (i=0; i<14; i++)
		w[i] = ustrlen(param_name[i])-1;

	#define MAX_ALIGN_SIZE 40
	#define UPDATE_W(sparam, j)        \
	  n = ustrlen(td[i].sparam);       \
	  if (n>w[j] && n<=MAX_ALIGN_SIZE) \
	    w[j] = n;

	for (i=0; i<tdsize; i++) {
		UPDATE_W(sproc,  0);
		UPDATE_W(sx,     1);
		UPDATE_W(sy,     2);
		UPDATE_W(sw,     3);
		UPDATE_W(sh,     4);
		UPDATE_W(sfg,    5);
		UPDATE_W(sbg,    6);
		UPDATE_W(skey,   7);
		UPDATE_W(sflags, 8);
		UPDATE_W(sd1,    9);
		UPDATE_W(sd2,    10);
		UPDATE_W(sdp,    11);
		UPDATE_W(sdp2,   12);
		UPDATE_W(sdp3,   13);
	}

	/* allocate a string */
	nalloc = 7;
	for (i=0; i<14; i++)
		nalloc += w[i]+2;
	nalloc *= (tdsize+2);
	nalloc += ustrsize(name)+20;
	s = malloc(nalloc);

	/* and finally fill the string */
	if (s) {
		is = s;
		is += usprintf(is, "DIALOG %s=\n", name);
		is += usprintf(is, "{\n   /* %-*s", w[0]+1, param_name[0]);
		for (i=1; i<14; i++)
			is += usprintf(is, "%-*s", w[i]+2, param_name[i]);
		is += usprintf(is, "*/\n");

		#define PRINT(fmt, sparam, j)  \
		  n = ustrlen(td[i].sparam);   \
		  if (n > w[j]) {              \
		    nalloc += (n - w[j]);      \
		    d = is - s;                \
		    s = realloc(s, nalloc);    \
		    is = s + d;                \
		  }                            \
		  is += usprintf(is, fmt "%*s", td[i].sparam, MAX(0, w[j]-n)+1, " ");

		for (i=0; i<tdsize; i++) {
			is += usprintf(is, "   { ");
			PRINT("%s,", sproc,  0);
			PRINT("%s,", sx,     1);
			PRINT("%s,", sy,     2);
			PRINT("%s,", sw,     3);
			PRINT("%s,", sh,     4);
			PRINT("%s,", sfg,    5);
			PRINT("%s,", sbg,    6);
			PRINT("%s,", skey,   7);
			PRINT("%s,", sflags, 8);
			PRINT("%s,", sd1,    9);
			PRINT("%s,", sd2,    10);
			PRINT("%s,", sdp,    11);
			PRINT("%s,", sdp2,   12);
			PRINT("%s",  sdp3,   13);
			is += usprintf(is, "},\n");
		}

		is += usprintf(
			is, "   { NULL,%*s0,%*s0,%*s0,%*s0,%*s0,%*s0,%*s0,%*s0,%*s0,%*s0,%*sNULL,%*sNULL,%*sNULL%*s}\n};",
			w[0]-3, " ", w[1], " ", w[2], " ", w[3], " ", w[4], " ", w[5], " ", w[6], " ", w[7], " ",
			w[8], " ", w[9], " ", w[10], " ", w[11]-3, " ", w[12]-3, " ", w[13]-3, " "
		);

		if (is-s > nalloc) {
			char error[150];
			n = (is-s) - nalloc;
			usprintf(error, "make_source() just overflowed its buffer by %d bytes", n);
			alert("Ooops!", error, "Please send a bug report to uos@free.fr", "Damn", NULL, 0, 0);
		}
	}
	return s;
}



/* write a dialog in a file, either at the head (offset=0), or by replacing
 * an existing dialog in the middle of the file. NOTE: pack_fputs is the
 * only packfile function to handle LF/CR-LF conversion.
 */
void write_dialog(TDIALOG *td, int tdsize, char *path, char *cname, long offset, int backup, int cpp)
{
	FILE *f;
	PACKFILE *pf;
	char bakpath[1024], buff[1024], *s;
	char name[1024] = "dlg[] ";
	int i, n;

	/* ensure that the dialog name is correct */
	if (cname)
		strcpy(name, cname);
	i = strlen(name)-1;
	while (i >= 0 && isspace(name[i]))
		i--;
	if (i < 0)
		strcpy(name, "dlg[] ");
	else {
		i++;
		if (name[i-1] != ']') {
			name[i++] = '[';
			name[i++] = ']';
		}
		name[i++] = ' ';
		name[i++] = 0;
	}

	if (!exists(path)) {
		f = fopen(path, "w");
		if (!f)
			alert(path, NULL, "Could not open the file for writing", "&Ok", NULL, 'o', 0);
		else {
			s = make_source(td, tdsize, name, cpp);
			fputs(s, f);
			fclose(f);
			free(s);
		}
		return;
	}

	/* first, prepare the files */
	replace_extension(bakpath, path, "bak", 1024);
	remove(bakpath);
	if (rename(path, bakpath) != 0) {
		alert(path, "Error renaming to", bakpath, "&Ok", NULL, 'o', 0);
		return;
	}

	f = fopen(bakpath, "rb");
	if (!f) {
		alert(bakpath, NULL, "Could not open backup file for reading ?!", "&uh?", NULL, 'u', 0);
		return;
	}

	pf = pack_fopen(path, "w");
	if (!pf) {
		alert(path, NULL, "Could not open file for writing.", "&Ok", NULL, 'o', 0);
		fclose(f);
		return;
	}

	/* now, write the dialog to the file */
	while (offset>1024 && !feof(f)) {
		n = fread(buff, sizeof(char), 1024, f);
		pack_fwrite(buff, sizeof(char)*n, pf);
		offset -= n;
	}
	n = fread(buff, sizeof(char), offset, f);
	pack_fwrite(buff, sizeof(char)*n, pf);

	s = make_source(td, tdsize, name, cpp);
	if (offset == 0) {
		pack_fputs(s, pf);
		pack_fputs("\n", pf);
	}
	else {
		pack_fputs((s+7), pf);
		fcheck(f, "*;");
	}

	while (!feof(f)) {
		n = fread(buff, sizeof(char), 1024, f);
		pack_fwrite(buff, sizeof(char)*n, pf);
	}

	pack_fclose(pf);
	fclose(f);
	free(s);

	/* remove the backup copy if we need to */
	if (!backup)
		if (remove(bakpath) != 0)
			alert("Could not delete the backup file:", bakpath, NULL, "&Ok", NULL, 'o', 0);
}



