#ifndef __RM_H__
#define __RM_H__

#include <stdlib.h>


// -- RM (Resource Manager) --


// Set DEBUG level - (RM_DEBUG):
// -----------------------------

// 0 = Normal - no extra information saved or logged.

// 1 = Light - Extra file & line information stored.
//     Any remaining resources are logged when rm_shutdown is called.

// 2 = Heavy - Extra file & line information stored.
//     Every resource added & removed is logged.


#define RM_DEBUG                  2


// Logging method
// --------------

// Use a console window for output.
//#define RM_LOG                    printf

// Custom logging function.
#define RM_LOG                    write_to_log_no_cr



// Internals.
// ----------

#define RM_TYPE_BITMAP            0
#define RM_TYPE_SAMPLE            1

struct rm
{
 #ifdef RM_DEBUG
 char name[120];
 #endif
 
 char *dat;
 char type;
 struct rm *next;
} rm;

struct rm_system
{
 struct rm *first;
 struct rm *last;
} rm_system;


// Main RM functions.
int add_to_rm_list(char *name, char type, char *dat);
int delete_from_rm_list(char *dat);
void rm_shutdown(void);

#if RM_DEBUG > 0

// Bitmap functions.
BITMAP *_load_bitmap_rm(char *file, int line, AL_CONST char *filename, RGB *pal);
BITMAP *_load_bmp_rm(char *file, int line, AL_CONST char *filename, RGB *pal);
BITMAP *_load_pcx_rm(char *file, int line, AL_CONST char *filename, RGB *pal);
BITMAP *_load_lbm_rm(char *file, int line, AL_CONST char *filename, RGB *pal);
BITMAP *_load_tga_rm(char *file, int line, AL_CONST char *filename, RGB *pal);
BITMAP *_create_bitmap_rm(char *file, int line, int width, int height);
BITMAP *_create_system_bitmap_rm(char *file, int line, int width, int height);
BITMAP *_create_video_bitmap_rm(char *file, int line, int width, int height);
BITMAP *_create_bitmap_ex_rm(char *file, int line, int depth, int width, int height);

// Sample functions.
SAMPLE *_load_sample_rm(char *file, int line, const char *filename);
SAMPLE *_load_wav_rm(char *file, int line, const char *filename);
SAMPLE *_load_voc_rm(char *file, int line, const char *filename);
SAMPLE *_create_sample_rm(char *file, int line, int bits, int stereo, int freq, int len);
void _rm_add_sample(char *file, int line, SAMPLE *sam);

#else

// Bitmap functions.
BITMAP *_load_bitmap_rm(AL_CONST char *filename, RGB *pal);
BITMAP *_load_bmp_rm(AL_CONST char *filename, RGB *pal);
BITMAP *_load_pcx_rm(AL_CONST char *filename, RGB *pal);
BITMAP *_load_lbm_rm(AL_CONST char *filename, RGB *pal);
BITMAP *_load_tga_rm(AL_CONST char *filename, RGB *pal);
BITMAP *_create_bitmap_rm(int width, int height);
BITMAP *_create_system_bitmap_rm(int width, int height);
BITMAP *_create_video_bitmap_rm(int width, int height);
BITMAP *_create_bitmap_ex_rm(int depth, int width, int height);

// Sample functions.
SAMPLE *_load_sample_rm(const char *filename);
SAMPLE *_load_wav_rm(const char *filename);
SAMPLE *_load_voc_rm(const char *filename);
SAMPLE *_create_sample_rm(int bits, int stereo, int freq, int len);
void _rm_add_sample(SAMPLE *sam);

#endif

void _destroy_bitmap_rm(BITMAP *bmp);
void _destroy_sample_rm(SAMPLE *sam);

// special defines to automatically add RM support.

#ifndef __RM_C__

// first the debug and normal specific versions.
#if RM_DEBUG > 0

// Bitmap functions.
#define load_bitmap(name, pal) _load_bitmap_rm(__FILE__, __LINE__, name, pal)
#define load_bmp(name, pal) _load_bmp_rm(__FILE__, __LINE__, name, pal)
#define load_pcx(name, pal) _load_pcx_rm(__FILE__, __LINE__, name, pal)
#define load_lbm(name, pal) _load_lbm_rm(__FILE__, __LINE__, name, pal)
#define load_tga(name, pal) _load_tga_rm(__FILE__, __LINE__, name, pal)
#define create_bitmap(width, height) _create_bitmap_rm(__FILE__, __LINE__, width, height)
#define create_bitmap_ex(depth, width, height) _create_bitmap_ex_rm(__FILE__, __LINE__, depth, width, height)
#define create_system_bitmap(width, height) _create_system_bitmap_rm(__FILE__, __LINE__, width, height)
#define create_video_bitmap(width, height) _create_video_bitmap_rm(__FILE__, __LINE__, width, height)

// Sample functions.
#define load_sample(name) _load_sample_rm(__FILE__, __LINE__, name)
#define load_wav(name) _load_wav_rm(__FILE__, __LINE__, name)
#define load_voc(name) _load_voc_rm(__FILE__, __LINE__, name)
#define create_sample(bits, stereo, freq, len) _create_sample_rm(__FILE__, __LINE__, bits, stereo, freq, len)
#define rm_add_sample(sam) _rm_add_sample(__FILE__, __LINE__, sam)


#else

// Bitmap functions.
#define load_bitmap(name, pal) _load_bitmap_rm(name, pal)
#define load_bmp(name, pal) _load_bmp_rm(name, pal)
#define load_pcx(name, pal) _load_pcx_rm(name, pal)
#define load_lbm(name, pal) _load_lbm_rm(name, pal)
#define load_tga(name, pal) _load_tga_rm(name, pal)
#define create_bitmap(width, height) _create_bitmap_rm(width, height)
#define create_bitmap_ex(depth, width, height) _create_bitmap_ex_rm(depth, width, height)
#define create_system_bitmap(width, height) _create_bitmap_rm(width, height)
#define create_video_bitmap(width, height) _create_video_bitmap_rm(width, height)

// Sample functions.
#define load_sample(name) _load_sample_rm(name)
#define load_wav(name) _load_wav_rm(name)
#define load_voc(name) _load_voc_rm(name)
#define create_sample(bits, stereo, freq, len) _create_sample_rm(bits, stereo, freq, len)
#define rm_add_sample(sam) _rm_add_sample(sam)

#endif

#define destroy_bitmap(b) _destroy_bitmap_rm(b)
#define destroy_sample(s) _destroy_sample_rm(s)

#endif // __RM_C__
#endif // __RM_H__
