
#include <allegro.h>
#include <stdio.h>
#include <stdarg.h>

#define __RM_C__ // So rm.h knows not to include special defines.
#include "rm.h"

#include "log.h"

// Resource Manager (RM)
// ---------------------

// A set of functions that wrap around the normal bitmap and sample creation 
// and destruction functions, to provide automatic resource cleanup on exit.

// Optional logging of all resources destroyed at exit is possible, as is
// more detailed logging of all resources as they are created and destroyed ingame.

// To use: Simple #include "rm.h" in each source code file and call rm_shutdown
// before your program exits.

   
int add_to_rm_list(char *name, char type, char *dat)
{
 struct rm *current;
 struct rm *last;
 
 current = (struct rm *)malloc(sizeof(rm));
 if (!current)
 {
  RM_LOG("Out of memory when trying to add to RM list: %s\n", name);
  
  return 0;
 }
 
 if (rm_system.first == NULL)
 {
  // This must be the first item in the list.
  rm_system.first = current;
  rm_system.last = current;
 }
 else
 {
  // We already have at least one item in the list.
  last = rm_system.last;
  last->next = current;
  rm_system.last = current;
 }
 
 current->dat = dat;
 current->type = type;
 
 #if RM_DEBUG > 0
 sprintf(current->name, "%s", name);

  #if RM_DEBUG == 2
  if (type == RM_TYPE_BITMAP) RM_LOG("RM - Bitmap added : %s\n", name);
  if (type == RM_TYPE_SAMPLE) RM_LOG("RM - Sample added : %s\n", name);
  #endif
 
 #endif
 
 return 1;
}

int delete_from_rm_list(char *dat)
{
 struct rm *current;
 struct rm *old;
 int y;
 
 y = 0;
 
 if (rm_system.first == NULL) return 0;
 current = rm_system.first;
 
 do
 {
  old = current;
  
  if (y != 0) current = current->next;
  
  if (current->dat == dat)
  {
   // Found specific entry.
   old->next = current->next;
   
   // Is this the first entry in the list?
   if (y == 0) rm_system.first = current->next;
   
   // Or the last?
   if (current == rm_system.last)
   {
    rm_system.last = old;
    
    if (y == 0)
    {
     // The first and last?
     rm_system.first = NULL;
     rm_system.last = NULL;
    }
   }
   
   // Destroy object in list based on type.
   if (current->type == RM_TYPE_BITMAP)
   {
    destroy_bitmap((BITMAP *)current->dat);
   
    #if RM_DEBUG == 2
    RM_LOG("RM - Bitmap deleted (from) : %s\n", current->name);
    #endif
   }
   if (current->type == RM_TYPE_SAMPLE)
   {
    destroy_sample((SAMPLE *)current->dat);
   
    #if RM_DEBUG == 2
    RM_LOG("RM - Sample deleted (from) : %s\n", current->name);
    #endif
   }
   // Free list entry.
   free(current);
   return 1;
  }
  y++;
  
 }
 while (current != rm_system.last);
 return 1;
}

void rm_shutdown(void)
{
 struct rm *current;
 struct rm *next;

 #if RM_DEBUG > 0
 {
  RM_LOG("\n");
  RM_LOG("Resource Manager :\n");
  RM_LOG("--\n");
 }
 #endif
 
 if (rm_system.first == NULL)
 {
  #if RM_DEBUG > 0
  {
   RM_LOG("No Resources in list.\n");
  }
  #endif
  return;
 }
 
 current = rm_system.first;
 
 do
 {

  next=current->next;
    
  // Destroy object in list based on type.
  if (current->type == RM_TYPE_BITMAP)
  {
   destroy_bitmap((BITMAP *)current->dat);
   
   #if RM_DEBUG > 0
   RM_LOG("RM - Bitmap deleted (from) : %s\n", current->name);
   #endif
  }
  if (current->type == RM_TYPE_SAMPLE)
  {
   stop_sample((SAMPLE *)current->dat);
   destroy_sample((SAMPLE *)current->dat);
   
   #if RM_DEBUG > 0
   RM_LOG("RM - Sample deleted (from) : %s\n", current->name);
   #endif
  }
  
  free(current);
  current=next;
  
 }
 while (current != rm_system.last);
 
 #if RM_DEBUG > 0
 {
  RM_LOG("--\n");
 }
 #endif
}


// Bitmap Functions.
// -----------------

#if RM_DEBUG > 0
BITMAP *_create_bitmap_rm(char *file, int line, int width, int height)
#else
BITMAP *_create_bitmap_rm(int width, int height)
#endif
{
 BITMAP *bmp;
 char name[120];
 
 #if RM_DEBUG > 0
 uszprintf(name, sizeof(name), "%s, line %d", file, line);
 #endif
 
 bmp = create_bitmap(width, height);
 
 if (bmp) add_to_rm_list(name, RM_TYPE_BITMAP, (char *)bmp);
 
 return bmp;
}

#if RM_DEBUG > 0
BITMAP *_create_video_bitmap_rm(char *file, int line, int width, int height)
#else
BITMAP *_create_video_bitmap_rm(int width, int height)
#endif
{
 BITMAP *bmp;
 char name[120];
 
 #if RM_DEBUG > 0
 uszprintf(name, sizeof(name), "%s, line %d", file, line);
 #endif
 
 bmp = create_video_bitmap(width, height);
 
 if (bmp) add_to_rm_list(name, RM_TYPE_BITMAP, (char *)bmp);
 
 return bmp;
}

#if RM_DEBUG > 0
BITMAP *_create_system_bitmap_rm(char *file, int line, int width, int height)
#else
BITMAP *_create_system_bitmap_rm(int width, int height)
#endif
{
 BITMAP *bmp;
 char name[120];
 
 #if RM_DEBUG > 0
 uszprintf(name, sizeof(name), "%s, line %d", file, line);
 #endif
 
 bmp = create_system_bitmap(width, height);
 
 if (bmp) add_to_rm_list(name, RM_TYPE_BITMAP, (char *)bmp);
 
 return bmp;
}

#if RM_DEBUG > 0
BITMAP *_create_bitmap_ex_rm(char *file, int line, int depth, int width, int height)
#else
BITMAP *_create_bitmap_ex_rm(int depth, int width, int height)
#endif
{
 BITMAP *bmp;
 char name[120];
 
 #if RM_DEBUG > 0
 uszprintf(name, sizeof(name), "%s, line %d", file, line);
 #endif
 
 bmp = create_bitmap_ex(depth, width, height);
 
 if (bmp) add_to_rm_list(name, RM_TYPE_BITMAP, (char *)bmp);

 return bmp;
}

#if RM_DEBUG > 0
BITMAP *_load_bitmap_rm(char *file, int line, AL_CONST char *filename, RGB *pal)
#else
BITMAP *_load_bitmap_rm(AL_CONST char *filename, RGB *pal)
#endif
{
 BITMAP *bmp;
 char name[120];
 
 bmp = load_bitmap(filename, pal);

 #if RM_DEBUG > 0
 uszprintf(name, sizeof(name), "%s - %s, line %d", filename, file, line);
 #endif
 
 if (bmp) add_to_rm_list(name, RM_TYPE_BITMAP, (char *)bmp);
 
 return bmp;
}

#if RM_DEBUG > 0
BITMAP *_load_bmp_rm(char *file, int line, AL_CONST char *filename, RGB *pal)
#else
BITMAP *_load_bmp_rm(AL_CONST char *filename, RGB *pal)
#endif
{
 BITMAP *bmp;
 char name[120];
 
 bmp = load_bmp(filename, pal);

 #if RM_DEBUG > 0
 uszprintf(name, sizeof(name), "%s - %s, line %d", filename, file, line);
 #endif
 
 if (bmp) add_to_rm_list(name, RM_TYPE_BITMAP, (char *)bmp);
 
 return bmp;
}

#if RM_DEBUG > 0
BITMAP *_load_pcx_rm(char *file, int line, AL_CONST char *filename, RGB *pal)
#else
BITMAP *_load_pcx_rm(AL_CONST char *filename, RGB *pal)
#endif
{
 BITMAP *bmp;
 char name[120];
 
 bmp = load_pcx(filename, pal);

 #if RM_DEBUG > 0
 uszprintf(name, sizeof(name), "%s - %s, line %d", filename, file, line);
 #endif
 
 if (bmp) add_to_rm_list(name, RM_TYPE_BITMAP, (char *)bmp);
 
 return bmp;
}

#if RM_DEBUG > 0
BITMAP *_load_lbm_rm(char *file, int line, AL_CONST char *filename, RGB *pal)
#else
BITMAP *_load_lbm_rm(AL_CONST char *filename, RGB *pal)
#endif
{
 BITMAP *bmp;
 char name[120];
 
 bmp = load_lbm(filename, pal);

 #if RM_DEBUG > 0
 uszprintf(name, sizeof(name), "%s - %s, line %d", filename, file, line);
 #endif
 
 if (bmp) add_to_rm_list(name, RM_TYPE_BITMAP, (char *)bmp);
 
 return bmp;
}

#if RM_DEBUG > 0
BITMAP *_load_tga_rm(char *file, int line, AL_CONST char *filename, RGB *pal)
#else
BITMAP *_load_tga_rm(AL_CONST char *filename, RGB *pal)
#endif
{
 BITMAP *bmp;
 char name[120];
 
 bmp = load_tga(filename, pal);

 #if RM_DEBUG > 0
 uszprintf(name, sizeof(name), "%s - %s, line %d", filename, file, line);
 #endif
 
 if (bmp) add_to_rm_list(name, RM_TYPE_BITMAP, (char *)bmp);
 
 return bmp;
}

void _destroy_bitmap_rm(BITMAP *bmp)
{
 if (bmp) delete_from_rm_list((char *)bmp);
}

// Sample Functions
// ----------------

#if RM_DEBUG > 0
SAMPLE *_load_sample_rm(char *file, int line, AL_CONST char *filename)
#else
SAMPLE *_load_sample_rm(AL_CONST char *filename)
#endif
{
 SAMPLE *sam;
 char name[120];
 
 sam = load_sample(filename);

 #if RM_DEBUG > 0
 uszprintf(name, sizeof(name), "%s - %s, line %d", filename, file, line);
 #endif
 
 if (sam) add_to_rm_list(name, RM_TYPE_SAMPLE, (char *)sam);
 
 return sam;
}

#if RM_DEBUG > 0
SAMPLE *_load_wav_rm(char *file, int line, AL_CONST char *filename)
#else
SAMPLE *_load_wav_rm(AL_CONST char *filename)
#endif
{
 SAMPLE *sam;
 char name[120];
 
 sam = load_sample(filename);

 #if RM_DEBUG > 0
 uszprintf(name, sizeof(name), "%s - %s, line %d", filename, file, line);
 #endif
 
 if (sam) add_to_rm_list(name, RM_TYPE_SAMPLE, (char *)sam);
 
 return sam;
}

#if RM_DEBUG > 0
SAMPLE *_load_voc_rm(char *file, int line, AL_CONST char *filename)
#else
SAMPLE *_load_voc_rm(AL_CONST char *filename)
#endif
{
 SAMPLE *sam;
 char name[120];
 
 sam = load_voc(filename);

 #if RM_DEBUG > 0
 uszprintf(name, sizeof(name), "%s - %s, line %d", filename, file, line);
 #endif
 
 if (sam) add_to_rm_list(name, RM_TYPE_SAMPLE, (char *)sam);
 
 return sam;
}

#if RM_DEBUG > 0
SAMPLE *_create_sample_rm(char *file, int line, int bits, int stereo, int freq, int len)
#else
SAMPLE *_create_sample_rm(int bits, int stereo, int freq, int len)
#endif
{
 SAMPLE *sam;
 char name[120];
 
 sam = create_sample(bits, stereo, freq, len);

 #if RM_DEBUG > 0
 uszprintf(name, sizeof(name), "%s, line %d", file, line);
 #endif

 if (sam) add_to_rm_list(name, RM_TYPE_SAMPLE, (char *)sam);
 
 return sam;
}

#if RM_DEBUG > 0
void _rm_add_sample(char *file, int line, SAMPLE *sam)
#else
void _rm_add_sample(SAMPLE *sam)
#endif
{
 char name[120];
 
 #if RM_DEBUG > 0
 uszprintf(name, sizeof(name), "%s, line %d", file, line);
 #endif

 if (sam) add_to_rm_list(name, RM_TYPE_SAMPLE, (char *)sam);
}

void _destroy_sample_rm(SAMPLE *sam)
{
 if (sam) delete_from_rm_list((char *)sam);
}


