// ------------------------------------------------------------------
// Vector Math Class Library
// (c) 1995 - 2000 Christian Schler
// ------------------------------------------------------------------

#include <math.h>

#include "vector.h"

// ------------------------------------------------------------------
// REST OF VECTOR STUFF
// ------------------------------------------------------------------

vector &vector::normalize()
{
   number q = 1 / sqrt( x * x + y * y + z * z );
   x *= q; 
   y *= q;
   z *= q;
   return *this;
}

number distance( const vector &a, const vector &b )
{
   return sqrt(
      ( a.x - b.x ) * ( a.x - b.x ) +
      ( a.y - b.y ) * ( a.y - b.y ) +
      ( a.z - b.z ) * ( a.z - b.z ) );
}

base &base::spin( const vector &omega )
{
   i += i % omega;
   j += j % omega;
   i.normalize();
   j.normalize();
   k = i % j;
   i = j % k;
   return *this;
}

// ------------------------------------------------------------------
// QUATERNION STUFF
// ------------------------------------------------------------------

quaternion::quaternion( const base &B )
{
   number s;
   number t = B.i.x + B.j.y + B.k.z + 1;

   if( t > 0 )
   {
      s = 0.5 / sqrt( t );
      w = 0.25 / s;
      v.x = ( B.j.z - B.k.y ) * s;
      v.y = ( B.k.x - B.i.z ) * s;
      v.z = ( B.i.y - B.j.x ) * s;
   }

   else
   {
      int i = B.i.x > B.j.y ? ( B.i.x > B.k.z ? 0 : 2 ) :
         ( B.j.y > B.k.z ? 1 : 2 );

      switch( i )
      {
         case 0:
         s = 0.5 / sqrt( B.i.x - B.j.y - B.k.z + 1 );
         v.x = 0.25 / s;
         w = ( B.j.z - B.k.y ) * s;
         v.y = ( B.i.y + B.j.x ) * s;
         v.z = ( B.i.z + B.k.x ) * s;
         break;

         case 1:
         s = 0.5 / sqrt( B.j.y - B.i.x - B.k.z + 1 );
         v.y = 0.25 / s;
         w = ( B.k.x - B.i.z ) * s;
         v.z = ( B.j.z + B.k.y ) * s;
         v.x = ( B.j.x + B.i.y ) * s;
         break;

         case 2:
         s = 0.5 / sqrt( B.k.z - B.i.x - B.j.y + 1 );
         v.z = 0.25 / s;
         w = ( B.i.y - B.j.x ) * s;
         v.x = ( B.k.x + B.i.z ) * s;
         v.y = ( B.k.y + B.j.z ) * s;
         break;
      }
   }

   normalize();
}

quaternion::operator base() const
{
   float wx, wy, wz, xx, yy, yz, xy, xz, zz, x2, y2, z2;
 
   x2 = 2 * v.x;  y2 = 2 * v.y;  z2 = 2 * v.z;
   xx = v.x * x2;  xy = v.x * y2;  xz = v.x * z2;
   yy = v.y * y2;  yz = v.y * z2;  zz = v.z * z2;
   wx = w * x2;  wy = w * y2;  wz = w * z2;
 
   base B;
 
   B.i.x = 1 - (yy + zz);
   B.j.x = xy - wz;
   B.k.x = xz + wy;
  
   B.i.y = xy + wz;
   B.j.y = 1 - (xx + zz);
   B.k.y = yz - wx;
 
   B.i.z = xz - wy;
   B.j.z = yz + wx;
   B.k.z = 1 - (xx + yy);
 
   return B;
}

quaternion &quaternion::normalize()
{
   number q = 1 / sqrt( w * w + v.x * v.x + v.y * v.y + v.z * v.z );
   w *= q;
   v.x *= q;
   v.y *= q;
   v.z *= q; 
   return *this;
}

quaternion &quaternion::spin( const vector &omega )
{
   number tmp = 0.5 * ( omega * v );
   v -= 0.5 * ( omega % v + w * omega );
   w += tmp;
   tmp = 1 / sqrt( w * w + v.x * v.x + v.y * v.y + v.z * v.z );
   w *= tmp; v.x *= tmp; v.y *= tmp; v.z *= tmp; 
   return *this;
}

// ------------------------------------------------------------------
// GLOBAL DEFINITIONS
// ------------------------------------------------------------------

vector zerovector
   = vector( 0, 0, 0 );

base identitybase
   = base( vector( 1, 0, 0 ), vector( 0, 1, 0 ), vector( 0, 0, 1 ) );

quaternion identityquaternion
   = quaternion( 1, zerovector );

space worldspace
   = space( zerovector, identitybase );

// ------------------------------------------------------------------
// REST OF OPERATORS
// ------------------------------------------------------------------

#ifndef __GNUC__
#error This code uses a GNU-C extension
#endif

vector operator *( const vector &s, const base &B ) return res
{
   res.x = s.x * B.i.x + s.y * B.j.x + s.z * B.k.x;
   res.y = s.x * B.i.y + s.y * B.j.y + s.z * B.k.y;
   res.z = s.x * B.i.z + s.y * B.j.z + s.z * B.k.z;
}

vector operator %( const vector &s, const base &B ) return res
{
   res.x = s.x * B.i.x + s.y * B.i.y + s.z * B.i.z;
   res.y = s.x * B.j.x + s.y * B.j.y + s.z * B.j.z;
   res.z = s.x * B.k.x + s.y * B.k.y + s.z * B.k.z;
}

base operator *( const base &A, const base &B ) return res
{
   res.i.x = A.i.x * B.i.x + A.i.y * B.j.x + A.i.z * B.k.x;
   res.i.y = A.i.x * B.i.y + A.i.y * B.j.y + A.i.z * B.k.y;
   res.i.z = A.i.x * B.i.z + A.i.y * B.j.z + A.i.z * B.k.z;
   res.j.x = A.j.x * B.i.x + A.j.y * B.j.x + A.j.z * B.k.x;
   res.j.y = A.j.x * B.i.y + A.j.y * B.j.y + A.j.z * B.k.y;
   res.j.z = A.j.x * B.i.z + A.j.y * B.j.z + A.j.z * B.k.z;
   res.k.x = A.k.x * B.i.x + A.k.y * B.j.x + A.k.z * B.k.x;
   res.k.y = A.k.x * B.i.y + A.k.y * B.j.y + A.k.z * B.k.y;
   res.k.z = A.k.x * B.i.z + A.k.y * B.j.z + A.k.z * B.k.z;
}

base operator %( const base &A, const base &B ) return res
{
   res.i.x = A.i.x * B.i.x + A.i.y * B.i.y + A.i.z * B.i.z;
   res.i.y = A.i.x * B.j.x + A.i.y * B.j.y + A.i.z * B.j.z;
   res.i.z = A.i.x * B.k.x + A.i.y * B.k.y + A.i.z * B.k.z;
   res.j.x = A.j.x * B.i.x + A.j.y * B.i.y + A.j.z * B.i.z;
   res.j.y = A.j.x * B.j.x + A.j.y * B.j.y + A.j.z * B.j.z;
   res.j.z = A.j.x * B.k.x + A.j.y * B.k.y + A.j.z * B.k.z;
   res.k.x = A.k.x * B.i.x + A.k.y * B.i.y + A.k.z * B.i.z;
   res.k.y = A.k.x * B.j.x + A.k.y * B.j.y + A.k.z * B.j.z;
   res.k.z = A.k.x * B.k.x + A.k.y * B.k.y + A.k.z * B.k.z;
}

quaternion operator *( const quaternion &a, const quaternion &b ) return res
{
   res.w = a.w * b.w - a.v * b.v;
   res.v = b.w * a.v + a.w * b.v + b.v % a.v;
}

quaternion operator %( const quaternion &a, const quaternion &b ) return res
{
   res.w = a.w * b.w + a.v * b.v;
   res.v = b.w * a.v - a.w * b.v - b.v % a.v;
}

