
// ------------------------------------------------------------------
// Vector Math Class Library
// (c) 1995 - 2000 Christian Schler
// ------------------------------------------------------------------

#ifndef VECTOR_H
#define VECTOR_H

#include <iostream.h>
#include <stdlib.h>

#include "node.h"

// ------------------------------------------------------------------
// NUMBER + VECTOR
// ------------------------------------------------------------------

typedef float number;

inline number unit_random()
{ return number( rand() & 65535 ) / 65536.0 - 0.5; }

struct vector
{
   number x, y, z;

   vector() {}
   vector( const number x, const number y, const number z )
      : x( x ), y( y ), z( z ) {}

   number quadrat() const { return x * x + y * y + z * z; }
   vector &normalize();
   vector &turn() { x = -x; y = -y; z = -z; return *this; }

   static vector random()
      { return vector( unit_random(), unit_random(), unit_random() ); }
};

inline istream &operator >>( istream &is, vector &v )
{ return is >> v.x >> v.y >> v.z; }

inline ostream &operator <<( ostream &os, const vector &v )
{ return os << v.x << ' ' << v.y << ' ' << v.z; }

inline bool operator ==( const vector &A, const vector &B )
{ return A.x == B.x && A.y == B.y && A.z == B.z; }

number distance( const vector &, const vector & );

// ------------------------------------------------------------------
// HALFSPACE + VOLUME
// ------------------------------------------------------------------

struct halfspace : node
{
   number d;
   vector n;
   halfspace() {}
   halfspace( const number &d, const vector &n )
      : d( d ), n( n ) {}
};

typedef list< halfspace > volume;

// ------------------------------------------------------------------
// BASE + SPACE
// ------------------------------------------------------------------

struct base
{
   vector i, j, k;
   base() {}
   base( const vector &i, const vector &j, const vector &k )
      : i( i ), j( j ), k( k ) {}

   base &spin( const vector & );
};

struct space
{
   vector s;
   base B;
	space() {}
   space( const vector &s, const base &B )
      : s( s ), B( B ) {}
};

typedef space matrix;

inline istream &operator >>( istream &is, base &B )
{ return is >> B.i >> B.j >> B.k; }

inline ostream &operator <<( ostream &os, const base &B )
{ return os << B.i << ' ' << B.j << ' ' << B.k; }

inline istream &operator >>( istream &is, space &spc )
{ return is >> spc.s >> spc.B; }

inline ostream &operator <<( ostream &os, const space &spc )
{ return os << spc.s << ' ' << spc.B; }

// ------------------------------------------------------------------
// QUATERNION
// ------------------------------------------------------------------

struct quaternion
{
   number w;
   vector v;

   quaternion() {};
   quaternion( const number w, const vector &v )
      : w( w ), v( v ) {}

   quaternion( const base & );
   operator base() const;

   quaternion &normalize();
   quaternion &conjugate() { v.turn(); return *this; }
   quaternion &spin( const vector & );
};

inline istream &operator >>( istream &is, quaternion Q )
{ return is >> Q.w >> Q.v; }

inline ostream &operator >>( ostream &os, const quaternion Q )
{ return os << Q.w << ' ' << Q.v; }

// ------------------------------------------------------------------
// EXTERN GLOBAL DEFINITIONS
// ------------------------------------------------------------------

extern vector zerovector;
extern base identitybase;
extern space worldspace;
extern quaternion identityquaternion;

// ------------------------------------------------------------------
// OPERATORS
// ------------------------------------------------------------------

// vector turn
inline vector operator -( const vector &v )
{ return vector( -v.x, -v.y, -v.z ); }

// vector add
inline vector operator +( const vector &a, const vector &b )
{ return vector( a.x + b.x, a.y + b.y, a.z + b.z ); }

inline vector &operator +=( vector &a, const vector &b )
{ a.x += b.x; a.y += b.y; a.z += b.z; return a; }

// vector sub
inline vector operator -( const vector &a, const vector &b )
{ return vector( a.x - b.x, a.y - b.y, a.z - b.z ); }

inline vector &operator -=( vector &a, const vector &b )
{ a.x -= b.x; a.y -= b.y; a.z -= b.z; return a; }

// vector scale
inline vector operator *( const number a, const vector &b )
{ return vector( a * b.x, a * b.y, a * b.z ); }

inline vector &operator *=( vector &a, const number b )
{ a.x *= b; a.y *= b; a.z *= b; return a; }

// vector dot product
inline number operator *( const vector &a, const vector &b )
{ return a.x * b.x + a.y * b.y + a.z * b.z; }

// vector cross product
inline vector operator %( const vector &a, const vector &b )
{ return vector(
     a.y * b.z - a.z * b.y,
     a.z * b.x - a.x * b.z,
     a.x * b.y - a.y * b.x ); }

// vector forward transform by base
vector operator *( const vector &, const base & );

// vector backward transform by base
vector operator %( const vector &, const base & );

// base forward transform by base
base operator *( const base &, const base & );

// base backward transform by base
base operator %( const base &, const base & );

// vector forward transform by space
inline vector operator *( const vector &s, const space &spc )
{ return ( s * spc.B ) + spc.s; }

// vector backward transform by space
inline vector operator %( const vector &s, const space &spc )
{ return ( s - spc.s ) % spc.B; }

// space forward transform by space
inline space operator *( const space &S, const space &P )
{ return space( ( S.s * P.B ) + P.s, S.B * P.B ); }

// space backward transform by space
inline space operator %( const space &S, const space &P )
{ return space( ( S.s - P.s ) % P.B, S.B % P.B ); }

// quaternion add
inline quaternion operator +( const quaternion &a, const quaternion &b )
{ return quaternion( a.w + b.w, a.v + b.v ); }

inline quaternion &operator +=( quaternion &a, const quaternion &b )
{ a.w += b.w; a.v += b.v; return a; }

// quaternion sub
inline quaternion operator -( const quaternion &a, const quaternion &b )
{ return quaternion( a.w - b.w, a.v - b.v ); }

inline quaternion &operator -=( quaternion &a, const quaternion &b )
{ a.w -= b.w; a.v -= b.v; return a; }

// quaternion scale
inline quaternion operator *( const number a, const quaternion &b )
{ return quaternion( a * b.w, a * b.v ); }

// quaternion forward transform by quaternion
quaternion operator *( const quaternion &a, const quaternion &b );

// quaternion backward transform by quaternion
quaternion operator %( const quaternion &a, const quaternion &b );

#endif

