
#include <allegro.h>
#include <fstream.h>
#include <math.h>

#include "game.h"
#include "inputs.h"
#include "synth.h"

const float M_PI = 3.14159265;

namespace synth
{

typedef int sample;
typedef int parameter;
typedef sample buffer[10];

struct sintable
{
   sample y[65536];
   sintable( float, float );
   sample operator[]( unsigned i ) const { return y[ i & 65535 ]; }
};

sintable::sintable( float prec, float f )
{ for( int i = 0; i < 65536; ++i ) y[i] = int( prec * sin( number( i ) * 2 * M_PI / f ) ); }

sintable sinus( 65535, 65536 );

extern sintable sinus;

struct noise
{
   buffer out;

   buffer &pace()
   {
      register int R = rand();
      out[0] = ( R & 32767 ) - 16384;
      out[1] = ( ( R = R * 98734 + 23847 ) & 32767 ) - 16384;
      out[2] = ( ( R = R * 34987 + 23478 ) & 32767 ) - 16384;
      out[3] = ( ( R = R * 23678 + 12389 ) & 32767 ) - 16384;
      out[4] = ( ( R = R * 98732 + 12378 ) & 32767 ) - 16384;
      out[5] = ( ( R = R * 39873 + 38764 ) & 32767 ) - 16384;
      out[6] = ( ( R = R * 23789 + 93844 ) & 32767 ) - 16384;
      out[7] = ( ( R = R * 43876 + 38499 ) & 32767 ) - 16384;
      out[8] = ( ( R = R * 39873 + 37893 ) & 32767 ) - 16384;
      out[9] = ( ( R = R * 39874 + 23873 ) & 32767 ) - 16384;
      return out;
   }
};


struct zero
{
   sample save;
   buffer out;

   zero() { save = 0; }

   buffer &pace( const buffer& in )
   {
      out[0] = in[0] - save;
      out[1] = in[1] - out[0];
      out[2] = in[2] - out[1];
      out[3] = in[3] - out[2];
      out[4] = in[4] - out[3];
      out[5] = in[5] - out[4];
      out[6] = in[6] - out[5];
      out[7] = in[7] - out[6];
      out[8] = in[8] - out[7];
      save = out[9] = in[9] - out[8];
      return out;
   }
};

struct sinustone
{
   int freq;
   int phase;
   buffer out;
   sinustone() { freq = phase = 0; }

   buffer &pace()
   {
      register int ptmp = phase;
      out[0] = sinus.y[ ptmp = ( ptmp + freq ) & 65535 ] >> 1;
      out[1] = sinus.y[ ptmp = ( ptmp + freq ) & 65535 ] >> 1;
      out[2] = sinus.y[ ptmp = ( ptmp + freq ) & 65535 ] >> 1;
      out[3] = sinus.y[ ptmp = ( ptmp + freq ) & 65535 ] >> 1;
      out[4] = sinus.y[ ptmp = ( ptmp + freq ) & 65535 ] >> 1;
      out[5] = sinus.y[ ptmp = ( ptmp + freq ) & 65535 ] >> 1;
      out[6] = sinus.y[ ptmp = ( ptmp + freq ) & 65535 ] >> 1;
      out[7] = sinus.y[ ptmp = ( ptmp + freq ) & 65535 ] >> 1;
      out[8] = sinus.y[ ptmp = ( ptmp + freq ) & 65535 ] >> 1;
      out[9] = sinus.y[ ptmp = ( ptmp + freq ) & 65535 ] >> 1;
      phase = ptmp;
      return out;
   }
};

inline buffer &add( buffer &a, const buffer &b )
{
   a[0] += b[0];
   a[1] += b[1];
   a[2] += b[2];
   a[3] += b[3];
   a[4] += b[4];
   a[5] += b[5];
   a[6] += b[6];
   a[7] += b[7];
   a[8] += b[8];
   a[9] += b[9];
   return a;
};

inline buffer &mul( buffer &a, const buffer &b )
{
   a[0] = ( a[0] * b[0] ) >> 16;
   a[1] = ( a[1] * b[1] ) >> 16;
   a[2] = ( a[2] * b[2] ) >> 16;
   a[3] = ( a[3] * b[3] ) >> 16;
   a[4] = ( a[4] * b[4] ) >> 16;
   a[5] = ( a[5] * b[5] ) >> 16;
   a[6] = ( a[6] * b[6] ) >> 16;
   a[7] = ( a[7] * b[7] ) >> 16;
   a[8] = ( a[8] * b[8] ) >> 16;
   a[9] = ( a[9] * b[9] ) >> 16;
   return a;
};

inline buffer &mul( const parameter a, buffer &b )
{
   b[0] = ( a * b[0] ) >> 16;
   b[1] = ( a * b[1] ) >> 16;
   b[2] = ( a * b[2] ) >> 16;
   b[3] = ( a * b[3] ) >> 16;
   b[4] = ( a * b[4] ) >> 16;
   b[5] = ( a * b[5] ) >> 16;
   b[6] = ( a * b[6] ) >> 16;
   b[7] = ( a * b[7] ) >> 16;
   b[8] = ( a * b[8] ) >> 16;
   b[9] = ( a * b[9] ) >> 16;
   return b;
}

/*
struct pole1
{
   parameter f;
   sample s;
   sample save;
   pole1() { save = s = f = 0; }

   void pace( buffer &out, const buffer &in )
   {
      out[0] = ( s += f * ( in[0] - save   ) ) >> 16;
      out[1] = ( s += f * ( in[1] - out[0] ) ) >> 16;
      out[2] = ( s += f * ( in[2] - out[1] ) ) >> 16;
      out[3] = ( s += f * ( in[3] - out[2] ) ) >> 16;
      out[4] = ( s += f * ( in[4] - out[3] ) ) >> 16;
      out[5] = ( s += f * ( in[5] - out[4] ) ) >> 16;
      out[6] = ( s += f * ( in[6] - out[5] ) ) >> 16;
      out[7] = ( s += f * ( in[7] - out[6] ) ) >> 16;
      out[8] = ( s += f * ( in[8] - out[7] ) ) >> 16;
      save =
      out[9] = ( s += f * ( in[9] - out[8] ) ) >> 16;
   }
};
*/

struct pole2
{
   parameter f;
   parameter q;
   sample s;
   sample v;
   sample save;
   pole2()
   { save = s = v = f = q = 0; for( int i = 0; i < 10; ++i ) out[i] = 0; }
   buffer out;

   buffer &pace( const buffer &in )
   {
      out[0] = ( s += ( v = f * ( in[0] - save ) + q * ( v >> 12 ) ) ) >> 16;
      out[1] = ( s += ( v = f * ( in[1] - out[0] ) + q * ( v >> 12 ) ) ) >> 16;
      out[2] = ( s += ( v = f * ( in[2] - out[1] ) + q * ( v >> 12 ) ) ) >> 16;
      out[3] = ( s += ( v = f * ( in[3] - out[2] ) + q * ( v >> 12 ) ) ) >> 16;
      out[4] = ( s += ( v = f * ( in[4] - out[3] ) + q * ( v >> 12 ) ) ) >> 16;
      out[5] = ( s += ( v = f * ( in[5] - out[4] ) + q * ( v >> 12 ) ) ) >> 16;
      out[6] = ( s += ( v = f * ( in[6] - out[5] ) + q * ( v >> 12 ) ) ) >> 16;
      out[7] = ( s += ( v = f * ( in[7] - out[6] ) + q * ( v >> 12 ) ) ) >> 16;
      out[8] = ( s += ( v = f * ( in[8] - out[7] ) + q * ( v >> 12 ) ) ) >> 16;
      save = out[9] = ( s += ( v = f * ( in[9] - out[8] ) + q * ( v >> 12 ) ) ) >> 16;
      return out;
   }
};

struct disk_read
{
   sample *readbuffer;
   int readindex;
   int readindexref;

   fstream file;
   int scale;

   disk_read( const char *filename, const int scale ) : readbuffer( new int[8000] ),
      file( filename, ios::in + ios::binary ), scale( scale ) { file.unsetf( ios::skipws );
      readindexref = readindex = 0; read( 8000 ); }

   ~disk_read() { file.close(); }

   void read( const int n )
   {
      unsigned char in_lo;
      signed char in_hi;
      for( int i = 0; i < n; ++i )
      {
         file >> in_lo >> in_hi;
         //file >> in_lo >> in_hi;
         readbuffer[ readindexref ] = ( ( in_lo + ( in_hi << 8 ) ) * scale ) >> 16;
         readindexref++; if( readindexref > 8000 ) readindexref = 0;
      }
   }

   void restart()
   {
      file.clear();
      file.seekp( 0 );
      readindexref = readindex = 0; 
      read( 8000 );
   }

   buffer &pace()
   {
      buffer &res = *( (int(*)[10]) (readbuffer + readindex) );
      readindex += 10;
      if( readindex > 7990 ) readindex = 0;
      return res;
   }

   void pace_out_of_timer()
   {
      int n = readindex - readindexref;
      if( n < 0 ) n += 8000;
      read( n );
   }
};

struct disk_write
{
   fstream file;
   disk_write( const char *filename ) : file( filename, ios::out + ios::binary )
      { file.unsetf( ios::skipws ); }

   ~disk_write() { file.close(); }

   void pace( const buffer &in )
   {
      for( int i = 0; i < 10; ++i )
      {
         int out = in[i] > 32767 ? 32767 : ( in[i] < -32768 ) ? -32878 : in[i];
         file << (char) ( out & 255 ) << (char) ( out >> 8 );
      }
   }
};

struct BLIT
{
   int phase;
   int freq;
   buffer out;

   BLIT() { phase = freq = 0; }

   buffer &pace();
};

struct ctriangle_tone
{
   BLIT B1, B2;
   sample s1, s2;
   parameter freq;
   buffer out;

   ctriangle_tone() { freq = s1 = s2 = 0; }

   buffer &pace()
   {
      B1.freq = freq;
      B2.freq = freq;
      B2.phase = ( B1.phase + 32768 ) & 65535;
      B1.pace();
      B2.pace();

      for( int i = 0; i < 10; ++i )
         out[i] =
            freq *
               ( s1 +=
                  ( s2 +=
                     ( ( B1.out[i] - B2.out[i] ) >> 1 ) -
                     ( s2 >> 6 ) ) -
               ( s1 >> 6 ) )
            >> 16;

      return out;
   }
};


struct speaker : BLIT
{
   bool running;
   const char *text;
   int textlength;
   int countdown;
   int i;

   speaker() { running = false; }

   void speak( const char *text )
   {
      this->text = text;
      textlength = strlen( text );
      countdown = i = 0;
      running = true;
   }

   buffer &pace();
};

struct csignal : ctriangle_tone
{
   char *p;
   bool end;
   int pause, F, dF;

   csignal() { reset(); p = 0; end = true; }

   void reset()
      { dF = F = 0; }

   void command( const char *p )
      { this->p = (char*) p; reset(); end = false; }

   buffer &pace();
};

buffer &csignal::pace()
{
   if( pause )
      pause --;

   else
   if( p )
   {
      while( !pause && !end )
      switch( *p )
      {
         case 'f':
         ++p;
         F = strtol( p, &p, 0 );
         break;

         case 'd':
         ++p;
         dF = strtol( p, &p, 0 );
         break;

         case 'p':
         ++p;
         pause = strtol( p, &p, 0 );
         break;

         case 0:
         p = 0;
         F = 0;
         end = true;
         break;

         default:
         ++p;
      }
   }

   if( !end )
      F += dF;

   freq = F >> 8;
   return ctriangle_tone::pace();
}



buffer &BLIT::pace()
{
   if( freq )
   {
      int M = ( int( 32768 / freq ) * 2 ) + 1;
      int a = freq >> 2;
      int ph = phase >> 1;
      int f = freq >> 1;
      int den;

      {
         den = sinus.y[ ph ];
         if( !den ) out[0] = a * M;
         else out[0] = a * sinus.y[ ( M * ph ) & 65535 ] / den;
         den = sinus.y[ ph = ( ph + f ) & 32767 ];
         if( !den ) out[1] = a * M;
         else out[1] = a * sinus.y[ ( M * ph ) & 65535 ] / den;
         den = sinus.y[ ph = ( ph + f ) & 32767 ];
         if( !den ) out[2] = a * M;
         else out[2] = a * sinus.y[ ( M * ph ) & 65535 ] / den;
         den = sinus.y[ ph = ( ph + f ) & 32767 ];
         if( !den ) out[3] = a * M;
         else out[3] = a * sinus.y[ ( M * ph ) & 65535 ] / den;
         den = sinus.y[ ph = ( ph + f ) & 32767 ];
         if( !den ) out[4] = a * M;
         else out[4] = a * sinus.y[ ( M * ph ) & 65535 ] / den;
         den = sinus.y[ ph = ( ph + f ) & 32767 ];
         if( !den ) out[5] = a * M;
         else out[5] = a * sinus.y[ ( M * ph ) & 65535 ] / den;
         den = sinus.y[ ph = ( ph + f ) & 32767 ];
         if( !den ) out[6] = a * M;
         else out[6] = a * sinus.y[ ( M * ph ) & 65535 ] / den;
         den = sinus.y[ ph = ( ph + f ) & 32767 ];
         if( !den ) out[7] = a * M;
         else out[7] = a * sinus.y[ ( M * ph ) & 65535 ] / den;
         den = sinus.y[ ph = ( ph + f ) & 32767 ];
         if( !den ) out[8] = a * M;
         else out[8] = a * sinus.y[ ( M * ph ) & 65535 ] / den;
         den = sinus.y[ ph = ( ph + f ) & 32767 ];
         if( !den ) out[9] = a * M;
         else out[9] = a * sinus.y[ ( M * ph ) & 65535 ] / den;
      }

      phase = ( ( ph + f ) << 1 ) & 65535;
   }

   else
   {
      out[0] = 0;
      out[1] = 0;
      out[2] = 0;
      out[3] = 0;
      out[4] = 0;
      out[5] = 0;
      out[6] = 0;
      out[7] = 0;
      out[8] = 0;
      out[9] = 0;
   }

   return out;
}

buffer &speaker::pace()
{
   switch( running )
   {
      case false:
      break;

      case true:
      if( !countdown )
      {
         char buf[2];
         buf[0] = text[i];
         buf[1] = 0;
         countdown = 5 * text_length( global::BACKSLANT, buf );
         freq = text[i] > 32 ? text[i] - 16 : 0;
         if( freq ) freq = 32768 * 10 / freq;
         i++;
         if( i == textlength ) { running = false; freq = 0; }
      }
      countdown --;
      break;
   }

   return BLIT::pace();
}


#define BUFSIZE 1000

AUDIOSTREAM *stream;

bool musicenable = false;
bool writeenable = false;

disk_read dr( "input.raw", 65536 );
disk_write wr( "output.raw" );
zero ze;
speaker read;
sinustone st;
noise ns;
pole2 filter;
pole2 filter2;
pole2 filter3;
pole2 raindrops;
buffer rainimpulse = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
static int lasta = 0;
csignal sig;

int enginesound = 0;
int intersound = 0;
int windsoundf = 0;
int windsoundv = 0;
int rainsound = 0;

void pace( unsigned short *streambuffer )
{
   int a = 0;
   int b = 0;
   int c = 0;

   filter.f = int( 4 + enginesound );
   if( filter.f > 100000 ) filter.f = 100000;
   filter.q = parameter( 0.5 * sqrt( filter.f / 100000.0 ) * 4096.0 );
   filter.q = parameter( 4096 - 8192 * filter.q / ( 8192 + filter.q ) );
   st.freq = 0;
   a = 20000;
   c = intersound;

   filter3.q = 2500;
   filter3.f = windsoundf;
   if( filter3.f > 100000 ) filter3.f = 100000;
   b = windsoundv;

   raindrops.f = rainsound;
   raindrops.q = parameter( 0.7 * sqrt( raindrops.f / 100000.0 ) * 4096.0 );
   raindrops.q = parameter( 4096 - 8192 * raindrops.q / ( 8192 + raindrops.q ) );

   for( int i = 0; i < BUFSIZE / 10; ++i )
   {
      // message brabbler
      buffer &buf = mul( 16384, read.pace() );

      // signal tone
      add( buf, sig.pace() );

      // engines
      add( buf, filter.pace( mul( a, ns.pace() ) ) );
      add( buf, filter2.pace( mul( c, ns.pace() ) ) );

      // atm-wind
      add( buf, filter3.pace( mul( b, ns.pace() ) ) );

      // rain
      if( rainsound )
      {

         for( int i = 0; i < 10; ++i )
         {
            int a = rand() % 29000;
            a = a * a >> 15;
            a = a * a >> 15;
            a = a * a >> 15;
            a = a * a >> 15;
            rainimpulse[i] = a - lasta;
            lasta = a;
         }

         add( buf, raindrops.pace( rainimpulse ) );
      }

      if( musicenable ) add( buf, dr.pace() );
      if( writeenable ) wr.pace( buf );

      *streambuffer++ = buf[0] + 32768;
      *streambuffer++ = buf[1] + 32768;
      *streambuffer++ = buf[2] + 32768;
      *streambuffer++ = buf[3] + 32768;
      *streambuffer++ = buf[4] + 32768;
      *streambuffer++ = buf[5] + 32768;
      *streambuffer++ = buf[6] + 32768;
      *streambuffer++ = buf[7] + 32768;
      *streambuffer++ = buf[8] + 32768;
      *streambuffer++ = buf[9] + 32768;
   }
}

void audio_update()
{
   unsigned short *streambuffer;
   if( stream )
   if( ( streambuffer = (unsigned short *) get_audio_stream_buffer( stream ) ) )
      { synth::pace( streambuffer );
      free_audio_stream_buffer( stream ); }
}

void init()
{
   read.freq = 0;
   stream = 0;
   filter.q = 3900;
   filter2.f = 50;
   filter2.q = 4050;
   //install_int_ex( pace, BPS_TO_TIMER( int( 22727.0 / 200.0 ) ) );
   reserve_voices( 1, 0 );
   install_sound( DIGI_AUTODETECT, MIDI_NONE, 0 );
   set_volume( 255, 0 );
   stream = play_audio_stream( BUFSIZE, 16, false, 22727, 255, 128 );
   raindrops.f = 20000;
   raindrops.q = 2500;
}

void brabble( const char *c )
{ read.speak( c ); }

void signal( const char *c )
{ sig.command( c ); }

void pace_out_of_timer()
{
   audio_update();
   if( musicenable ) dr.pace_out_of_timer();

   switch( inputs::keypress )
   {
      case 'm':
      if( !musicenable ) { musicenable = true; dr.restart(); }
      else musicenable = false;
      //game::msg( musicenable ? "MUSIC ON" : "MUSIC OFF" );
      break;

      /*
      case 'w':
      writeenable ^= true;
      game::msg( writeenable ? "WRITE ON" : "WRITE OFF" );
      break;
      */
   }
}

}


