
#ifndef ENGINE_H
#define ENGINE_H

#include <allegro.h>

#include "node.h"
#include "vector.h"

#define BSP_MAX_POLYGONS 4

namespace engine
{
extern bool halftone_enable;
extern bool mask_enable;
extern int mask_xanchor;
extern int mask_yanchor;
extern BITMAP *mask_bitmap;
extern bool outline_polygons;

void makecircle( vector *, const int, const vector &,
   const vector &, const vector &, const number = 0 );

void init();
}

// ------------------------------------------------------------------
// MESH
// ------------------------------------------------------------------

struct cview;

struct cmesh
{
   // --- defines ---------------------------------------------------

   enum
   {
      CULLING = 1,
      NORMALS = 2,
      SORTING = 4,
      RELATIVE = 8,
      BSP = 16
   };

   enum sidecode
   { SPLIT, SIDEA, SIDEB, COPLANAR };

   struct polygon3d : node
   {
      unsigned nvertices;
      unsigned polygonnormal_i;
      unsigned vertices_i[4];
      sidecode sc;
   };

   struct BSPspan
   { polygon3d *A, *B, *C, *end; };

   struct BSPnode
   {
      halfspace H;
      polygon3d *start;
      polygon3d *end;
      BSPnode *A;
      BSPnode *B;
   };

   // --- data members ----------------------------------------------

   char name[17];
   number msize;
   int flags;

   unsigned nvertices;
   unsigned nnormals;
   unsigned npolygons;

   unsigned vertexcapacity;
   unsigned polygoncapacity;
   unsigned normalcapacity;

   vector *vertices;
   vector *normals;
   polygon3d *polygons;
   unsigned *colors;

   node anchor;
   BSPnode BSProot;

   // --- push polygon & co -----------------------------------------

   void push_polygon( const int n, const int v1 )
   {
      polygons[ npolygons ].nvertices = 1;
      polygons[ npolygons ].polygonnormal_i = nnormals + n;
      polygons[ npolygons ].vertices_i[0] = nvertices + v1;
      polygons[ npolygons ].insertbefore( &anchor );
      npolygons++;
   }

   void push_polygon( const int n, const int v1, const int v2 )
   {
      polygons[ npolygons ].nvertices = 2;
      polygons[ npolygons ].polygonnormal_i = nnormals + n;
      polygons[ npolygons ].vertices_i[0] = nvertices + v1;
      polygons[ npolygons ].vertices_i[1] = nvertices + v2;
      polygons[ npolygons ].insertbefore( &anchor );
      npolygons++;
   }

   void push_polygon( const int n, const int v1, const int v2, const int v3 )
   {
      polygons[ npolygons ].nvertices = 3;
      polygons[ npolygons ].polygonnormal_i = nnormals + n;
      polygons[ npolygons ].vertices_i[0] = nvertices + v1;
      polygons[ npolygons ].vertices_i[1] = nvertices + v2;
      polygons[ npolygons ].vertices_i[2] = nvertices + v3;
      polygons[ npolygons ].insertbefore( &anchor );
      npolygons++;
   }

   void
   push_polygon( const int n, const int v1, const int v2, const int v3, const int v4 )
   {
      polygons[ npolygons ].nvertices = 4;
      polygons[ npolygons ].polygonnormal_i = nnormals + n;
      polygons[ npolygons ].vertices_i[0] = nvertices + v1;
      polygons[ npolygons ].vertices_i[1] = nvertices + v2;
      polygons[ npolygons ].vertices_i[2] = nvertices + v3;
      polygons[ npolygons ].vertices_i[3] = nvertices + v4;
      polygons[ npolygons ].insertbefore( &anchor );
      npolygons++;
   }

   void push_normal( const int color )
   { colors[ nnormals++ ] = color; }

   void push_normal( const int color, const vector &normal )
   { colors[ nnormals ] = color; normals[ nnormals++ ] = normal; }

   void push_vertex( const vector &vertex )
   { vertices[ nvertices++ ] = vertex; }

   // --- BSP stuff -------------------------------------------------

   sidecode sidecheck( const polygon3d *, const halfspace &, number = 0.00001 );
   void rearrange( BSPspan &, const halfspace &, number );
   halfspace findbest( const polygon3d *, const polygon3d *, number & );
   void BSPcreate( BSPnode & );

   // --- other methods ---------------------------------------------

   void clip( polygon3d *, polygon3d *, const halfspace &, int );

   void draw( cview &, const space &, const volume & );
   void make_normals()
      { generate_normals( normals, polygons, polygons + npolygons, vertices ); }

   static cmesh *load( const char *, number size );
   static void generate_normals( vector *, const polygon3d *, const polygon3d *, const vector * );

   // --- constructors ----------------------------------------------

   cmesh() :
      flags( 0 ),
      nvertices( 0 ),
      nnormals( 0 ),
      npolygons( 0 )
      { anchor.succ = anchor.pred = &anchor; name[0] = 0; }

   cmesh( istream &, const number = 1, const char * = 0 );

   #ifdef PEDANTIC
   bool pedantic_basic( const char *, const polygon3d * ) const;
   #else
   bool pedantic_basic( const char *, const polygon3d * ) const { return false; }
   #endif
};

// ------------------------------------------------------------------
// VIEW
// ------------------------------------------------------------------

#define CLIP_KEEP 256
#define CLIP_SHADOW 512

struct BITMAP;

struct cview : cmesh
{
   BITMAP *framebuffer;

   int self_x, self_y, w, h, blit_x, blit_y;
   int save_w, save_h, blit_w, blit_h;

   number centerx, centery, vx, vy, aspect;
   number focusx, focusy;
   number save_centerx, save_centery, save_vx, save_vy;
   number save_focusx, save_focusy;

   bool sub;

   int polygoncount;
   typedef polygon3d *chandle;

   cview( unsigned, unsigned, unsigned );

   void set_viewport( int, int, int, int, const number );
   void set_sub_viewport( int, int, int, int );
   void restore_viewport();
   void set_focus( const number );
   void blast_onto( BITMAP * );

   void reset() { anchor.succ = anchor.pred = &anchor;
      nvertices = 0; nnormals = 0; npolygons = 0; polygoncount = 0; }

   chandle get_handle() const { return (polygon3d*) anchor.pred; }

   chandle transform( const cmesh &, const space & );
   void cull_backfaces( chandle );
   void cull_backfaces( chandle, const vector & );
   void calc_criterion( chandle );
   void sort_faces( chandle );

   volume get_view_volume() const;
   volume get_view_volume( const number, const number ) const;

   void clip_halfspace( chandle handle, const halfspace &H, const int flags = 0 )
      { cmesh::clip( handle, (polygon3d*) &anchor, H, flags ); }

   void clip_volume( chandle, const volume & );
   void light_direct( chandle, const vector &, const bool = false );

   void sort( chandle );
   void draw();

   #ifdef PEDANTIC
   bool pedantic_screen( const char *, const polygon3d * ) const;
   #else
   bool pedantic_screen( const char *, const polygon3d * ) const { return false; }
   #endif
};

// ------------------------------------------------------------------
// DRAW & PACE - BASE CLASS
// ------------------------------------------------------------------

struct DP : node
{
   virtual ~DP() {}
   virtual void draw( cview &, const space &, const volume & ) {}
   virtual void pace( const number ) {}
};

// ------------------------------------------------------------------
// PORTAL
// ------------------------------------------------------------------

#define PORTAL_MAX_VERTICES 8

struct portal : DP
{
   DP *dest;
   int nvertices;
   vector vertices[ PORTAL_MAX_VERTICES ];
   vector normal;
   portal( istream & );
   void draw( cview &, const space &, const volume & );
   void pace( const number dt ) { if( dest ) dest->pace( dt ); }
};

// ------------------------------------------------------------------
// NEWTON + RIGID BODY
// ------------------------------------------------------------------

struct newton_object : DP, space
{
   vector v, L;
   quaternion Q;
   newton_object();
   newton_object( istream & );
   void pace( const number );
};

struct single_cmesh_newton_object: newton_object, cmesh
{
   single_cmesh_newton_object() {}
   single_cmesh_newton_object( istream & );
   void draw( cview &, const space &, const volume & );
};

typedef single_cmesh_newton_object rigid_body;

// ------------------------------------------------------------------
// WORLD
// ------------------------------------------------------------------

struct cworld : DP
{};

struct cworld_dplist : cworld
{
   list< DP > dplist;

   cworld_dplist() {};
   cworld_dplist( istream & );

   void draw( cview &, const space &, const volume & );
   void pace( const number );
};

#endif
