
#include <allegro.h>
#include <fstream.h>
#include <math.h>
#include <string.h>

#include "engine.h"
#include "synth.h"

const float M_PI = 3.14159265;

//                   . ...
//                ...
// 3d engine     ..
//               __    ___
//               \/ '' | |  ...
//              ######## | #####
//              ##########-#####
//               O o o o O  o o
//
// tuuuut tuuuut pfffffff

#ifdef PEDANTIC
bool cmesh::pedantic_basic( const char *method, const polygon3d* P ) const
{
   if( !P )
      { cerr << method << " in mesh " << name << " - polygon null-pointer\n"; return true; }

   if( P < polygons || P > polygons + npolygons )
      { cerr << method << " in mesh " << name << " - polygon at " << P << " out of range of array\n"; return true; }

   if( !P->nvertices )
      { cerr << method << " in mesh " << name << " - polygon of 0 points should be deleted\n"; return true; }

   if( P->nvertices < 0 || P->nvertices > 4 )
      { cerr << method << " in mesh " << name << " - nvertices out of range: " << P->nvertices << endl; return true; }

   if( P->polygonnormal_i < 0 || P->polygonnormal_i > nnormals )
      { cerr << method << " in mesh " << name << " - polygonnormal index out of range: " << P->polygonnormal_i << endl; return true; }

   for( unsigned u = 0; u < P->nvertices; ++u )
      if( P->vertices_i[u] < 0 || P->vertices_i[u] > nvertices )
         { cerr << method << " in mesh " << name << " - vertex index out of range: " << P->vertices_i << endl; return true; }

   return false;
}

bool cview::pedantic_screen( const char *method, const polygon3d* P ) const
{
   for( unsigned u = 0; u < P->nvertices; ++u )
   {
      number vertex_z = vertices[ P->vertices_i[u] ].z;
      number vertex_x = vertices[ P->vertices_i[u] ].x / vertex_z;
      number vertex_y = vertices[ P->vertices_i[u] ].y / vertex_z;
      vertex_x = centerx + focusx * vertex_x;
      vertex_y = centery + focusy * vertex_y;

      if( vertex_x < self_x || vertex_x > self_x + w + 1 || vertex_y < self_y || vertex_y > self_y + h + 1 )
          { cerr << method << " in mesh " << name << " - vertex out of screen: " << vertex_x << ", " << vertex_y << endl; }
   }

   return false;
}
#endif


BITMAP *halftone;
COLOR_MAP OR_drawing;

namespace engine
{

bool halftone_enable;
bool mask_enable;
int mask_xanchor;
int mask_yanchor;
BITMAP *mask_bitmap;

void init()
{
   halftone_enable = true;
   mask_enable = false;
   mask_xanchor = 0;
   mask_yanchor = 0;
   mask_bitmap = 0;

   halftone = create_bitmap( 32, 32 );
   clear( halftone );
   for( int y = 0; y < 32; ++y )
   for( int x = y & 1; x < 32; x += 2 )
   putpixel( halftone, x, y, 1 );

   for( int y = 0; y < 256; ++y )
   {
      OR_drawing.data[0][y] = 0;
      for( int x = 1; x < 256; ++x )
         OR_drawing.data[y][x] = x | y;
   }

   color_map = &OR_drawing;
}

void makecircle( vector *output, const int npoints,
const vector &a, const vector &b, const vector &m, const number offs )
{
   for( int i = 0; i < npoints; ++i )
   output[i] = m + cos( 2 * M_PI * ( i + offs ) / npoints ) * a + sin( 2 * M_PI * ( i + offs ) / npoints ) * b;
}

}

using namespace engine;

// ------------------------------------------------------------------
// MESH
// ------------------------------------------------------------------

cmesh::cmesh( istream &is, const number size, const char *pname )
{
   anchor.succ = &anchor;
   anchor.pred = &anchor;

   if( pname ) { strncpy( name, pname, 16 ); name[16] = 0; }
   else name[0] = 0;

   flags = 0;
   bool tesselate = false;
   char c;
   is >> c;
   if( c == '-' )
   while( !is.eof() )
   {
      is >> c;
      switch( c )
      {
         case 'c':
         flags |= CULLING;
         break;

         case 'n':
         flags |= NORMALS;
         break;

         case 's':
         flags |= SORTING;
         break;

         case 'r':
         flags |= RELATIVE;
         break;

         case 'b':
         flags |= BSP;
         break;

         case 't':
         tesselate = true;
         break;

         default:
         goto next;
      }
   }

   else
   is.putback( c );

   next:
   is >> nvertices;

   vertexcapacity = flags & BSP ? 30 * nvertices : 2 * nvertices;
   vertices = new vector[ vertexcapacity ];

   for( unsigned i = 0; i < nvertices; )
   {
      is >> c;
      switch( c )
      {
         case 'c':
         is >> c;
         switch( c )
         {
            case 'y':
            {
               int n;
               number h, r;
               is >> n >> h >> r;
               engine::makecircle( vertices + i, n, size * vector( r, 0, 0 ),
                  size * vector( 0, 0, r ), size * vector( 0, h, 0 ) );
               i += n;
            }
            break;

            default:
            {
               is.putback( c );
               int n;
               vector a, b, c;
               is >> n >> a >> b >> c;
               engine::makecircle( vertices + i, n, size * a, size * b, size * c );
               i += n;
            }
            break;
         }
         break;

         default:
         is.putback( c );
         is >> vertices[i];
         vertices[i] *= size;
         i++;
      }
   }

   msize = 0;
   for( unsigned u = 0; u < nvertices; ++u )
   {
      number q = vertices[u].quadrat();
      if( q > msize ) msize = q;
   }
   msize = sqrt( msize );

   is >> npolygons;
   nnormals = npolygons;

   polygoncapacity = flags & BSP ? 30 * npolygons : 2 * npolygons;
   normalcapacity = polygoncapacity;
   polygons = new polygon3d[ polygoncapacity ];
   normals = new vector[ normalcapacity ];
   colors = new unsigned[ normalcapacity ];

   for( unsigned i = 0; i < npolygons; )
   {
      is >> c;
      switch( c )
      {
         case 's':
         {
            unsigned nv, np, ci[8], nc, vi[4], end[4];
            int x[4];
            is >> nv >> np;

            nc = 0;
            while( !is.eof() )
            { is >> c; if( c == '/' ) break; else { is.putback(c); is >> ci[nc]; nc++; } }

            for( unsigned j = 0; j < nv; ++j )
            { x[j] = 0;
              again: is >> c;
              if( c == '>' ) { ++x[j]; goto again; }
              else if( c == '<' ) { --x[j]; goto again; }
              else { is.putback(c); } is >> vi[j];
              end[j] = 0;
              is >> c; if( c == '%' ) is >> end[j]; else is.putback(c);
              end[j] += vi[j] + x[j] * np; }

            int ic = 0;
            for( unsigned j = i; j < i + np; ++j )
            {
               polygons[j].nvertices = nv;
               polygons[j].polygonnormal_i = j;
               for( unsigned k = 0; k < nv; ++k )
                  { polygons[j].vertices_i[k] = vi[k]; vi[k] += x[k];
                  if( vi[k] == end[k] ) vi[k] -= x[k] * np; }
               colors[j] = ci[ ic ];
               ic = ( ++ic ) % nc;
            }

            i += np;
         }
         break;

         case 'd':
         {
            char cc;
            is >> cc;
            bool minus = false;
            if( cc == '-' ) minus = true;
            else { is.putback(cc); }
            unsigned n, c, j;
            is >> n >> c >> j;
            int k = j;
            int ni = i;

            n -= 2;
            while( n )
            {
               colors[i] = c;
               polygons[i].nvertices = 4;
               polygons[i].polygonnormal_i = ni;
               polygons[i].vertices_i[0] = j;
               if( minus ) {
               polygons[i].vertices_i[1] = k + 3;
               polygons[i].vertices_i[2] = k + 2;
               polygons[i].vertices_i[3] = k + 1;
               } else {
               polygons[i].vertices_i[1] = k + 1;
               polygons[i].vertices_i[2] = k + 2;
               polygons[i].vertices_i[3] = k + 3;
               }
               k += 2;
               n -= 2;
               ++i;
            }
         }
         break;

         default:
         is.putback( c );
         is >> colors[i];
         is >> polygons[i].nvertices;
         polygons[i].polygonnormal_i = i;
         for( unsigned j = 0; j < polygons[i].nvertices; ++j )
            is >> polygons[i].vertices_i[j];
         i++;
      }
   }

   // set_gfx_mode( GFX_TEXT, 0, 0, 0, 0 );

   if( tesselate )
   {
      unsigned uend = npolygons;

      //cout << npolygons << " : ";
      for( unsigned u = 0; u < uend; ++u )
      if( polygons[u].nvertices > 3 )
      {
         polygons[u].nvertices = 3;
         polygons[npolygons].nvertices = 3;
         polygons[npolygons].polygonnormal_i = nnormals;
         polygons[npolygons].vertices_i[0] = polygons[u].vertices_i[0];
         polygons[npolygons].vertices_i[1] = polygons[u].vertices_i[2];
         polygons[npolygons].vertices_i[2] = polygons[u].vertices_i[3];
         ++npolygons;
         colors[nnormals] = colors[u];
         ++nnormals;
      }
      //cout << npolygons << endl;
      //readkey();
   }

   bool useless = false;
   is >> c;
   if( c == 'n' ) {
      useless = true;
      for( unsigned i = 0; i < nnormals; ++i ) {
         int j;
         is >> j;
         normals[i] = vertices[j];
         normals[i].normalize(); } }
   else {
      is.putback( c );
      if( flags & NORMALS ) make_normals(); }

   #ifdef PEDANTIC
   if( flags & BSP && !useless )
   for( unsigned u = 0; u < npolygons; ++u )
   {
      if( polygons[u].nvertices > 3 )
      {
         number T = ( vertices[ polygons[u].vertices_i[3] ] -
                    vertices[ polygons[u].vertices_i[0] ] ) *
                    normals[ polygons[u].polygonnormal_i ];
         T = T / sqrt( vertices[ polygons[u].vertices_i[3] ].quadrat() );
         if( T < -1e-6 || T > 1e-6 )
            cerr << "Mesh " << name << " Polygon " << u << " not planar\n";
      }
   }
   #endif


   for( unsigned u = 0; u < npolygons; ++u )
      polygons[u].insertbefore( &anchor );

   if( flags & BSP )
   {
      BSProot.start = (polygon3d*) &anchor;
      BSProot.end = (polygon3d*) &anchor;
      BSPcreate( BSProot );
   }
}

cmesh *cmesh::load( const char *name, number size )
{
   fstream is( "objects.txt", ios::in );
   if( !is ) return 0;

   char buf[80];
   while( !is.eof() )
   {
      is.getline( buf, 80 );
      if( !strcmp( buf, name ) ) { return new cmesh( is, size, name ); }
   }

   return 0;
}

void cmesh::generate_normals( vector *normals, const polygon3d *P,
const polygon3d *P_end, const vector *vertices )
{
   while( P < P_end )
   {
      if( P->nvertices > 2 )
      {
         *normals++ = ( ( vertices[ P->vertices_i[2] ] -
         vertices[ P->vertices_i[0] ] ) % (
         vertices[ P->vertices_i[1] ] -
         vertices[ P->vertices_i[0] ] ) ).normalize();
      }
      ++P;
   }
}

   // --- MESH DRAW -------------------------------------------------

static cmesh::BSPnode *BSPstack[200];
static int BSPcase[200];

void cmesh::draw( cview &view, const space &spc, const volume &vol )
{
   if( flags & BSP )
   {
      // --- BSP traversal here -------------------------------------

      cview::chandle handle;
      unsigned vbase = view.nvertices;
      unsigned nbase = view.nnormals;

      for( unsigned i = 0; i < nvertices; ++i )
         view.vertices[view.nvertices++] = vertices[i] % spc;

      for( unsigned i = 0; i < nnormals; ++i )
      {
         view.normals[view.nnormals] = normals[i] % spc.B;
         view.colors[view.nnormals++] = colors[i];
      }


      BSPnode *bnode = &BSProot;
      int sp = 0;
      handle = view.get_handle();
      goto recycle;

      while( true )
      switch( BSPcase[sp] )
      {
         case 0:
         recycle:
         if( bnode->H.n == zerovector ) goto next;

         if( spc.s * bnode->H.n > bnode->H.d )
         {
            if( bnode->B )
            { BSPcase[sp] = 1; BSPstack[sp++] = bnode; bnode = bnode->B; goto recycle; }

            case 1:
            for( polygon3d *P = (polygon3d*) bnode->start->succ;
                 P != bnode->end;
                 P = (polygon3d*) P->succ )
            {
               pedantic_basic( "BSPdraw1", P );
               view.polygons[view.npolygons].nvertices = P->nvertices;
               view.polygons[view.npolygons].polygonnormal_i = P->polygonnormal_i + nbase;
               view.polygons[view.npolygons].vertices_i[0] = P->vertices_i[0] + vbase;
               view.polygons[view.npolygons].vertices_i[1] = P->vertices_i[1] + vbase;
               view.polygons[view.npolygons].vertices_i[2] = P->vertices_i[2] + vbase;
               view.polygons[view.npolygons].vertices_i[3] = P->vertices_i[3] + vbase;
               view.polygons[view.npolygons].insertbefore( &view.anchor );
               view.npolygons++;
            }

            if( bnode->A )
            { BSPcase[sp] = 3; BSPstack[sp++] = bnode; bnode = bnode->A; goto recycle; }
         }

         else
         {
            if( bnode->A )
            { BSPcase[sp] = 2; BSPstack[sp++] = bnode; bnode = bnode->A; goto recycle; }

            case 2:
            for( polygon3d *P = (polygon3d*) bnode->start->succ;
                 P != bnode->end;
                 P = (polygon3d*) P->succ )
            {
               pedantic_basic( "BSPdraw2", P );
               view.polygons[view.npolygons].nvertices = P->nvertices;
               view.polygons[view.npolygons].polygonnormal_i = P->polygonnormal_i + nbase;
               view.polygons[view.npolygons].vertices_i[0] = P->vertices_i[0] + vbase;
               view.polygons[view.npolygons].vertices_i[1] = P->vertices_i[1] + vbase;
               view.polygons[view.npolygons].vertices_i[2] = P->vertices_i[2] + vbase;
               view.polygons[view.npolygons].vertices_i[3] = P->vertices_i[3] + vbase;
               view.polygons[view.npolygons].insertbefore( &view.anchor );
               view.npolygons++;
            }

            if( bnode->B )
            { BSPcase[sp] = 3; BSPstack[sp++] = bnode; bnode = bnode->B; goto recycle; }
         }

         case 3:
         next:
         if( !sp ) goto out;
         bnode = BSPstack[--sp];
      }

      out:
      if( flags & CULLING ) view.cull_backfaces( handle );
      view.clip_volume( handle, vol );
   }

   else
   {
      // --- ordinary drawing here ----------------------------------

      cview::chandle handle = view.transform( *this, spc );
      if( flags & CULLING ) view.cull_backfaces( handle );
      view.clip_volume( handle, vol );
   }

   return;
}

   // ---------------------------------------------------------------
   // BSP STUFF
   // ---------------------------------------------------------------

cmesh::sidecode cmesh::sidecheck( const polygon3d *Q, const halfspace &H,
    number epsilon )
{
   unsigned A = 0, B = 0, C = 0;

   for( unsigned i = 0; i < Q->nvertices; ++i )
   {
      number T = H.n * vertices[ Q->vertices_i[i] ] - H.d;
      if( ( T < epsilon ) && ( T > -epsilon ) ) C++;
      else if( T > 0 ) A++;
      else B++;
   }

   // cout << " A: " << A << " B: " << B << " C: " << C << endl;

   if( A && B ) return SPLIT;
   else if( A ) return SIDEA;
   else if( B ) return SIDEB;
   else return COPLANAR;
}

void cmesh::rearrange( BSPspan &span, const halfspace &H, number epsilon )
{
   #define SWAP(P,Q) \
   { \
      node *TMP2 = (Q)->pred; \
      \
      polygon3d PTMP = *P; \
      *P = *(polygon3d*)TMP2; \
      *(polygon3d*)TMP2 = PTMP; \
      \
      node NTMP = *(node*)P; \
      *(node*) P = *TMP2; \
      *TMP2 = NTMP; \
   }

   polygon3d *recover = (polygon3d*) span.A->pred;
   polygon3d *P;
   polygon3d *Q;

   P = (polygon3d*) recover->succ;
   Q = span.end;

   while( P != Q )
   {
      P->sc = sidecheck( P, H, epsilon );

      if( P->sc == SPLIT )
      {
         polygon3d *save = (polygon3d*) P->pred;
         clip( (polygon3d*) P->pred, (polygon3d*) P->succ, H, CLIP_KEEP );
         P = save;

         /*
         nexttry:
         polygon3d *PP = (polygon3d*) save->succ;
         while( PP != (polygon3d*) P->succ )
         {
            PP->sc = sidecheck( PP, H, epsilon );
            if( PP->sc == SPLIT ) {
               cerr << "Split in split in mesh " << name << endl;
               epsilon *= 10; goto nexttry; }
            PP = (polygon3d*) PP->succ;
         }
         */
      }

      P = (polygon3d*) P->succ;
   }

   P = (polygon3d*) recover->succ;
   span.A = P;
   Q = span.end;

   while( true )
   {
      while( P->sc != SIDEB && P != Q ) P = (polygon3d*) P->succ;
      while( ((polygon3d*)Q->pred)->sc == SIDEB ) { Q = (polygon3d*) Q->pred; if( P == Q ) break; }
      if( P == Q ) break;
      SWAP( P, Q );
   }

   span.B = P;
   P = (polygon3d*) recover->succ;

   while( true )
   {
      while( P->sc != COPLANAR && P != Q ) P = (polygon3d*) P->succ;
      while( ((polygon3d*)Q->pred)->sc == COPLANAR ) { Q = (polygon3d*) Q->pred; if( P == Q ) break; }
      if( P == Q ) break;
      SWAP( P, Q );
   }

   span.C = P;
}

halfspace cmesh::findbest( const polygon3d *start, const polygon3d *end,
                           number &epsilon )
{
   recycle:
   halfspace resultH;
   int score = 1 << 30;
   unsigned osp = 0;
   unsigned osa = 0;
   unsigned osb = 0;
   unsigned oco = 0;

   const polygon3d *P = start;
   while( P != end )
   {
      if( P->nvertices > 2 )
      {
         halfspace H;
         H.n = ( ( vertices[ P->vertices_i[2] ] - vertices[ P->vertices_i[0] ] ) %
               ( vertices[ P->vertices_i[1] ] - vertices[ P->vertices_i[0] ] ) );
         number q = sqrt( H.n.quadrat() );
         if( q < epsilon * epsilon ) goto next;
         H.n = ( 1 / q ) * H.n;
         H.d = vertices[ P->vertices_i[0] ] * H.n;
      
         unsigned sp = 0;
         unsigned sa = 0;
         unsigned sb = 0;
         unsigned co = 0;

         const polygon3d *Q = start;
         while( Q != end )
         {
            int sc = sidecheck( Q, H, epsilon ) ;
            //cout << sc << ' ';
            switch( sc )
            {
               case SPLIT: sp++; break;
               case SIDEA: sa++; break;
               case SIDEB: sb++; break;
               case COPLANAR: co++; break;
            }
            Q = (polygon3d*) Q->succ;
         }
         // do not split is priority in this BSP
         //cout << endl;

         int scorethis =
            int( sa - sb ) * int( sa - sb ) + 65536 * sp * sp - co * co;
      
         if( ( scorethis < score ) )
         {
            osp = sp;
            osa = sa;
            osb = sb;
            oco = co;
            resultH = H;
            score = scorethis;
         }
      }

      next:
      P = (polygon3d*) P->succ;
   }

   /*
   cout << endl;
   cout << "Findbest " << epsilon  << endl;
   cout << resultH.n << " " << resultH.d << endl;
   cout << " " << osp << " " << osa << " " << osb << " " << oco << endl;
   readkey();
   */

   if( !( osp + osa + osb + oco ) )
      return halfspace( 0, zerovector );

   if( !oco )
      { epsilon *= 10; goto recycle; }

   return resultH;
}

// the big one...
void cmesh::BSPcreate( BSPnode &bnode )
{
   /*
   set_gfx_mode( GFX_TEXT, 0, 0, 0, 0 );
   cout << "BSP CREATE " << name << endl;
   cout << endl;
   */

   BSPspan span;
   span.A = (polygon3d*) bnode.start->succ;
   span.end = bnode.end;

   number epsilon = 1e-6 * msize;
   bnode.H = findbest( span.A, span.end, epsilon );

   if( bnode.H.n == zerovector )
   {
      bnode.A = 0;
      bnode.B = 0;
      bnode.end = (polygon3d*) bnode.start->succ;
      return;
   }

   rearrange( span, bnode.H, epsilon );
   bnode.start = (polygon3d*) span.C->pred;
   bnode.end = span.B;

   if( span.C != span.A )
   {
      bnode.A = new BSPnode;
      bnode.A->start = (polygon3d*) span.A->pred;
      bnode.A->end = span.C;
      BSPcreate( *bnode.A );
   }
   else
      bnode.A = 0;

   if( span.end != span.B )
   {
      bnode.B = new BSPnode;
      bnode.B->start = (polygon3d*) span.B->pred;
      bnode.B->end = span.end;
      BSPcreate( *bnode.B );
   }
   else
      bnode.B = 0;
}


// ------------------------------------------------------------------
// VIEW
// ------------------------------------------------------------------

cview::cview( const unsigned nvertices,
              const unsigned nnormals,
              const unsigned npolygons )
{
   vertexcapacity = nvertices;
   vertices = new vector[ nvertices ];
   polygoncapacity = npolygons;
   polygons = new polygon3d[ npolygons ];
   normalcapacity = nnormals;
   normals = new vector[ nnormals ];
   colors = new unsigned[ nnormals ];
   reset();
   framebuffer = 0;
}

void cview::set_viewport(
   int px,
   int py,
   int pw,
   int ph,
   const number paspect )
{
   strcpy( name, "cview" );

   blit_x = px;
   blit_y = py;
   blit_w = pw;
   blit_h = ph;
   self_x = 0;
   self_y = 0;
   w = pw;
   h = ph;
   centerx = 0.5 + pw / 2;
   centery = 0.5 + ph / 2;
   vx = pw / 2;
   vy = ph / 2;
   aspect = paspect;
   sub = false;

   if( framebuffer ) destroy_bitmap( framebuffer );
   framebuffer = create_bitmap( w + 1, h + 1 );
   clear( framebuffer );
   #ifdef CLIPPING
   set_clip( framebuffer, 0, 0, 0, 0 );
   #endif
}

void cview::set_sub_viewport(
   int px,
   int py,
   int pw,
   int ph )
{
   if( !sub )
   {
      save_w = w;
      save_h = h;
      save_centerx = centerx;
      save_centery = centery;
      save_vx = vx;
      save_vy = vy;
      save_focusx = focusx;
      save_focusy = focusy;
      sub = true;
   }

   self_x = px;
   self_y = py;
   w = pw;
   h = ph;
   centerx = 0.5 + px + pw / 2;
   centery = 0.5 + py + ph / 2;
   vx = pw / 2;
   vy = ph / 2;
}

void cview::restore_viewport()
{
   if( sub )
   {
      self_x = 0;
      self_y = 0;
      w = save_w;
      h = save_h;
      centerx = save_centerx;
      centery = save_centery;
      vx = save_vx;
      vy = save_vy;
      focusx = save_focusx;
      focusy = save_focusy;
      sub = false;
   }
}

void cview::set_focus( const number F )
{
   focusx = w * F / 2;
   focusy = focusx * aspect;
}

void cview::blast_onto( BITMAP *dest )
{
   blit(
      framebuffer,
      dest,
      0, 0,
      blit_x, blit_y,
      blit_w, blit_h
   );
}

   // ---------------------------------------------------------------
   // TRANSFORMATION
   // ---------------------------------------------------------------

cview::chandle cview::transform( const cmesh &sourcemesh,
const space &spc )
{
   polygon3d * const P_return = (polygon3d*) anchor.pred;

   {
      const polygon3d *P = sourcemesh.polygons;
      const polygon3d * const P_end = P + sourcemesh.npolygons;
      polygon3d *P_dest = polygons + npolygons;
      while( P < P_end )
      {
         sourcemesh.pedantic_basic( "cview::transform", P );
         P_dest->nvertices = P->nvertices;
         P_dest->polygonnormal_i = P->polygonnormal_i + nnormals;
         P_dest->vertices_i[0] = P->vertices_i[0] + nvertices;
         P_dest->vertices_i[1] = P->vertices_i[1] + nvertices;
         P_dest->vertices_i[2] = P->vertices_i[2] + nvertices;
         P_dest->vertices_i[3] = P->vertices_i[3] + nvertices;
         P_dest->insertbefore( &anchor );
         ++P;
         ++P_dest;
      }
      npolygons += sourcemesh.npolygons;
   }

   {
      const vector *v = sourcemesh.vertices;
      const vector * const v_end = v + sourcemesh.nvertices;
      vector *v_dest = vertices + nvertices;
      while( v < v_end ) *v_dest++ = *v++ % spc;
      nvertices += sourcemesh.nvertices;
   }

   if( sourcemesh.flags & NORMALS )
   {
      const vector *n = sourcemesh.normals;
      const vector * const n_end = n + sourcemesh.nnormals;
      vector *n_dest = normals + nnormals;
      while( n < n_end ) *n_dest++ = *n++ % spc.B;
   }

   {
      const unsigned *c = sourcemesh.colors;
      const unsigned * const c_end = c + sourcemesh.nnormals;
      unsigned *c_dest = colors + nnormals;
      while( c < c_end ) *c_dest++ = *c++;
   }
   nnormals += sourcemesh.nnormals;

   return P_return;
}

   // ---------------------------------------------------------------
   // BACKFACE CULLING
   // ---------------------------------------------------------------

void cview::cull_backfaces( cview::chandle P )
{
   P = (polygon3d*) P->succ;
   while( P != (polygon3d*) &anchor )
   {
      if( P->nvertices > 2 ) {
         if( normals[ P->polygonnormal_i ] * vertices[ *P->vertices_i ] > 0 )
            { P->nvertices = 0; P->pop(); } }
      P = (polygon3d*) P->succ;
   }
}

void cview::cull_backfaces( cview::chandle P, const vector &ref )
{
   P = (polygon3d*) P->succ;
   while( P != (polygon3d*) &anchor )
   {
      if( P->nvertices > 2 ) {
         if( normals[ P->polygonnormal_i ] * ref > 0 )
            { P->nvertices = 0; P->pop(); } }
      P = (polygon3d*) P->succ;
   }
}

   // ---------------------------------------------------------------
   // LIGHTING
   // ---------------------------------------------------------------

void cview::light_direct(
   cview::chandle P,
   const vector &light,
   const bool rel )
{
   P = (polygon3d*) P->succ;

   if( rel )
   while( P != (polygon3d*) &anchor )
   {
      if( P->nvertices > 2 && !( colors[ P->polygonnormal_i ] & 1024 ) )
      {
         number q = ( light * normals[ P->polygonnormal_i ] )
           * ( vertices[ P->vertices_i[0] ] * normals[ P->polygonnormal_i ] );
    
         if( q < 0 ) colors[ P->polygonnormal_i ] |= 128;
      }

      P = (polygon3d*) P->succ;
   }

   else
   while( P != (polygon3d*) &anchor )
   {
      if( P->nvertices > 2 && !( colors[ P->polygonnormal_i ] & 1024 ) )
      if( light * normals[ P->polygonnormal_i ] > 0 )
      colors[ P->polygonnormal_i ] |= 128;

      P = (polygon3d*) P->succ;
   }
}

   // ---------------------------------------------------------------
   // SORTING
   // ---------------------------------------------------------------

void cview::calc_criterion( cview::chandle )
{}

/*
   cmesh::polygon3d *P = P_begin;
   const cmesh::polygon3d * const P_end = polygons + npolygons;
   while( P < P_end )
   {
      if( P->nvertices )
      {
         vector x = zerovector;
         for( unsigned i = 0; i < P->nvertices; ++i ) x += vertices[ P->vertices_i[i] ];
         P->criterion = ( 1.0 / P->nvertices / P->nvertices ) * x.quadrat();
      }
      ++P;
   }
}
*/

void cview::sort_faces( cview::chandle )
{}

   // ------------------------------------------------------------
   // VIEW VOLUME
   // ------------------------------------------------------------

volume cview::get_view_volume() const
{
   volume vol;
   vol.push( new halfspace( 0, vector( focusx, 0, vx ) ) );
   vol.push( new halfspace( 0, vector( - focusx, 0, vx ) ) );
   vol.push( new halfspace( 0, vector( 0, focusy, vy ) ) );
   vol.push( new halfspace( 0, vector( 0, - focusy, vy ) ) );
   return vol;
}

volume cview::get_view_volume( const number px, const number py ) const
{
   volume vol;
   vol.push( new halfspace( 0, vector( focusx, 0, px ) ) );
   vol.push( new halfspace( 0, vector( - focusx, 0, px ) ) );
   vol.push( new halfspace( 0, vector( 0, focusy, py ) ) );
   vol.push( new halfspace( 0, vector( 0, - focusy, py ) ) );
   return vol;
}

   // ---------------------------------------------------------------
   // CLIPPING
   // ---------------------------------------------------------------

static int counter = 20;

void
cmesh::clip( polygon3d *start,
             polygon3d *end,
             const halfspace &H,
             const int flags )
{
   #define FINISH( POLY, NVERTICES ) \
   { \
      (POLY)->nvertices = (NVERTICES); \
      (POLY)->polygonnormal_i = P->polygonnormal_i; \
      (POLY)->vertices_i[0] = vertices_i[0];\
      (POLY)->vertices_i[1] = vertices_i[1];\
      (POLY)->vertices_i[2] = vertices_i[2];\
      (POLY)->vertices_i[3] = vertices_i[3];\
   }
   
   #define FINISHKEEP( POLY, NVERTICES ) \
   { \
      (POLY)->nvertices = (NVERTICES); \
      if( flags & CLIP_SHADOW ) \
      {  \
         colors[ nnormals ] = colors[ P->polygonnormal_i ] | ( ( ( colors[P->polygonnormal_i] & 1024 ) >> 3 ) ^ 128 ); \
         normals[ nnormals ] = normals[ P->polygonnormal_i ]; \
         (POLY)->polygonnormal_i = nnormals; \
         ++nnormals; \
      } \
      else \
         (POLY)->polygonnormal_i = P->polygonnormal_i + ( flags & 255 ); \
      (POLY)->vertices_i[0] = vertices_ikeep[0];\
      (POLY)->vertices_i[1] = vertices_ikeep[1];\
      (POLY)->vertices_i[2] = vertices_ikeep[2];\
      (POLY)->vertices_i[3] = vertices_ikeep[3];\
   }
   
   #define INSERT( POINT ) \
   { \
      if( nvertices == 4 ) \
      { \
         FINISH( P_new, 4 ); \
         P_new->insertbefore( P ); \
         ++P_new; \
         ++npolygons; \
         vertices_i[1] = vertices_i[3]; \
         vi_dest = vertices_i + 2; \
         nvertices = 2; \
      } \
      \
      *vi_dest++ = (POINT); \
      ++nvertices; \
      if( nvertices < 0 || nvertices > 4 ) \
         { cerr << "clip - internal nvertices out of range: " << nvertices << endl; } \
      if( vi_dest < vertices_i || vi_dest > vertices_i + 4 ) \
         { cerr << "clip - internal vi_dest out of range: " << vi_dest << endl; } \
   }
   
   #define INSERTKEEP( POINT ) \
   { \
      if( nverticeskeep == 4 ) \
      { \
         FINISHKEEP( P_new, 4 ); \
         P_new->insertbefore( P ); \
         ++P_new; \
         ++npolygons; \
         vertices_ikeep[1] = vertices_ikeep[3]; \
         vi_destkeep = vertices_ikeep + 2; \
         nverticeskeep = 2; \
      } \
      \
      *vi_destkeep++ = (POINT); \
      ++nverticeskeep; \
      if( nverticeskeep < 0 || nverticeskeep > 4 ) \
         { cerr << "clip - internal nverticeskeep out of range: " << nverticeskeep << endl; } \
      if( vi_destkeep < vertices_ikeep || vi_destkeep > vertices_ikeep + 4 ) \
         { cerr << "clip - internal vi_destkeep out of range: " << vi_destkeep << endl; } \
   }

   unsigned vertices_i[4];
   unsigned vertices_ikeep[4];
   unsigned nvertices = 0;
   unsigned nverticeskeep = 0;

   vector *v_new = vertices + this->nvertices;
   cmesh::polygon3d *P_new = polygons + npolygons;

   polygon3d *P = (polygon3d*) start->succ;
   while( P != end )
   {
      pedantic_basic( "cmesh::clip", P );

      if( !P->nvertices ) goto next;

      if( P->nvertices == 1 )
      {
         if( ( vertices[ P->vertices_i[0] ] * H.n - H.d ) <= 0 )
         P->nvertices = 0;
         P->pop();
      }

      else if( P->nvertices == 2 )
      {
         vector *As = vertices + P->vertices_i[0];
         vector *Bs = vertices + P->vertices_i[1];
         number ASN = *As * H.n - H.d;
         number BSN = *Bs * H.n - H.d;
         bool sideA = ASN > 0;
         bool sideB = BSN > 0;

         if( (!sideA) && (!sideB) )
         {
            P->nvertices = 0;
            P->pop();
         }

         else if( sideA ^ sideB )
         {
            if( sideA )
            {
               vector ds = *Bs - *As;
               number BAN = ds * H.n;
               if( BAN )
               {
               /*
                  if( false && flags & CLIP_KEEP )
                  {
                     P_new->nvertices = 2;
                     P_new->polygonnormal_i = P->polygonnormal_i;
                     P_new->vertices_i[0] = this->nvertices;
                     P_new->vertices_i[1] = P->vertices_i[1];
                     //P_new->insertbefore( P );
                     //++P_new;
                     //++npolygons;
                  }
                  */

                  P->vertices_i[1] = this->nvertices;

                  number q = BSN / BAN;
                  *v_new++ = *Bs - q * ds;
                  this->nvertices++;
               }
            }

            else
            {
               vector ds = *As - *Bs;
               number ABN = ds * H.n;
               if( ABN )
               {
               /*
                  if( false && flags & CLIP_KEEP )
                  {
                     P_new->nvertices = 2;
                     P_new->polygonnormal_i = P->polygonnormal_i;
                     P_new->vertices_i[0] = P->vertices_i[0];
                     P_new->vertices_i[1] = this->nvertices;
                     //P_new->insertbefore( P );
                     //++P_new;
                     //++npolygons;
                  }
                  */

                  P->vertices_i[0] = this->nvertices;

                  number q = ASN / ABN;
                  *v_new++ = *As - q * ds;
                  this->nvertices++;
               }
            }
         }
      }

      else if( P->nvertices > 2 )
      {
         nvertices = 0;
         nverticeskeep = 0;
         unsigned *vi = P->vertices_i;
         unsigned *vi_end = vi + P->nvertices;
         unsigned *vi_dest = vertices_i;
         unsigned *vi_destkeep = vertices_ikeep;
         vector *As = vertices + *( vi_end - 1 );
         bool sideA = *As * H.n - H.d > 0;
         while( vi < vi_end )
         {
            vector *Bs = vertices + *vi;
            number BSN = *Bs * H.n - H.d;
            bool sideB = BSN > 0;

            if( sideA ^ sideB )
            {
               vector ds = *Bs - *As;
               number BAN = ds * H.n;
               if( BAN )
               {
                  number q = BSN / BAN;
                  *v_new++ = *Bs - q * ds;
                  INSERT( this->nvertices )
                  if( flags & CLIP_KEEP ) INSERTKEEP( this->nvertices )
                  ++this->nvertices;
               }
            }

            if( sideB ) INSERT( *vi )
            else if( flags & CLIP_KEEP ) INSERTKEEP( *vi );

            As = Bs;
            sideA = sideB;
            ++vi;
         }

         if( flags & CLIP_KEEP )
         if( nverticeskeep > 2 )
         {
            FINISHKEEP( P_new, nverticeskeep );
            P_new->insertbefore( P );
            ++P_new;
            ++npolygons;
         }

         if( nvertices > 2 ) FINISH( P, nvertices )
         else { P->nvertices = 0; P->pop(); }
      }

      next:
      #ifdef PEDANTIC
      if( this->nvertices > vertexcapacity )
         { cerr << "cmesh::clip in mesh " << name << " running short of vertex space: " << nvertices << endl; }
      if( npolygons > polygoncapacity )
         { cerr << "cmesh::clip in mesh " << name << " running short of polygon space: " << npolygons << endl; }
      if( nnormals > normalcapacity )
         { cerr << "cmesh::clip in mesh " << name << " running short of normal space: " << nnormals << endl; }
      #endif

      P = (polygon3d *) P->succ;
   }

   #undef FINISH
   #undef INSERT
   #undef FINISHKEEP
   #undef INSERTKEEP
}

void cview::clip_volume( cview::chandle handle, const volume &vol )
{
   counter--;
   if( counter ) { counter = 20; synth::audio_update(); }

   for( pointer< halfspace > p = vol.begin(); !p.finish(); ++p )
   clip_halfspace( handle, *p );
}

   // ---------------------------------------------------------------
   // RENDERINGELINGELING
   // ---------------------------------------------------------------

static V3D_f drawbuffer[ 4 ];

/*
static V3D_f *drawbufferp[] =
{ drawbuffer, drawbuffer + 1, drawbuffer + 2, drawbuffer + 3 };
*/


namespace engine {
  static bool outline_polygons = false;
}

static void draw_polygon( BITMAP *framebuffer, unsigned nvertices )
{
   #define INC( X ) (X)=(X)++; if((X)>=nvertices) (X)=0;
   #define DEC( X ) (X)=(X)--; if((X)>=nvertices) (X)=nvertices-1;

   // #define PRINT

   if( nvertices < 3 ) return;
   if( nvertices > 10 ) return;

   unsigned Ai = 0, Bi = 0;
   unsigned An = 0, Bn = 0;
   unsigned Acount = nvertices, Bcount = nvertices;
   number Ax = 0, Bx = 0;
   number Ay = 0, By = 0;
   number Adx = 0, Bdx = 0;
   number Ady = 0, Bdy = 0;
   number Aincx = 0, Bincx = 0;

   int Aix = 0, Bix = 0;
   int Aiincx = 0, Biincx = 0;
   int Aixx = 0, Bixx = 0;
   int Aiincxx = 0, Biincxx = 0;

   int curry = 0;
   number ymin = 10000;

   for( unsigned u = 0; u < nvertices; ++u )
      if( drawbuffer[u].y < ymin )
         { ymin = drawbuffer[u].y; Ai = u; Bi = u; }

   curry = int( ymin );
   An = Bn = 0;

   #ifdef PRINT
      cerr << "Polygon: " << nvertices << endl;
   #endif

   do
   {
      if( !An ) {
         Ax = drawbuffer[ Ai ].x;
         Ay = drawbuffer[ Ai ].y;
         #ifdef PRINT
            cerr << "Ax[" << Ai <<"] = " << Ax << " Ay[" << Ai << "] = " << Ay << endl;
         #endif
         INC( Ai );
         --Acount;
         Ady = drawbuffer[ Ai ].y - Ay;
         if( !Acount || Ady < 0 ) break;
         An = int( drawbuffer[ Ai ].y ) - curry;
         if( An ) {
            Adx = drawbuffer[ Ai ].x - Ax;
            Ax = Ax + Adx * ( ceil( Ay ) - Ay ) / Ady;
            Aincx = Adx / Ady;
            #ifdef PRINT
               cerr << "Adx = " << Adx << " Ady = " << Ady << endl;
               cerr << "advance Ai = " << Ai << endl;
               cerr << "new Ax = " << Ax;
               cerr << " Aincx = " << Aincx;
               cerr << " An = " << An << endl;
            #endif
            Aix = int( floor( Ax ) );
            Aiincx = int( floor( Aincx ) );
            Aixx = int( ( Ax - floor( Ax ) ) * 65536 );
            Aiincxx = int( ( Aincx - floor( Aincx ) ) * 65536 );
         }
      }

      if( !Bn ) {
         Bx = drawbuffer[ Bi ].x;
         By = drawbuffer[ Bi ].y;
         #ifdef PRINT
            cerr << "Bx[" << Bi << "] = " << Bx << " By[" << Bi << "] = " << By << endl;
         #endif
         DEC( Bi );
         --Bcount;
         Bdy = drawbuffer[ Bi ].y - By;
         if( !Bcount || Bdy < 0 ) break;
         Bn = int( drawbuffer[ Bi ].y ) - curry;
         if( Bn ) {
            Bdx = drawbuffer[ Bi ].x - Bx;
            Bx = Bx + Bdx * ( ceil( By ) - By ) / Bdy;
            Bincx = Bdx / Bdy;
            #ifdef PRINT
               cerr << "Bdx = " << Bdx << " Bdy = " << Bdy << endl;
               cerr << "advance Bi = " << Bi << endl;
               cerr << "new Bx = " << Bx;
               cerr << " Bincx = " << Bincx;
               cerr << " Bn = " << Bn << endl;
            #endif
            Bix = int( floor( Bx ) );
            Biincx = int( floor( Bincx ) );
            Bixx = int( ( Bx - floor( Bx ) ) * 65536 );
            Biincxx = int( ( Bincx - floor( Bincx ) ) * 65536 );
         }
      }

      if( An && Bn ) {
         #ifdef PRINT
            cerr << "LINE " << int( Ax ) << " - " << int( Bx ) << endl;
         #endif


         if( Bix > Aix ) hline( framebuffer, Aix, curry, Bix-1, drawbuffer[0].c );
         else if( Bix < Aix ) hline( framebuffer, Bix, curry, Aix-1, drawbuffer[0].c );

         if( engine::outline_polygons ) {
            _putpixel( framebuffer, Aix, curry, 5 );//drawbuffer[0].c );
            _putpixel( framebuffer, Bix, curry, 5 );//drawbuffer[0].c );
         }

         Aix += Aiincx;
         Aixx += Aiincxx;
         if( Aixx >= 65536 ) { Aixx -= 65536; ++Aix; }

         Bix += Biincx;
         Bixx += Biincxx;
         if( Bixx >= 65536 ) { Bixx -= 65536; ++Bix; }

         ++curry;
         --An;
         --Bn;
      }
   }
   while( true );

   #undef INC;
   #undef DEC;
}

void cview::draw()
{
   const polygon3d *P = (polygon3d *) anchor.succ;

   while( P != (polygon3d *) &anchor )
   {
      #ifdef PEDANTIC
      pedantic_basic( "cview::draw", P );
      pedantic_screen( "cview::draw", P );
      #endif

      if( !P->nvertices ) goto next;

      if( colors[ P->polygonnormal_i ] & 256 )
      { if( halftone_enable )
        { if( mask_enable )
            drawing_mode( DRAW_MODE_MASKED_PATTERN, mask_bitmap,
              mask_xanchor, mask_yanchor );
          else
            drawing_mode( DRAW_MODE_MASKED_PATTERN, halftone, 0, 0 );
        }
        else goto next;
      }

      else if( colors[ P->polygonnormal_i ] & 512 )
        drawing_mode( DRAW_MODE_TRANS, 0, 0, 0 );

      else
      { if( mask_enable )
          drawing_mode( DRAW_MODE_MASKED_PATTERN, mask_bitmap,
            mask_xanchor, mask_yanchor );
        else
          drawing_mode( DRAW_MODE_SOLID, 0, 0, 0 );
      }

      switch( P->nvertices )
      {
         case 1:
         {
            polygoncount++;
            const vector &vertex = vertices[ P->vertices_i[0] ];
            if( vertex.z == 0 ) goto next;
            number z = 1 / vertex.z;
            putpixel( framebuffer, int( centerx + focusx * z * vertex.x ),
               int( centery + focusy * z * vertex.y ), colors[ P->polygonnormal_i ] );
         }

         break;

         case 2:
         {
            polygoncount++;
            const vector &vertex0 = vertices[ P->vertices_i[0] ];
            const vector &vertex1 = vertices[ P->vertices_i[1] ];
            if( vertex0.z == 0 || vertex1.z == 0 ) goto next;
            number z0 = 1 / vertex0.z;
            number z1 = 1 / vertex1.z;
            line( framebuffer,
               int( centerx + focusx * z0 * vertex0.x ),
               int( centery + focusy * z0 * vertex0.y ),
               int( centerx + focusx * z1 * vertex1.x ),
               int( centery + focusy * z1 * vertex1.y ),
               colors[ P->polygonnormal_i ] );
         }
         break;

         default:
         // draw a polygon
         {
            polygoncount++;
            counter--;
            if( counter ) { counter = 20; synth::audio_update(); }

            const unsigned *vi = P->vertices_i;
            const unsigned * const vi_end = vi + P->nvertices;
            V3D_f *dest = drawbuffer;
   
            dest->c = colors[ P->polygonnormal_i ] & 255;
            while( vi < vi_end )
            {
               const vector &vertex = vertices[ *vi ];
               if( !vertex.z ) goto next;
               number z = 1 / vertex.z;
               dest->z = vertex.z;
               dest->x = centerx + focusx * z * vertex.x;
               dest->y = centery + focusy * z * vertex.y;
               ++dest;
               ++vi;
            }

            draw_polygon( framebuffer, P->nvertices );
         }
      }
      next:
      P = (polygon3d *) P->succ;
   }
}

// ---------------------------------------------------------------
// PORTAL
// ---------------------------------------------------------------

portal::portal( istream &is )
{
   is >> nvertices;
   for( int i = 0; i < nvertices; ++i ) is >> vertices[i];
   normal = ( vertices[2] - vertices[0] ) % ( vertices[1] - vertices[0] );
   normal.normalize();
   dest = new cworld_dplist( is );
}

void portal::draw( cview &v, const space &spc, const volume &vol )
{
   volume newvol = vol;

   if( ( vertices[0] - spc.s ) * normal < 0 )
   {
      for( int i = 0; i < nvertices; ++i )
      {
         vector a = vertices[i] % spc;
         vector b = vertices[ (i+1) % nvertices ] % spc;
         newvol.push( new halfspace( 0, b % ( b - a ) ) );
      }
   
      if( dest ) dest->draw( v, spc, newvol );
   }
}

// ---------------------------------------------------------------
// NEWTON
// ---------------------------------------------------------------

newton_object::newton_object()
{
   s = zerovector;
   v = zerovector;
   B = identitybase;
   L = zerovector;
   Q = identityquaternion;
};

newton_object::newton_object( istream &is )
{
   s = zerovector;
   v = zerovector;
   B = identitybase;
   L = zerovector;
   Q = identityquaternion;

   char c;
   is >> c;
   if( c == '(' )
   while( !is.eof() )
   {
        is >> c;
        switch( c )
        {
         case 's':
         is >> s;
         break;

         case 'v':
         is >> v;
         break;

         case 'Q':
         is >> Q;

         case 'B':
         is >> B;
         Q = B;
         break;

         case 'L':
         is >> L;
         break;

         case ')':
         return;
      }
   }
}

void newton_object::pace( const number dt )
{
   s += dt * v;
   B = Q.spin( dt * L );
}

// ---------------------------------------------------------------
// RIGID BODY
// ---------------------------------------------------------------

single_cmesh_newton_object::single_cmesh_newton_object( istream &is )
{
   char c;
   is >> c;
   if( c == '(' )
   while( !is.eof() )
   {
        is >> c;
      switch( c )
      {
/*              case 'M':
         {
                char name[20];
            is.width( 20 );
            is >> name;
            fstream file( "objects.txt", ios::in );
            if( file ) cmesh::load( *this, name, file );
            file.close();
         }
         break;*/

         case 'N':
         (*(newton_object*)this) = newton_object( is );
         break;

         case ')':
         return;
      }
   }
}

void
single_cmesh_newton_object::draw( cview &V, const space &spc, const volume &vol )
{
   cview::chandle handle = V.transform( *this, spc % (*this) );
   V.cull_backfaces( handle );
   V.clip_volume( handle, vol );
}

// ---------------------------------------------------------------
// WORLD
// ---------------------------------------------------------------

cworld_dplist::cworld_dplist( istream &is )
{
   char buf[20];
   is >> *buf;
   if( *buf == '(' )
   while( !is.eof() )
   {
           is.width( 20 );
      is >> buf;

//      if( !strcmp( buf, "solid_sg" ) )
//      dplist.push( new solid_sg( is ) );

//      if( !strcmp( buf, "checked_ground" ) )
//      dplist.push( new checked_ground( is ) );

//           if( !strcmp( buf, "rigid_body" ) )
//      dplist.push( new rigid_body( is ) );

//      if( !strcmp( buf, "portal" ) )
//      dplist.push( new portal( is ) );

      if( *buf == ')' ) return;
   }
}

void cworld_dplist::draw( cview &V, const space &spc, const volume &vol )
{
   pointer< DP > p = dplist.begin();
        while( !p.finish() ) { (*p).draw( V, spc, vol ); ++p; }
}

void cworld_dplist::pace( const number dt )
{
   pointer< DP > p = dplist.begin();
        while( !p.finish() ) { (*p).pace( dt ); ++p; }
}

