
#include <math.h>
#include <fstream.h>
#include <string.h>

#include "game.h"
#include "global.h"

namespace game
{

list< cMOB > prototypes;

bool load_prototypes()
{
   char buf[20];
   ifstream file( "objects.txt" );
   file.width(20);
   if( seekline( file, "@prototypes" ) )
   while( !file.eof() )
   {
      file >> buf;
      cMOB *mob = 0;
      if( !strcmp( buf, "#" ) ) break;
      if( !strcmp( buf, "static" ) ) mob = new cMOB;
      if( !strcmp( buf, "auto" ) ) mob = new cMOB_auto;
      if( !strcmp( buf, "airplane" ) ) mob = new cMOB_airplane;
      if( mob ) { mob->load( file ); prototypes.push( mob ); }
   }

   else
   { cerr << "Could not load prototypes!" << endl; return false; }

   return true;
}

cMOB *give_prototype( const char *sname )
{
   for( pointer< cMOB > p = prototypes.begin(); !p.finish(); ++p )
      if( (*p).name )
         if( !strcmp( (*p).name, sname ) ) return &(*p);

   return 0;
}

// ----------------------------------------------------------------------------
// CMOB
// ----------------------------------------------------------------------------

cMOB::cMOB()
{
   mode = PLANET;
   kill_me_please = false;
   name = 0;
   mesh = meshlow = 0;
   spaces = zerovector;
   spaceB = identitybase;
}

void cMOB::pace( const number )
{
   if( mode == SPACE )
   {
      s = ( spaces - actplanet->s ) % actplanet->rotation_B;
      B = spaceB % actplanet->rotation_B;
   }
}

void cMOB::draw( cview &view,
const space &spc, const volume &vol )
{
   // which mesh?
   // much detail or less detail?
   cmesh *tmpmesh = mesh;
   if( ( s - spc.s ).quadrat() > lowdist ) tmpmesh = meshlow;

   // object is behind camera?
   if( ( s - spc.s ) * spc.B.k < - tmpmesh->msize ) return;

   // transform that mesh
   cview::chandle handle = view.get_handle();
   tmpmesh->draw( view, spc % *this, vol );

   // perform lighting
   vector a = actplanet->s % actplanet->rotation_B;
   view.light_direct( handle, a % spc.B, tmpmesh->flags & cmesh::RELATIVE );

   if( s * a > 0 )
   {
      vector b = actplanet_data->radius * ( s - ( s * a / a.quadrat() * a ) ).normalize();
      view.clip_halfspace( handle, halfspace( b * ( b - spc.s ), b % spc.B ),
         CLIP_KEEP + CLIP_SHADOW );
   }
}

// Will draw the shadow of the cmob
// projectes s onto sphere
// use tanget plane to cast shadow

void cMOB::draw_shadow( cview &view,
const space &spc, const volume &vol )
{
   vector sun_in_worldspace =
      ( zerovector - actplanet->s ) %
      actplanet->rotation_B;

   // are we already on the dark side of the planet?
   if( s * sun_in_worldspace < 0 ) return;

   // sp will contain projected s onto the sphere
   vector ds = s - ( 30 * s + sun_in_worldspace );
   number a = ds * ds;
   number b = 2 * ( s * ds );
   number c = s * s - actplanet_data->radius * actplanet_data->radius;
   number d = b * b - 4 * a * c;
   if( d < 0 ) return; // no solution
   number t = ( - b - sqrt( d ) ) / ( 2 * a );
   vector sp = s + t * ds;

   // shadow behind horizon?
   if( ( sp - spc.s ) * nearplanetdx >
      nearplanetdx2s * nearplanetx ) return;

   // which mesh for the shadow?
   // low or high detail?
   cmesh *tmpmesh = mesh;
   if( ( sp - spc.s ).quadrat() > lowdist ) tmpmesh = meshlow;

   // shadow behind camera?
   if( ( sp - spc.s ) * spc.B.k < - tmpmesh->msize ) return;

   // create a matrix that does shadow-casting,
   // ie. projection onto the ground
   vector dx = ( sun_in_worldspace + 100 * sp );
   matrix M = matrix( sp, base(
      B.i - ( sp * B.i ) / ( sp * dx ) * dx,
      B.j - ( sp * B.j ) / ( sp * dx ) * dx,
      B.k - ( sp * B.k ) / ( sp * dx ) * dx ) );




   // transfer polygons
   // but will use only 1 color
   cview::chandle handle = view.get_handle();
   for( unsigned i = 0; i < tmpmesh->npolygons; ++i )
   {
      if( tmpmesh->flags & cmesh::CULLING )
         if( tmpmesh->normals[ tmpmesh->polygons[i].polygonnormal_i ] *
         this->B * dx < 0 ) continue;

      view.polygons[view.npolygons].polygonnormal_i = view.nnormals;
      view.polygons[view.npolygons].nvertices = tmpmesh->polygons[i].nvertices;
      view.polygons[view.npolygons].vertices_i[0] = tmpmesh->polygons[i].vertices_i[0] + view.nvertices;
      view.polygons[view.npolygons].vertices_i[1] = tmpmesh->polygons[i].vertices_i[1] + view.nvertices;
      view.polygons[view.npolygons].vertices_i[2] = tmpmesh->polygons[i].vertices_i[2] + view.nvertices;
      view.polygons[view.npolygons].vertices_i[3] = tmpmesh->polygons[i].vertices_i[3] + view.nvertices;
      view.polygons[view.npolygons].insertbefore( &view.anchor );
      ++view.npolygons;
   }

   M = M % spc;

   // transfer the vertices
   for( unsigned i = 0; i < tmpmesh->nvertices; ++i )
      //view.push_vertex( actplanet_data->radius * ( tmpmesh->vertices[i] * M ).normalize() % spc );
      view.push_vertex( tmpmesh->vertices[i] * M );

   // transfer only a single color: OR 128
   view.push_normal( 512 + 128 );

   view.clip_volume( handle, vol );
}

void cMOB::load( istream &is )
{
   char buf[20];
   number size;
   if( name ) delete[] name;
   name = new char[20];
   is.width( 20 );
   is >> name;
   if( !strcmp( name, "space" ) ) { mode = SPACE; is >> name; }
   is >> lowdist;
   if( lowdist > 0 ) { is >> size >> buf; mesh = cmesh::load( buf, size ); }
   is >> visdist >> size >> buf;
   meshlow = cmesh::load( buf, size );

   if( mesh )
   for( unsigned u = 0; u < mesh->nnormals; ++u )
      if( mesh->colors[u] == 255 ) mesh->colors[u] = backgroundcolor;

   for( unsigned u = 0; u < meshlow->nnormals; ++u )
      if( meshlow->colors[u] == 255 ) meshlow->colors[u] = backgroundcolor;
}

void cMOB_airplane::pace( const number dt )
{
   vector sn = s;
   sn.normalize();

   if( ( rand() & 65535 ) < 100 || iflag )
      hdest = 0.5 * number( rand() & 65535 ) / 65536.0 + 0.05;

   if( ( rand() & 65535 ) < 150 || iflag )
   { turn = true; headvector = vector::random();
     headvector -= headvector * sn * sn; headvector.normalize(); }

   // dmpfung
   v -= dt * ( 0.45 * v * B.k * B.k + 2.2 * ( v * B.i * B.i + v * B.j * B.j ) );
   L -= dt * 0.5 * L;

   // thrust
   v += dt * 0.1 * B.k;

   switch( loop )
   {
      case 0:
      {
         // normal autopilot
         number h = sqrt( s * s ) - actplanet_data->radius;
         number dh = 4 * ( h - hdest );
         dh = dh > 0 ? dh * dh : - dh * dh;
         dh = dh > 0 ? dh / ( 1 + dh ) : dh / ( 1 - dh );

         number dx = 0;
         if( turn )
         {
            if( headvector * B.k < -0.9 ) loop = 1;
            dx = headvector * B.i;
            if( dx * dx < 0.01 ) turn = false;
         }
         // autolevel
         L -= dt * ( ( ( ( s % B.i ).normalize() + 0.5 * dh * sn ) % B.k ) +
         ( ( B.k % s ).normalize() + dx * sn ) % B.i );

         // auto yaw
         L += dt * 2 * v * B.k * sn * B.i * sn;
      }
      break;

      case 1:
      // loop phase 1
      L += dt * B.i;
      if( B.j * sn < 0 ) { wait = 0.5; loop = 2; }
      break;

      case 2:
      wait -= dt;
      if( wait < 0 ) loop = 3;
      break;

      case 3:
      // loop phase 2
      L -= 2 * ( B.j * sn + 0.2 ) * dt * B.k;
      if( B.j * sn > 0 ) loop = 0;
      break;
   }

   newton_object::pace( dt );

   iflag = false;
}

cMOB_auto::cMOB_auto()
{
   front_ax = rear_ax = speed = lenkrad = lenkradv = 0;
   headvector = zerovector;
   iflag = false;
}

void cMOB_auto::pace( const number dt )
{
   vector sn = s;
   sn.normalize();
   s = actplanet_data->radius * sn;
   B.j = sn;
   B.i = B.j % B.k;
   B.k = B.i % B.j;

   if( ( rand() & 65535 ) < 150 || iflag )
   { turn = true; headvector = vector::random();
     headvector -= headvector * sn * sn; headvector.normalize(); }

   vector dvector = headvector - ( B.k + ( lenkrad - L * B.j ) * B.i ).normalize();
   number dx = dvector * B.i;
   if( headvector * B.k < 0 ) dx = dx < 0 ? -1 : 1;
   lenkradv += dt * ( dx - 5 * lenkradv - lenkrad );
   lenkrad += dt * lenkradv; 
   if( lenkrad > ( 3 / ( 1 + 10000 * speed * speed ) ) ) lenkrad = 3 / ( 1 + 10000 * speed * speed );
   if( lenkrad < ( -3 / ( 1 + 10000 * speed * speed ) ) ) lenkrad = -3 / ( 1 + 10000 * speed * speed );

   //speed += dt * 0.2 * ( 0.1 - 0.025 * lenkrad * lenkrad - speed );
   speed = maxspeed;
   number axial = speed / sqrt( 1 + lenkrad * lenkrad );
   vector lateral = - speed * lenkrad / sqrt( 1 + lenkrad * lenkrad ) / ( front_ax - rear_ax ) * B.j;

   v = axial * B.k + lateral % ( rear_ax * B.k );
   L = lateral;
   s += dt * v;
   B.spin( dt * L );
}

void cMOB_auto::load( istream &is )
{
   cMOB::load( is );
   is >> rear_ax >> front_ax >> maxspeed;
}

static bool lenkrad_visible = false;

void cMOB_auto::draw( cview &view, const space &spc, const volume &vol )
{
   cMOB::draw( view, spc, vol );

   if( key[ KEY_F7 ] ) lenkrad_visible = true;
   if( key[ KEY_F8 ] ) lenkrad_visible = false;

   if( lenkrad_visible )
   {
      cview::chandle handle = view.get_handle();
      view.push_polygon( 0, 0, 1 );
      view.push_polygon( 0, 0, 2 );
      view.push_vertex( s % spc );
      view.push_vertex( ( s + 0.01 * headvector ) % spc );
      view.push_vertex( ( s + 0.01 * ( B.k + lenkrad * B.i ).normalize() ) % spc );
      view.push_normal( 5 );
      view.clip_volume( handle, vol );
   }
}

// ----------------------------------------------------------------------
// MOBLIST
// ----------------------------------------------------------------------

// --- helper functions ---------------------

// false if vector x (relative to state.playerlocals) is beyond the horizon
// relys on
//    nearplantx, nearplanetdx, nearplanetdx2
//    filled by game::pace()

static bool check_vis( const vector &x )
{
   return( ( x * nearplanetdx ) < ( nearplanetdx2s * nearplanetx ) );
}

static vector cs;

int sortfunc3( const void *s1, const void *s2 )
{
   if( ( (*(cMOB**)s1)->s - cs ).quadrat() <
      ( (*(cMOB**)s2)->s - cs ).quadrat() ) return 1;
   else return -1;
}

unsigned cMOBlist::extract( cMOB **extrlist, unsigned numhalf, const unsigned maxhalf )
{
   for( pointer< cMOB > p = list.begin(); !p.finish(); ++p )
      if( (*p).s * state.playerlocals > 0 ) { extrlist[ numhalf++ ] = &(*p);
         if( numhalf == maxhalf ) break; }
   update = false;
   return numhalf;
}

void cMOBlist::pace( const number dt )
{
   for( pointer< cMOB > p = list.begin(); !p.finish(); ++p )
   {
      (*p).pace( dt );
      if( (*p).kill_me_please ) kill( &(*p) );
   }
}

void cMOBlist_cloud::put( const number distance )
{
   if( !prototype ) return;

   if( prototype->mode == cMOB::SPACE )
   {
      cMOB *mob = prototype->copy();
      vector v = state.player.v;
      v.normalize();
   
      do
      {
         number z = distance * ( 0.5 * unit_random() + 0.8 );
         mob->spaces = z * v;
         mob->spaces += z * 2 * unit_random() * state.player.B.i;
         mob->spaces += z * 2 * unit_random() * state.player.B.j;
      }
      while( mob->spaces.quadrat() < ( distance * distance * 0.2 ) );
   
      mob->spaces += state.player.s;
   
      mob->spaceB.j = vector::random();
      mob->spaceB.j.normalize();
      mob->spaceB.i = ( vector::random() % mob->spaceB.j ).normalize();
      mob->spaceB.k = mob->spaceB.i % mob->spaceB.j;
      mob->pace( 0.001 );
      
      list.push( mob );
   }

   else
   {
      cMOB *mob = prototype->copy();
      vector v = playergroundv;
      v.normalize();
   
      do
      {
         number z = distance * ( 0.5 * unit_random() + 0.8 );
         mob->s = z * v;
         mob->s += z * 2 * unit_random() * state.player.B.i;
         mob->s += z * 2 * unit_random() * state.player.B.j;
      }
      while( mob->s.quadrat() < ( distance * distance * 0.2 ) );
   
      mob->s += state.playerlocals;
      mob->s *= ( h + actplanet_data->radius ) / sqrt( mob->s.quadrat() );
   
      mob->B.j = mob->s;
      mob->B.j.normalize();
      mob->B.i = ( vector::random() % mob->B.j ).normalize();
      mob->B.k = mob->B.i % mob->B.j;
   
      list.push( mob );
   }
};

cMOBlist_cloud::cMOBlist_cloud()
{
   d = 0;
   h = 0;
   prototype = 0;
}

static int funny_counter = 0;

void cMOBlist_cloud::pace( const number dt )
{
   if( !prototype ) return;

   if( prototype->mode == cMOB::SPACE )
   {
      number h = d * dt * sqrt( state.player.v.quadrat() );
      int i = int( h );
      for( ; i > 0; --i ) put( 2 * sqrt( prototype->visdist ) );
      h = h - i;
      if( ( unit_random() + 0.5 ) < h ) put( 2 * sqrt( prototype->visdist ) );

      funny_counter --;
      if( funny_counter < 0 )
      {
         funny_counter = 10;
         for( pointer< cMOB > p = list.begin(); !p.finish(); ++p )
         if( ( (*p).s - state.player.s ).quadrat() > ( 10 * (*p).visdist ) )
            /*kill( &(*p) )*/;
      }
   }

   else
   {
      number h = d * dt * sqrt( playergroundv.quadrat() );
      int i = int( h );
      for( ; i > 0; --i ) put( 2 * sqrt( prototype->visdist ) );
      h = h - i;
      if( ( unit_random() + 0.5 ) < h ) put( 2 * sqrt( prototype->visdist ) );
    
      funny_counter --;
      if( funny_counter < 0 )
      {
         funny_counter = 10;
         for( pointer< cMOB > p = list.begin(); !p.finish(); ++p )
         if(
            ( (*p).s * dsn < 0 )
            ||
            ( (*p).s - actplanet_data->radius * dsn ).quadrat() > ( 1.5 * (*p).visdist )
         ) kill( &(*p) );
      }
   }

   cMOBlist::pace( dt );
}

void cMOBlist_cloud::init()
{
   if( !prototype ) return;

   if( prototype->mode == cMOB::SPACE ) {}

   else
   {
      number svd4 = sqrt( prototype->visdist ) * 4;
      number dens = d / svd4;
      number area = svd4 * svd4;
    
      int n = int( dens * area ) / 2;
      for( int i = 0; i < n; ++i )
      {
         cMOB *mob = prototype->copy();
         mob->s = actplanet_data->radius * dsn + svd4 * unit_random() * dsx + svd4 * unit_random() * dsz;
         mob->s *= ( h + actplanet_data->radius ) / sqrt( mob->s.quadrat() );
      
         mob->B.j = mob->s;
         mob->B.j.normalize();
         mob->B.i = ( vector::random() % mob->B.j ).normalize();
         mob->B.k = mob->B.i % mob->B.j;
    
         list.push( mob );
      }
   }

   update = true;
}

// ------------------------------------------------------------------

 list< cMOBlist > MOBS;

 cMOB **extrlist;         // pointer array for storage of extracted MOBS
 cMOB **farlist;          // pointer array for all MOBS behind horizon
 cMOB **nearlist;         // pointer array for all MOBS in front of horizon

 int maxextr, maxfar, maxnear;   // capacity of the pointer arrays
 int numextr, numfar, numnear;   // current occupation of the pointer arrays
 int extr_counter, visible_counter;   // timers

void MOBS_init()
{
   maxextr = 5000; extrlist = new (cMOB *) [ maxextr ];
   maxfar = 500; farlist = new (cMOB *) [ maxfar ];
   maxnear = 500; nearlist = new (cMOB *) [ maxnear ];
   MOBS_flush();
}

void MOBS_extract()
{
   bool update = false;

   for( pointer< cMOBlist > p = MOBS.begin(); !p.finish(); ++p )
      update |= (*p).update;

   extr_counter--;
   update |= extr_counter < 0;

   if( update )
      { MOBS_extract_lists(); extr_counter = 29; }

   visible_counter--;
   update |= visible_counter < 0;

   if( update )
   {
      numfar = 0;
      numnear = 0;
   
      for( int i = 0; i < numextr; ++i )
      {
         // vector x = line of sight
         vector x = extrlist[i]->s - state.playerlocals;
    
         // within visible range?
         // then decide between near and far
         if( ( ( x - x * dsn * dsn ).quadrat() + 0.1 * ( x * dsn ) * ( x * dsn ) )
         < extrlist[i]->visdist )
            if( check_vis( x ) )
               { if( numnear < maxnear ) nearlist[ numnear++ ] = extrlist[i]; }
            else
               { if( numfar < maxfar ) farlist[ numfar++ ] = extrlist[i]; }
      }
      visible_counter = 2;
   }
}

void MOBS_draw_far( cview &view, const space &spc, const volume &vol )
{
   cs = spc.s;
   // qsort( farlist, numfar, sizeof( cmob * ), sortfunc3 );
   for( int i = 0; i < numfar; ++i ) farlist[i]->draw( view, spc, vol );
}

void MOBS_draw_near( cview &view, const space &spc, const volume &vol )
{
   cs = spc.s;
   qsort( nearlist, numnear, sizeof( cMOB * ), sortfunc3 );
   for( int i = 0; i < numnear; ++i ) nearlist[i]->draw( view, spc, vol );
}

void MOBS_draw_shadow( cview &view, const space &spc, const volume &vol )
{
   for( int i = 0; i < numfar; ++i )
   farlist[i]->draw_shadow( view, spc, vol );
   for( int i = 0; i < numnear; ++i )
   nearlist[i]->draw_shadow( view, spc, vol );
}

}

