
#ifndef GAME_H
#define GAME_H

#include <allegro.h>
#include "engine.h"
#include "special.h"

namespace game
{

extern cview *view;

extern int polygons1;
extern int polygons2;

// ----------------------------------------------------------------------------
// GAME STATE
// ----------------------------------------------------------------------------

struct cplanet : solid_sphere
{
   cring *ring;
   int num;
   vector rotation_L;
   vector dv;
   base rotation_B;

   void draw( cview &, const space &, const volume & );
   void draw_local( cview &, const space &, const volume & );
   void draw_ringdist( cview &, const space &, const volume & );
   void draw_ringnear( cview &, const space &, const volume & );
};

struct cstate
{
   newton_object player;
   short locmode;
   short movemode;
   short planetnumber;
   short inventory[20];
   short weapons[10];

   bool interdrive;
   number interthrust;
   bool interdefect;
   number intercrash;

   cplanet planets[20];
   vector playerlocals;
   vector playerlocalv;
   base playerlocalB;
   vector playerlocalL;

   short flags[20];

   bool init();
};

extern cstate state;
extern cworld *world;

extern number thrusters;
extern const number power;

// ----------------------------------------------------------------------------
// GENERAL DATA
// ----------------------------------------------------------------------------

// this is for the initial splash logo

extern number titletime;    // how long (countdown)
extern int titlex, titley;  // and where

// some fundamental color indices

extern int
   backgroundcolor,    // physical index for sky-color
   hudcolor,           // physical index for HUD-color
   acthudcolor,        // virtual index for actual HUD-color
   nexthudcolor[32];   // translate current -> next HUD-color

// the global 3D-objects

extern starfield stars;          // the outer-space starfield
extern checked_ground checked;   // the checkerboard ground clutter

// some utility variables

struct cplanet_data;
extern int nearplanet;                 // the current (nearest) planet's index
extern cplanet *actplanet;             // and a pointer to the current planet
extern cplanet_data *actplanet_data;   // and a pointer to it's data

extern vector ds;              // ds = actplanet's center -> player
extern vector dsn;             // dsn = ds normalized
extern vector dsx, dsz;        // orthogonal complements to dsn
extern vector playergroundv;   // velocitiy vector over ground
extern space groundspace;      // a space where always y = altitude
extern base groundnormalbase;  // a base which rotates as player moves over surface

extern number nearplanetc,  // variables that contain
   nearplanetc2,            // precalculated values
   nearplanetx,             // for optimised checking
   nearplanetx2,            // against the current
   nearplanetdx2,           // planet's horizon
   nearplanetdx2s;
extern vector nearplanetdx;

// all stuff from the palette manager

void palinit();
extern bool palused[128];    // an allocation table
int palalloc();              // give unused index else -1
void palfree( const int );   // free an index
void palset( const int, const RGB & ); // palset() sets a color and sets
                             // color+128 to the shadowed color

// all stuff from the message brabbler

#define MSG_NUMBUFFER 3
void msg_init();
void msg( const char * );    // text is copied into internal buffers
void msg_draw( cview & );    // draw the msg onto a view
void msg_pace( number );     // brabble onward
extern int msgi;             // current message-line
extern int msgphase[ MSG_NUMBUFFER ]; // 0 = idle, 1 = coming, 2 = standing, 3 = going
extern char msgline[ MSG_NUMBUFFER ][80]; // contains text for the lines
extern number msgscroll[ MSG_NUMBUFFER ]; // x-offsets for the respective lines
extern BITMAP *msgbitmap;    // a sub-bitmap of the framebuffer for output

// all the stuff for the infrared-
// camera motion-blurrer

void IR_init();
extern bool IRenable;      // true or false
extern int IRxanchor[];    // an anchor-table for
extern int IRyanchor[];    // the masked pattern mode
extern int IRcycle;        // 0..3
extern float IRwarmup;     // a countdown (blank screen while > 0)
extern BITMAP *IRcanvas;   // drawing is redirected to this in IR mode
extern BITMAP *IRbitmap;   // a 32 x 32 bitmap for masked pattern
void IR_pace( number );    // countdown and cycle update

/*
The blinker structure provides a bool "out" variable
which toggles true/false at regular intervals when
paced with successive dt values; T = period, phase = duty cycle
*/

struct blinker
{
   number t, T, phase;
   bool out;
   blinker( const number phase = 0.5, const number T = 1 )
      : t( 0 ), T( 1 ), phase( 0.5 ), out( false ) {};
   void pace( const number dt )
      { t += dt; if( t > T ) t -= T; out = t < ( T * phase ); }
};

extern blinker interblinker;  // this is for blinking the velocity-bar
                              // while in interdrive


extern bool destenable;
extern number dest_loc_lat;
extern number dest_loc_long;

extern int command;

extern number map_zoom;
extern base map_rotate;
extern bool map_center;

// ----------------------------------------------------------------------------
// PLANETS DATA
// ----------------------------------------------------------------------------

struct cMOBlist;
struct cMOBlist_cloud;

/*
The structure "cplanet_data" holds the invariant data
as read from the file game.txt.
The structure "cplanet" holds the dynamical data.
*/

extern int STARTNUMBER,    // index of start-planet
   NPLANETS,               // number of planets
   NMAJORBODIES;           // number of planets without parents

// enum for the planet's flags
enum { RING = 1 << 16, GIANT = 1 << 17, RAIN = 1 << 18 };
// max. number of cloud-type MOB-lists
#define NCLOUDS 8

struct cplanet_data
{
   char name[20];       // these are in the same order as
   int parent;          // read from the file game.txt
   number distance;     // name, parent, distance, radius,
   number radius;       // optional opposition specifier,
   int opp;             // GC, CC, SC, atmosphere dens (=mag)
   int groundcolor1;    // and the status-flags
   int groundcolor2;
   int skycolor;
   number mag;
   int flags;

   number ringradius;     // if planet has a ring, data goes here
   number ringdiameter;   // radius, thickness and color
   int ringcolor;

   number rain_length;    // if planet has rain, data goes here
   number rain_speed;     // strip length, strip speed, top altitude,
   number rain_height;    // and color
   int rain_color;

   RGB spacecol;     // The planet's color as seen from space is a mixture
   int ground_IR;    // between groundcolor and skycolor dependant of
   int check_IR;     // the atm-density. The IR components are attenuated
   int sky_IR;       // with distance from the sun.
   int spacecol_IR;

   // The MOB-lists as read from the file game.txt
   // one for all fixed MOBS and a number of different
   // cloud-type lists.

   cMOBlist *MOB_fixed;
   cMOBlist_cloud *MOB_cloud[ NCLOUDS ];
};

extern cplanet_data *planet_data;

/*
The following stuff is for sorting
planet-indices within a linked list.
2 linked lists one for major bodies
only and one for children only. This
is used by the draw routine to depth-sort
planets correctly with respect to relative
position to the parent's ring
*/

struct planetindex : node
{
   int i;
   planetindex( const int i ) : i( i ) {}
};

extern list< planetindex > major_bodies;
extern list< planetindex > *children;

// workspace for the rain effect generator

#define NRAIN 500                    // how many raindrops
extern vector rain_vertex[ NRAIN ];  // an array of random vertices
extern vector rain_shift;            // moves down continuously

// ----------------------------------------------------------------------------
// MOB / MOBLIST - STRUCTURES & DATA
// ----------------------------------------------------------------------------

// MOB = "movable object"

/*
The base class of the MOBS contains some basic data,
virtual draw() and pace() methods and a virtual copy()
function that returns a clone of the MOB itself, so that
one piece of code can create any type of MOB if given
the right prototype object.
The pointers to the meshes are meant to be shared
among all MOBS of the same type so the destructor of one
MOB doesn't kill them.
*/

struct cMOB : newton_object
{
   vector spaces;
   base spaceB;

   bool kill_me_please;
   char *name;
   cmesh *mesh;      // pointer to high-detail mesh
   cmesh *meshlow;   // pointer to low-detail mesh
   number lowdist;   // square of distance to switch to low-detail
   number visdist;   // square of visibility boundary
   enum { PLANET, SPACE } mode;

   cMOB();
   virtual void pace( const number );
   virtual void draw( cview &, const space &, const volume & );
   virtual void draw_shadow( cview &, const space &, const volume & );
   virtual cMOB *copy() const { return new cMOB( *this ); };
   virtual void load( istream & );
};

// The bullet type of MOB has a pace() function for
// linear advance, does not draw shadows and introduces
// a suicide-timer.

struct cMOB_bullet : cMOB
{
   number time_to_live;
   void draw_shadow( cview &, const space &, const volume & ) {}
   void pace( const number dt ) { s += dt * v; time_to_live -= dt;
      if( time_to_live < 0 ) kill_me_please = true; }
   cMOB *copy() const { return new cMOB_bullet( *this ); }
};

// The airplane type of MOB has some more parameters and
// an external pace() function for the movement.

struct cMOB_airplane : cMOB
{
   number hdest;        // destination altitude
   vector headvector;   // destination heading vector
   bool turn;           // true when turning
   bool iflag;          // flag for initial parameter set
   int loop;            // 0 = normal flight, 1..3 = looping phases
   number wait;         // wait counter between looping phases

   cMOB_airplane() { hdest = 0; loop = 0; wait = 0; iflag = true; }
   void pace( const number );
   cMOB *copy() const { return new cMOB_airplane( *this ); }
};

struct cMOB_auto : cMOB
{
   number front_ax;
   number rear_ax;
   number speed;
   number maxspeed;
   number lenkrad;
   number lenkradv;
   vector headvector;
   bool turn, iflag;

   cMOB_auto();
   void draw( cview &, const space &, const volume & );
   void pace( const number );
   cMOB *copy() const { return new cMOB_auto( *this ); }
   void load( istream & );
};


              // -----------------------------------------

// The prototype manager

extern list< cMOB > prototypes;
bool load_prototypes();
cMOB *give_prototype( const char * );

              // -----------------------------------------

/*
The cMOBlist structure is a container for a number
of MOBS which are held in a linked list. From this
list, there are extractions made at every 30 frames
into a global buffer. Only MOBS in that buffer get drawn
onto screen. The extraction rule could be anything,
currently it is that the MOB must be on the same hemisphere
of the planet than the player.
MOBS are in the ownership of the list ie. the destructor
of the list kills all MOBS.
*/

struct cMOBlist : node
{
   list< cMOB > list;   // the list where the MOBS are
   bool update;         // list has updated, perform new extraction immediately
   void zzap()          // wipe out the entire list
      { list.zzap(); update = true; }

   cMOBlist() { update = false; }    // the friendly constructor
   virtual ~cMOBlist() { zzap(); };  // the wicked destructor

   unsigned extract( cMOB **, unsigned, const unsigned );
   virtual void pace( const number );   // Calls pace() of all MOBS

   void push( cMOB *pcMOB )   // a new MOB into the list
      { list.push( pcMOB ); update = true; }

   void kill( cMOB *pcMOB )   // a single MOB from the list
      { pcMOB->pop(); delete pcMOB; update = true; }
};

// The cloud type of cMOBlist generates MOBS at random position
// in the vincinity of the player, and destructs the MOBS that
// get too far away as the player moves.

struct cMOBlist_cloud : cMOBlist
{
   number d, h;       // density, altitude;
   cMOB *prototype;   // create new MOBS from this prototype
   cMOBlist_cloud();

   void pace( const number );   // check player movement and put() if necessary
   void put( const number );    // put a MOB, but far away enough
   void init();                 // initialize with a number of MOBS
};

              // -----------------------------------------

/*
The MOBS manager is finally a container of all cMOBlists of the
game.
*/

extern list< cMOBlist > MOBS;   // a collection of all active MOB-lists

extern cMOB **extrlist;         // pointer array for storage of extracted MOBS
extern cMOB **farlist;          // pointer array for all MOBS behind horizon
extern cMOB **nearlist;         // pointer array for all MOBS in front of horizon

extern int maxextr, maxfar, maxnear;   // capacity of the pointer arrays
extern int numextr, numfar, numnear;   // current occupation of the pointer arrays

extern int extr_counter, visible_counter;   // timers

void MOBS_init();
inline void MOBS_flush() { extr_counter = visible_counter = 0; }

void MOBS_draw_far( cview &view, const space &spc, const volume &vol );
void MOBS_draw_near( cview &view, const space &spc, const volume &vol );
void MOBS_draw_shadow( cview &view, const space &spc, const volume &vol );

inline void MOBS_extract_lists()
{
   numextr = 0;
   for( pointer< cMOBlist > p = MOBS.begin(); !p.finish(); ++p )
      numextr = (*p).extract( extrlist, numextr, maxextr );
}

void MOBS_extract();

inline void MOBS_pace( const number dt )
   { for( pointer< cMOBlist > p = MOBS.begin(); !p.finish(); ++p ) (*p).pace( dt ); }

// extern cMOBlist *MOBlist_bullets;

extern number altitude, altitude2;

extern const char *welcomemsg;

extern cmesh *meshes;

#define ADDCOLOR( B, A, M ) \
B.r = (int) min( 63.0, sqrt( A.r * A.r + B.r * B.r * (M) ) + 0.5 ); \
B.g = (int) min( 63.0, sqrt( A.g * A.g + B.g * B.g * (M) ) + 0.5 ); \
B.b = (int) min( 63.0, sqrt( A.b * A.b + B.b * B.b * (M) ) + 0.5 ); \

#define BLENDCOLOR( C, A, B, M ) \
C.r = (int) ( sqrt( A.r * A.r * ( 1 - (M) ) + B.r * B.r * (M) ) + 0.5 ); \
C.g = (int) ( sqrt( A.g * A.g * ( 1 - (M) ) + B.g * B.g * (M) ) + 0.5 ); \
C.b = (int) ( sqrt( A.b * A.b * ( 1 - (M) ) + B.b * B.b * (M) ) + 0.5 ); \

#define SCALECOLOR( B, A, M ) \
B.r = (int) min( 63.0, double((M) * A.r) ); \
B.g = (int) min( 63.0, double((M) * A.g) ); \
B.b = (int) min( 63.0, double((M) * A.b) ); \

template< typename X >
inline X min( X a, X b ) { return a < b ? a : b; }

// ----------------------------------------------------------------------------
// FUNCTIONS
// ----------------------------------------------------------------------------

bool init();
void draw();
void pace( const number );

}

#endif

