#include "BBall.h"
#include <math.h>


BBall::BBall() : BObject() {
	g = -0.75;
	air = 0.985;
	bounce = 0.75;
}


BBall::~BBall() {
}


void BBall::MsgStart() {
	BObject::MsgStart();
	Place(minx,miny);
	v = 0.0;
	dir = DTOR(90);
	is_in_play = false;
	splayer = 0;
	cplayer = -1;
}

// Move the ball and check for collisions
// Returns:
//    -1 if there was no collision
//     0 if player 1 hit the ball
//     1 if player 2 hit the ball
//     2 if both players hit the ball
int BBall::Update(BPlayer *p1, BPlayer *p2) {
	bool ret = -1;
	if (key[KEY_ESC]) {
		Parent()->HandleEvent(*this, ESCAPE_GAME);
		return ret;
	}

	if (!is_in_play && v <= 1 && y() == miny)
		return ret;

	int newx = x();
	int newy = y();
	oldx = x();
	oldy = y();
	int olds = side;

	float vx = v*cos(dir);
	float vy = v*sin(dir);

	// adjust vertical speed (gravity)
	vy += g;
	vy *= air;

	// adjust horizontal speed
	vx *= air;

	// move the ball
	newx += (int)vx;
	newy -= (int)vy;

	// make sure we're not out of bounds
	if (newx < minx) {
		newx = minx + (minx - newx);
		vx = -vx;
	}
	else if (newx > maxx) {
		newx = maxx - (newx - maxx);
		vx = -vx;
	}
	if (newy > miny) {
		newy = miny;
		vy *= bounce;
		if (vy > -8)
			vy = 0;
		else
			vy = -vy;

		// ball is on the ground -> someone either gets a point or the serve changes
		if (is_in_play) {
			theSkin->PlaySample(SAMPLE_WHISTLE);
			if (x() > SCREEN_W/2) {
				if (splayer == 0) {
					++(*p1);
					Parent()->HandleEvent(*this, UPDATE_SCORE1);
				}
				else
					splayer = 0;
			}
			else {
				if (splayer == 1) {
					++(*p2);
					Parent()->HandleEvent(*this, UPDATE_SCORE2);
				}
				else
					splayer = 1;
			}

			// if a player has 15 points and at least two more than the opponent
			// then the game is over
			int s1 = p1->Score();
			int s2 = p2->Score();
			if ((s1 >= 15 || s2 >= 15) && (s1-s2 > 1 || s1-s2 < -1))
				Parent()->HandleEvent(*this, GAME_OVER);
			is_in_play = false;
		}
	}

	Place(newx,newy);
	UpdateSpeedAndDirection(vx, vy);

	// do collision detection with the players
	BPlayer *pl[2];
	pl[0] = p1;
	pl[1] = p2;

	if (is_in_play) {
		int i;
		for (i=0; i<2; i++) {
			if (BounceTheBall((BObject *)pl[i], newx, newy)) {
				theSkin->PlaySample(SAMPLE_HIT);
				ret++;
				newx = oldx;
				newy = oldy;

				// give the ball a bit of a speedup if hit by the player
				vx = v*cos(dir);
				vy = v*sin(dir);
				vx += (pl[i]->moved_x*bounce);
				vy -= (pl[i]->moved_y*bounce);

				// move the ball if the player moved
				newx += (pl[i]->moved_x);
				newy += (pl[i]->moved_y);

				newx += (int)vx;
				newy -= (int)vy;
				Place(newx,newy);

				UpdateSpeedAndDirection(vx, vy);

				// see if the same player hit the ball too many times
				if (hplayer == i && cplayer != i) {
					cplayer = i;
					hit++;
					if (hit > 3) {
						theSkin->PlaySample(SAMPLE_WHISTLE);
						hit = 0;
						is_in_play = false;
						hplayer = -1;
						int j = i==0 ? 1 : 0;
						if (splayer == j) {
							++(*pl[j]);
							Parent()->HandleEvent(*this, UPDATE_SCORE1+j);
							if (pl[j]->Score() >= 15 && pl[j]->Score() - pl[i]->Score() > 1)
								Parent()->HandleEvent(*this, GAME_OVER);
						}
						else
							splayer = j;
					}
				}
				else {
					hit = 1;
					hplayer = i;
					cplayer = i;
				}
			}
			else {
				if (cplayer == i)
					cplayer = -1;
			}
		}
	}

	// bounce of the bar if necessary
	int barx1 = SCREEN_W/2 - 8 - (w()>>1);
	int barx2 = SCREEN_W/2 + 8 + (w()>>1);
	int bary  = SCREEN_H - 312 - (h()>>1);
	if (newx < barx1)
		side = LEFT;
	else if (newx > barx2)
		side = RIGHT;
	else
		side = BETWEEN;

	if ((side != olds || side == BETWEEN) && newy > bary) {
		theSkin->PlaySample(SAMPLE_NET);
		// collision!
		vx = v*cos(dir);
		vy = v*sin(dir);

		if (olds == LEFT) {
			newx = barx1 - (newx - barx1);
			vx = -vx;
			side = LEFT;
		}
		else if (olds == RIGHT) {
			newx = barx2 + (barx2 - newx);
			vx = -vx;
			side = RIGHT;
		}
		else if (olds == BETWEEN) {
			newy = bary - (newy - bary);
			vy = -vy*bounce;
			if (newx > (SCREEN_W>>1))
				vx += 1.0f;
			else
				vx -= 1.0f;
		}

		newx += (int)vx;
		newy -= (int)vy;
		Place(newx,newy);
		UpdateSpeedAndDirection(vx, vy);
	}

	df = -vx/4.0;
	if (df > -0.125 && df < 0.125) df = 0;

	frame += df;
	if (frame >= N_OF_FRAMES)
		frame = 0;
	else if (frame < 0)
		frame = N_OF_FRAMES-1;

	// Prevent unnecessary redraw
	MakeClean();

	return ret;
}


// Collision detection: detects whether the ball is on top of a player
// (meaning there was a collision).
bool BBall::CollisionDetection(BObject *p, int& cx, int& cy) {
	BITMAP *bp = p->GetBitmap();	// the player bitmap
	BITMAP *bb = GetBitmap();		// the ball bitmap
	int magic_pink = getpixel(bb, 0,0);
	int dx = x() - p->x();
	int dy = y() - p->y();

	int plpix;
	for (cy=h(); cy>=0; --cy) {
		for (cx=0; cx<w(); cx++) {
			if (getpixel(bb, cx, cy) != magic_pink) {
				plpix = getpixel(bp, dx+cx, dy+cy);
				if (plpix != magic_pink && plpix != -1)
					return true;
			}
		}
	}

	return false;
}


void BBall::UpdateSpeedAndDirection(float vx, float vy) {
	if (vx == 0.0) vx = 0.0000001;
	if (vy>0) {
		if (vx>0)
			dir = atan(vy/vx);
		else
			dir = DTOR(180) + atan(vy/vx);
	}
	else {
		if (vx>0)
			dir = DTOR(360) + atan(vy/vx);
		else
			dir = DTOR(180) + atan(vy/vx);
	}
	v = sqrt(vx*vx + vy*vy);
}


bool BBall::BounceTheBall(BObject *obj, int newx, int newy) {
	int px, px2, py, py2;
	int cx, cy;		// point of collision

	// make the players bounding rectangle (slightly bigger than
	// the players bitmaps)
	px = obj->x() - (obj->w()>>1) - (w()>>1);
	px2 = obj->x() + (obj->w()>>1) + (w()>>1);
	py = obj->y() - (obj->h()>>1) - (h()>>1);
	py2 = obj->y() + (obj->h()>>1) + (h()>>1);

	// see if the new ball position is inside the players bounding box
	if (!MASIsInRect(newx, newy, px, py, px2, py2))
		return false;

	// see if there is actually a collision
	if (!CollisionDetection(obj, cx, cy))
		return false;

	// bounce the ball
	int dx = (w()>>1) - cx;
	int dy = cy - (h()>>1);

	dx *= 1.25f;

	if (dx==0) {
		if (dy>0)
			dir = DTOR(270);
		else
			dir = DTOR(90);
	}
	else {
		float gama = atan(dy/dx);
		if (dy>0) {
			if (dx<0)
				gama += DTOR(180);
		}
		else {
			if (dx>0)
				gama += DTOR(180);
			else
				gama += DTOR(360);
		}

		if (gama > DTOR(360))
			gama -= DTOR(360);
		dir = gama;
	}

	return true;
}

void BBall::Serve(BPlayer *player) {
	int p=0;
	if (player->x() > SCREEN_W/2) p=1;

	if (p != splayer || is_in_play) {
		return;
	}

	if (splayer == 0) {
		if (player->x() > SCREEN_W/4)
			return;
		Place(player->x(),player->y()-100);
		side = LEFT;
		dir = DTOR(85);
	}
	else {
		if (player->x() < 3*SCREEN_W/4)
			return;
		Place(player->x(),player->y()-100);
		side = RIGHT;
		dir = DTOR(95);
	}
	is_in_play = true;
	v = 28.0;
	df = v;
	hit = 0;
	hplayer = -1;
}
