/* BlitzHack 2003 code snippets */

#include "snippets.h"
#include <malloc.h>
#include <stdio.h>
#include <string.h>

/* --- data --- */
AL_CONST char *sn_datafile_name(AL_CONST char *def)
{
    AL_CONST char *ret;
    if(!exists("datapath.cfg"))
    {
        return def;
    }
    push_config_state();
    set_config_file("datapath.cfg");
    ret = get_config_string("data", "path", def);
    pop_config_state();
    return ret;
}

/* --- snow --- */
typedef struct
{
    float x, y;
} Vector2d;

static Vector2d *_snow = NULL;
static int _snow_quantity = 0;

int sn_create_snow(int quantity)
{
    int s;
    Vector2d *tSnow = NULL;
    if (_snow==NULL)
    {
        _snow_quantity = quantity;
        _snow = (Vector2d*)malloc(_snow_quantity * sizeof(Vector2d));
        if (_snow==NULL) return FALSE;
        for (s=0, tSnow=_snow; s<_snow_quantity; s++, tSnow++)
        {
            tSnow->x = rand()%SCREEN_W;
            tSnow->y = rand()%SCREEN_H;
        }
    }
    return TRUE;
}

void sn_destroy_snow(void)
{
    free(_snow);
    _snow=NULL;
    _snow_quantity = 0;
}

void sn_move_snow(void)
{
    int s;
    Vector2d *tSnow = NULL;
    if (_snow!=NULL)
    {
        for (s=0, tSnow=_snow; s<_snow_quantity; s++, tSnow++)
        {
            tSnow->x += rand()%2-2;
            tSnow->y += 1+rand()%2;
            if (tSnow->y>SCREEN_H) tSnow->y = 0;
            if (tSnow->x>SCREEN_W) tSnow->x = 0;
            if (tSnow->x<0) tSnow->x = SCREEN_W;
        }
    }
}


void sn_show_snow(BITMAP* bmp, BITMAP* snow)
{
    int s;
    Vector2d *tSnow = NULL;
    if (_snow!=NULL&&bmp!=NULL)
    {
        for (s=0, tSnow=_snow; s<_snow_quantity; s++, tSnow++)
        {
            if (snow == NULL) putpixel(bmp, (int)tSnow->x, (int)tSnow->y, makecol(255,255,255));
            else masked_blit(snow, bmp, 0, 0, (int)tSnow->x, (int)tSnow->y, snow->w, snow->h);
        }
    }
}



/* --- scoreblock --- */
struct _score_structure
{
   BITMAP *img;
   int *dval;
   int w, h;
   unsigned int current;
   int digits;
};

SN_SCOREBLOCK *sn_create_scoreblock (void)
{
   /* Allocate stuff */
   SN_SCOREBLOCK *o = (SN_SCOREBLOCK*)malloc (sizeof (struct _score_structure));
   if (o)
     {
        o->img = NULL;
        o->current = 0;
        o->digits = 0;
     }
   return o;
}

void sn_destroy_scoreblock (SN_SCOREBLOCK * o)
{
   /* clean up */
   if (o)
     {
        if (o->dval)
           free (o->dval);
        if (o->img)
           destroy_bitmap (o->img);
        free (o);
     }
}

int sn_init_scoreblock (SN_SCOREBLOCK * o, int d, FONT * f, int foreground, int background)
{
   /* initialise various fields */
   int w = 0, h;
   int i, t;
   static const char *digs[] =
      { "0", "1", "2", "3", "4", "5", "6", "7", "8", "9", "0" };
   /* it's forbidden to re-initialise a SCORE */
   if (o->img)
      return -1;

   /* Find the widest digit */
   for (i = 0; i < 10; ++i)
     {
        t = text_length (f, digs[i]);
        if (t > w)
           w = t;
     }
   /* allocate space for each digit's value */
   o->dval = (int*)malloc (d * sizeof (int));
   o->digits = d;
   for (i = 0; i < d; ++i)
      o->dval[i] = 0;
   w += 2;
   o->w = w;
   o->h = h = text_height (f) + 2;
   o->img = create_bitmap (w, 11 * h);
   clear_bitmap (o->img);
   rectfill (o->img, 0, 0, w - 1, 11 * h, background);
   text_mode (background);
   /* draw each digit onto the bitmap */
   for (i = 0; i <= 10; ++i)
     {
        textout (o->img, f, digs[i], 1, i * h + 1, foreground);
     }
   text_mode (0);
   return 0;
}

void sn_update_scoreblock (SN_SCOREBLOCK * o, unsigned int v)
{
   int vi;
   int lim = o->h * 5;
   int c;
   int *pv;
   /* got through each digit, move it up or down to
    * get it to the desired position.
    */
   for (pv = o->dval; pv < o->dval + o->digits; ++pv)
     {
        c = *pv;
        vi = (v % 10) * o->h - c;
        if (vi < -lim)
           ++c;
        else if (vi < 0)
           --c;
        else if (vi > lim)
           --c;
        else if (vi > 0)
           ++c;
        (*pv) = (10 * o->h + c) % (10 * o->h);
        v /= 10;
     }
}

void sn_display_scoreblock (SN_SCOREBLOCK * o, BITMAP * target, int x, int y)
{
   int i;
   /* Just blit each digit in turn */
   for (i = 0; i < o->digits; ++i)
     {
        blit (o->img, target, 0, o->dval[o->digits - 1 - i], x + i * o->w, y,
              o->w, o->h);
     }

}



/* --- particle --- */
struct ptc
{
   fixed x, y;
   fixed dx, dy;
   int life;
};

struct _particle_generator
{
   int count, max;
   struct ptc *ptcs;
   int *colours;
   int alpha;
   int size;
   int speed;
   int life;
   int last;
};

static fixed iso2d[256][2];
static fixed iso3d[256][2];
static fixed fan[256][2];
static int firsttime=1;
void firstinit(void) {
  int i;
  fixed theta, phi;
  for (i=0; i<256; ++i) {
    theta=itofix(rand()%256);
    phi=itofix(rand()%64);
    iso2d[i][0]=fixcos(theta);
    iso2d[i][1]=fixsin(theta);
    iso3d[i][0]=fixmul(fixcos(phi), iso2d[i][0]);
    iso3d[i][1]=fixmul(fixcos(phi), iso2d[i][1]);
    fan[i][0]=fixdiv(itofix((rand()%100-50)*(rand()%100-50)), itofix(2500));
    fan[i][1]=fixdiv(itofix((rand()%100-50)*(rand()%100-50)), itofix(2500));
  }
}

SN_PGEN *sn_create_pgen (void)
{
   SN_PGEN *g = (SN_PGEN *) malloc (sizeof (SN_PGEN));
   g->colours = NULL;
   g->ptcs = NULL;
   g->count = g->max = g->alpha = g->size = 0;
   if (firsttime) {
     firsttime=0;
     firstinit();
   }

   return g;
}

void sn_destroy_pgen (SN_PGEN * g)
{
   if (g)
     {
        if (g->colours)
           free (g->colours);
        if (g->ptcs)
           free (g->ptcs);
        free (g);
     }
}

void ptc_draw (struct ptc *p, SN_PGEN * g, BITMAP * bmp)
{
   int clr, x, y;
   x = fixtoi (p->x);
   y = fixtoi (p->y);
   if (p->life > 0)
     {
        p->x = fixadd(p->x, p->dx);
        p->y =fixadd(p->y, p->dy);
        --p->life;

        clr = g->colours[p->life];
        if (g->size == 0)
          {
             putpixel (bmp, x, y, clr);
          }
        else
          {
             circlefill (bmp, x, y, g->size, clr);
          }
     }
}

struct ptc *pgen_find (SN_PGEN * g)
{
    int i, j;
   for (j = 0; j < g->max; ++j)
     {

       i=(j+g->last)%g->max;

        if (g->ptcs[i].life <= 0)
          {
        g->last=i;
             if (i > g->count)
                g->count = i;
             return &g->ptcs[i];
          }
     }
   return NULL;
}

int pgen_init_ex (SN_PGEN * g, int size, int life, int speed, int alpha, int (*getramp) (int),
                  int cap)
{
   int i;
   g->size = size;
   g->life = life;
   g->speed=speed;
   g->alpha=alpha;
   g->colours = (int*)malloc (sizeof (int) * life);
   for (i = 0; i < life; ++i)
     {
        g->colours[i] = getramp (i * 256 / life);
     }
   g->max = cap;
   g->ptcs = (struct ptc*)malloc (g->max * sizeof (struct ptc));
   for (i=0; i<cap; ++i) {
     g->ptcs[i].life=-1;
   }
   return 0;
}

int myramp (int x)
{
   int r = x;
   int g = 2 * x - 256;
   int b = 3 * x - 512;
   if (g < 0)
      g = 0;
   if (b < 0)
      b = 0;
   return makecol (r, g, b);
}
int myramp2 (int x)
{
   int r = 255 - x;
   int g = 255 - x * x;
   int b = 255 - x * x * x;
   if (r < 0)
      r = 0;
   if (g < 0)
      g = 0;
   if (b < 0)
      b = 0;
   return makecol (r, g, b);
}

int sn_init_pgen (SN_PGEN * g, int size, int life, int speed)
{
   return pgen_init_ex (g, size, life, speed, 0, myramp, 256);
}


int sn_fire_pgen (SN_PGEN * g, int xi, int yi, int angle, int count)
{
  fixed dx, dy;
  int i;
  static int ctr=0;
  struct ptc *p;
  dx = fixcos(itofix(angle));
  dy = fixsin(itofix (angle));
  for (i = 0; i < count; ++i)
    {
      ctr=(ctr+1)%256;
      p = pgen_find (g);
      if (p == NULL)
    return i;
      p->x = itofix (xi);
      p->y = itofix (yi);
      p->life = g->life;
      switch(angle) {
      case SN_PGEN_ISO2D:

    p->dx = fixmul (itofix (g->speed), iso2d[ctr][0]);
    p->dy = fixmul (itofix (g->speed), iso2d[ctr][1]);
    break;
      case SN_PGEN_ISO3D:
    p->dx = fixmul (itofix (g->speed), iso3d[ctr][0]);
    p->dy = fixmul (itofix (g->speed), iso3d[ctr][1]);
    break;
      default:
    p->dx = fixmul (itofix (g->speed), fixadd(dx, fan[ctr][0]));
    p->dy = fixmul (itofix (g->speed), fixadd(dy, fan[ctr][1]));
    break;
      }
    }
  return count;
}

int sn_display_pgen (SN_PGEN * g, BITMAP * bmp)
{
   int i;

   for (i = 0; i < g->count; ++i)
     {
        ptc_draw (&g->ptcs[i], g, bmp);
     }
   return 0;
}



/* --- menu --- */
int sn_menu(FONT *tf, char *title, FONT *stf, char *subtitle,
            unsigned int n_opt, FONT* of, char *options, int col1, int col2)
{
    char** opt_addr;
    int i, a, oy, y, yinc, o, tm;

    tm = text_mode(-1);

    if(title)
        textout_centre(screen, tf, title, SCREEN_W / 2, 10, col1);
    if(subtitle)
        textout_centre(screen, stf, subtitle, SCREEN_W / 2, text_height(tf) + 20, col1);

    if(n_opt == 0)
    {
        clear_keybuf();
        readkey();
        return 0;
    }

    opt_addr = (char**)malloc(sizeof(char *) * n_opt);
    opt_addr[0] = options;
    a = 1;
    for(i = 0; a < (int)n_opt; ++i)
    {
        if(options[i] == '\0')
        {
            opt_addr[a] = options + i + 1;
            ++a;
        }
    }

    oy = text_height(tf) + text_height(stf) + 30;
    oy += (SCREEN_H - oy) / 2 - ((text_height(of) + 10) * n_opt / 2 - 10);
    y = oy;
    yinc = text_height(of) + 10;

    for(i = 0; i < (int)n_opt; i++)
    {
        textout_centre(screen, of, opt_addr[i], SCREEN_W / 2, y, col1);
        y += yinc;
    }

    o = 0;
    while(!key[KEY_ENTER] && !key[KEY_SPACE])
    {
        textout_centre(screen, of, opt_addr[o], SCREEN_W / 2, oy + yinc * o, col2);
        clear_keybuf();
        while(!keypressed())    yield_timeslice();
        textout_centre(screen, of, opt_addr[o], SCREEN_W / 2, oy + yinc * o, col1);
        if(key[KEY_UP])     o = MAX(0, o - 1);
        if(key[KEY_DOWN])   o = MIN((int)n_opt - 1, o + 1);
        if(key[KEY_ESC])
        {
            o = n_opt - 1;
            break;
        }
    }

    free(opt_addr);
    text_mode(tm);

    return o;
}



/* --- hiscore --- */
//Creates a hiscore list and returns it
//Ex.
//SN_HISCORE_LIST *hl=CreateHiscoreList(5,"hl.txt");
//Creates a hiscore list with 5 elements and saves and loads it's elements from hl.txt
SN_HISCORE_LIST *sn_create_hiscore_list(int size,char *filename)
{
    SN_HISCORE_LIST *hl=(SN_HISCORE_LIST *)malloc(sizeof(SN_HISCORE_LIST));

    hl->hiscores=(SN_HISCORE *)malloc(sizeof(SN_HISCORE)*size);
    strcpy(hl->filename,filename);
    hl->size=size;

    sn_reset_hiscore_list(hl);
    sn_load_hiscore_list(hl);

    return hl;
}

//Free all allocated memory
void sn_destroy_hiscore_list(SN_HISCORE_LIST *hl)
{
    free(hl->hiscores);
    free(hl);
}

//Resets a hiscore list and sets all names to "" and the score to 0
void sn_reset_hiscore_list(SN_HISCORE_LIST *hl)
{
    int i;

    for(i=0;i<hl->size;i++)
    {
        strcpy(hl->hiscores[i].name,"");
        hl->hiscores[i].score=0;
    }
}

// Load hiscores from disk
void sn_load_hiscore_list(SN_HISCORE_LIST *hl)
{
    FILE *file=NULL;

    if((file=fopen(hl->filename,"rb"))!=NULL)
    {
        fread(hl->hiscores,sizeof(SN_HISCORE),hl->size,file);
        fclose(file);
    }
}

//Save hiscores to disk
void sn_save_hiscore_list(SN_HISCORE_LIST *hl)
{
    FILE *file=NULL;

    if((file=fopen(hl->filename,"wb"))!=NULL)
    {
        fwrite(hl->hiscores,sizeof(SN_HISCORE),hl->size,file);
        fclose(file);
    }
}

//Checks if a given score is a hiscore
int sn_is_hiscore(SN_HISCORE_LIST *hl,int score)
{
    int i;

    for(i=0;i<hl->size;i++)
        if(score>hl->hiscores[i].score)
            return 1;

    return 0;
}

// Inserts a name and a score if it is a hiscore.
// After insertion the hiscore list is saved to disk.
void sn_insert_hiscore(SN_HISCORE_LIST *hl,char *name,int score)
{
    int index=-1;
    int i;
    SN_HISCORE tmp,old;

    for(i=0;i<hl->size;i++)
        if(score>hl->hiscores[i].score)
        {
            index=i;
            break;
        }

    if(index!=-1)
    {// We have a hiscore. Insert it and move all other elements down one step
        strcpy(tmp.name,name);
        tmp.score=score;

        for(i=index;i<hl->size;i++)
        {
            old=hl->hiscores[i];
            hl->hiscores[i]=tmp;
            tmp=old;
        }

        sn_save_hiscore_list(hl);
    }
}



/* --- collide --- */
int sn_collision(BITMAP *s1, int s1x1, int s1y1, BITMAP *s2, int s2x1, int s2y1)
{
    int mask = bitmap_mask_color(s1);
    int s1x2 = s1x1 + s1->w - 1;
    int s1y2 = s1y1 + s1->h - 1;
    int s2x2 = s2x1 + s2->w - 1;
    int s2y2 = s2y1 + s2->h - 1;

    int left = s1x1;
    int right = s1x2;
    int top = s1y1;
    int bottom = s1y2;

    int x, y;

    if(s1x1 > s2x2 || s1x2 < s2x1 || s1y1 > s2y2 || s1y2 < s2y1)
    {
        return FALSE;
    }

    if(s1x1 <= s2x1)    left = s2x1;
    if(s1x2 >= s2x2)    right = s2x2;
    if(s1y1 <= s2y1)    top = s2y1;
    if(s1y2 >= s2y2)    bottom = s2y2;

    for(y = top; y <= bottom; y++)
    {
        for(x = left; x <= right; x++)
        {
            if((getpixel(s1, x - s1x1, y - s1y1) != mask) &&
               (getpixel(s2, x - s2x1, y - s2y1) != mask))
            {
                 return TRUE;
            }
        }
    }

    return FALSE;
}
