/*==============================================================================

  BAMBOOZLE MACHINAE by Inphernic (http://edu.kauhajoki.fi/~juipe)
  for Speedhack 2003
  
  Compilation requires both Allegro and AllegroGL libraries.

==============================================================================*/

/*==============================================================================
 INCLUDES, DEFINITIONS, AND GLOBAL STRUCTURES
==============================================================================*/

#include "allegro.h"
#include "alleggl.h"
#include "math.h"

#define _blockdepth             .1                    // depth of the blocks
#define _blocksize              30                    // block dimensions
#define _gfxcard                GFX_OPENGL_WINDOWED   // "gfxcard to use"
#define _screenwidth            640                   // desired screen width
#define _screenheight           480                   // desired screen height
#define _screendepth            16                    // desired screen depth
#define _stardepth              40                    // this * 4 = star amount
#define _starsize               20                    // size for a single star

// Block definition
typedef struct {
  unsigned char type;          // block type (0=empty, 1-9 different blocks)
  unsigned char killswitch;    // death flag for the block
  unsigned char phase;         // effect counter
} machinae_block;

// Plane definition (a plane consists of n x n blocks)
typedef struct {
  machinae_block *blocks;      // blocks in this plane
  float angle;                 // better name = displacement in x-axis
  float angledir;              // better name = direction of the displacement
  float opacity;               // plane opacity
} machinae_plane;

// Level definition (a level consists of n planes)
typedef struct {
  machinae_plane *planes;      // planes in this level
  unsigned char size;          // level size (size*size*size = amount of blocks)
  unsigned int blocks_curr;    // blocks currently in play
  unsigned int blocks_clear;   // clear limit for this level
  unsigned int blocks_cleared; // amount of cleared blocks
  unsigned int blocks_init;    // how many blocks this level initially had
} machinae_level;

// The Game (global data structure)
struct {
  char killswitch;             // exit flag for the game
  char mitemp;                 // temporary value used for ingame menus
  char raised;                 // flag set if a block is raised
  char armin;                  // cheat mode flag (easter egg)
  machinae_level *current_level; // pointer to current level
  float fx_temp;               // temporary value used for effects
  float fx_temp2;              // same as above
  float camera_x;              // camera x-position
  float camera_y;              // camera y-position
  float camera_z;              // camera z-position
  int time_left, time_init, px, py, keyok, updok, updating; unsigned char phase, levc, level; unsigned int score;
  FONT *font_menu;             // font used for text output
  MIDI *ingame;                // MIDI played ingame
  MIDI *menu;                  // MIDI played in the main menu
  MIDI *endgame;               // MIDI played after the game has ended
  BITMAP *frbmp;               // temporary bitmap for the fractal
  GLuint frtxt;                // fractal texture
} bamboozle_game;

// Highscore list (global data structure)
struct {
  char n[20];                  // name in this entry
  int s;                       // score in this entry
  int l;                       // level reached in this entry
} hisc[10];                    // ten results are tracked

// Wonkastar (global data structure)
struct {
  struct {
    float c[4];                // holds the angle of this star
  } depth[_stardepth];         // the depth of the star (iterations)
} bambi_star;

char name[20]; // a temporary placeholder for highscore list name

/*==============================================================================
 FUNCTION PROTOTYPES AND EXPLANATIONS (IN THE ORDER OF APPEARANCE)
==============================================================================*/

// read highscores from hiscore.dat
void hi_read(void);

// write highscores to hiscore.dat
void hi_write(void);

// ticks 20 times per second
void bambi_fx_timer(void);

// ticks 60 times per second
void bambi_game_timer(void);

// ticks every second
void bambi_game_timer_sec(void);

// updates logic for the Wonkastar
void bambi_star_logic(void);

// initialize the game
void bambi_init(void);

// clear the backbuffer
void bambi_clear(void);

// pageflip backbuffer->screen
void bambi_blit(void);

// deletes a level structure
//    temp - pointer to the level
machinae_level *machinae_delete(machinae_level *temp);

// generates a level (it must be created first!)
//    temp - pointer to the level
//    type_lo - lowest type of block available
//    type_hi - highest type of block available
void machinae_generate(machinae_level *temp, unsigned char type_lo, unsigned char type_hi);

// exit the game
//    message - optional exit message for errors etc
void bambi_exit(char *message);

// creates a level and returns a pointer to it
//    size - size of the level (size*size*size)
//    blocks_clear - block clear limit for this level
machinae_level *machinae_create(unsigned char size, unsigned int blocks_clear);

// draw the front of the block
//    x, y, z - coordinates
void machinae_block_front(float x, float y, float z);

// draw the back of the block
//    x, y, z - coordinates
void machinae_block_back(float x, float y, float z);

// draw the bottom of the block
//    x, y, z - coordinates
void machinae_block_bottom(float x, float y, float z);

// draw the top of the block
//    x, y, z - coordinates
void machinae_block_top(float x, float y, float z);

// draw the left of the block
//    x, y, z - coordinates
void machinae_block_left(float x, float y, float z);

// draw the right of the block
//    x, y, z - coordinates
void machinae_block_right(float x, float y, float z);

// draw the black wireframe to the back of the block (before block gets drawn)
//    x, y, z - coordinates
void machinae_block_wire_back(float x, float y, float z);

// draw the black wireframe to the front of the block (after block gets drawn)
//    x, y, z - coordinates
void machinae_block_wire_front(float x, float y, float z);

// draw the block
//    type - type of the block
//    x, y, z - coordinates
//    o - opacity
void machinae_draw_block(unsigned char type, float x, float y, float z, float o);

// update a plane, drop pieces that should fall
//    temp - pointer to the level
//    plane - number of the plane
void machinae_fall_plane(machinae_level *temp, unsigned char plane);

// draw the plane
//    temp - pointer to the level
//    yp - where to place it in the y-axis
//    plane - number of the plane
void machinae_draw_plane(machinae_level *temp, int yp, unsigned char plane);

// draw the whole level
//    temp - pointer to the level
void machinae_draw_level(machinae_level *temp);

// rotate a plane counter-clockwise
//    temp - pointer to the level
//    plane - number of the plane
void machinae_rotate_plane_ccw(machinae_level *temp, unsigned char plane);

// rotate a plane clockwise
//    temp - pointer to the level
//    plane - number of the plane
void machinae_rotate_plane_cw(machinae_level *temp, unsigned char plane);

// ingame keyboard logic
void machinae_game_keylogic(void);

// kill vertical lines from the level
//    temp - pointer to the level
void machinae_killvertical(machinae_level *temp);

// kill horizontal lines from the level
//    temp - pointer to the level
void machinae_killhorizontal(machinae_level *temp);

// kill all lines from the level
//    temp - pointer to the level
void machinae_killmatrix(machinae_level *temp);

// draw some of the ingame time and statistics (lower hud)
void hud_draw_timer(void);

// draw some of the ingame statistics (higher hud)
//    temp - pointer to the level
void hud_draw_status(machinae_level *temp);

// set time allotted for level completion
//    time - self-explanatory
void bambi_set_time(unsigned char time);

// if the global data structure contains a level, destroys it
void bambi_clear_level(void);

// sets the variables to match those of the given level
//    level - number of the level to set up
void bambi_set_level(unsigned char level);

// remove the ingame timers
void balboa_rem_game_timers(void);

// install the ingame timers
void balboa_set_game_timers(void);

// start a new game
void bambi_new_game(void);

// ends the current game
void bambi_end_game(void);

// leads to game exit
void set_bambi_free(void);

// draw a fire "bullet" to given position
//    x, y, z - coordinates
void bambi_gets_the_bullet(float x, float y, float z);

// draw a blue "bullet" to given position
//    x, y, z - coordinates
void bambi_gets_the_silver_bullet(float x, float y, float z);

// draw a green "bullet" to given position
//    x, y, z - coordinates
void bambi_gets_the_other_bullet(float x, float y, float z);

// draw a red "bullet" to given position
//    x, y, z - coordinates
void bambi_gets_the_red_bullet(float x, float y, float z);

// draw simple menu
//    y - y-position where to draw the menu
//    char1-3 - menu items
void bambi_on_the_menu(unsigned short y, char *item1, char *item2, char *item3);

// draw "centered" text to given position
//    y - y-position where to draw the text
//    str - string to draw
void all_this_for_one_piece_of_text(float y, char *str);

// sets up the first depth in the Wonkastar
//    c - angle value
void set_star(float c);

// draw the Wonkastar
void draw_star(void);

// generate a fractal
void fractalgen(void);

// main function
//    argc - argument count
//    argv - pointers to argument strings
int main(int argc, char *argv[]);

/*==============================================================================
 FUNCTIONS
==============================================================================*/

void hi_read(void) {
  packfile_password("ROFFLELOLASLOMG");
  set_config_file("hiscore.dat");
  strncpy(hisc[0].n, get_config_string("Hiscore", "S0", "Inph"), 20);
  hisc[0].s = get_config_int("Hiscore", "I0", 100000);
  hisc[0].l = get_config_int("Hiscore", "L0", 10);
  strncpy(hisc[1].n, get_config_string("Hiscore", "S1", "Inph"), 20);
  hisc[1].s = get_config_int("Hiscore", "I1", 90000);
  hisc[1].l = get_config_int("Hiscore", "L1", 9);
  strncpy(hisc[2].n, get_config_string("Hiscore", "S2", "Inph"), 20);
  hisc[2].s = get_config_int("Hiscore", "I2", 80000);
  hisc[2].l = get_config_int("Hiscore", "L2", 8);
  strncpy(hisc[3].n, get_config_string("Hiscore", "S3", "Inph"), 20);
  hisc[3].s = get_config_int("Hiscore", "I3", 70000);
  hisc[3].l = get_config_int("Hiscore", "L3", 7);
  strncpy(hisc[4].n, get_config_string("Hiscore", "S4", "Inph"), 20);
  hisc[4].s = get_config_int("Hiscore", "I4", 60000);
  hisc[4].l = get_config_int("Hiscore", "L4", 6);
  strncpy(hisc[5].n, get_config_string("Hiscore", "S5", "Inph"), 20);
  hisc[5].s = get_config_int("Hiscore", "I5", 50000);
  hisc[5].l = get_config_int("Hiscore", "L5", 5);
  strncpy(hisc[6].n, get_config_string("Hiscore", "S6", "Inph"), 20);
  hisc[6].s = get_config_int("Hiscore", "I6", 40000);
  hisc[6].l = get_config_int("Hiscore", "L6", 4);
  strncpy(hisc[7].n, get_config_string("Hiscore", "S7", "Inph"), 20);
  hisc[7].s = get_config_int("Hiscore", "I7", 30000);
  hisc[7].l = get_config_int("Hiscore", "L7", 3);
  strncpy(hisc[8].n, get_config_string("Hiscore", "S8", "Inph"), 20);
  hisc[8].s = get_config_int("Hiscore", "I8", 20000);
  hisc[8].l = get_config_int("Hiscore", "L8", 2);
  strncpy(hisc[9].n, get_config_string("Hiscore", "S9", "Inph"), 20);
  hisc[9].s = get_config_int("Hiscore", "I9", 10000);
  hisc[9].l = get_config_int("Hiscore", "L9", 1);
}

void hi_write(void) {
  packfile_password("ROFFLELOLASLOMG");
  set_config_file("hiscore.dat");
  set_config_string("Hiscore", "S0", hisc[0].n);
  set_config_int("Hiscore", "I0", hisc[0].s);
  set_config_int("Hiscore", "L0", hisc[0].l);
  set_config_string("Hiscore", "S1", hisc[1].n);
  set_config_int("Hiscore", "I1", hisc[1].s);
  set_config_int("Hiscore", "L1", hisc[1].l);
  set_config_string("Hiscore", "S2", hisc[2].n);
  set_config_int("Hiscore", "I2", hisc[2].s);
  set_config_int("Hiscore", "L2", hisc[2].l);
  set_config_string("Hiscore", "S3", hisc[3].n);
  set_config_int("Hiscore", "I3", hisc[3].s);
  set_config_int("Hiscore", "L3", hisc[3].l);
  set_config_string("Hiscore", "S4", hisc[4].n);
  set_config_int("Hiscore", "I4", hisc[4].s);
  set_config_int("Hiscore", "L4", hisc[4].l);
  set_config_string("Hiscore", "S5", hisc[5].n);
  set_config_int("Hiscore", "I5", hisc[5].s);
  set_config_int("Hiscore", "L5", hisc[5].l);
  set_config_string("Hiscore", "S6", hisc[6].n);
  set_config_int("Hiscore", "I6", hisc[6].s);
  set_config_int("Hiscore", "L6", hisc[6].l);
  set_config_string("Hiscore", "S7", hisc[7].n);
  set_config_int("Hiscore", "I7", hisc[7].s);
  set_config_int("Hiscore", "L7", hisc[7].l);
  set_config_string("Hiscore", "S8", hisc[8].n);
  set_config_int("Hiscore", "I8", hisc[8].s);
  set_config_int("Hiscore", "L8", hisc[8].l);
  set_config_string("Hiscore", "S9", hisc[9].n);
  set_config_int("Hiscore", "I9", hisc[9].s);
  set_config_int("Hiscore", "L9", hisc[9].l);
}

void bambi_fx_timer(void) {
  bamboozle_game.fx_temp+=.15;
  bamboozle_game.fx_temp2+=.05;
  if (bamboozle_game.fx_temp>6.28) bamboozle_game.fx_temp-=6.28;
  if (bamboozle_game.fx_temp2>6.28) bamboozle_game.fx_temp2-=6.28;
      bambi_star_logic();
}

void bambi_game_timer(void) {
  bamboozle_game.keyok=1;
  bamboozle_game.updok=1;
  if (bamboozle_game.levc) {
    bamboozle_game.time_left--;
    if (bamboozle_game.time_left<0) bamboozle_game.time_left=0;
    bamboozle_game.score+=20;
  }
}

void bambi_game_timer_sec(void) {
  if (bamboozle_game.time_left>0) if (!bamboozle_game.levc) bamboozle_game.time_left--;
  if (bamboozle_game.time_left<0) bamboozle_game.time_left=0;
}

void bambi_star_logic(void) {
  int i, x;
  for(i=_stardepth-1; i>0; i--) {
    for (x=0; x<4; x++) {
      bambi_star.depth[i].c[x]=bambi_star.depth[i-1].c[x];
    }
  }
}

void bambi_init(void) {
  srand(time(NULL));
  bamboozle_game.killswitch=0;
  allegro_init();
  set_color_depth(_screendepth);
  install_allegro_gl();
  allegro_gl_begin();
  if (_gfxcard==GFX_OPENGL_WINDOWED) allegro_gl_set(AGL_WINDOWED, TRUE);
  else allegro_gl_set(AGL_WINDOWED, FALSE);
  allegro_gl_set(AGL_COLOR_DEPTH, _screendepth);
  allegro_gl_set(AGL_Z_DEPTH, _screendepth);
  allegro_gl_set(AGL_DOUBLEBUFFER, 1);
  allegro_gl_set(AGL_SUGGEST, AGL_DOUBLEBUFFER | AGL_COLOR_DEPTH | AGL_RENDERMETHOD | AGL_Z_DEPTH | AGL_COLOR_DEPTH | AGL_WINDOWED);
  allegro_gl_end();
  if (_gfxcard==GFX_OPENGL_WINDOWED) set_gfx_mode(GFX_OPENGL_WINDOWED, _screenwidth, _screenheight, 0, 0);
  else set_gfx_mode(GFX_OPENGL, _screenwidth, _screenheight, 0, 0);
  allegro_gl_begin();
  glClearColor(0,0,.4,0);
  glHint(GL_FOG_HINT, GL_NICEST);
  glHint(GL_POLYGON_SMOOTH_HINT, GL_DONT_CARE);
  glHint(GL_PERSPECTIVE_CORRECTION_HINT,GL_NICEST);
  glShadeModel(GL_SMOOTH);
  glPolygonMode(GL_FRONT, GL_FILL);
  glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LEQUAL);
  glMatrixMode(GL_PROJECTION); glLoadIdentity();
  glFrustum(-320.0, 320.0, -240, 240, 1, 600.0);
  glFrontFace(GL_CCW);
  glClearDepth(60.0);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  allegro_gl_end();
  install_keyboard();
  install_int_ex(bambi_fx_timer, BPS_TO_TIMER(20));
  install_int_ex(bambi_game_timer, BPS_TO_TIMER(60));
  install_int_ex(bambi_game_timer_sec, BPS_TO_TIMER(1));
  fractalgen();
  bamboozle_game.keyok=0;
  bamboozle_game.updok=0;
  bamboozle_game.phase=0;
  bamboozle_game.updating=0;
  bamboozle_game.levc=0;
  bamboozle_game.font_menu = allegro_gl_convert_allegro_font(font, AGL_FONT_TYPE_TEXTURED, 120.0);
  install_sound(DIGI_AUTODETECT, MIDI_AUTODETECT, "");
  bamboozle_game.ingame=NULL;
  bamboozle_game.menu=NULL;
  bamboozle_game.endgame=NULL;
  bamboozle_game.ingame=load_midi("ingame.mid");
  bamboozle_game.menu=load_midi("theme.mid");
  bamboozle_game.endgame=load_midi("endgame.mid");
  set_display_switch_mode(SWITCH_BACKGROUND);
  set_window_close_button(1);
  set_window_close_hook(set_bambi_free);
  bamboozle_game.armin=0;
  glEnable(GL_TEXTURE);
}

void bambi_clear(void) {
  allegro_gl_begin();
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  glLoadIdentity();
}

void bambi_blit(void) {
  allegro_gl_flip();
  allegro_gl_end();
}

machinae_level *machinae_delete(machinae_level *temp) {
  unsigned char i;
  if (temp) {
    if (temp->planes) {
      for(i=0; i<temp->size; i++) {
        if (temp->planes[i].blocks) {
          free(temp->planes[i].blocks);
          temp->planes[i].blocks=NULL;
        }
      }
      free(temp->planes);
      temp->planes=NULL;
    }
    free(temp);
    temp=NULL;
  }
  return temp;
}

void machinae_generate(machinae_level *temp, unsigned char type_lo, unsigned char type_hi) {
  unsigned char x, y, p;
  if (temp) {
    for (p=0; p<temp->size; p++) {
      for (y=0; y<temp->size; y++) {
        for (x=0; x<temp->size; x++) {
          temp->planes[p].blocks[(y*temp->size)+x].type=type_lo+(rand()%(type_hi-type_lo));
          temp->planes[p].blocks[(y*temp->size)+x].killswitch=0;
          temp->planes[p].blocks[(y*temp->size)+x].phase=0;
          if (temp->planes[p].blocks[(y*temp->size)+x].type) temp->blocks_init++;
        }
      }
    }
  }
}

void bambi_exit(char *message) {
  remove_int(bambi_game_timer_sec);
  remove_int(bambi_fx_timer);
  remove_int(bambi_game_timer);
  bamboozle_game.current_level=machinae_delete(bamboozle_game.current_level);
  bamboozle_game.killswitch=1;
  remove_keyboard();
  remove_allegro_gl();
  if (message) allegro_message("%s", message);
  allegro_exit();
  exit(0);
}

machinae_level *machinae_create(unsigned char size, unsigned int blocks_clear) {
  machinae_level *temp; unsigned char i, error=0;
  if (size<4) size=4;
  temp=malloc(sizeof(machinae_level));
  if (temp) {
    temp->planes=malloc(sizeof(machinae_plane)*size);
    if (temp->planes) {
      for(i=0; i<size; i++) {
        temp->planes[i].angle=0.0;
        temp->planes[i].opacity=0.0;
        temp->planes[i].angledir=0;
        temp->planes[i].blocks=malloc(sizeof(machinae_block)*(size*size));
        if (!temp->planes[i].blocks) error=1;
      }
    }
    else error=1;
  }
  if (error) bambi_exit("Not enough memory to build The Machinae!");
  else { temp->size=size; temp->blocks_clear=blocks_clear; temp->blocks_cleared=0; }
  return temp;
}

void machinae_block_front(float x, float y, float z) {
  glBegin(GL_QUADS);
  glVertex3f(x-_blocksize, y-_blocksize, z);
  glVertex3f(x+_blocksize, y-_blocksize, z);
  glVertex3f(x+_blocksize, y+_blocksize, z);
  glVertex3f(x-_blocksize, y+_blocksize, z);
  glEnd();
}
void machinae_block_back(float x, float y, float z) {
  glBegin(GL_QUADS);
  glVertex3f(x-_blocksize, y-_blocksize, z-_blockdepth);
  glVertex3f(x+_blocksize, y-_blocksize, z-_blockdepth);
  glVertex3f(x+_blocksize, y+_blocksize, z-_blockdepth);
  glVertex3f(x-_blocksize, y+_blocksize, z-_blockdepth);
  glEnd();
}
void machinae_block_bottom(float x, float y, float z) {
  glBegin(GL_QUADS);
  glVertex3f(x-_blocksize, y-_blocksize, z);
  glVertex3f(x+_blocksize, y-_blocksize, z);
  glVertex3f(x+_blocksize, y-_blocksize, z-_blockdepth);
  glVertex3f(x-_blocksize, y-_blocksize, z-_blockdepth);
  glEnd();
}
void machinae_block_top(float x, float y, float z) {
  glBegin(GL_QUADS);
  glVertex3f(x-_blocksize, y+_blocksize, z);
  glVertex3f(x+_blocksize, y+_blocksize, z);
  glVertex3f(x+_blocksize, y+_blocksize, z-_blockdepth);
  glVertex3f(x-_blocksize, y+_blocksize, z-_blockdepth);
  glEnd();
}
void machinae_block_left(float x, float y, float z) {
  glBegin(GL_QUADS);
  glVertex3f(x-_blocksize, y-_blocksize, z);
  glVertex3f(x-_blocksize, y+_blocksize, z);
  glVertex3f(x-_blocksize, y+_blocksize, z-_blockdepth);
  glVertex3f(x-_blocksize, y-_blocksize, z-_blockdepth);
  glEnd();
}
void machinae_block_right(float x, float y, float z) {
  glBegin(GL_QUADS);
  glVertex3f(x+_blocksize, y-_blocksize, z);
  glVertex3f(x+_blocksize, y+_blocksize, z);
  glVertex3f(x+_blocksize, y+_blocksize, z-_blockdepth);
  glVertex3f(x+_blocksize, y-_blocksize, z-_blockdepth);
  glEnd();
}
void machinae_block_wire_back(float x, float y, float z) {
  glColor3f(0,0,0);
  glBegin(GL_LINES);
  glVertex3f(x-_blocksize, y-_blocksize, z-_blockdepth);
  glVertex3f(x-_blocksize, y-_blocksize, z);
  glEnd();
  glBegin(GL_LINES);
  glVertex3f(x+_blocksize, y+_blocksize, z-_blockdepth);
  glVertex3f(x+_blocksize, y+_blocksize, z);
  glEnd();
  glBegin(GL_LINES);
  glVertex3f(x-_blocksize, y+_blocksize, z-_blockdepth);
  glVertex3f(x-_blocksize, y+_blocksize, z);
  glEnd();
  glBegin(GL_LINES);
  glVertex3f(x+_blocksize, y-_blocksize, z-_blockdepth);
  glVertex3f(x+_blocksize, y-_blocksize, z);
  glEnd();
  glBegin(GL_LINES);
  glVertex3f(x-_blocksize, y-_blocksize, z-_blockdepth);
  glVertex3f(x+_blocksize, y-_blocksize, z-_blockdepth);
  glEnd();
  glBegin(GL_LINES);
  glVertex3f(x+_blocksize, y-_blocksize, z-_blockdepth);
  glVertex3f(x+_blocksize, y+_blocksize, z-_blockdepth);
  glEnd();
  glBegin(GL_LINES);
  glVertex3f(x+_blocksize, y+_blocksize, z-_blockdepth);
  glVertex3f(x-_blocksize, y+_blocksize, z-_blockdepth);
  glEnd();
  glBegin(GL_LINES);
  glVertex3f(x-_blocksize, y+_blocksize, z-_blockdepth);
  glVertex3f(x-_blocksize, y-_blocksize, z-_blockdepth);
  glEnd();
}
void machinae_block_wire_front(float x, float y, float z) {
  glColor3f(0,0,0);
  glBegin(GL_LINES);
  glVertex3f(x-_blocksize, y-_blocksize, z);
  glVertex3f(x+_blocksize, y-_blocksize, z);
  glEnd();
  glBegin(GL_LINES);
  glVertex3f(x+_blocksize, y-_blocksize, z);
  glVertex3f(x+_blocksize, y+_blocksize, z);
  glEnd();
  glBegin(GL_LINES);
  glVertex3f(x+_blocksize, y+_blocksize, z);
  glVertex3f(x-_blocksize, y+_blocksize, z);
  glEnd();
  glBegin(GL_LINES);
  glVertex3f(x-_blocksize, y+_blocksize, z);
  glVertex3f(x-_blocksize, y-_blocksize, z);
  glEnd();
}

void machinae_draw_block(unsigned char type, float x, float y, float z, float o) {
  float i=sin(bamboozle_game.fx_temp);
  float j=cos(bamboozle_game.fx_temp);
  switch (type) {
    case 1:
      machinae_block_wire_back(x,y,z);
      glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
      glEnable(GL_BLEND);
      glColor4f(.8,.8,.8,o); machinae_block_front(x,y,z);
      glColor4f(.8,.8,.8,o); machinae_block_back(x,y,z);
      glColor4f(1,1,1,o); machinae_block_top(x,y,z);
      glColor4f(.4,.4,.4,o); machinae_block_bottom(x,y,z);
      glColor4f(.8,.8,.8,o); machinae_block_left(x,y,z);
      glColor4f(.8,.8,.8,o); machinae_block_right(x,y,z);
      glDisable(GL_BLEND);
      machinae_block_wire_front(x,y,z);
      break;
    case 2:
      machinae_block_wire_back(x,y,z);
      glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
      glEnable(GL_BLEND);
      glColor4f(.8,0,0,o); machinae_block_front(x,y,z);
      glColor4f(.8,0,0,o); machinae_block_back(x,y,z);
      glColor4f(1,0,0,o); machinae_block_top(x,y,z);
      glColor4f(.4,0,0,o); machinae_block_bottom(x,y,z);
      glColor4f(.8,0,0,o); machinae_block_left(x,y,z);
      glColor4f(.8,0,0,o); machinae_block_right(x,y,z);
      glDisable(GL_BLEND);
      machinae_block_wire_front(x,y,z);
      break;
    case 3:
      machinae_block_wire_back(x,y,z);
      glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
      glEnable(GL_BLEND);
      glColor4f(.8,.8,0,o); machinae_block_front(x,y,z);
      glColor4f(.8,.8,0,o); machinae_block_back(x,y,z);
      glColor4f(1,1,0,o); machinae_block_top(x,y,z);
      glColor4f(.4,.4,0,o); machinae_block_bottom(x,y,z);
      glColor4f(.8,.8,0,o); machinae_block_left(x,y,z);
      glColor4f(.8,.8,0,o); machinae_block_right(x,y,z);
      glDisable(GL_BLEND);
      machinae_block_wire_front(x,y,z);
      break;
    case 4:
      machinae_block_wire_back(x,y,z);
      glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
      glEnable(GL_BLEND);
      glColor4f(0,.8,0,o); machinae_block_front(x,y,z);
      glColor4f(0,.8,0,o); machinae_block_back(x,y,z);
      glColor4f(0,1,0,o); machinae_block_top(x,y,z);
      glColor4f(0,.4,0,o); machinae_block_bottom(x,y,z);
      glColor4f(0,.8,0,o); machinae_block_left(x,y,z);
      glColor4f(0,.8,0,o); machinae_block_right(x,y,z);
      glDisable(GL_BLEND);
      machinae_block_wire_front(x,y,z);
      break;
    case 5:
      machinae_block_wire_back(x,y,z);
      glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
      glEnable(GL_BLEND);
      glColor4f(0,0,.8,o); machinae_block_front(x,y,z);
      glColor4f(0,0,.8,o); machinae_block_back(x,y,z);
      glColor4f(0,0,1,o); machinae_block_top(x,y,z);
      glColor4f(0,0,.4,o); machinae_block_bottom(x,y,z);
      glColor4f(0,0,.8,o); machinae_block_left(x,y,z);
      glColor4f(0,0,.8,o); machinae_block_right(x,y,z);
      glDisable(GL_BLEND);
      machinae_block_wire_front(x,y,z);
      break;
    case 6:
      machinae_block_wire_back(x,y,z);
      glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
      glEnable(GL_BLEND);
      glColor4f(.8,0,.8,o); machinae_block_front(x,y,z);
      glColor4f(.8,0,.8,o); machinae_block_back(x,y,z);
      glColor4f(1,0,1,o); machinae_block_top(x,y,z);
      glColor4f(.4,0,.4,o); machinae_block_bottom(x,y,z);
      glColor4f(.8,0,.8,o); machinae_block_left(x,y,z);
      glColor4f(.8,0,.8,o); machinae_block_right(x,y,z);
      glDisable(GL_BLEND);
      machinae_block_wire_front(x,y,z);
      break;
    case 7:
      machinae_block_wire_back(x,y,z);
      glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
      glEnable(GL_BLEND);
      glColor4f(0,.8,.8,o); machinae_block_front(x,y,z);
      glColor4f(0,.8,.8,o); machinae_block_back(x,y,z);
      glColor4f(0,1,1,o); machinae_block_top(x,y,z);
      glColor4f(0,.4,.4,o); machinae_block_bottom(x,y,z);
      glColor4f(0,.8,.8,o); machinae_block_left(x,y,z);
      glColor4f(0,.8,.8,o); machinae_block_right(x,y,z);
      glDisable(GL_BLEND);
      machinae_block_wire_front(x,y,z);
      break;
    case 8:
      machinae_block_wire_back(x,y,z);
      glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
      glEnable(GL_BLEND);
      glColor4f(0,.3,.3,o); machinae_block_front(x,y,z);
      glColor4f(0,.3,.3,o); machinae_block_back(x,y,z);
      glColor4f(0,.4,.4,o); machinae_block_top(x,y,z);
      glColor4f(0,.2,.2,o); machinae_block_bottom(x,y,z);
      glColor4f(0,.3,.3,o); machinae_block_left(x,y,z);
      glColor4f(0,.3,.3,o); machinae_block_right(x,y,z);
      glDisable(GL_BLEND);
      machinae_block_wire_front(x,y,z);
      break;
    case 9:
      machinae_block_wire_back(x,y,z);
      glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
      glEnable(GL_BLEND);
      glColor4f(1,i+j,i,o); machinae_block_front(x,y,z);
      glColor4f(1,i+j,i,o); machinae_block_back(x,y,z);
      glColor4f(1,i+j,i,o); machinae_block_top(x,y,z);
      glColor4f(1,i+j,i,o); machinae_block_bottom(x,y,z);
      glColor4f(1,i+j,i,o); machinae_block_left(x,y,z);
      glColor4f(1,i+j,i,o); machinae_block_right(x,y,z);
      glDisable(GL_BLEND);
      machinae_block_wire_front(x,y,z);
      break;
  }
}

void machinae_fall_plane(machinae_level *temp, unsigned char plane) {
  unsigned char x, y, kf=0;
  for (y=0; y<(temp->size-1); y++) {
    for (x=0; x<temp->size; x++) {
      if (temp->planes[plane].blocks[(y*temp->size)+x].type) {
        if (!temp->planes[plane].blocks[((y+1)*temp->size)+x].type) {
          temp->planes[plane].blocks[((y+1)*temp->size)+x].type=temp->planes[plane].blocks[(y*temp->size)+x].type;
          kf=1;
          temp->planes[plane].blocks[(y*temp->size)+x].type=0;
        }
      }
    }
  }
  if (kf) {
    machinae_killmatrix(temp);
  }
}

void machinae_draw_plane(machinae_level *temp, int yp, unsigned char plane) {
  int x, y; float sx, sy;
  if (temp) {
    if (bamboozle_game.updating && temp->planes[plane].opacity<0.9) temp->planes[plane].opacity+=.05;
    if (temp->planes[plane].angle) {
      glPushMatrix();
      glTranslatef(temp->planes[plane].angle*-20,0,0);
    }
    for (y=0; y<temp->size; y++) {
      for (x=0; x<temp->size; x++) {
        sx = 0-(((temp->size*(_blocksize*2))/2)-_blocksize)+(x*(_blocksize*2));
        sy = (-2-(temp->size*_blockdepth))+(y*_blockdepth);
        if (temp->planes[plane].blocks[(y*temp->size)+x].killswitch && bamboozle_game.updating) {
          temp->planes[plane].blocks[(y*temp->size)+x].phase++;
          if (temp->planes[plane].blocks[(y*temp->size)+x].phase>100) {
            temp->planes[plane].blocks[(y*temp->size)+x].phase=0;
            temp->planes[plane].blocks[(y*temp->size)+x].killswitch=0;
            temp->planes[plane].blocks[(y*temp->size)+x].type=0;
            temp->blocks_cleared++;
          }
        }
        if (plane==bamboozle_game.py && x==bamboozle_game.px && bamboozle_game.raised && (y==temp->size-1)) {
          machinae_draw_block(temp->planes[plane].blocks[(y*temp->size)+x].type, sx, yp, sy+(temp->planes[plane].blocks[(y*temp->size)+x].phase/400.0)+.1, temp->planes[plane].opacity-((temp->planes[plane].opacity/200.0)*temp->planes[plane].blocks[(y*temp->size)+x].phase));
          bambi_gets_the_bullet(sx, yp, sy+(temp->planes[plane].blocks[(y*temp->size)+x].phase/500.0)+.1);
        }
        else {
          machinae_draw_block(temp->planes[plane].blocks[(y*temp->size)+x].type, sx, yp, sy+(temp->planes[plane].blocks[(y*temp->size)+x].phase/400.0), temp->planes[plane].opacity-((temp->planes[plane].opacity/200.0)*temp->planes[plane].blocks[(y*temp->size)+x].phase));
          if (plane==bamboozle_game.py && x==bamboozle_game.px) bambi_gets_the_bullet(sx, yp, sy+(temp->planes[plane].blocks[(y*temp->size)+x].phase/500.0));
        }
        if (temp->planes[plane].blocks[(y*temp->size)+x].type) {
          temp->blocks_curr++;
        }
      }
    }
    if (temp->planes[plane].angle) {
      glPopMatrix();
      if (bamboozle_game.updating) temp->planes[plane].angle+=temp->planes[plane].angledir;
      if (temp->planes[plane].angle<=-90.0) {
        for (y=0; y<temp->size; y++) {
        for (x=0; x<temp->size; x++) {
          if (temp->planes[plane].blocks[(y*temp->size)+x].killswitch) {
            temp->planes[plane].blocks[(y*temp->size)+x].phase=0;
            temp->planes[plane].blocks[(y*temp->size)+x].killswitch=0;
            temp->planes[plane].blocks[(y*temp->size)+x].type=0;
            temp->blocks_cleared++;
          }
        }
        }
        temp->planes[plane].angle=0;
        temp->planes[plane].angledir=0;
        machinae_rotate_plane_ccw(temp, plane);
        temp->planes[plane].opacity=0;
        machinae_killmatrix(temp);
      }
      if (temp->planes[plane].angle>=90.0) {
        for (y=0; y<temp->size; y++) {
        for (x=0; x<temp->size; x++) {
          if (temp->planes[plane].blocks[(y*temp->size)+x].killswitch) {
            temp->planes[plane].blocks[(y*temp->size)+x].phase=0;
            temp->planes[plane].blocks[(y*temp->size)+x].killswitch=0;
            temp->planes[plane].blocks[(y*temp->size)+x].type=0;
            temp->blocks_cleared++;
          }
        }
        }
        temp->planes[plane].angle=0;
        temp->planes[plane].angledir=0;
        machinae_rotate_plane_cw(temp, plane);
        temp->planes[plane].opacity=0;
        machinae_killmatrix(temp);
      }
    }
  }
}

void machinae_draw_level(machinae_level *temp) {
  unsigned char y;
  if (temp) {
    temp->blocks_curr=0;
    for (y=0; y<temp->size; y++) {
      machinae_fall_plane(temp, y);
      machinae_draw_plane(temp, (((temp->size*(_blocksize*2))/2)-_blocksize)-((_blocksize*2)*y), y);
    }
  }
}

void machinae_rotate_plane_ccw(machinae_level *temp, unsigned char plane) {
  int x, y, c=0;
  machinae_block *btemp;
  btemp=malloc(sizeof(machinae_block)*(temp->size*temp->size));
  if (temp) {
    for (x=temp->size-1; x>=0; x--) for (y=0; y<temp->size; y++) {
      btemp[c]=temp->planes[plane].blocks[(y*temp->size)+x]; c++;
    }
    free(temp->planes[plane].blocks);
    temp->planes[plane].blocks=btemp;
  }
}

void machinae_rotate_plane_cw(machinae_level *temp, unsigned char plane) {
  int x, y, c=0;
  machinae_block *btemp;
  btemp=malloc(sizeof(machinae_block)*(temp->size*temp->size));
  if (temp) {
    for (x=0; x<temp->size; x++) for (y=temp->size-1; y>=0; y--) {
      btemp[c]=temp->planes[plane].blocks[(y*temp->size)+x]; c++;
    }
    free(temp->planes[plane].blocks);
    temp->planes[plane].blocks=btemp;
  }
}

void machinae_game_keylogic(void) {
  int temp;
    if (key[KEY_E]) {
      if (!bamboozle_game.current_level->planes[bamboozle_game.py].angledir && bamboozle_game.current_level->planes[bamboozle_game.py].opacity>=0.9) {
        bamboozle_game.current_level->planes[bamboozle_game.py].angle=1.0;
        bamboozle_game.current_level->planes[bamboozle_game.py].angledir=1;
      }
      key[KEY_E]=0;
    }
    if (key[KEY_Q]) {
      if (!bamboozle_game.current_level->planes[bamboozle_game.py].angledir && bamboozle_game.current_level->planes[bamboozle_game.py].opacity>=0.9) {
        bamboozle_game.current_level->planes[bamboozle_game.py].angle=-1.0;
        bamboozle_game.current_level->planes[bamboozle_game.py].angledir=-1;
      }
      key[KEY_Q]=0;
    }
    if (key[KEY_D]) {
      bamboozle_game.camera_x-=20.0;
      bamboozle_game.camera_z+=.1;
      if (bamboozle_game.camera_x<-440.0-((bamboozle_game.current_level->size-5)*20)) bamboozle_game.camera_x=-440.0-((bamboozle_game.current_level->size-5)*20);
    }
    else if (key[KEY_A]) {
      bamboozle_game.camera_x+=20.0;
      bamboozle_game.camera_z+=.1;
      if (bamboozle_game.camera_x>440.0+((bamboozle_game.current_level->size-5)*20)) bamboozle_game.camera_x=440.0+((bamboozle_game.current_level->size-5)*20);
    }
    else { bamboozle_game.camera_x/=1.1; bamboozle_game.camera_z/=1.1; }

    if (bamboozle_game.camera_z>.8) bamboozle_game.camera_z=.8;

    if (!bamboozle_game.raised) {
      if (key[KEY_ESC]) { bamboozle_game.phase=3; }
      if (key[KEY_LEFT]) { bamboozle_game.px--; if (bamboozle_game.px<0) bamboozle_game.px=0; key[KEY_LEFT]=0; }
      if (key[KEY_RIGHT]) { bamboozle_game.px++; if (bamboozle_game.px>bamboozle_game.current_level->size-1) bamboozle_game.px=bamboozle_game.current_level->size-1; key[KEY_RIGHT]=0; }
      if (key[KEY_UP]) { bamboozle_game.py--; if (bamboozle_game.py<0) bamboozle_game.py=0; key[KEY_UP]=0; }
      if (key[KEY_DOWN]) { bamboozle_game.py++; if (bamboozle_game.py>bamboozle_game.current_level->size-1) bamboozle_game.py=bamboozle_game.current_level->size-1; key[KEY_DOWN]=0; }
    }
    else {
      if (key[KEY_LEFT] && bamboozle_game.px>0) {
        if (!bamboozle_game.current_level->planes[bamboozle_game.py].blocks[((bamboozle_game.current_level->size-1)*bamboozle_game.current_level->size)+bamboozle_game.px].killswitch) {
        if (!bamboozle_game.current_level->planes[bamboozle_game.py].blocks[((bamboozle_game.current_level->size-1)*bamboozle_game.current_level->size)+(bamboozle_game.px-1)].killswitch) {
          temp=bamboozle_game.current_level->planes[bamboozle_game.py].blocks[((bamboozle_game.current_level->size-1)*bamboozle_game.current_level->size)+bamboozle_game.px].type;
          bamboozle_game.current_level->planes[bamboozle_game.py].blocks[((bamboozle_game.current_level->size-1)*bamboozle_game.current_level->size)+bamboozle_game.px].type=bamboozle_game.current_level->planes[bamboozle_game.py].blocks[((bamboozle_game.current_level->size-1)*bamboozle_game.current_level->size)+(bamboozle_game.px-1)].type;
          bamboozle_game.current_level->planes[bamboozle_game.py].blocks[((bamboozle_game.current_level->size-1)*bamboozle_game.current_level->size)+(bamboozle_game.px-1)].type=temp;
          bamboozle_game.raised=0;
          machinae_killmatrix(bamboozle_game.current_level);
          if (((int)bamboozle_game.score-20)<0) bamboozle_game.score=0;
          else bamboozle_game.score-=20;
        }
        }
      }
      if (key[KEY_RIGHT] && bamboozle_game.px<(bamboozle_game.current_level->size-1)) {
        if (!bamboozle_game.current_level->planes[bamboozle_game.py].blocks[((bamboozle_game.current_level->size-1)*bamboozle_game.current_level->size)+bamboozle_game.px].killswitch) {
        if (!bamboozle_game.current_level->planes[bamboozle_game.py].blocks[((bamboozle_game.current_level->size-1)*bamboozle_game.current_level->size)+(bamboozle_game.px+1)].killswitch) {
          temp=bamboozle_game.current_level->planes[bamboozle_game.py].blocks[((bamboozle_game.current_level->size-1)*bamboozle_game.current_level->size)+bamboozle_game.px].type;
          bamboozle_game.current_level->planes[bamboozle_game.py].blocks[((bamboozle_game.current_level->size-1)*bamboozle_game.current_level->size)+bamboozle_game.px].type=bamboozle_game.current_level->planes[bamboozle_game.py].blocks[((bamboozle_game.current_level->size-1)*bamboozle_game.current_level->size)+(bamboozle_game.px+1)].type;
          bamboozle_game.current_level->planes[bamboozle_game.py].blocks[((bamboozle_game.current_level->size-1)*bamboozle_game.current_level->size)+(bamboozle_game.px+1)].type=temp;
          bamboozle_game.raised=0;
          machinae_killmatrix(bamboozle_game.current_level);
          if (((int)bamboozle_game.score-20)<0) bamboozle_game.score=0;
          else bamboozle_game.score-=20;
        }
        }
      }
      if (key[KEY_UP] && bamboozle_game.py>0) {
        if (!bamboozle_game.current_level->planes[bamboozle_game.py].blocks[((bamboozle_game.current_level->size-1)*bamboozle_game.current_level->size)+bamboozle_game.px].killswitch) {
        if (!bamboozle_game.current_level->planes[bamboozle_game.py-1].blocks[((bamboozle_game.current_level->size-1)*bamboozle_game.current_level->size)+bamboozle_game.px].killswitch) {
          temp=bamboozle_game.current_level->planes[bamboozle_game.py].blocks[((bamboozle_game.current_level->size-1)*bamboozle_game.current_level->size)+bamboozle_game.px].type;
          bamboozle_game.current_level->planes[bamboozle_game.py].blocks[((bamboozle_game.current_level->size-1)*bamboozle_game.current_level->size)+bamboozle_game.px].type=bamboozle_game.current_level->planes[bamboozle_game.py-1].blocks[((bamboozle_game.current_level->size-1)*bamboozle_game.current_level->size)+bamboozle_game.px].type;
          bamboozle_game.current_level->planes[bamboozle_game.py-1].blocks[((bamboozle_game.current_level->size-1)*bamboozle_game.current_level->size)+bamboozle_game.px].type=temp;
          bamboozle_game.raised=0;
          machinae_killmatrix(bamboozle_game.current_level);
          if (((int)bamboozle_game.score-20)<0) bamboozle_game.score=0;
          else bamboozle_game.score-=20;
        }
        }
      }
      if (key[KEY_DOWN] && bamboozle_game.py<(bamboozle_game.current_level->size-1)) {
        if (!bamboozle_game.current_level->planes[bamboozle_game.py].blocks[((bamboozle_game.current_level->size-1)*bamboozle_game.current_level->size)+bamboozle_game.px].killswitch) {
        if (!bamboozle_game.current_level->planes[bamboozle_game.py+1].blocks[((bamboozle_game.current_level->size-1)*bamboozle_game.current_level->size)+bamboozle_game.px].killswitch) {
          temp=bamboozle_game.current_level->planes[bamboozle_game.py].blocks[((bamboozle_game.current_level->size-1)*bamboozle_game.current_level->size)+bamboozle_game.px].type;
          bamboozle_game.current_level->planes[bamboozle_game.py].blocks[((bamboozle_game.current_level->size-1)*bamboozle_game.current_level->size)+bamboozle_game.px].type=bamboozle_game.current_level->planes[bamboozle_game.py+1].blocks[((bamboozle_game.current_level->size-1)*bamboozle_game.current_level->size)+bamboozle_game.px].type;
          bamboozle_game.current_level->planes[bamboozle_game.py+1].blocks[((bamboozle_game.current_level->size-1)*bamboozle_game.current_level->size)+bamboozle_game.px].type=temp;
          bamboozle_game.raised=0;
          machinae_killmatrix(bamboozle_game.current_level);
          if (((int)bamboozle_game.score-20)<0) bamboozle_game.score=0;
          else bamboozle_game.score-=20;
        }
        }
      }
    }
    if (key[KEY_SPACE]) {
      if (bamboozle_game.raised) bamboozle_game.raised=0;
      else {
        if (bamboozle_game.current_level->planes[bamboozle_game.py].blocks[((bamboozle_game.current_level->size-1)*bamboozle_game.current_level->size)+bamboozle_game.px].type) {
          bamboozle_game.raised=1;
        }
      }
      key[KEY_SPACE]=0;
    }

    if (key[KEY_PGUP] && bamboozle_game.armin) {
      bamboozle_game.armin=2;
      bamboozle_game.levc=1;
      key[KEY_PGUP]=0;
    }
}

void machinae_killvertical(machinae_level *temp) {
  int curt, curc; unsigned char tx, p, x, i;
  for (p=0; p<temp->size; p++) {
    curt=99; curc=0;
    for (x=0; x<temp->size; x++) {
      tx=temp->planes[p].blocks[((temp->size-1)*temp->size)+x].type;
      if (tx==curt && !temp->planes[p].blocks[((temp->size-1)*temp->size)+x].phase && curt!=0) curc++;
      if (tx!=curt || temp->planes[p].blocks[((temp->size-1)*temp->size)+x].phase || (x==(temp->size-1))) {
        if (curc>=3) {
          for(i=0; i<curc; i++) {
            if (x!=(temp->size-1)) temp->planes[p].blocks[((temp->size-1)*temp->size)+(x-1-i)].killswitch=1;
            else {
              if (tx==curt) temp->planes[p].blocks[((temp->size-1)*temp->size)+(x-i)].killswitch=1;
              else temp->planes[p].blocks[((temp->size-1)*temp->size)+(x-1-i)].killswitch=1;
            }
          }
          bamboozle_game.score+=(curc*100)*(curc-2);
        }
        curt=tx; curc=1;
      }
    }
  }
}

void machinae_killhorizontal(machinae_level *temp) {
  int curt, curc; unsigned char tx, p, x, i;
  for (x=0; x<temp->size; x++) {
    curt=99; curc=0;
    for (p=0; p<temp->size; p++) {
      tx=temp->planes[p].blocks[((temp->size-1)*temp->size)+x].type;
      if (tx==curt && !temp->planes[p].blocks[((temp->size-1)*temp->size)+x].phase && curt!=0) curc++;
      if (tx!=curt || temp->planes[p].blocks[((temp->size-1)*temp->size)+x].phase || (p==(temp->size-1))) {
        if (curc>=3) {
          for(i=0; i<curc; i++) {
            if (p!=(temp->size-1)) temp->planes[p-1-i].blocks[((temp->size-1)*temp->size)+x].killswitch=1;
            else {
              if (tx==curt) temp->planes[p-i].blocks[((temp->size-1)*temp->size)+x].killswitch=1;
              else temp->planes[p-1-i].blocks[((temp->size-1)*temp->size)+x].killswitch=1;
            }
          }
          bamboozle_game.score+=(curc*100)*(curc-2);
        }
        curt=tx; curc=1;
      }
    }
  }
}

void machinae_killmatrix(machinae_level *temp) {
  if (temp) {
    machinae_killhorizontal(temp);
    machinae_killvertical(temp);
  }
}

void hud_draw_timer(void) {
  char temp[80];
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glEnable(GL_BLEND);
  glBegin(GL_QUADS);
  glColor4f(1,.5,0,0);
  glVertex3f(-320-bamboozle_game.camera_x, -235, -1.15-bamboozle_game.camera_z);
  glVertex3f(320-bamboozle_game.camera_x, -235, -1.15-bamboozle_game.camera_z);
  glColor4f(1,0,0,1);
  glVertex3f(320-bamboozle_game.camera_x, -225, -1.15-bamboozle_game.camera_z);
  glVertex3f(-320-bamboozle_game.camera_x, -225, -1.15-bamboozle_game.camera_z);
  glEnd();
  glBegin(GL_QUADS);
  glColor4f(1,1,1,0);
  glVertex3f(-320-bamboozle_game.camera_x, -240, -1.15-bamboozle_game.camera_z);
  glVertex3f(-320-bamboozle_game.camera_x+((640.0/bamboozle_game.time_init)*bamboozle_game.time_left), -240, -1.15-bamboozle_game.camera_z);
  glColor4f(0,1,0,1);
  glVertex3f(-320-bamboozle_game.camera_x+((640.0/bamboozle_game.time_init)*bamboozle_game.time_left), -220, -1.15-bamboozle_game.camera_z);
  glVertex3f(-320-bamboozle_game.camera_x, -220, -1.15-bamboozle_game.camera_z);
  glEnd();
  glDisable(GL_BLEND);
  sprintf(temp,"%02d/%02d",bamboozle_game.time_left,bamboozle_game.time_init);
  all_this_for_one_piece_of_text(-0.9, temp);
  sprintf(temp,"Score: %08d  Level : %02d   ",bamboozle_game.score,bamboozle_game.level);
  all_this_for_one_piece_of_text(-0.7, temp);
  if (bamboozle_game.armin==2) {
    sprintf(temp,"THIS IS NOT GIANA SISTERS   ");
    all_this_for_one_piece_of_text(0.7, temp);
    sprintf(temp,"BUT I'LL LET IT PASS THIS TIME    ");
    all_this_for_one_piece_of_text(0.6, temp);
  }
  glTranslatef(bamboozle_game.camera_x,bamboozle_game.camera_y,bamboozle_game.camera_z);

}

void hud_draw_status(machinae_level *temp) {
  int tx; char stemp[8];
  tx=(int)temp->blocks_clear-(int)temp->blocks_cleared;
  if (tx<0) tx=0;
  else {
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glEnable(GL_BLEND);
    glBegin(GL_QUADS);
    glColor4f(1,.5,0,0);
    glVertex3f(-320-bamboozle_game.camera_x, 235, -1.15-bamboozle_game.camera_z);
    glVertex3f(320-bamboozle_game.camera_x, 235, -1.15-bamboozle_game.camera_z);
    glColor4f(1,0,0,1);
    glVertex3f(320-bamboozle_game.camera_x, 225, -1.15-bamboozle_game.camera_z);
    glVertex3f(-320-bamboozle_game.camera_x, 225, -1.15-bamboozle_game.camera_z);
    glEnd();
    glBegin(GL_QUADS);
    glColor4f(1,1,1,0);
    glVertex3f(-320-bamboozle_game.camera_x, 240, -1.15-bamboozle_game.camera_z);
    glVertex3f(-320-bamboozle_game.camera_x+((640.0/temp->blocks_clear)*tx), 240, -1.15-bamboozle_game.camera_z);
    glColor4f(1,1,0,1);
    glVertex3f(-320-bamboozle_game.camera_x+((640.0/temp->blocks_clear)*tx), 220, -1.15-bamboozle_game.camera_z);
    glVertex3f(-320-bamboozle_game.camera_x, 220, -1.15-bamboozle_game.camera_z);
    glEnd();
    glDisable(GL_BLEND);
    sprintf(stemp,"%03d/%03d",temp->blocks_cleared,temp->blocks_clear);
    all_this_for_one_piece_of_text(0.93, stemp);
    glTranslatef(bamboozle_game.camera_x,bamboozle_game.camera_y,bamboozle_game.camera_z);
  }
}

void bambi_set_time(unsigned char time) {
  bamboozle_game.time_init=time;
  bamboozle_game.time_left=time;
}

void bambi_clear_level(void) {
  if (bamboozle_game.current_level) bamboozle_game.current_level=machinae_delete(bamboozle_game.current_level);
}

void bambi_set_level(unsigned char level) {
  bamboozle_game.level=level;
  if (bamboozle_game.armin==2) bamboozle_game.armin=1;
  switch(level) {
    case 1:
      bamboozle_game.current_level=machinae_create(5, 40);
      machinae_generate(bamboozle_game.current_level, 1, 6);
      bambi_set_time(90);
      break;
    case 2:
      bamboozle_game.current_level=machinae_create(5, 50);
      machinae_generate(bamboozle_game.current_level, 1, 6);
      bambi_set_time(90);
      break;
    case 3:
      bamboozle_game.current_level=machinae_create(6, 60);
      machinae_generate(bamboozle_game.current_level, 1, 7);
      bambi_set_time(85);
      break;
    case 4:
      bamboozle_game.current_level=machinae_create(6, 70);
      machinae_generate(bamboozle_game.current_level, 1, 7);
      bambi_set_time(80);
      break;
    case 5:
      bamboozle_game.current_level=machinae_create(7, 80);
      machinae_generate(bamboozle_game.current_level, 1, 8);
      bambi_set_time(75);
      break;
    case 6:
      bamboozle_game.current_level=machinae_create(7, 90);
      machinae_generate(bamboozle_game.current_level, 1, 8);
      bambi_set_time(70);
      break;
    case 7:
      bamboozle_game.current_level=machinae_create(8, 100);
      machinae_generate(bamboozle_game.current_level, 1, 9);
      bambi_set_time(65);
      break;
    case 8:
      bamboozle_game.current_level=machinae_create(8, 120);
      machinae_generate(bamboozle_game.current_level, 1, 9);
      bambi_set_time(60);
      break;
    case 9:
      bamboozle_game.current_level=machinae_create(9, 140);
      machinae_generate(bamboozle_game.current_level, 1, 10);
      bambi_set_time(50);
      break;
    case 10:
      bamboozle_game.current_level=machinae_create(9, 200);
      machinae_generate(bamboozle_game.current_level, 1, 10);
      bambi_set_time(40);
      break;
  }
  bamboozle_game.px=0;
  bamboozle_game.py=0;
  bamboozle_game.raised=0;
  machinae_killmatrix(bamboozle_game.current_level);
}

void balboa_rem_game_timers(void) {
  remove_int(bambi_game_timer_sec);
  remove_int(bambi_game_timer);
}

void balboa_set_game_timers(void) {
  install_int_ex(bambi_game_timer_sec, BPS_TO_TIMER(1));
  install_int_ex(bambi_game_timer, BPS_TO_TIMER(60));
}

void bambi_new_game(void) {
  stop_midi();
  bambi_clear_level();
  bambi_set_level(1);
  bamboozle_game.phase=2;
  balboa_set_game_timers();
  play_midi(bamboozle_game.ingame, 1);
}

void bambi_end_game(void) {
  stop_midi();
  balboa_rem_game_timers();
  bambi_clear_level();
  bamboozle_game.phase=0;
}

void set_bambi_free(void) {
  stop_midi();
  bamboozle_game.killswitch=1;
  bamboozle_game.phase=99;
}

void bambi_gets_the_bullet(float x, float y, float z) {
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glEnable(GL_BLEND);
  glBegin(GL_TRIANGLE_FAN);
  glColor4f(1,1,0,1);
  glVertex3f(x, y, z);
  glColor4f(1,0,0,0);
  glVertex3f(x, y-_starsize, z);
  glVertex3f(x+_starsize, y, z);
  glVertex3f(x, y+_starsize, z);
  glVertex3f(x-_starsize, y, z);
  glVertex3f(x, y-_starsize, z);
  glEnd();
  glDisable(GL_BLEND);
}

void bambi_gets_the_silver_bullet(float x, float y, float z) {
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glEnable(GL_BLEND);
  glBegin(GL_TRIANGLE_FAN);
  glColor4f(.7,.7,1,1);
  glVertex3f(x, y, z);
  glColor4f(0,0,1,0);
  glVertex3f(x, y-_starsize, z);
  glVertex3f(x+_starsize, y, z);
  glVertex3f(x, y+_starsize, z);
  glVertex3f(x-_starsize, y, z);
  glVertex3f(x, y-_starsize, z);
  glEnd();
  glDisable(GL_BLEND);
}

void bambi_gets_the_other_bullet(float x, float y, float z) {
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glEnable(GL_BLEND);
  glBegin(GL_TRIANGLE_FAN);
  glColor4f(.7,1,.7,.8);
  glVertex3f(x, y, z);
  glColor4f(0,1,0,0);
  glVertex3f(x, y-_starsize, z);
  glVertex3f(x+_starsize, y, z);
  glVertex3f(x, y+_starsize, z);
  glVertex3f(x-_starsize, y, z);
  glVertex3f(x, y-_starsize, z);
  glEnd();
  glDisable(GL_BLEND);
}
void bambi_gets_the_red_bullet(float x, float y, float z) {
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glEnable(GL_BLEND);
  glBegin(GL_TRIANGLE_FAN);
  glColor4f(1,.7,.7,.8);
  glVertex3f(x, y, z);
  glColor4f(1,0,0,0);
  glVertex3f(x, y-_starsize, z);
  glVertex3f(x+_starsize, y, z);
  glVertex3f(x, y+_starsize, z);
  glVertex3f(x-_starsize, y, z);
  glVertex3f(x, y-_starsize, z);
  glEnd();
  glDisable(GL_BLEND);
}

void bambi_on_the_menu(unsigned short y, char *item1, char *item2, char *item3) {
  glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glFrontFace(GL_CCW);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
  glEnable(GL_TEXTURE_2D);
  glEnable(GL_BLEND);
  glBlendFunc(GL_ONE, GL_ONE_MINUS_SRC_COLOR);
  if (item1) allegro_gl_printf(bamboozle_game.font_menu, -.01-(((strlen(item1)+1)*9.0)/320.0), .03, -1, makecol(255,255,255), item1);
  if (item2) allegro_gl_printf(bamboozle_game.font_menu, -.01-(((strlen(item2)+1)*9.0)/320.0), -.07, -1, makecol(255,255,255), item2);
  if (item3) allegro_gl_printf(bamboozle_game.font_menu, -.01-(((strlen(item3)+1)*9.0)/320.0), -.17, -1, makecol(255,255,255), item3);
  glDisable(GL_TEXTURE_2D);
  glDisable(GL_BLEND);
  glMatrixMode(GL_PROJECTION); glLoadIdentity();
  glFrustum(-320.0, 320.0, -240, 240, 1, 600.0);
  glFrontFace(GL_CCW);
  glClearDepth(60.0);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  bambi_gets_the_bullet(-280-(50*cos(bamboozle_game.fx_temp)), y-(bamboozle_game.mitemp*50), -2);
  bambi_gets_the_bullet(280-(-50*cos(bamboozle_game.fx_temp)), y-(bamboozle_game.mitemp*50), -2);
}

void all_this_for_one_piece_of_text(float y, char *str) {
  glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glFrontFace(GL_CCW);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
  glEnable(GL_TEXTURE_2D);
  glEnable(GL_BLEND);
  glBlendFunc(GL_ONE, GL_ONE_MINUS_SRC_COLOR);
  allegro_gl_printf(bamboozle_game.font_menu, -.01-(((strlen(str)+1)*9.0)/320.0), y, -1, makecol(255,255,255), str);
  glDisable(GL_TEXTURE_2D);
  glDisable(GL_BLEND);
  glMatrixMode(GL_PROJECTION); glLoadIdentity();
  glFrustum(-320.0, 320.0, -240, 240, 1, 600.0);
  glFrontFace(GL_CCW);
  glClearDepth(60.0);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
}

void set_star(float c) {
  bambi_star.depth[0].c[0]=c;
  bambi_star.depth[0].c[1]=c+1.57;
  bambi_star.depth[0].c[2]=c+3.14;
  bambi_star.depth[0].c[3]=c+4.71;
}

void draw_star(void) {
  unsigned char i, x;
  for (i=0; i<_stardepth; i++) {
    for (x=0; x<4; x++) {
      bambi_gets_the_other_bullet(
        (sin(bambi_star.depth[i].c[x])*i*i),
        (cos(bambi_star.depth[i].c[x])*i*i),
        -1-(i/10.0)+(1.3*sin(bamboozle_game.fx_temp2))
      );
      bambi_gets_the_red_bullet(
        (sin(bambi_star.depth[i].c[x]/2)*i*i),
        (cos(bambi_star.depth[i].c[x]/2)*i*i),
        -1-(i/10.0)+(1.3*sin(bamboozle_game.fx_temp2/2))
      );
      bambi_gets_the_silver_bullet(
        (sin(-1*bambi_star.depth[i].c[x])*i*i),
        (cos(-1*bambi_star.depth[i].c[x])*i*i),
        -1-(i/10.0)+(1.3*sin(bamboozle_game.fx_temp2))
      );
      bambi_gets_the_red_bullet(
        (sin(-1*bambi_star.depth[i].c[x]/2)*i*i),
        (cos(-1*bambi_star.depth[i].c[x]/2)*i*i),
        -1-(i/10.0)+(1.3*sin(bamboozle_game.fx_temp2/2))
      );
    }
  }
}

void fractalgen(void) {
	float rmin= -2, rmax= .5, imin = -1.25, imax = 1.25, maxsizesq, deltar, deltai, x, y, xsq, ysq, cr, ci;
	short maxsize=2, maxit=256, maxcol=256, maxrow=256, color, row, col;
  bamboozle_game.frbmp=create_bitmap(256,256);
  if (bamboozle_game.frbmp) {
	  maxsizesq=maxsize*maxsize;
	  deltar=(rmax-rmin)/(maxcol-1);
	  deltai=(imax-imin)/(maxrow-1);
	  for (row=0; row<maxrow+1; row++) {
 		  for (col=0; col<maxcol+1; col++) {
 			  x=0; y=0; ci=(imin+row*deltai); cr=(rmin+col*deltar); color=0;
			  do {
				  color++;
				  xsq=x*x;
				  ysq=y*y;
				  if ((xsq+ysq)<(maxsizesq)) { y=x*2*y+ci; x=xsq-ysq+cr; }
			  }
			  while ((color<maxit)&&((xsq+ysq)<(maxsizesq)));
			  if (color>=maxit) color=0;
			  if (color) putpixel(bamboozle_game.frbmp, row, col, makecol(color,color,col));
			  else putpixel(bamboozle_game.frbmp, row, col, 0);
		  }
	  }
	  bamboozle_game.frtxt=allegro_gl_make_texture(bamboozle_game.frbmp);
    destroy_bitmap(bamboozle_game.frbmp);
	}
	else bambi_exit("Not enough memory to build the fractal!");
}

int main(int argc, char *argv[]) {
  int c, h, i, p=0, temp, temp2; char ntf[64];
  bambi_init();
  set_window_title("Bamboozle Machinae - Speedhack '03");
  // Activate the easter egg by starting the game with "-legends_never_die_ARMIN" as the first parameter
  if (argc>=2 && argv[1]) { if (!strcmp(argv[1], "-legends_never_die_ARMIN")) bamboozle_game.armin=1; }
  balboa_rem_game_timers();
  bamboozle_game.mitemp=0;
  hi_read();
  while (!bamboozle_game.killswitch) {
    // Highscore list
    if (bamboozle_game.phase==13) clear_keybuf();
    while (bamboozle_game.phase==13) {
      set_star(sin(bamboozle_game.fx_temp));
      bambi_clear();
      glEnable(GL_TEXTURE_2D);
      glBindTexture(GL_TEXTURE_2D, bamboozle_game.frtxt);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
      glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
      glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
      glBegin(GL_QUADS);
      glTexCoord2f(0,0); glVertex3f(-320, -240, -1);
      glTexCoord2f(1,0); glVertex3f(320, -240, -1);
      glTexCoord2f(1,1); glVertex3f(320, 240, -1);
      glTexCoord2f(0,1); glVertex3f(-320, 240, -1);
      glEnd();
      glDisable(GL_TEXTURE_2D);
      glDisable(GL_DEPTH_TEST);
      glEnable(GL_DEPTH_TEST);
      if (keypressed()) bamboozle_game.phase=0;
      all_this_for_one_piece_of_text(0.9, "HIGHSCORES ");
      all_this_for_one_piece_of_text(0.6, "NAME         LEVEL    SCORE  ");
      for (i=0; i<10; i++) {
        sprintf(ntf, "%-15s %2d %8d  ", hisc[i].n, hisc[i].l, hisc[i].s);
        all_this_for_one_piece_of_text(0.5-(0.1*i), ntf);
      }
      bambi_blit();
      key[KEY_ENTER]=0;
      key[KEY_ESC]=0;
      key[KEY_DOWN]=0;
      key[KEY_UP]=0;
    }
    // Main menu
    if (bamboozle_game.phase==0) { bamboozle_game.mitemp=0; play_midi(bamboozle_game.menu, 1); }
    while (bamboozle_game.phase==0) {
      set_star(sin(bamboozle_game.fx_temp));
      bambi_clear();
      glDisable(GL_DEPTH_TEST);
      draw_star();
      glEnable(GL_DEPTH_TEST);
      if (key[KEY_DOWN]) { bamboozle_game.mitemp++; key[KEY_DOWN]=0; }
      if (key[KEY_UP]) { bamboozle_game.mitemp--; key[KEY_UP]=0; }
      if (key[KEY_ESC]) { bamboozle_game.mitemp=2; key[KEY_ESC]=0; }
      if (key[KEY_ENTER]) {
        if (bamboozle_game.mitemp==0) bambi_new_game();
        if (bamboozle_game.mitemp==1) bamboozle_game.phase=13;
        if (bamboozle_game.mitemp==2) set_bambi_free();
        key[KEY_ENTER]=0;
      }
      if (bamboozle_game.mitemp<0) bamboozle_game.mitemp=2;
      if (bamboozle_game.mitemp>2) bamboozle_game.mitemp=0;
      bambi_on_the_menu(0, "NEW GAME", "HIGHSCORES", "EXIT GAME");
      all_this_for_one_piece_of_text(0.6, "BAMBOOZLE MACHINAE  ");
      all_this_for_one_piece_of_text(0.5, "SPEEDHACK 2003  ");
      if (bamboozle_game.armin) all_this_for_one_piece_of_text(-0.6, "ROFFLEFANT LIKES MILK ");
      all_this_for_one_piece_of_text(-0.7, "INPH SOFTWORKS ");
      bambi_blit();
    }
    // Game
    while (bamboozle_game.phase==2) {
      if (bamboozle_game.updok) { bamboozle_game.updating=1; bamboozle_game.updok=0; }
      bambi_clear();
      if (bamboozle_game.keyok) { bamboozle_game.keyok=0; machinae_game_keylogic(); }
      glTranslatef(bamboozle_game.camera_x,bamboozle_game.camera_y,bamboozle_game.camera_z);
      machinae_draw_level(bamboozle_game.current_level);
      hud_draw_timer();
      hud_draw_status(bamboozle_game.current_level);
      bambi_blit();
      if (bamboozle_game.current_level->blocks_cleared >=bamboozle_game.current_level->blocks_clear) bamboozle_game.levc=1;
      if (bamboozle_game.time_left<=0) {
        if (bamboozle_game.levc) {
          bamboozle_game.levc=0;
          bambi_clear_level();
          bamboozle_game.level++;
          if (bamboozle_game.level<=10) bambi_set_level(bamboozle_game.level);
          else {
            bamboozle_game.phase=5;
          }
        }
        else {
          bamboozle_game.phase=4;
        }
      }
      if (bamboozle_game.updating) bamboozle_game.updating=0;
    }
    // Pause game
    if (bamboozle_game.phase==3) { balboa_rem_game_timers(); bamboozle_game.mitemp=0; midi_pause(); }
    while (bamboozle_game.phase==3) {
      bambi_clear();
      if (key[KEY_DOWN]) { bamboozle_game.mitemp++; key[KEY_DOWN]=0; }
      if (key[KEY_UP]) { bamboozle_game.mitemp--; key[KEY_UP]=0; }
      if (key[KEY_ENTER]) {
        if (bamboozle_game.mitemp==0) bamboozle_game.phase=2;
        if (bamboozle_game.mitemp==1) bamboozle_game.phase=6;
        key[KEY_ENTER]=0;
      }
      if (bamboozle_game.mitemp<0) bamboozle_game.mitemp=1;
      if (bamboozle_game.mitemp>1) bamboozle_game.mitemp=0;
      all_this_for_one_piece_of_text(0.5, "GAME PAUSED. QUIT?  ");
      bambi_on_the_menu(0, "NO", "YES", NULL);
      hud_draw_timer();
      hud_draw_status(bamboozle_game.current_level);
      bambi_blit();
      if (bamboozle_game.phase!=3) { balboa_set_game_timers(); midi_resume(); }
    }
    // Game Over
    if (bamboozle_game.phase==4) { play_midi(bamboozle_game.endgame, 1); }
    while (bamboozle_game.phase==4) {
      while(!key[KEY_SPACE]) {
        bambi_clear();
        set_star(sin(bamboozle_game.fx_temp));
        bambi_clear();
        glDisable(GL_DEPTH_TEST);
        draw_star();
        glEnable(GL_DEPTH_TEST);
        all_this_for_one_piece_of_text(0.6, "GAME OVER ");
        all_this_for_one_piece_of_text(0.3, "BETTER LUCK NEXT TIME ");
        all_this_for_one_piece_of_text(-0.5, "PRESS SPACE ");
        bambi_blit();
      }
      bambi_end_game();
      bamboozle_game.phase=7;
      key[KEY_SPACE]=0;
      clear_keybuf();
    }
    // Game Completed
    while (bamboozle_game.phase==5) {
      bambi_end_game();
      play_midi(bamboozle_game.endgame, 1);
      while(!key[KEY_SPACE]) {
        bambi_clear();
        set_star(sin(bamboozle_game.fx_temp));
        bambi_clear();
        glDisable(GL_DEPTH_TEST);
        draw_star();
        glEnable(GL_DEPTH_TEST);
        all_this_for_one_piece_of_text(0.6, "GAME COMPLETE! ");
        all_this_for_one_piece_of_text(0.4, "I HOPE YOU LIKED THE GAME ");
        all_this_for_one_piece_of_text(0.3, "AS MUCH AS I DID MAKING IT! ");
        all_this_for_one_piece_of_text(-0.5, "PRESS SPACE ");
        bambi_blit();
      }
      bamboozle_game.phase=7;
      key[KEY_SPACE]=0;
      clear_keybuf();
    }
    // Just back to main menu
    while (bamboozle_game.phase==6) {
      bambi_end_game();
      play_midi(bamboozle_game.endgame, 1);
      bamboozle_game.phase=7;
    }
    // Highscore, enter it plz
    while (bamboozle_game.phase==7) {
      bamboozle_game.phase=13; i=0;
      key[KEY_ESC]=0;
      h=-1;
      for (c=9; c>=0; c--) if ((int)bamboozle_game.score>hisc[c].s) h=c;
      c=0;
      clear_keybuf();
      while(h>-1) {
        bambi_clear();
        set_star(sin(bamboozle_game.fx_temp));
        bambi_clear();
        glDisable(GL_DEPTH_TEST);
        draw_star();
        glEnable(GL_DEPTH_TEST);
        all_this_for_one_piece_of_text(0.5, "NEW HIGHSCORE! ");
        all_this_for_one_piece_of_text(0.4, "ENTER YOUR NAME  ");
        sprintf(ntf, "<%s>", name);
        all_this_for_one_piece_of_text(-0.1, ntf);
        bambi_blit();
        if (keypressed()==TRUE) {
          temp = (readkey() & 0xff);
          if (temp == 8) {
            if (c!=19) name[c]=0;
            c--;
            for(temp2=c; temp2<15; temp2++) name[temp2]=0;
            if (c<0) c=0;
            name[c] = 0;
          }
          else if (temp>=32 && temp<=126) {
            if (c!=14) name[c] = temp;
            c++; if (c>14) c=14;
          }
          clear_keybuf();
        }
        if (key[KEY_ENTER]) {
          for(c=8; c>=h; c--) {
            strncpy(hisc[c+1].n, hisc[c].n, 20);
            hisc[c+1].l = hisc[c].l;
            hisc[c+1].s = hisc[c].s;
          }
          name[19]='\0';
          strncpy(hisc[h].n, name, 20);
          hisc[h].l = bamboozle_game.level;
          if (hisc[h].l>10) hisc[h].l=10;
          hisc[h].s = bamboozle_game.score;
          h=-1;
          bamboozle_game.score=0;
          for(c=0; c<20; c++) name[c]=0;
          hi_write(); hi_read();
          key[KEY_ENTER]=0;
        }
        if (i==0) p++;
        if (i==1) p--;
        if (p<0) { p=0; i=0; }
        if (p>50) { p=50; i=1; }
      }
    }
  }
  if (bamboozle_game.frbmp) destroy_bitmap(bamboozle_game.frbmp);
  if (bamboozle_game.ingame) destroy_midi(bamboozle_game.ingame);
  if (bamboozle_game.menu) destroy_midi(bamboozle_game.menu);
  bambi_exit(NULL);
  return 0;
}
END_OF_MAIN();
