// This code is LGPL, see licence.txt for details
//
#include <allegro.h>
#include <stdio.h>
//#include "fortify.h"
#include "mystring.h"
#include "func.h"
#include "symtab.h"
#include "synttree.h"
#include "intcode.h"

// Number: assigns line numbers to this block of intermediate code, starting with ln
void IntInstr::Number (int ln)   {
   IntInstr *num = this;
   while (num != NULL)   {
      num->n = ln++; num = num->next;
   }
}

// Names of the opcodes
char *op_name[] = {
   "OP_NOP",           // no operation
   "OP_PUSH",          // push string [var]
   "OP_GETTOP",        // get string from top of stack (=assign) [var]
   "OP_DISCARD",       // discard top value from the stack
   "OP_FUNCTION",      // call a function
   "OP_JMP",           // unconditional jump [dest]
   "OP_OR",            // jump if TRUE and don't pop the stack else pop it
   "OP_AND",           // jump if FALSE and don't pop the stack else pop it
   "OP_JMPF",          // jump if FALSE [dest]
   "OP_JMPT",          // jump if TRUE [dest]
   "OP_VAR_LESS",      // test if one var is less than another
   "OP_VAR_GREATER",   // test if one var is greater than another
   "OP_VAR_EQUAL",     // test whether two vars are equal
   "OP_BOOL_EQUAL",    // test whether two bools are equal
   "OP_BOOL_NOT",      // flip the TRUE/FALSE status
   "OP_CONCAT",        // concatenate two strings
   "OP_ADD",           // some number stuff...
   "OP_SUB",
   "OP_MUL",
   "OP_DIV",
   "OP_POW",
   "OP_NEG",
   "OP_MOD",
   "JUMPTARGET"        // not an opcode but a jump target;
                     // the target field points to the jump instruction
};

// Show this block of intermediate code
FILE *ff;
extern IntInstr *intcode;

void IntInstr::Show ()   {
   if(this==intcode) {
     ff=fopen("temp.log","wt");
   }
   if(opcode==JUMPTARGET) {
     if (target)  fprintf (ff,"%3d: %s:",n, target);
   }
   else {
     fprintf (ff,"%3d: %s ", n, op_name[opcode]);
     if (str)     fprintf (ff,"%s ", str->name);
     if (target)  fprintf (ff,"%s", target);
   }
   fprintf (ff,"\n");
   if (next != NULL)   next->Show();

   if(this==intcode) {
     fclose(ff);
   }

}

// Concatenates two blocks of instructions
IntInstr *Concatenate (IntInstr *blk1, IntInstr *blk2)  {
   IntInstr *search = blk1;
   while (search->next != NULL)   search = search->next;
   search->next = blk2;
   return blk1;
}
void SetupJT (IntInstr *jt,char *name) {
  static int n = 0;
  if(jt->opcode != JUMPTARGET) return;

  if(name) {
    jt->target=new char[strlen(name)+1];
    strcpy(jt->target,name);
  }
  else {  // create a target name
    jt->target = new char [20];
    sprintf (jt->target, "jmpt%010d", ++n);
  }

}
void SetupJ (IntInstr *jt,char *name) {
  if(jt->opcode == JUMPTARGET) return;

  if(name) {
    jt->target=new char[strlen(name)+1];
    strcpy(jt->target,name);
  }

}

// Prefix a jump target to a block; returns the new block
IntInstr *PrefixJT (IntInstr *blk)   {
   IntInstr *jt = new IntInstr;
   jt->opcode = JUMPTARGET;
   jt->next = blk;
   return jt;
}

// Recursively generate intermediate code
IntInstr *GenIntCode (SyntTree tree)  {
   if(tree==NULL) return new IntInstr (OP_NOP);
   TreeNode *root = tree;
   IntInstr *blk1, *blk2,
            *cond, *jump2else, *thenpart, *jump2end, *elsepart, *endif,
            *beginloop, *endloop, *jump2begin;

   switch (root->type)  {
   case STMT_LIST:
      blk1 = GenIntCode (root->child[0]);
      blk2 = GenIntCode (root->child[1]);
      Concatenate (blk1, blk2);
      return blk1;
   case EMPTY_STMT:
      return new IntInstr (OP_NOP);
   case LABEL_STMT:
      blk1 = new IntInstr (JUMPTARGET);
      SetupJT(blk1,root->symbol->name);
      return blk1;
   case GOTO_STMT:
      blk1 = new IntInstr (OP_JMP);
      SetupJ(blk1,root->symbol->name);
      return blk1;
   case EXPR_STMT:
      blk1 = GenIntCode (root->child[0]);
      blk2 = new IntInstr (OP_DISCARD);
      return Concatenate (blk1, blk2);
   case FUNCTION_STMT:
      blk1 = GenIntCode (root->child[0]);
      blk2 = new IntInstr (OP_FUNCTION);
      blk2->call=root->call;
      return Concatenate (blk1, blk2);
   case ARG_LIST:
      blk1 = GenIntCode (root->child[1]);
      blk2 = GenIntCode (root->child[0]);
      
      return Concatenate (blk1, blk2);
   case WHILE_STMT:
      // First, create the necessary code parts
      beginloop = new IntInstr (JUMPTARGET);
      cond      = GenIntCode (root->child[0]);
      jump2end  = new IntInstr (OP_JMPF);      // set target below
      thenpart  = GenIntCode (root->child[1]);
      jump2begin = new IntInstr (OP_JMP);
      endloop     = new IntInstr (JUMPTARGET);
      
      // Set up the jumps
      SetupJT(beginloop,NULL);
      SetupJT(endloop,NULL);
      SetupJ(jump2end,endloop->target);
      SetupJ(jump2begin,beginloop->target);
      
      // Now, concatenate them all
      Concatenate (beginloop, cond);
      Concatenate (cond, jump2end);
      Concatenate (jump2end, thenpart);
      Concatenate (thenpart, jump2begin);
      Concatenate (jump2begin, endloop);
      return beginloop;
   case DO_STMT:
      // First, create the necessary code parts
      beginloop = new IntInstr (JUMPTARGET);
      thenpart  = GenIntCode (root->child[1]);
      cond      = GenIntCode (root->child[0]);
      jump2begin = new IntInstr (OP_JMPT);
      endloop = new IntInstr (JUMPTARGET);

      // Set up the jumps
      SetupJT(beginloop,NULL);
      SetupJT(endloop,NULL);
      SetupJ(jump2begin,beginloop->target);
      
      // Now, concatenate them all
      Concatenate (beginloop, thenpart);
      Concatenate (thenpart, cond);
      Concatenate (cond, jump2begin);
      Concatenate (jump2begin, endloop);
      return beginloop;
   case IFTHEN_STMT:
      // First, create the necessary code parts
      cond      = GenIntCode (root->child[0]);
      jump2end  = new IntInstr (OP_JMPF);      // set target below
      thenpart  = GenIntCode (root->child[1]);
      endif     = new IntInstr (JUMPTARGET);

      // Set up the jumps
      SetupJT(endif,NULL);
      SetupJ(jump2end,endif->target);

      // Now, concatenate them all
      Concatenate (cond, jump2end);
      Concatenate (jump2end, thenpart);
      Concatenate (thenpart, endif);
      return cond;
   case IFTHENELSE_STMT:
      // First, create the necessary code parts
      cond      = GenIntCode (root->child[0]);
      jump2else = new IntInstr (OP_JMPF);      // set target below
      thenpart  = GenIntCode (root->child[1]);
      jump2end  = new IntInstr (OP_JMP);       // set target below
      elsepart  = PrefixJT(GenIntCode (root->child[2]));
      endif     = new IntInstr (JUMPTARGET);

      // Set up the jumps
      SetupJT(elsepart,NULL);
      SetupJT(endif,NULL);
      SetupJ(jump2else,elsepart->target);
      SetupJ(jump2end,endif->target);

      // Now, concatenate them all
      Concatenate (cond, jump2else);
      Concatenate (jump2else, thenpart);
      Concatenate (thenpart, jump2end);
      Concatenate (jump2end, elsepart);
      Concatenate (elsepart, endif);
      return cond;
   case ERROR_STMT:
      return new IntInstr (OP_NOP);
   case EQUAL_EXPR:
      blk1 = GenIntCode (root->child[0]);
      blk2 = GenIntCode (root->child[1]);
      Concatenate (blk1, blk2);
      if (root->child[0]->rettype == T_VAR)
         return Concatenate (blk1, new IntInstr (OP_VAR_EQUAL));
      else
         return Concatenate (blk1, new IntInstr (OP_BOOL_EQUAL));
   case NOTEQUAL_EXPR:
      blk1 = GenIntCode (root->child[0]);
      blk2 = GenIntCode (root->child[1]);
      Concatenate (blk1, blk2);
      if (root->child[0]->rettype == T_VAR)
         Concatenate (blk1, new IntInstr (OP_VAR_EQUAL));
      else
         Concatenate (blk1, new IntInstr (OP_BOOL_EQUAL));
      return Concatenate (blk1, new IntInstr (OP_BOOL_NOT));
   case LT_EXPR:
      blk1 = GenIntCode (root->child[0]);
      blk2 = GenIntCode (root->child[1]);
      Concatenate (blk1, blk2);
      return Concatenate (blk1, new IntInstr (OP_VAR_LESS));
   case GT_EXPR:
      blk1 = GenIntCode (root->child[0]);
      blk2 = GenIntCode (root->child[1]);
      Concatenate (blk1, blk2);
      return Concatenate (blk1, new IntInstr (OP_VAR_GREATER));
   case AND_EXPR:
      // First, create the necessary code parts
      cond = GenIntCode (root->child[0]);
      jump2end = new IntInstr (OP_AND);      // set target below
      thenpart = GenIntCode (root->child[1]);
      endif = new IntInstr (JUMPTARGET);

      // Set up the jumps
      SetupJT(endif,NULL);
      SetupJ(jump2end,endif->target);
      
      // Now, concatenate them all
      Concatenate (cond, jump2end);
      Concatenate (jump2end, thenpart);
      Concatenate (thenpart, endif);
      return cond;
   case OR_EXPR:
      // First, create the necessary code parts
      cond = GenIntCode (root->child[0]);
      jump2end = new IntInstr (OP_OR);      // set target below
      thenpart = GenIntCode (root->child[1]);
      endif = new IntInstr (JUMPTARGET);

      // Set up the jumps
      SetupJT(endif,NULL);
      SetupJ(jump2end,endif->target);
      
      // Now, concatenate them all
      Concatenate (cond, jump2end);
      Concatenate (jump2end, thenpart);
      Concatenate (thenpart, endif);
      return cond;
   case LTET_EXPR:
      blk1 = GenIntCode (root->child[0]);
      blk2 = GenIntCode (root->child[1]);
      Concatenate (blk1, blk2);
      Concatenate (blk1, new IntInstr (OP_VAR_GREATER));
      return Concatenate (blk1, new IntInstr (OP_BOOL_NOT));
   case GTET_EXPR:
      blk1 = GenIntCode (root->child[0]);
      blk2 = GenIntCode (root->child[1]);
      Concatenate (blk1, blk2);
      Concatenate (blk1, new IntInstr (OP_VAR_LESS));
      return Concatenate (blk1, new IntInstr (OP_BOOL_NOT));
   case NOT_EXPR:
      blk1 = GenIntCode (root->child[0]);
      return Concatenate (blk1, new IntInstr (OP_BOOL_NOT));
   case ASSIGN_EXPR:
      blk1 = GenIntCode (root->child[0]);
      blk2 = new IntInstr (OP_GETTOP, root->symbol);
      return Concatenate (blk1, blk2);
   case CONCAT_EXPR:
      blk1 = GenIntCode (root->child[0]);
      blk2 = GenIntCode (root->child[1]);
      Concatenate (blk1, blk2);
      return Concatenate (blk1, new IntInstr (OP_CONCAT));
   case ADD_EXPR:
      blk1 = GenIntCode (root->child[0]);
      blk2 = GenIntCode (root->child[1]);
      Concatenate (blk1, blk2);
      return Concatenate (blk1, new IntInstr (OP_ADD));
   case SUB_EXPR:
      blk1 = GenIntCode (root->child[0]);
      blk2 = GenIntCode (root->child[1]);
      Concatenate (blk1, blk2);
      return Concatenate (blk1, new IntInstr (OP_SUB));
   case MUL_EXPR:
      blk1 = GenIntCode (root->child[0]);
      blk2 = GenIntCode (root->child[1]);
      Concatenate (blk1, blk2);
      return Concatenate (blk1, new IntInstr (OP_MUL));
   case DIV_EXPR:
      blk1 = GenIntCode (root->child[0]);
      blk2 = GenIntCode (root->child[1]);
      Concatenate (blk1, blk2);
      return Concatenate (blk1, new IntInstr (OP_DIV));
   case NEG_EXPR:
      blk1 = GenIntCode (root->child[0]);
      return Concatenate (blk1, new IntInstr (OP_NEG));
   case MOD_EXPR:
      blk1 = GenIntCode (root->child[0]);
      blk2 = GenIntCode (root->child[1]);
      Concatenate (blk1, blk2);
      return Concatenate (blk1, new IntInstr (OP_MOD));
   case POW_EXPR:
      blk1 = GenIntCode (root->child[0]);
      blk2 = GenIntCode (root->child[1]);
      Concatenate (blk1, blk2);
      return Concatenate (blk1, new IntInstr (OP_POW));
   case IDENT_EXPR:
   case CONST_EXPR:
      return new IntInstr (OP_PUSH, root->symbol);
   case INT2STR_EXPR:
   case INT2REAL_EXPR:
      return new IntInstr (OP_NOP);
   }
   return new IntInstr (OP_NOP); // shouldn't happen
}
