// This code is LGPL, see licence.txt for details
//
#include <math.h>
#include <string.h>
#include <stdio.h>
#include <allegro.h>
#include "helper.h"
#include "gfx_diag.h"

extern "C" {

#include <allegro/internal/aintern.h>



typedef struct FLOODED_LINE      /* store segments which have been flooded */
{
   short flags;                  /* status of the segment */
   short lpos, rpos;             /* left and right ends of segment */
   short y;                      /* y coordinate of the segment */
   short next;                   /* linked list if several per line */
} FLOODED_LINE;


static int flood_count;          /* number of flooded segments */

#define FLOOD_IN_USE             1
#define FLOOD_TODO_ABOVE         2
#define FLOOD_TODO_BELOW         4

#define FLOOD_LINE(c)            (((FLOODED_LINE *)_scratch_mem) + c)



/* flooder:
 *  Fills a horizontal line around the specified position, and adds it
 *  to the list of drawn segments. Returns the first x coordinate after 
 *  the part of the line which it has dealt with.
 */
static int flooder(BITMAP *bmp, int x, int y, int src_color, void *d, void (*proc)(BITMAP *bmp, int x1, int y, int x2, void *d))
{
   FLOODED_LINE *p;
   int left = 0, right = 0;
   unsigned long addr;
   int c;

   /* helper for doing checks in each color depth */
   #define FLOODER(bits, size)                                               \
   {                                                                         \
      /* check start pixel */                                                \
      if ((int)bmp_read##bits(addr+x*size) != src_color)                     \
	 return x+1;                                                         \
									     \
      /* work left from starting point */                                    \
      for (left=x-1; left>=bmp->cl; left--) {                                \
	 if ((int)bmp_read##bits(addr+left*size) != src_color)               \
	    break;                                                           \
      }                                                                      \
									     \
      /* work right from starting point */                                   \
      for (right=x+1; right<bmp->cr; right++) {                              \
	 if ((int)bmp_read##bits(addr+right*size) != src_color)              \
	    break;                                                           \
      }                                                                      \
   }

   if (is_linear_bitmap(bmp)) {     /* use direct access for linear bitmaps */
      addr = bmp_read_line(bmp, y);
      bmp_select(bmp);

      switch (bitmap_color_depth(bmp)) {

	 #ifdef ALLEGRO_COLOR8
	    case 8:
	       FLOODER(8, 1);
	       break;
	 #endif

	 #ifdef ALLEGRO_COLOR16
	    case 15:
	    case 16:
	       FLOODER(16, sizeof(short));
	       break;
	 #endif

	 #ifdef ALLEGRO_COLOR24
	    case 24:
	       FLOODER(24, 3);
	       break;
	 #endif

	 #ifdef ALLEGRO_COLOR32
	    case 32:
	       FLOODER(32, sizeof(long));
	       break;
	 #endif
      }

      bmp_unwrite_line(bmp);
   }
   else {                           /* have to use getpixel() for mode-X */
      /* check start pixel */
      if (getpixel(bmp, x, y) != src_color)
	 return x+1;

      /* work left from starting point */ 
      for (left=x-1; left>=bmp->cl; left--)
	 if (getpixel(bmp, left, y) != src_color)
	    break;

      /* work right from starting point */ 
      for (right=x+1; right<bmp->cr; right++)
	 if (getpixel(bmp, right, y) != src_color)
	    break;
   } 

   left++;
   right--;

   /* draw the line */
   proc(bmp, left, y, right, d);

   /* store it in the list of flooded segments */
   c = y;
   p = FLOOD_LINE(c);

   if (p->flags) {
      while (p->next) {
	 c = p->next;
	 p = FLOOD_LINE(c);
      }

      p->next = c = flood_count++;
      _grow_scratch_mem(sizeof(FLOODED_LINE) * flood_count);
      p = FLOOD_LINE(c);
   }

   p->flags = FLOOD_IN_USE;
   p->lpos = left;
   p->rpos = right;
   p->y = y;
   p->next = 0;

   if (y > bmp->ct)
      p->flags |= FLOOD_TODO_ABOVE;

   if (y+1 < bmp->cb)
      p->flags |= FLOOD_TODO_BELOW;

   return right+2;
}



/* check_flood_line:
 *  Checks a line segment, using the scratch buffer is to store a list of 
 *  segments which have already been drawn in order to minimise the required 
 *  number of tests.
 */
static int check_flood_line(BITMAP *bmp, int y, int left, int right, int src_color, void *d, void (*proc)(BITMAP *bmp, int x1, int y, int x2, void *d))
{
   int c;
   FLOODED_LINE *p;
   int ret = FALSE;

   while (left <= right) {
      c = y;

      for (;;) {
	 p = FLOOD_LINE(c);

	 if ((left >= p->lpos) && (left <= p->rpos)) {
	    left = p->rpos+2;
	    break;
	 }

	 c = p->next;

	 if (!c) {
	    left = flooder(bmp, left, y, src_color, d, proc);
	    ret = TRUE;
	    break; 
	 }
      }
   }

   return ret;
}



/* do_floodfill:
 *  Fills an enclosed area (starting at point x, y) with the specified color.
 */
void do_floodfill(BITMAP *bmp, int x, int y, void *d, void (*proc)(BITMAP *bmp, int x1, int y, int x2, void *d))
{
	floodfill(bmp,10,10,makecol(0,0,200));
	return;
   int src_color;
   int c, done;
   FLOODED_LINE *p;

   /* make sure we have a valid starting point */ 
   if ((x < bmp->cl) || (x >= bmp->cr) || (y < bmp->ct) || (y >= bmp->cb))
      return;

   acquire_bitmap(bmp);

   /* what color to replace? */
   src_color = getpixel(bmp, x, y);

   /* set up the list of flooded segments */
   _grow_scratch_mem(sizeof(FLOODED_LINE) * bmp->cb);
   flood_count = bmp->cb;
   p = (FLOODED_LINE *)_scratch_mem;
   for (c=0; c<flood_count; c++) {
      p[c].flags = 0;
      p[c].lpos = SHRT_MAX;
      p[c].rpos = SHRT_MIN;
      p[c].y = y;
      p[c].next = 0;
   }

   /* start up the flood algorithm */
   flooder(bmp, x, y, src_color, d, proc);

   /* continue as long as there are some segments still to test */
   do {
      done = TRUE;

      /* for each line on the screen */
      for (c=0; c<flood_count; c++) {

	 p = FLOOD_LINE(c);

	 /* check below the segment? */
	 if (p->flags & FLOOD_TODO_BELOW) {
	    p->flags &= ~FLOOD_TODO_BELOW;
	    if (check_flood_line(bmp, p->y+1, p->lpos, p->rpos, src_color, d, proc)) {
	       done = FALSE;
	       p = FLOOD_LINE(c);
	    }
	 }

	 /* check above the segment? */
	 if (p->flags & FLOOD_TODO_ABOVE) {
	    p->flags &= ~FLOOD_TODO_ABOVE;
	    if (check_flood_line(bmp, p->y-1, p->lpos, p->rpos, src_color, d, proc)) {
	       done = FALSE;
	       /* special case shortcut for going backwards */
	       if ((c < bmp->cb) && (c > 0))
		  c -= 2;
	    }
	 }
      }

   } while (!done);

   release_bitmap(bmp);
}
}

struct gradient
{
  RGB c1, c2;    // first and second colors
  float ax,ay;   // the vector of the first point (A)
  float abx,aby; // the normalised vector of the second point (B) from the first point (A)
  float ab;      // the distance between A and B
};

void gradient_filler(BITMAP *bmp,int x1,int y,int x2,void *d)
{
  gradient *g=(gradient *)d;
  int i;
  
  float r1,g1,b1;
  float r2,g2,b2;
  float sr,sg,sb;
  
  float afx,afy,aex,aey;
  float ac,ad;
  float acs;
  
  if(x1>=x2) return;
  
  afx = g->ax - x1;
  afy = g->ay - y;
  aex = g->ax - x2;
  aey = g->ay - y;
  
  ac = afx*g->abx + afy*g->aby;
  ad = aex*g->abx + aey*g->aby;

  if(ac<0) {
    r1=g->c1.r;
    g1=g->c1.g;
    b1=g->c1.b;
  }
  else if(ac>g->ab) {
    r1=g->c2.r;
    g1=g->c2.g;
    b1=g->c2.b;
  }
  else {
    r1=int(g->c1.r*ac/g->ab+g->c2.r*(1-ac/g->ab)+.5);
    g1=int(g->c1.g*ac/g->ab+g->c2.g*(1-ac/g->ab)+.5);
    b1=int(g->c1.b*ac/g->ab+g->c2.b*(1-ac/g->ab)+.5);
  }
  
  if(ad<0) {
    r2=g->c1.r;
    g2=g->c1.g;
    b2=g->c1.b;
  }  
  else if(ad>g->ab) {
    r2=g->c2.r;
    g2=g->c2.g;
    b2=g->c2.b;
  }
  else {
    r2=int(g->c1.r*ad/g->ab+g->c2.r*(1-ad/g->ab)+.5);
    g2=int(g->c1.g*ad/g->ab+g->c2.g*(1-ad/g->ab)+.5);
    b2=int(g->c1.b*ad/g->ab+g->c2.b*(1-ad/g->ab)+.5);
  }
  
  sr=(r2-r1)/float(x2-x1);
  sg=(g2-g1)/float(x2-x1);
  sb=(b2-b1)/float(x2-x1);
  
  acs=(ad-ac)/(x2-x1);
  
  for(i=x1;i<x2;i++,ac+=acs) {
    putpixel(bmp,i,y,makecol(int(r1+.5),int(g1+.5),int(b1+.5)));
    
    if(ac>=0&&ac<g->ab) {
      r1+=sr;
      g1+=sg;
      b1+=sb;
    }
  }
  
}

void do_gradient_fill(BITMAP *bmp, int x, int y, int x1,int y1,int x2,int y2,int c1,int c2,float slope)
{
  gradient g;
  float l,c;
  
  g.c1.r=getr(c1);
  g.c1.g=getg(c1);
  g.c1.b=getb(c1);
  g.c2.r=getr(c2);
  g.c2.g=getg(c2);
  g.c2.b=getb(c2);
  
  g.ax=x1;
  g.ay=y1;
  
  c=sqrt((x2-x1)*(x2-x1)+(y2-y1)*(y2-y1));
  
  g.abx=g.ax-c*sin(slope);
  g.aby=g.ay-c*cos(slope);
  
  g.ab=sqrt(g.abx*g.abx + g.aby*g.aby);
  
  l = 1.0f / sqrt(g.abx*g.abx + g.aby*g.aby); 
  g.abx*=l; 
  g.aby*=l;
  
  do_floodfill(bmp, x, y, (void *)&g, gradient_filler);
  return;
}



int get_text_height(char *text)
{
  int i=1;
  while(*text) {
    while(*text&&*text!='^') text++;
    if(!*text) break;
    text++;
    if(!*text) break;
    if(*text=='n') i++;
    text++;
    
  }
  return font_height()*i;
}
int get_text_width(char *text)
{
  int i,j,t=0;
  char temp[128];
  j=0;
  while(*text) {
//    j=0;
    while(*text&&*text!='^') {temp[j++]=*text;text++;}
    if(!*text) break;
    text++;
    if(!*text) break;

    if(*text=='n') {
      temp[j]=0; j=0;
      i=font_length(temp);
      if(i>t) t=i;
    }
    if(*text=='^') {
      temp[j++]=*text;
    }
    text++;
  }
   
  temp[j]='\0';
  i=font_length(temp);
  if(i>t) t=i; 
   
  return t; 
}

int check_pad(int x)
{
        if(num_joysticks<=0) return FALSE;
        if(!(joy[0].flags&JOYFLAG_DIGITAL))   // we only want gamepads..
          return FALSE;                      // we could add a calibrate later.

        if(joy[0].num_sticks<=0) return FALSE;
        if(joy[0].stick[0].num_axis<=1) return FALSE;

        poll_joystick();

        switch(x) {
          case key_left:
            return joy[0].stick[0].axis[0].d1;
          case key_right:
            return joy[0].stick[0].axis[0].d2;
          case key_up:
            return joy[0].stick[0].axis[1].d1;
          case key_down:
            return joy[0].stick[0].axis[1].d2;
          case key_action:
            if(joy[0].num_buttons<1) return FALSE;
            return joy[0].button[0].b;
          case key_cancel:
            if(joy[0].num_buttons<2) return FALSE;
            return joy[0].button[1].b;
          case key_menu:
            if(joy[0].num_buttons<3) return FALSE;
            return joy[0].button[2].b;
          case key_select:
            if(joy[0].num_buttons<4) return FALSE;
            return joy[0].button[3].b;
          default: return FALSE;
        }
        
}
int check_key(int x)
{
        if(keyboard_needs_poll())
          poll_keyboard();

        switch(x) {
          case key_left:
            return key[KEY_LEFT]||check_pad(x);
          case key_right:
            return key[KEY_RIGHT]||check_pad(x);
          case key_up:
            return key[KEY_UP]||check_pad(x);
          case key_down:
            return key[KEY_DOWN]||check_pad(x);
          case key_action:
            return (key_shifts & KB_ALT_FLAG)||check_pad(x);
          case key_cancel:
            return key[KEY_ESC]||(key_shifts & KB_CTRL_FLAG)||check_pad(x);
          case key_menu:
            return key[KEY_ENTER]||check_pad(x);
          case key_select:
            return (key_shifts & KB_SHIFT_FLAG)||check_pad(x);
          default: return FALSE;
        }
        
}


DATAFILE *fonts;
FONT *old_font;
int total_fonts;

frame frames[MAX_FRAME];
frame *current_frame;
int total_frames=0;
int cf_type;

int get_fonts()
{
  return total_fonts;
}

void set_font(int n)
{

  if(n>=0&&n<total_fonts){
    font=(FONT *)fonts[n].dat;
  }
  else font=old_font;
} 

int font_height()
{
  return text_height(font);
}
int font_length(char *text)
{
  return text_length(font,text);
}

void load_frames()
{
  char filename[80];
  total_frames=0;
  
  sprintf(filename,"gfxdiag%d.dat",total_frames+1);

  while(exists(filename)&&total_frames<MAX_FRAME) {
    frames[total_frames].dat=load_datafile(filename);
    if(!frames[total_frames].dat) {
      set_gfx_mode(GFX_TEXT,0,0,0,0);
      allegro_message("Error in %s\n",filename);
      allegro_exit();exit(1);

    }

    frames[total_frames].left = (BITMAP *)frames[total_frames].dat[dia_left].dat;
    frames[total_frames].top = (BITMAP *)frames[total_frames].dat[dia_up].dat;
    frames[total_frames].bottom = (BITMAP *)frames[total_frames].dat[dia_down].dat;
    frames[total_frames].right = (BITMAP *)frames[total_frames].dat[dia_right].dat;

    frames[total_frames].top_left = (BITMAP *)frames[total_frames].dat[dia_leftup].dat;
    frames[total_frames].top_right = (BITMAP *)frames[total_frames].dat[dia_rightup].dat;
    frames[total_frames].bottom_left = (BITMAP *)frames[total_frames].dat[dia_leftdown].dat;
    frames[total_frames].bottom_right = (BITMAP *)frames[total_frames].dat[dia_rightdown].dat;

    frames[total_frames].c1=makecol(0,0,0);
    frames[total_frames].c2=makecol(0,0,0);
    frames[total_frames].slope=0;

    total_frames++;
    sprintf(filename,"gfxdiag%d.dat",total_frames+1);
  }
  if(total_frames>0) current_frame=&frames[0];
  else current_frame=NULL;

  if(current_frame==NULL) {
    set_gfx_mode(GFX_TEXT,0,0,0,0);
    allegro_message("Error: can't find dialog frames!\n");
    allegro_exit();exit(1);
  }
  cf_type=0;
  old_font=font;
  
  total_fonts=0;
  fonts=load_datafile("fnt.dat");

  if(fonts) {
    while(fonts[total_fonts].type==DAT_FONT) total_fonts++;
  }
  set_font(0);

  frames[0].c1=makecol(0,0,64);
  frames[0].c2=makecol(0,0,255);
  frames[0].slope=0;

}

int get_frames()
{
  return total_frames;
}

int get_current_frame()
{
  for(int i=0;i<total_frames;i++)
    if(current_frame==&frames[i])
      return i;

  return -1;
}

void set_frame(int number)
{ 
  cf_type=number;
  if(number>=0&&number<total_frames)
    current_frame=&frames[number];
  else if(total_frames>0) {
    current_frame=&frames[0];
    cf_type=0;
  }
  else current_frame=NULL;

  if(current_frame==NULL) {
    set_gfx_mode(GFX_TEXT,0,0,0,0);
    allegro_message("Error: can't find dialog frames!\n");
    allegro_exit();exit(1);
  }
  

}

void unload_frames()
{
  int i;

  for(i=0;i<total_frames;i++) {
    unload_datafile(frames[i].dat);
    frames[i].dat=NULL;
  }
  total_frames=0;

  if(fonts)
    unload_datafile(fonts);
}


void progress_bar(BITMAP *bmp,int x,int y,int w,int w2,int h,int c1,int c2)
{
  double r1=getr(c1);
  double g1=getg(c1);
  double b1=getb(c1);
  double r2=getr(c2);
  double g2=getg(c2);
  double b2=getb(c2);
  double sr=(r2-r1)/double(w2-1);
  double sg=(g2-g1)/double(w2-1);
  double sb=(b2-b1)/double(w2-1);
   
  for(int i=0;i<w;i++) {
    vline(bmp,x+i,y,y+h-1,makecol(int(r1),int(g1),int(b1)));
     
    r1+=sr;
    g1+=sg;
    b1+=sb;
  }
   
   
}

/*
void fill_hort_gradient(BITMAP *bmp,int x,int y,int c1,int c2)
{
  int w=1;
  while(w<bmp->w) w<<=1;
  BITMAP *b=create_bitmap(w,2);
  double r1=getr(c1);
  double g1=getg(c1);
  double b1=getb(c1);
  double r2=getr(c2);
  double g2=getg(c2);
  double b2=getb(c2);
  double w1=bmp->w;
  double sr=(r2-r1)/w1;
  double sg=(g2-g1)/w1;
  double sb=(b2-b1)/w1;

  for(int i=0;i<int(w1);i++) {
    vline(b,i,0,bmp->h,makecol(int(r1+.5),int(g1+.5),int(b1+.5)));
      
    r1+=sr;
    g1+=sg;
    b1+=sb;
  }
  drawing_mode(DRAW_MODE_COPY_PATTERN,b,0,0);
 
  floodfill(bmp,x,y,makecol(255,255,255));

  destroy_bitmap(b);
  solid_mode();
}
void fill_vert_gradient(BITMAP *bmp,int x,int y,int c1,int c2)
{
  int h=1;
  while(h<bmp->h) h<<=1;
  
  BITMAP *b=create_bitmap(2,h);
  double r1=getr(c1);
  double g1=getg(c1);
  double b1=getb(c1);
  double r2=getr(c2);
  double g2=getg(c2);
  double b2=getb(c2);
  double h1=bmp->h;
  double sr=(r2-r1)/h1;
  double sg=(g2-g1)/h1;
  double sb=(b2-b1)/h1;

  for(int i=0;i<int(h1);i++) {
    hline(b,0,i,bmp->w,makecol(int(r1+.5),int(g1+.5),int(b1+.5)));
      
    r1+=sr;
    g1+=sg;
    b1+=sb;
  }
  drawing_mode(DRAW_MODE_COPY_PATTERN,b,0,0);
 
  floodfill(bmp,x,y,makecol(255,255,255));

  destroy_bitmap(b);
  solid_mode();
}

void fill_gradient(BITMAP *bmp,int x,int y,int c1,int c2,float slope)
{
  if(slope==HUGE_VAL||slope==-HUGE_VAL) {
    fill_hort_gradient(bmp,x,y,c1,c2);

    return;
  }
  if(slope==0) {
    fill_vert_gradient(bmp,x,y,c1,c2);
    
    return;
  }
  int w=1,h=1;
  while(w<bmp->w) w<<=1;
  while(h<bmp->h) h<<=1;
  
  BITMAP *b=create_bitmap(w,h);
  
  double nx=bmp->w;
  double ny=nx*slope; 
  double r1=getr(c1);
  double g1=getg(c1);
  double b1=getb(c1);
  double r2=getr(c2);
  double g2=getg(c2);
  double b2=getb(c2);
  double h1=bmp->h+ny;
  double sr=(r2-r1)/h1;
  double sg=(g2-g1)/h1;
  double sb=(b2-b1)/h1;

  for(int i=0;i<int(h1);i++) {
    line(b,0,i,int(i+nx+.5),-int(ny+.5),makecol(int(r1+.5),int(g1+.5),int(b1+.5)));
      
    r1+=sr;
    g1+=sg;
    b1+=sb;
  }    
  
  drawing_mode(DRAW_MODE_COPY_PATTERN,b,0,0);
 
  floodfill(bmp,x,y,makecol(255,255,255));

  destroy_bitmap(b);
  solid_mode();

}

*/

void display_text(BITMAP *bmp,int x,int y,int w,char *text,int nc)
{
  int cx=x,cy=y;
  char line[100];
  line[0]=0;
  int nl=0;
  int tc=0;
  int pl=0;
  char a,b;
  int white=makecol(255,255,255);
  int grey=makecol(128,128,128);
  int red=makecol(255,0,0);
  int blue=makecol(0,0,255);
  int green=makecol(0,255,0);
  int yellow=makecol(255,255,0);

  int color=white;

  
  if(nc==-1) nc=strlen(text);
  while(text&&tc<nc) {
    a=*text;tc++;
    text++;
    if(a=='^'&&text) {
      b=*text;tc++;
      text++;
      if(b!='^') {
        if(nl!=0) 
          textout(bmp,font,line,cx,cy,color);
        if(b=='n') {
          cx=x;
          cy+=text_height(font);
          line[0]=0; nl=0;
          pl=0;
        }
        if(b=='r') color=red;
        if(b=='g') color=green;
        if(b=='b') color=blue;
        if(b=='y') color=yellow;
        if(b=='w') color=white;
        if(b=='G') color=grey;

        cx+=text_length(font,line);          
        pl+=text_length(font,line);
        line[0]=0; nl=0;
        continue;
      }
    }
    
    line[nl++]=a;
    line[nl]=0;

    
    if(tc>=nc||text_length(font,line)+pl>w) {
      pl=0;
      if(nl!=0) {
        textout(bmp,font,line,cx,cy,color);
      }
      cx=x;
      cy+=text_height(font);
      
      line[0]=0; nl=0;
    }
  }
  
}
