// This code is LGPL...
#include <allegro.h>
#include <string.h>


#include "alleg.h"
#include "music.h"



#ifdef USE_FMOD           
#include <fmod.h>
#endif

#ifdef USE_DUMB
#include <dumb.h>
#include <aldumb.h>
#endif

struct song
{
  char *name;
  
#ifdef USE_FMOD
  FMUSIC_MODULE *data;   
#endif

#ifdef USE_DUMB
  DUH *data;

  AUDIOSTREAM *stream;
  DUH_RENDERER *dr;
  unsigned short *sptr;
  int spos;
#endif        

  song *next;   
};

struct music_data
{
  song *top;
  song *current;
  int play;
  int loop;

  int frequency;
  int n_channels;
  int buffer_size;
  
  int music_volume;
} song_list;

song *add_song()
{
  song *temp=new song;
  temp->name=NULL;
  temp->next=song_list.top;
  song_list.top=temp;

  #ifdef USE_FMOD
    temp->data=NULL;
  #endif

  #ifdef USE_DUMB
    temp->data=NULL;
    temp->stream=NULL;
    temp->dr=NULL;
  #endif

  return temp;
}

void delete_songs()
{
  music_stop();

  song *temp = song_list.top;
  
  while(temp) {
    song_list.top=song_list.top->next;

    if(temp->name) delete [] temp->name;
    
#ifdef USE_FMOD
    FMUSIC_FreeSong(temp->data);
#endif
#ifdef USE_DUMB
    unload_duh(temp->data);
#endif
    delete temp;
    temp=song_list.top;
  }
}

#ifdef USE_DUMB
volatile static int render_dumb;

void do_dumb()
{
  render_dumb++;
}
END_OF_FUNCTION(do_dumb);

#endif


void music_init()
{   
#ifndef NO_MUSIC
  int i;
  song *temp;
#endif

  
  song_list.top=NULL;
  song_list.current=NULL; 
  
  song_list.play=FALSE; 
  song_list.loop=FALSE;

  song_list.music_volume=get_config_int("[sound]","music_volume",196);
  song_list.frequency=get_config_int("[sound]","music_frequency",22050);
  song_list.buffer_size=get_config_int("[sound]","music_buffer_size",32768);
  song_list.n_channels=get_config_int("[sound]","music_sterio",2);

  #ifdef USE_DUMB
    dumb_resampling_quality =  get_config_int("[sound]","music_quality",0);
    dumb_it_max_to_mix = get_config_int("[sound]","music_max_channels",32);
  #endif

  DATAFILE *music;
  
  #ifdef USE_FMOD
    set_pass();
    music=load_datafile("music.dat");
    packfile_password(NULL);
  
    if(!music)
      return;

    FSOUND_Init(song_list.frequency,32,0);
                 
    for(i=0;music[i].type!=DAT_END;i++)
      if(music[i].type==DAT_IT) {
        temp=add_song();
        if(temp) {
          temp->data=FMUSIC_LoadSongMemory(music[i].dat,music[i].size);
          temp->name=new char[strlen(get_datafile_property(&music[i], DAT_ID('N','A','M','E')))+1];
          strcpy(temp->name,get_datafile_property(&music[i], DAT_ID('N','A','M','E')));
        }
      }
  #endif

  #ifdef USE_DUMB

    LOCK_VARIABLE(render_dumb);
    LOCK_FUNCTION(do_dumb);

    install_int_ex(do_dumb,(1193181*256)/song_list.frequency);

    set_volume_per_voice(1);

    if (install_sound(DIGI_AUTODETECT, MIDI_NONE, NULL)) {
      return;
    }

    dumb_register_dat_it(DUMB_DAT_IT);

    set_pass();
    music=load_datafile("music.dat");
    packfile_password(NULL);
  
    if(!music) {
      dumb_exit();
      return;
    }

    for(i=0;music[i].type!=DAT_END;i++)
      if(music[i].type==DUMB_DAT_IT) {
        temp=add_song();
        if(temp) {
          temp->data=(DUH *)music[i].dat;
          music[i].dat=NULL;
          temp->name=new char[strlen(get_datafile_property(&music[i], DAT_ID('N','A','M','E')))+1];
          strcpy(temp->name,get_datafile_property(&music[i],
                            DAT_ID('N','A','M','E')));
        }
      }
    


  #endif
  
  #ifdef NO_MUSIC
    music=NULL;
  #endif  

  
  unload_datafile(music);
      
  return;
}

void music_close()
{
  delete_songs();

  #ifdef USE_FMOD
    FSOUND_Close();
  #endif           

  #ifdef USE_DUMB
    remove_int(do_dumb);

    dumb_exit();
  #endif
  
  set_config_int("[sound]","music_volume",song_list.music_volume);
  set_config_int("[sound]","music_frequency",song_list.frequency);
  set_config_int("[sound]","music_buffer_size",song_list.buffer_size);
  set_config_int("[sound]","music_lines",song_list.n_channels);

  #ifdef USE_DUMB
    set_config_int("[sound]","music_quality",dumb_resampling_quality);
    set_config_int("[sound]","music_max_channels",dumb_it_max_to_mix);
  #endif

    
}

void music_set(char *song_name,int loop)
{
  song *temp=song_list.top;
  
  while(temp) {
    if(strcmp(song_name,temp->name)==0)
      break;
    temp=temp->next;  
  }                 
  
  song_list.current=temp;
  song_list.loop=loop;  

}

void music_update()
{      
  #ifdef USE_DUMB
    unsigned short *sptr;
    long n;
    long size;

    if(!song_list.current||!song_list.current->dr) return;

    size = song_list.buffer_size * song_list.n_channels;

    // Part 1: The dumb renderer.

    while(render_dumb>0) {
      if(song_list.current->spos+256>=size)
        break;

      song_list.current->spos += song_list.n_channels * duh_render(song_list.current->dr, 16, 1, (float(song_list.music_volume))/256, 65536.0 / song_list.frequency, 256, song_list.current->sptr + song_list.current->spos);

      render_dumb--;
    }

    // Part 2: the allegro buffer 
    
    sptr = (unsigned short *)get_audio_stream_buffer(song_list.current->stream);

    if (!sptr)
      return;

    if(song_list.current->spos<size)
      song_list.current->spos += song_list.n_channels * duh_render(song_list.current->dr, 16, 1, (float(song_list.music_volume))/256, 65536.0 / song_list.frequency, (size-song_list.current->spos-1)/song_list.n_channels, song_list.current->sptr + song_list.current->spos);

    memcpy(sptr,song_list.current->sptr,song_list.current->spos*sizeof(short));

    n=song_list.current->spos;

    song_list.current->spos=0;

    for (; n < size; n++)
      sptr[n] = 0x8000;

    free_audio_stream_buffer(song_list.current->stream);

  #endif
}

void music_play()
{
  song_list.play=TRUE;
#ifdef USE_FMOD
  if(!song_list.current||!song_list.current->data) return;
    
  if(!FMUSIC_IsPlaying(song_list.current->data)) {
    FMUSIC_SetMasterVolume(song_list.current->data,song_list.music_volume);
    FMUSIC_PlaySong(song_list.current->data);
  }
#endif

#ifdef USE_DUMB
  if(!song_list.current||!song_list.current->data) return;

  song_list.current->stream = play_audio_stream(song_list.buffer_size, 16, song_list.n_channels - 1, song_list.frequency, 255, 128);
  
  if(!song_list.current) {
    song_list.play=FALSE;
    return;
  }

  song_list.current->dr = duh_start_renderer(song_list.current->data, song_list.n_channels, 0);

  if (!song_list.current->dr) {
    stop_audio_stream(song_list.current->stream);
    song_list.current->stream=NULL;
    return;
  }

  song_list.current->sptr = new unsigned short [song_list.buffer_size*song_list.n_channels];
  song_list.current->spos = 0;

  render_dumb=0;

#endif
}

void music_stop()
{
  song_list.play=FALSE;

#ifdef USE_FMOD
  if(!song_list.current||!song_list.current->data) return;    
  FMUSIC_StopSong(song_list.current->data);
#endif

#ifdef USE_DUMB
  if(!song_list.current||!song_list.current->dr) return;
  duh_end_renderer(song_list.current->dr);
  stop_audio_stream(song_list.current->stream);
  delete song_list.current->sptr;
  song_list.current->sptr=NULL;
  song_list.current->dr=NULL;
  song_list.current->stream=NULL;
#endif
}

void music_pause()
{  

#ifdef USE_FMOD
  if(!song_list.current||!song_list.current->data) return;
  if(!FMUSIC_GetPaused(song_list.current->data)) 
    FMUSIC_SetPaused(song_list.current->data,TRUE);
#endif

#ifdef USE_DUMB
  if(!song_list.current||!song_list.current->dr) return;

  voice_stop(song_list.current->stream->voice);

#endif
}
void music_unpause()
{    
#ifdef USE_FMOD
  if(!song_list.current||!song_list.current->data) return;
  if(FMUSIC_GetPaused(song_list.current->data)) 
    FMUSIC_SetPaused(song_list.current->data,FALSE);
#endif    

#ifdef USE_DUMB
  if(!song_list.current||!song_list.current->dr) return;

  voice_start(song_list.current->stream->voice);

#endif
}

void music_set_vol(int vol)
{
  song_list.music_volume=vol;
#ifdef USE_FMOD
  if(!song_list.current||!song_list.current->data) return;    
  FMUSIC_SetMasterVolume(song_list.current->data,song_list.music_volume);
#endif  
}

int music_get_vol()
{
  return song_list.music_volume;
}
