/********************************************************************
 *                        AllegroDarts v1.02                         *
 *                               by                                 *
 *               Miran Amon (miran_amon@hotmail.com)                *
 *                  copyright (c) Miran Amon 2003                   *
 ********************************************************************/

#include "game.h"
#include "params.h"
#include "sounds.h"
#include "data.h"

#ifndef ALLEGRO_DOS
static bool closed = false;
static void closehook(void) { closed = true; }
#else
#define closed 0
#endif


#ifdef ALLEGRO_WINDOWS
#include <winalleg.h>
#define YIELD() Sleep(1)
#else
#ifdef ALLEGRO_UNIX
#include <sys/time.h>
static void YIELD(void)
{
	struct timeval tv;
	tv.tv_sec = 0;
	tv.tv_usec = 1;
	select(0, NULL, NULL, NULL, &tv);
}
#else
#define YIELD() yield_timeslice()
#endif
#endif


static volatile int timer = 0;
static void timer_f() {
	++timer;
} END_OF_STATIC_FUNCTION(timer_f);


Game::Game()
	:lastScore(0),
	total(0),
	throwing(0),
	activePlayer(0),
	counter(-1),
	menuOn(true)
{
}


Game::~Game() {
	if (buffer) destroy_bitmap(buffer);
	if (board) delete board;
	if (pad) delete pad;
	if (scoreBoard) delete scoreBoard;
	if (player[0]) delete player[0];
	if (player[1]) delete player[1];
	if (holder) destroy_bitmap(holder);

	Data::Unload();
	Dart::Destroy();
	Sound::Destroy();
	remove_int(timer_f);
	Params::Save("darts.ini");
}


Error Game::Init() {
	if (allegro_init() != 0) return Error(Error::ALLEGRO);
	if (install_keyboard() != 0) return Error(Error::KEYBOARD);
	if (install_timer() != 0) return Error(Error::TIMER);
	if (install_mouse() < 0) return Error(Error::MOUSE);
	if (install_sound(DIGI_AUTODETECT, MIDI_AUTODETECT, NULL) != 0) return Error(Error::SOUND);
	set_volume(-1, -1);
	Params::Load("darts.ini");
#ifndef ALLEGRO_DOS
	set_window_title("AllegroDarts");
	set_window_close_hook(&closehook);
#endif
	set_display_switch_mode(SWITCH_BACKGROUND);
	set_color_depth(Params::bpp);
	if (set_gfx_mode(Params::fullscreen ? GFX_AUTODETECT : GFX_AUTODETECT_WINDOWED, Params::w, Params::h, 0, 0) != 0) {
		int alternative;
		switch (Params::bpp) {
			case 15:	alternative = 16;	break;
			case 16:	alternative = 15;	break;
			case 24:	alternative = 32;	break;
			case 32:	alternative = 24;	break;
			default:	alternative = 16;	break;
		}
		set_color_depth(alternative);
		if (set_gfx_mode(Params::fullscreen ? GFX_AUTODETECT : GFX_AUTODETECT_WINDOWED, Params::w, Params::h, 0, 0) != 0) {
			return Error(Error::GFX);
		}
	}
	buffer = create_bitmap(SCREEN_W, SCREEN_H);
	if (!buffer) return Error(Error::MEMORY);
	
	LOCK_VARIABLE(timer);
	LOCK_FUNCTION(timer_f);
	if (install_int_ex(timer_f, BPS_TO_TIMER(Params::fps)) < 0) return Error(Error::TIMER);

	if (Data::Load() != 0) return Error(Error::MEMORY);
	
	srand((unsigned)time(NULL));
	board = new Board(SCREEN_H);
	if (!board) return Error(Error::MEMORY);
	pad = new Pad(SCREEN_H, (5*SCREEN_H-3*SCREEN_W)/2, SCREEN_W-SCREEN_H);
	if (!pad) return Error(Error::MEMORY);
	if (Dart::Load() != 0) return Error(Error::MEMORY);
	int dx = (SCREEN_W-SCREEN_H)/3;
	dart[0].InitDefaultPosition(SCREEN_H + dx/2, SCREEN_H - (SCREEN_W-SCREEN_H)/4);
	dart[1].InitDefaultPosition(SCREEN_H + 3*dx/2, SCREEN_H - (SCREEN_W-SCREEN_H)/4);
	dart[2].InitDefaultPosition(SCREEN_H + 5*dx/2, SCREEN_H - (SCREEN_W-SCREEN_H)/4);
	ResetDarts();
	
	player[0] = new Player;
	if (!player[0]) return Error(Error::MEMORY);
	player[1] = new Player;
	if (!player[1]) return Error(Error::MEMORY);
	scoreBoard = new ScoreBoard(SCREEN_H, 0);
	if (!scoreBoard) return Error(Error::MEMORY);
	holder = load_bitmap("holder.tga", NULL);
	Sound::Initialize();
	menu.Make();
	return Error(Error::NONE);
}


void Game::Run() {
	bool done = false;
	bool needToRedraw = true;
	
	player[activePlayer]->StartTurn();
	while (!done) {
		while (timer) {
			--timer;
			done = Update();
			needToRedraw = true;
		}
		
		if (needToRedraw) {
			Draw();
			DrawDoubleBuffer();
			needToRedraw = false;
		}
		
		if (closed) done = true;
		YIELD();
	}
}


bool Game::Update() {
	if (menuOn) {
		menu.Update();
		while (keypressed()) {
			int c = readkey() >> 8;
			menu.OnKey(c);
			if (menu.Done()) {
				if (menu.Selection() == MainMenu::NEW_GAME) {
					RestartGame();
				}
				menu.Reset();
				menuOn = false;
			}
			if (menu.WillExit()) {
				return true;
			}
		}
		return false;
	}
	else {
		if (mouse_b && pad->CanThrow() && counter==-1) {
			throwing = Params::fps/(Params::difficulty+4);
		}
		
		if (throwing) {
			int x, y;
			--throwing;
			if (pad->GetMousePos(x, y) || !throwing) {
				throwing = 0;
				int f;
				lastScore = board->GetPoints(x,y,f);
				player[activePlayer]->Throw();
				player[activePlayer]->AwardPoints(lastScore,f);
				if (player[activePlayer]->Score() != Params::maxPoints) {
					total += lastScore;
				}
				if (player[activePlayer]->Score() == 0) {
					Sound::Play(SMP_WIN);
					GameOver();
					return false;
				}
				if (!player[activePlayer]->DartsLeft()) {
					counter = 2*Params::fps;
				}
			}
		}
		
		if (counter > 0) {
			--counter;
		}
		else if (counter == 0) {
			ResetDarts();
			board->DrawBoard();
			counter = -1;
			lastScore = 0;
			total = 0;
			++activePlayer;
			activePlayer %= 2;
			player[activePlayer]->StartTurn();
		}
		
		if (key[KEY_ESC]) {
			menuOn = true;
		}
	}
	
	return false;
}


void Game::Draw() {
	// clear the buffer
	clear(buffer);
	
	// do game specific drawing
	board->Draw(buffer);
	pad->Draw(buffer);
	DrawScore();
	if (holder) blit(holder, buffer, 0, 0, SCREEN_H, SCREEN_H - (SCREEN_W-SCREEN_H)/2, holder->w, holder->h);
	DrawDarts();
	
	// draw the menu
	if (menuOn) {
		menu.Draw(buffer);
	}
	
	// draw the mouse
	DrawMouse();
}


void Game::DrawDoubleBuffer() {
	acquire_screen();
	blit(buffer, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
	release_screen();
}


void Game::DrawMouse() {
	if (throwing) {
		dart[player[activePlayer]->DartsLeft()-1].Move(mouse_x, mouse_y);
	}
	else {
		if (!menuOn) {
			draw_sprite(buffer, Data::cursor, mouse_x, mouse_y);
		}
	}
}


void Game::DrawScore() {
	if (scoreBoard->CanDraw()) {
		scoreBoard->Draw(buffer, activePlayer, player[0]->Score(), player[1]->Score(), lastScore, total);
	}
	else {
		int y = 0;
		int x = SCREEN_H+12;
		int dy = 10;
		int cName = makecol(255,255,255);
		int cActive = makecol(0,255,0);
		textprintf(buffer, font, x, y += dy, cName, "Player 1: %d", player[0]->Score());
		textprintf(buffer, font, x, y += dy, cName, "Player 2: %d", player[1]->Score());
		circlefill(buffer, x - 6, activePlayer == 0 ? y-dy+4 : y+4, 2, cActive);
	
		textprintf(buffer, font, x + 12, y += dy+4, makecol(255,255,0), "points = %d", lastScore);
		textprintf(buffer, font, x + 12, y += dy+4, makecol(255,255,0), "total = %d", total);
		
		if (counter > 0) {
			textprintf(buffer, font, x + 12, SCREEN_H - 30, makecol(255,255,255), "Next player...");
		}
	}
}


void Game::DrawDarts() {
	for (int i=0; i<3; ++i) {
		dart[i].Draw(buffer);
	}
}


void Game::ResetDarts() {
	for (int i=0; i<3; ++i) {
		dart[i].Reset();
	}
}


void Game::GameOver() {
	Draw();
	DrawDoubleBuffer();
	int w = text_length(Data::font1, "press any key to continue...") + 30;
	int h = text_height(Data::font1)*4 + 20;
	int y = (SCREEN_H - h)/2;
	BITMAP *back = create_bitmap(w, h);
	clear_to_color(back, makecol(128,128,144));
	set_trans_blender(0,0,0, 128);
	draw_trans_sprite(screen, back, (SCREEN_W-w)/2, y);
	destroy_bitmap(back);
	textprintf_centre(screen, Data::font2, SCREEN_W/2, y+=10, -1, "Game Over!");
	textprintf_centre(screen, Data::font2, SCREEN_W/2, y+=text_height(Data::font2), -1, "Player %d is the winner!", activePlayer+1);
	textprintf_centre(screen, Data::font1, SCREEN_W/2, y+=2*text_height(Data::font1), -1, "press any key to continue...");
	readkey();
	RestartGame();
	menuOn = false;
}


void Game::RestartGame() {
	player[0]->Reset();
	player[1]->Reset();
	ResetDarts();
	board->DrawBoard();
	activePlayer = 0;
	lastScore = 0;
	total = 0;
	throwing = 0;
	counter = -1;
	player[activePlayer]->StartTurn();
}
