#include "global.h"
#include "bookwidget.h"
using namespace MAS;
using namespace std;

BookWidget::BookWidget()
	:Widget(),
	book(NULL),
	linesPerPage(32),
	nPages(0),
	curPage(0),
	xOffset(0),
	yOffset(0),
	pageHeight(0),
	textAreaHeight(0)
{
	backColor = Color::gray;
}


BookWidget::~BookWidget() {
	book = NULL;
}


void BookWidget::SetBook(Book *book) {
	this->book = book;
	
	if (book) {
		nPages = book->lines.size()/linesPerPage + 1;
		curPage = 0;
		xOffset = 0;
		yOffset = 0;
		pageHeight = 0;
	}
	
	Redraw();
}


void BookWidget::Draw(Bitmap &canvas) {
	canvas.Clear(backColor);

	if (!book || book->lines.size()<1) return;

	// get the font and font colors
	int state = Disabled() ? 2 : (Selected() ? 1 : (HasFocus() ? 3 : 0));
	Font f = GetFont(state);
	Color fg = GetFontColor(state);
	Color bg = GetShadowColor(state);
	int rowheight = (int)(f.TextHeight()*1.25);

	// calculate page layout
	pageHeight = rowheight*linesPerPage;
	textAreaHeight = h() - 9*rowheight;

	int titleX = 40;
	int titleY = rowheight - yOffset;

	static char buf[16];
	usprintf(buf, "%d/%d\0", curPage+1, nPages);
	int pageX = w() - f.GUITextLength(buf) - 40;
	int pageY = titleY;
	
	// print the title
	int maxW = 3*(pageX - titleX)/4;
	maxW = MAX(maxW, 40);
	static char title[1024];
	ustrcpy(title, book->title);
	if (f.GUITextLength(title) > maxW) {
		char *titleName = get_filename(book->title);
		for (int i=ustrlen(book->title)-ustrlen(titleName); i>0; --i) {
			memset((char *)title, 0, 1024);
			ustrncpy(title, book->title, i);
			ustrcat(title, "...\\");
			ustrcat(title, titleName);
			if (f.GUITextLength(title) <= maxW) {
				break;
			}
		}
	}
	f.GUITextout(canvas, title, titleX, titleY, fg, bg, -1, 0);

	// print the current page
	f.GUITextout(canvas, buf, pageX, pageY, fg, bg, -1, 0);
	
	// print a separator
	canvas.Hline(30, 2*rowheight + 2 - yOffset, w() - 30, fg);

	// print the actual text
	if (book->lines.size() > 0) {
		vector<string>::iterator i;
		
		// find the longest line
		int c = 0;
		int maxw = 0;
		for (i = book->lines.begin() + (curPage*linesPerPage); i != book->lines.end() && c < linesPerPage; ++i, ++c) {
			int lw = f.TextLength(i->c_str());
			if (lw > maxw) {
				maxw = lw;
			}
		}

		// render the lines
		int y;
		c = 0;
		for (
			i = book->lines.begin() + (curPage*linesPerPage), y = -yOffset; 
			i != book->lines.end() && y <= canvas.h() && c < linesPerPage; 
			++i, y += rowheight, ++c)
		{
			int x = MAX((w() - maxw)/2 - xOffset, 10);
			RenderLine(canvas, f, i->c_str(), x, y + 5*rowheight, maxw, fg, bg);
			//f.GUITextout(canvas, i->c_str(), 50 - xOffset, y + 5*rowheight, fg, bg, -1, 0);
		}		
	}	

	// another separator
	canvas.Hline(30, (7 + linesPerPage)*rowheight - 2 - yOffset, w() - 30, fg);
	
	// page title again
	titleY = (7 + linesPerPage)*rowheight - yOffset;
	f.GUITextout(canvas, title, titleX, titleY, fg, bg, -1, 0);
	
	// current page again
	pageY = titleY;
	f.GUITextout(canvas, buf, pageX, pageY, fg, bg, -1, 0);
}


bool BookWidget::MsgWantfocus() {
	Widget::MsgWantfocus();
	return true;
}


bool BookWidget::MsgChar(int c) {
	Widget::MsgChar(c);
	
	int newPage = curPage;
	int newYOffset = yOffset;
	int d1 = SCREEN_H/20;
	int d2 = d1/8;
	switch (c>>8) {
		case KEY_LEFT:	--newPage;				break;
		case KEY_RIGHT:	++newPage;				break;
		case KEY_UP:	newYOffset -= d2;		break;
		case KEY_DOWN:	newYOffset += d2;		break;
		case KEY_PGUP:	newYOffset -= d1;		break;
		case KEY_PGDN:	newYOffset += d1;		break;
		case KEY_HOME:	newPage = 0;			break;
		case KEY_END:	newPage = nPages - 1;	break;
		
		default:
			return false;
	};
	
	newPage = MID(0, newPage, nPages-1);
	if (newPage != curPage) {
		curPage = newPage;
		xOffset = 0;
		newYOffset = 0;
		Redraw();
	}
	
	newYOffset = MID(0, newYOffset, MAX(0, pageHeight - textAreaHeight));
	if (newYOffset != yOffset) {
		yOffset = newYOffset;
		Redraw();
	}
	
	return true;
}


void BookWidget::MsgWheel(int d) {
	Widget::MsgWheel(d);
	
	for (int i=0; i<ABS(d); i++) {
		MsgChar((d < 0) ? (KEY_PGUP<<8) : (KEY_PGDN<<8));
	}
}


void BookWidget::MsgLPress() {
	Widget::MsgLPress();
	MsgChar(KEY_RIGHT<<8);
}


void BookWidget::MsgRPress() {
	Widget::MsgRPress();
	MsgChar(KEY_LEFT<<8);
}


void BookWidget::SetColors(Color back, Color font, Color shd) {
	backColor = back;
	for (int i=0; i<4; i++)	SetFontColor(font, shd, i);
}


void BookWidget::GotoPage(int i) {
	curPage = MID(0, i-1, nPages-1);
	Redraw();
}


int BookWidget::CurrentPage() {
	return curPage+1;
}


void BookWidget::RenderLine(Bitmap &canvas, Font &font, const char *line, int x, int y, int maxw, Color &fg, Color &bg) {
	//font.GUITextout(canvas, line, x, y, fg, bg, -1, 0);
	
	int l = ustrlen(line);
	int nSpaces = 0;
	int i;
	for (i=0; i<l; i++) {
		if (line[i] == ' ') {
			++nSpaces;
		}
	}
	
	int diff = maxw - font.TextLength(line);
	if (diff < 0 || diff > 0.2*maxw) {
		font.GUITextout(canvas, line, x, y, fg, bg, -1, 0);
	}
	else {
		float extraSpace = (float)diff / (float)nSpaces;
		float fx = (float)x;
		char buf[] = " \0";
		for (i=0; i<l; i++) {
			buf[0] = line[i];
			font.GUITextout(canvas, buf, (int)fx, y, fg, bg, -1, 0);
			fx += (float)font.TextLength(buf);
			if (line[i] == ' ') {
				fx += extraSpace;
			}
		}
	}
}


void BookWidget::SetLinesPerPage(int v) {
	int line = curPage*linesPerPage;
	linesPerPage = v;
	curPage = line/linesPerPage;
	nPages = book->lines.size()/linesPerPage + 1;
	xOffset = 0;
	yOffset = 0;
	Redraw();
}


int BookWidget::GetLinesPerPage() {
	return linesPerPage;
}


int BookWidget::NumberOfPages() {
	return nPages;
}
