
#ifndef _bitmap_h
#define _bitmap_h

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#ifndef M_PI
#define M_PI		3.14159265358979323846
#endif

#include <windows.h>
#include <mmsystem.h>
#include <gl/gl.h>
#define buffer_count 3

#ifndef rnd
#define rnd() (((rand() ^ rand()) / ((double) RAND_MAX + 1.0)))
#endif
#ifndef crnd
#define crnd() (rnd() * 2.0 - 1.0)
#endif

#define __timer_func() (timeGetTime() / 1000.0)

void die(char *format, ...) {
  char buf[1024];
  va_list ap;
  va_start(ap, format);
  vsprintf(buf, format, ap);
  va_end(ap);
  MessageBox(0, buf, "Error", MB_OK | MB_ICONEXCLAMATION);
  exit(1);
}
void update_frameinfo();

typedef unsigned char uint8;
typedef unsigned short uint16;
typedef unsigned long uint32;

#define BITMAP _BITMAP
class BITMAP { public:
  int w, h, depth;
  uint8 *data;
  uint8 **line;
  unsigned int mask, texture;
  double xtex, ytex;
};

int key[256];

#define KEY_ESC       VK_ESCAPE
#define KEY_LCONTROL  VK_CONTROL
#define KEY_RCONTROL  VK_CONTROL
#define KEY_DOWN      VK_DOWN
#define KEY_UP        VK_UP
#define KEY_LEFT      VK_LEFT
#define KEY_RIGHT     VK_RIGHT
#define KEY_PAUSE     VK_PAUSE
#define KEY_PLUS_PAD  VK_ADD
#define KEY_MINUS_PAD VK_SUBTRACT
#define KEY_SPACE     VK_SPACE
#define KEY_ENTER     VK_RETURN

#define KEY_A 'A'
#define KEY_B 'B'
#define KEY_C 'C'
#define KEY_D 'D'
#define KEY_E 'E'
#define KEY_F 'F'
#define KEY_G 'G'
#define KEY_H 'H'
#define KEY_I 'I'
#define KEY_J 'J'
#define KEY_K 'K'
#define KEY_L 'L'
#define KEY_M 'M'
#define KEY_N 'N'
#define KEY_O 'O'
#define KEY_P 'P'
#define KEY_Q 'Q'
#define KEY_R 'R'
#define KEY_S 'S'
#define KEY_T 'T'
#define KEY_U 'U'
#define KEY_V 'V'
#define KEY_W 'W'
#define KEY_X 'X'
#define KEY_Y 'Y'
#define KEY_Z 'Z'

#define windowClassName "GLWindow"
#define title           "Space"
LRESULT CALLBACK WndProc(HWND, UINT, WPARAM, LPARAM);
void InitGL() {
  glClearColor(0.0, 0.0, 0.0, 0.5);
}
#define xbias 0
#define ybias 0
void ResizeGLScene(int width, int height) {
  glViewport(0, 0, width, height);
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glOrtho(0, width, height, 0, -1, 1);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
}

class gl_window { public:
  int w, h, depth, active;

  HDC hDC;
  HGLRC hRC;
  HWND hWnd;

  gl_window() { depth = 24; }
  void remove() {
    ChangeDisplaySettings(0, 0);
    ShowCursor(1);
    if (hRC) {
      wglMakeCurrent(0, 0);
      wglDeleteContext(hRC);
    }
    if (hDC) { ReleaseDC(hWnd, hDC); }
    if (hWnd) { DestroyWindow(hWnd); }
  }
  ~gl_window() { remove(); }
  int create(int width, int height, int d) {
    hDC = 0; hRC = 0; hWnd = 0;

    w = width;
    h = height;
    depth = d;
    active = 0;

    GLuint PixelFormat;
    WNDCLASS wc;
    DWORD dwExStyle, dwStyle;
    RECT WindowRect;
    WindowRect.left = 0;
    WindowRect.right = width;
    WindowRect.top = 0;
    WindowRect.bottom = height;

    HINSTANCE hInstance = GetModuleHandle(0);
    wc.style         = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
    wc.lpfnWndProc   = (WNDPROC) WndProc;
    wc.cbClsExtra    = 0;
    wc.cbWndExtra    = 0;
    wc.hInstance     = hInstance;
    wc.hIcon         = LoadIcon(0, IDI_WINLOGO);
    wc.hCursor       = LoadCursor(0, IDC_ARROW);
    wc.hbrBackground = 0;
    wc.lpszMenuName  = 0;
    wc.lpszClassName = windowClassName;

    RegisterClass(&wc);
	  
    DEVMODE dmScreenSettings;
    memset(&dmScreenSettings, 0, sizeof(dmScreenSettings));
    dmScreenSettings.dmSize = sizeof(dmScreenSettings);
    dmScreenSettings.dmPelsWidth  = width;
    dmScreenSettings.dmPelsHeight = height;
    dmScreenSettings.dmBitsPerPel = depth;
    dmScreenSettings.dmFields = DM_BITSPERPEL | DM_PELSWIDTH | DM_PELSHEIGHT;

    if (ChangeDisplaySettings(&dmScreenSettings, CDS_FULLSCREEN) != DISP_CHANGE_SUCCESSFUL) {
      remove(); return 0;
//      die("Couldn't select video mode: %dx%dx%d", width, height, depth);
    }
    
    ShowCursor(FALSE);    // hide mouse

    dwExStyle = WS_EX_APPWINDOW;
    dwStyle = WS_POPUP;

    AdjustWindowRectEx(&WindowRect, dwStyle, FALSE, dwExStyle);
    if (!(hWnd = CreateWindowEx(dwExStyle, windowClassName, title, dwStyle | WS_CLIPSIBLINGS | WS_CLIPCHILDREN, 0, 0, WindowRect.right - WindowRect.left, WindowRect.bottom - WindowRect.top, 0, 0, hInstance, 0))) {
      remove(); return 0;
//      die("Window creation error");
    }

    static PIXELFORMATDESCRIPTOR pfd = {      // PFD Contents:
      sizeof(PIXELFORMATDESCRIPTOR),  // Size Of This Pixel Format Descriptor
      1,                              // Version Number
      PFD_DRAW_TO_WINDOW |            // Format Must Support Window
      PFD_SUPPORT_OPENGL |            // Format Must Support OpenGL
      PFD_DOUBLEBUFFER,               // Must Support Double Buffering
      PFD_TYPE_RGBA,                  // Request An RGBA Format
      depth,                           // Select Our Color Depth
      0, 0, 0, 0, 0, 0,               // Color Bits Ignored
      0,                              // No Alpha Buffer
      0,                              // Shift Bit Ignored
      0,                              // No Accumulation Buffer
      0, 0, 0, 0,                     // Accumulation Bits Ignored
      0,                              // 16Bit Z-Buffer (Depth Buffer)  
      0,                              // No Stencil Buffer
      0,                              // No Auxiliary Buffer
      PFD_MAIN_PLANE,                 // Main Drawing Layer
      0,                              // Reserved
      0, 0, 0                         // Layer Masks Ignored
    };
	  
    if (!(hDC = GetDC(hWnd))) { remove(); return 0; /* die("Couldn't create a GL Device Context"); */ }
    if (!(PixelFormat = ChoosePixelFormat(hDC,&pfd))) { remove(); return 0; /* die("Couldn't find pixel format: %d bpp", depth); */ }
    if (!SetPixelFormat(hDC,PixelFormat,&pfd)) { remove(); return 0; /* die("Found, but couldn't select pixel format: %d bpp", depth); */ }
    if (!(hRC = wglCreateContext(hDC))) { remove(); return 0; /* die("Couldn't create a GL Rendering Context"); */ }
    if (!wglMakeCurrent(hDC, hRC)) { remove(); return 0; /* die("Couldn't activate the GL Rendering Context"); */ }

    ShowWindow(hWnd, SW_SHOW);
    SetForegroundWindow(hWnd);      // Higher Priority
    SetFocus(hWnd);
    ResizeGLScene(width, height);

    InitGL();
    return 1;
  }
} _window;

void init_all(int w, int h, int d) {
  if (!_window.create(w, h, 32)) {
    if (!_window.create(w, h, 24)) {
      if (!_window.create(w, h, 16)) {
        die("Could not initialize video mode: %dx%d.", w, h);
      }
    }
  }
}

class MIDI;

void _replay_midi();
MIDI *_active_midi = 0;

char _keybuf[1024];
int _keycount = 0;
LRESULT CALLBACK WndProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam) {
  switch (uMsg) {
    case WM_ACTIVATE:
      _window.active = !HIWORD(wParam) && (LOWORD(wParam) != WA_INACTIVE);
      return 0;
    case WM_SYSCOMMAND:
      if (wParam == SC_SCREENSAVE || wParam == SC_MONITORPOWER) { return 0; }
      break;
    case WM_CLOSE:
      PostQuitMessage(0);
      return 0;
    case WM_SIZE:
      ResizeGLScene(LOWORD(lParam), HIWORD(lParam));
      return 0;
    case WM_KEYDOWN:
      key[wParam] = 1;
      return 0;
    case WM_KEYUP:
      key[wParam] = 0;
      return 0;
    case WM_CHAR:
      _keybuf[_keycount] = (char) wParam;
      if (_keycount < 1024) { _keycount++; }
      return 0;
    case MM_MCINOTIFY:
      if (_active_midi) {
        _replay_midi();
      }
      break;
  }
  return DefWindowProc(hWnd, uMsg, wParam, lParam);
}
int keypressed() { return _keycount != 0; }
int readkey() { if (_keycount) { return _keybuf[--_keycount]; } return 0; }

int bytes_per_pixel(BITMAP *bmp) {
  return bmp->depth / 8;
}

void unmask_bitmap(BITMAP *bmp);
void untexture_bitmap(BITMAP *bmp);
BITMAP *create_bitmap_ex(int depth, int w, int h) {
  BITMAP *result = new BITMAP;
  if (!result) { return 0; }
  result->w = w;
  result->h = h;
  result->depth = depth;
  result->data = new uint8[w * h * bytes_per_pixel(result)];
  if (!result->data) { delete result; return 0; }
  result->line = new uint8 *[h];
  if (!result->line) { delete[] result->data; delete result; return 0; }
  w *= bytes_per_pixel(result);
  for (int y = 0; y < h; y++) {
    result->line[y] = &result->data[y * w];
  }
  result->texture = 0;
  result->mask = 0;
  return result;
}

BITMAP *create_bitmap(int w, int h) { return create_bitmap_ex(_window.depth, w, h); }

void destroy_bitmap(BITMAP *bmp) {
  unmask_bitmap(bmp);
  untexture_bitmap(bmp);
  delete[] bmp->data;
  delete[] bmp->line;
  delete bmp;
}

#define u_getr16(c) (((c) >> 11) & 0x1F)
#define u_getg16(c) (((c) >>  5) & 0x3F)
#define u_getb16(c) ((c) & 0x1F)
#define u_makecol16(r, g, b) (((r) << 11) | ((g) << 5) | (b))
#define u_rmax16 31
#define u_gmax16 63
#define u_bmax16 31
#define u_readcol16(lineptr, x) (*((uint16 *) ((lineptr) + ((x)<<1))))
#define u_writecol16(lineptr, x, col) *((uint16 *) ((lineptr) + ((x)<<1))) = (col)

#define u_getr24(c) (((c) >> 16) & 0xFF)
#define u_getg24(c) (((c) >>  8) & 0xFF)
#define u_getb24(c) ((c) & 0xFF)
#define u_makecol24(r, g, b) (((r) << 16) | ((g) << 8) | (b))
#define u_rmax24 255
#define u_gmax24 255
#define u_bmax24 255
#define u_readcol24(lineptr, x) (*((int *) ((char *) (lineptr) + (x) * 3)) & 0xFFFFFF)
#define u_writecol24(lineptr, x, col) *((uint16 *) ((lineptr)+(x)*3)) = (col) & 0xFFFF; *((uint8 *) ((lineptr)+(x)*3+2)) = (col) >> 16

#define u_getr32(c) (((c) >> 16) & 0xFF)
#define u_getg32(c) (((c) >>  8) & 0xFF)
#define u_getb32(c) ((c) & 0xFF)
#define u_makecol32(r, g, b) (((r) << 16) | ((g) << 8) | (b))
#define u_rmax32 255
#define u_gmax32 255
#define u_bmax32 255
#define u_readcol32(lineptr, x) (*((uint32 *) ((lineptr) + ((x)<<2))))
#define u_writecol32(lineptr, x, col) *((uint32 *) ((lineptr) + ((x)<<2))) = (col)

int convert_24_to_16(int color, int x, int y) {
  int r = u_getr24(color), g = u_getg24(color), b = u_getb24(color);
  return u_makecol16(r*u_rmax16/u_rmax24, g*u_gmax16/u_gmax24, b*u_bmax16/u_bmax24);
}
int convert_16_to_24(int color, int x, int y) {
  int r = u_getr16(color), g = u_getg16(color), b = u_getb16(color);
  return u_makecol24(r*u_rmax24/u_rmax16, g*u_gmax24/u_gmax16, b*u_bmax24/u_bmax16);
}
#define convert_16_to_32(color, x, y) convert_16_to_24(color, x, y)
#define convert_32_to_16(color, x, y) convert_24_to_16(color, x, y)
#define convert_24_to_32(color, x, y) (color)
#define convert_32_to_24(color, x, y) (color)

int bitmap_mask_color(BITMAP *bmp) {
  if (bmp->depth == 15) { return 0x7C1F; }
  else if (bmp->depth == 16) { return 0xF81F; }
  else { return 0xFF00FF; }
}
int round_pow2(int val) {
  int i = 1;
  while (i < val) { i *= 2; }
  return i;
}

typedef gl_window screen_type;
screen_type *buffer = &_window;
screen_type *screen = &_window;

void blit(screen_type *src, screen_type *dest, int x1, int y1, int x2, int y2, int w, int h) {
  SwapBuffers(_window.hDC);
  update_frameinfo();
}

#define __blit(depth1, depth2)                                                    \
  for (y = 0; y < h; y++) {                                                       \
    uint8 *line1 = src->line[y+ysrc], *line2 = dest->line[y+ydest];               \
    for (x = 0; x < w; x++) {                                                     \
      color = u_readcol##depth1##(line1, x+xsrc);                                 \
      color = convert_##depth1##_to_##depth2##(color, x, y);                      \
      u_writecol##depth2##(line2, x+xdest, color);                                \
    }                                                                             \
  }
/*
      r = u_getr##depth1##(color)*u_rmax##depth2##/u_rmax##depth1##;                                                \
      g = u_getg##depth1##(color)*u_gmax##depth2##/u_gmax##depth1##;                                                \
      b = u_getb##depth1##(color)*u_bmax##depth2##/u_bmax##depth1##;                                                \
      color = u_makecol##depth2##(r, g, b);                                       \
*/

void blit(BITMAP *src, BITMAP *dest, int xsrc, int ysrc, int xdest, int ydest, int w, int h) {
  if (!src || !dest) { return; }
  int x, y, color, r, g, b;

  if (xsrc < 0) { w += xsrc; xdest -= xsrc; xsrc = 0; }
  if (ysrc < 0) { h += ysrc; ydest -= ysrc; ysrc = 0; }
  if (xsrc + w > src->w) { w = src->w - xsrc; }
  if (ysrc + h > src->h) { h = src->h - ysrc; }
  if (xdest < 0) { w += xdest; xsrc -= xdest; xdest = 0; }
  if (ydest < 0) { h += ydest; ysrc -= ydest; ydest = 0; }
  if (xdest + w > dest->w) { w = dest->w - xdest; }
  if (ydest + h > dest->h) { h = dest->h - ydest; }
  if (w <= 0 || h <= 0) { return; }

  if (src->depth == dest->depth) {
    int bt = bytes_per_pixel(src);
    int bytes = bt * w;
    for (y = 0; y < h; y++) {
      memcpy((void *) (dest->line[y+ydest] + xdest * bt), (void *) (src->line[y+ysrc] + xsrc * bt), bytes);
    }
  } else {
    if (src->depth == 16) {
      if (dest->depth == 24) { __blit(16, 24) }
      else if (dest->depth == 32) { __blit(16, 32) }
    } else if (src->depth == 24) {
      if (dest->depth == 16) { __blit(24, 16) }
      else if (dest->depth == 32) { __blit(24, 32) }
    } else if (src->depth == 32) {
      if (dest->depth == 16) { __blit(32, 16) }
      else if (dest->depth == 24) { __blit(32, 24) }
    }
  }
}

void clear(screen_type *scr) { glClear(GL_COLOR_BUFFER_BIT); }
void clear(BITMAP *bmp) { if (!bmp) { return; } memset((void *) bmp->data, 0, bmp->w * bmp->h * bytes_per_pixel(bmp)); }

void unmask_bitmap(BITMAP *bmp) {
  if (bmp->mask) {
    glDeleteTextures(1, &bmp->mask);
    bmp->mask = 0;
  }
}
void mask_bitmap(BITMAP *obmp) {
  if (!obmp->mask) {
    BITMAP *bmp = create_bitmap_ex(32, round_pow2(obmp->w), round_pow2(obmp->h));
    obmp->xtex = (obmp->w - 1) / double(bmp->w - 1);
    obmp->ytex = (obmp->h - 1) / double(bmp->h - 1);
    clear(bmp);
    blit(obmp, bmp, 0, 0, 0, 0, bmp->w, bmp->h);
    int mcolor = bitmap_mask_color(bmp);
    for (int y = 0; y < obmp->h; y++) {
      uint8 *line = bmp->line[y];
      for (int x = 0; x < obmp->w; x++) {
        int color = u_readcol32(line, x) & 0xFFFFFF;
        if (color != mcolor) { color |= (0xFF << 24); }
        else { color = 0; }
        u_writecol32(line, x, color);
      }
    }
    glGenTextures(1, &obmp->mask);
    if (obmp->mask == 0) { die("Bad texture ID."); }
    glBindTexture(GL_TEXTURE_2D, obmp->mask);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    glTexImage2D(GL_TEXTURE_2D, 0, 4, bmp->w, bmp->h, 0, GL_RGBA, GL_UNSIGNED_BYTE, (void *) bmp->data);
    destroy_bitmap(bmp);
  }
}

void texture_bitmap(BITMAP *obmp) {
  glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_FASTEST);
  if (!obmp->texture) {
    BITMAP *bmp = create_bitmap_ex(24, round_pow2(obmp->w), round_pow2(obmp->h));
    obmp->xtex = (obmp->w - 1) / double(bmp->w - 1);
    obmp->ytex = (obmp->h - 1) / double(bmp->h - 1);
    clear(bmp);
    blit(obmp, bmp, 0, 0, 0, 0, bmp->w, bmp->h);
    glGenTextures(1, &obmp->texture);
    if (obmp->texture == 0) { die("Bad texture ID."); }
    glBindTexture(GL_TEXTURE_2D, obmp->texture);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    glTexImage2D(GL_TEXTURE_2D, 0, 3, bmp->w, bmp->h, 0, GL_RGB, GL_UNSIGNED_BYTE, (void *) bmp->data);
    destroy_bitmap(bmp);
  }
}

void untexture_bitmap(BITMAP *bmp) {
  if (bmp->texture) {
    glDeleteTextures(1, &bmp->texture);
    bmp->texture = 0;
  }
}
#define modified_bitmap(bmp) untexture_bitmap(bmp); unmask_bitmap(bmp)

#define DRAW_MODE_SOLID 0
#define DRAW_MODE_TRANS 1
double _trans_alpha = 1.0;
int _drawing_mode=0;
void drawing_mode(int dmode, BITMAP *i2, int i3, int i4) { _drawing_mode = dmode; if (!dmode) { _trans_alpha = 0; } }
void enable_trans() {
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glColor4f(1, 1, 1, _trans_alpha);
}
void disable_trans() {
  glDisable(GL_BLEND);
  glColor3f(1, 1, 1);
}

void blit(BITMAP *src, screen_type *dest, int xsrc, int ysrc, int xdest, int ydest, int w, int h) {
  if (!src) { return; }
  glEnable(GL_TEXTURE_2D);
  glShadeModel(GL_FLAT);
  texture_bitmap(src);
  disable_trans();

  if (xsrc < 0) { w += xsrc; xdest -= xsrc; xsrc = 0; }
  if (ysrc < 0) { h += ysrc; ydest -= ysrc; ysrc = 0; }
  if (xsrc + w > src->w) { w = src->w - xsrc; }
  if (ysrc + h > src->h) { h = src->h - ysrc; }
  if (xdest < 0) { w += xdest; xsrc -= xdest; xdest = 0; }
  if (ydest < 0) { h += ydest; ysrc -= ydest; ydest = 0; }
  if (xdest + w > dest->w) { w = dest->w - xdest; }
  if (ydest + h > dest->h) { h = dest->h - ydest; }
  if (w <= 0 || h <= 0) { return; }

  int xmax = round_pow2(src->w) - 1;
  int ymax = round_pow2(src->h) - 1;
  double left   = xsrc / double(xmax);
  double top    = ysrc / double(ymax);
  double right  = (xsrc + w - 1) / double(xmax);
  double bottom = (ysrc + h - 1) / double(ymax);

  glBindTexture(GL_TEXTURE_2D, src->texture);
  glBegin(GL_QUADS);
    glTexCoord2d(left,  top);    glVertex2f(xdest, ydest);
    glTexCoord2d(right, top);    glVertex2f(xdest+w+xbias, ydest);
    glTexCoord2d(right, bottom); glVertex2f(xdest+w+xbias, ydest+h+ybias);
    glTexCoord2d(left,  bottom); glVertex2f(xdest, ydest+h+ybias);
  glEnd();
}
int bitmap_color_depth(BITMAP *bmp) { return bmp->depth; }
int bitmap_color_depth(screen_type *scr) { return _window.depth; }
void convert_depth(BITMAP *&bmp, int depth) {
  if (!bmp) { return; }
  if (bitmap_color_depth(bmp) == depth) { return; }
  BITMAP *old = bmp;
  bmp = create_bitmap_ex(depth, old->w, old->h);
  if (!bmp) { die("Not enough memory for conversion of bitmap: %dx%dx%d to %dx%dx%d", old->w, old->h, bitmap_color_depth(old), old->w, old->h, depth); }
  blit(old, bmp, 0, 0, 0, 0, old->w, old->h);
  destroy_bitmap(old);
}

void set_trans_blender(int i1, int i2, int i3, int alpha) {
  _trans_alpha = alpha / 255.0;
}

void texture_rect(BITMAP *src, int x, int y) {
  if (!src) { return; }
  int xmax = round_pow2(src->w) - 1;
  int ymax = round_pow2(src->h) - 1;

  double right  = (src->w-1) / double(xmax);
  double bottom = (src->h-1) / double(ymax);

  glBegin(GL_QUADS);
    glTexCoord2d(0, 0); glVertex2f(x, y);
    glTexCoord2d(right, 0); glVertex2f(x+src->w+xbias, y);
    glTexCoord2d(right, bottom); glVertex2f(x+src->w+xbias, y+src->h+ybias);
    glTexCoord2d(0, bottom); glVertex2f(x, y+src->h+ybias);
  glEnd();
}

void draw_sprite(screen_type *dest, BITMAP *src, int x, int y) {
  if (!src) { return; }

  glEnable(GL_TEXTURE_2D);
  glShadeModel(GL_FLAT);
  mask_bitmap(src);

  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glColor4f(1, 1, 1, 1);

  glBindTexture(GL_TEXTURE_2D, src->mask);
  texture_rect(src, x, y);

}

void draw_trans_sprite(screen_type *dest, BITMAP *src, int x, int y) {
  if (!src) { return; }
  glEnable(GL_TEXTURE_2D);
  glShadeModel(GL_FLAT);
  mask_bitmap(src);
  enable_trans();

  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glBindTexture(GL_TEXTURE_2D, src->mask);

  texture_rect(src, x, y);
}

void rect(screen_type *scr, int x1, int y1, int x2, int y2, int color) {
  glDisable(GL_TEXTURE_2D);
  glShadeModel(GL_FLAT);
  if (_drawing_mode == DRAW_MODE_TRANS) { enable_trans(); } else { disable_trans(); }

  glColor4f((color & 0xFF)/255.0, ((color >> 8) & 0xFF)/255.0, (color >> 16)/255.0, _trans_alpha);
  glBegin(GL_LINE_LOOP);
    glVertex2f(x1, y1);
    glVertex2f(x2, y1);
    glVertex2f(x2, y2);
    glVertex2f(x1, y2);
  glEnd();
}
void rectfill(screen_type *scr, int x1, int y1, int x2, int y2, int color) {
  glDisable(GL_TEXTURE_2D);
  glShadeModel(GL_FLAT);
  if (_drawing_mode == DRAW_MODE_TRANS) { enable_trans(); } else { disable_trans(); }

  glColor4f((color & 0xFF)/255.0, ((color >> 8) & 0xFF)/255.0, (color >> 16)/255.0, _trans_alpha);
  glBegin(GL_QUADS);
    glVertex2f(x1, y1);
    glVertex2f(x2, y1);
    glVertex2f(x2, y2);
    glVertex2f(x1, y2);
  glEnd();
}
void rectfill_gouraud(screen_type *scr, int x1, int y1, int x2, int y2, int r1,int g1,int b1, int r2,int g2,int b2, int r3,int g3,int b3, int r4,int g4,int b4) {
  glDisable(GL_TEXTURE_2D);
  glShadeModel(GL_SMOOTH);
  glDisable(GL_BLEND);
  glBegin(GL_QUADS);
    glColor3f(r1/255.0, g1/255.0, b1/255.0);
    glVertex2f(x1, y1);
    glColor3f(r2/255.0, g2/255.0, b2/255.0);
    glVertex2f(x2, y1);
    glColor3f(r3/255.0, g3/255.0, b3/255.0);
    glVertex2f(x2, y2);
    glColor3f(r4/255.0, g4/255.0, b4/255.0);
    glVertex2f(x1, y2);
  glEnd();
}

int makecol(int r, int g, int b) { return r | (g << 8) | (b << 16); }

//double time, dt, fps;

void update_frameinfo() {
  MSG msg;
  if (PeekMessage(&msg, 0, 0, 0, PM_REMOVE)) {
    if (msg.message == WM_QUIT) {
      return;
    } else {
      TranslateMessage(&msg);
      DispatchMessage(&msg);
    }
  }
  /*
  static int count = 0;
  static double last_frame=0;
  double last_time = time;
  time = __timer_func();
  dt = time - last_time;
  count++;
  if (time - last_frame > .5) {
    fps = count / (time - last_frame);
    last_frame = time;
    count = 0;
  }
  */
}

#define __pivot(src_x, src_y, dest_x, dest_y)                       \
  dest_x = (src_x - cx) * cos_t - (src_y - cy) * sin_t + px;        \
  dest_y = (src_x - cx) * sin_t + (src_y - cy) * cos_t + py;

void rotate_sprite_aa(screen_type *scr, BITMAP *src, double px, double py, double cx, double cy, double angle, double scale) {
  if (!src) { return; }
  int xmax = round_pow2(src->w) - 1;
  int ymax = round_pow2(src->h) - 1;
  double r = (src->w-1) / double(xmax);
  double b = (src->h-1) / double(ymax);

  glEnable(GL_TEXTURE_2D);
  glShadeModel(GL_FLAT);
  mask_bitmap(src);

  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glBindTexture(GL_TEXTURE_2D, src->mask);

  glColor4f(1, 1, 1, 1);

  double cos_t = cos(angle) * scale;
  double sin_t = sin(angle) * scale;

  int w = src->w, h = src->h;
  double x1, y1, x2, y2, x3, y3, x4, y4;
  __pivot(0, 0,     x1, y1)
  __pivot((w-1), 0,   x2, y2)
  __pivot((w-1), (h-1),   x3, y3)
  __pivot(0, (h-1), x4, y4)

  glBegin(GL_QUADS);
    glTexCoord2d(0, 0); glVertex2f(x1, y1);
    glTexCoord2d(r, 0); glVertex2f(x2, y2);
    glTexCoord2d(r, b); glVertex2f(x3, y3);
    glTexCoord2d(0, b); glVertex2f(x4, y4);
  glEnd();

}
void rotate_sprite(screen_type *scr, BITMAP *src, int x, int y, int angle) {
  rotate_sprite_aa(scr, src, x + src->w*.5, y + src->h*.5, src->w*.5, src->h*.5, angle/double(256<<16)*2*M_PI, 1);
}

void blit_additive(BITMAP *src, screen_type *dest, int xsrc, int ysrc, int xdest, int ydest, int w, int h, int red, int green, int blue, int alpha) {
  if (!src) { return; }
  glEnable(GL_TEXTURE_2D);
  glShadeModel(GL_FLAT);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE);
  glEnable(GL_BLEND);
  glColor4f(red/255.0, green/255.0, blue/255.0, alpha/255.0);

  texture_bitmap(src);

  int xmax = round_pow2(src->w) - 1;
  int ymax = round_pow2(src->h) - 1;
  double left   = xsrc / double(xmax);
  double top    = ysrc / double(ymax);
  double right  = (xsrc + w - 1) / double(xmax);
  double bottom = (ysrc + h - 1) / double(ymax);
  double maxr = (src->w - 1) / double(xmax);
  double maxb = (src->h - 1) / double(ymax);
  if (right > maxr) { right = maxr; }
  if (bottom > maxb) { bottom = maxb; }

  glBindTexture(GL_TEXTURE_2D, src->texture);
  glBegin(GL_QUADS);
    glTexCoord2d(left,  top);    glVertex2f(xdest, ydest);
    glTexCoord2d(right, top);    glVertex2f(xdest+w+xbias, ydest);
    glTexCoord2d(right, bottom); glVertex2f(xdest+w+xbias, ydest+h+ybias);
    glTexCoord2d(left,  bottom); glVertex2f(xdest, ydest+h+ybias);
  glEnd();

}

int getpixel(BITMAP *bmp, int x, int y) {
  if (((unsigned) x) >= bmp->w || ((unsigned) y) >= bmp->h) { return -1; }
  if (bmp->depth == 16) { return u_readcol16(bmp->line[y], x); }
  else if (bmp->depth == 24) { return u_readcol24(bmp->line[y], x); }
  else if (bmp->depth == 32) { return u_readcol32(bmp->line[y], x); }
  return -1;
}

void putpixel(BITMAP *bmp, int x, int y, int color) {
  if (((unsigned) x) >= bmp->w || ((unsigned) y) >= bmp->h) { return; }
  if (bmp->depth == 16) { u_writecol16(bmp->line[y], x, color); }
  else if (bmp->depth == 24) { u_writecol24(bmp->line[y], x, color); }
  else if (bmp->depth == 32) { u_writecol32(bmp->line[y], x, color); }
}

void hsv_to_rgb(float h, float s, float v, int *r, int *g, int *b) {
  float f, x, y, z;
  v *= 255.0;
  if (s == 0.0) { *r = *g = *b = (int)v; }
  else {
    while (h < 0) { h += 360; }
    h = fmod(h, 360) / 60.0;
    int i = (int)h;
    f = h - i;
    x = v * (1.0 - s);
    y = v * (1.0 - (s * f));
    z = v * (1.0 - (s * (1.0 - f)));

    switch (i) {
      case 0: *r = v; *g = z; *b = x; break;
      case 1: *r = y; *g = v; *b = x; break;
      case 2: *r = x; *g = v; *b = z; break;
      case 3: *r = x; *g = y; *b = v; break;
      case 4: *r = z; *g = x; *b = v; break;
      case 5: *r = v; *g = x; *b = y; break;
    }
  }
}

void idle_func() {
  update_frameinfo();
  blit(buffer, screen, 0, 0, 0, 0, buffer->w, buffer->h);
}

void tint_additive(screen_type *scr, int radd, int gadd, int badd) {
  glDisable(GL_DITHER);
  glDisable(GL_TEXTURE_2D);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE);
  glShadeModel(GL_FLAT);
  glColor4f(radd/255.0, gadd/255.0, badd/255.0, 1.0);
  int x1 = 0, y1 = 0, x2 = scr->w, y2 = scr->h;
  glBegin(GL_QUADS);
    glVertex2f(x1, y1);
    glVertex2f(x2, y1);
    glVertex2f(x2, y2);
    glVertex2f(x1, y2);
  glEnd();
}

#include "win32bmp.h"
#include "win32particle.h"
#include "win32font.h"
#include "win32fade.h"
#include "win32sound.h"
#include "win32midi.h"

#define time __time

#endif
