#ifndef __filter_h
#define __filter_h

#include <stdarg.h>
#include "global.h"

typedef unsigned char uint8;
typedef unsigned short uint16;
typedef unsigned int uint32;

#ifndef u_getr15
#define u_getr15(c) (((c) >> 10) & 0x1F)
#define u_getg15(c) (((c) >>  5) & 0x1F)
#define u_getb15(c) ((c) & 0x1F)
#define u_makecol15(r, g, b) (((r) << 10) | ((g) << 5) | (b))
#define u_rmax15 31
#define u_gmax15 31
#define u_bmax15 31
#define u_readcol15(lineptr, x) (*((uint16 *) ((lineptr) + ((x)<<1))))
#define u_readcol_mask15(lineptr, x) u_readcol15(lineptr, x)
#define u_writecol15(lineptr, x, col) *((uint16 *) ((lineptr) + ((x)<<1))) = (col)

#define u_getr16(c) (((c) >> 11) & 0x1F)
#define u_getg16(c) (((c) >>  5) & 0x3F)
#define u_getb16(c) ((c) & 0x1F)
#define u_makecol16(r, g, b) (((r) << 11) | ((g) << 5) | (b))
#define u_rmax16 31
#define u_gmax16 63
#define u_bmax16 31
#define u_readcol16(lineptr, x) (*((uint16 *) ((lineptr) + ((x)<<1))))
#define u_readcol_mask16(lineptr, x) u_readcol16(lineptr, x)
#define u_writecol16(lineptr, x, col) *((uint16 *) ((lineptr) + ((x)<<1))) = (col)

#define u_getr24(c) (((c) >> 16) & 0xFF)
#define u_getg24(c) (((c) >>  8) & 0xFF)
#define u_getb24(c) ((c) & 0xFF)
#define u_makecol24(r, g, b) (((r) << 16) | ((g) << 8) | (b))
#define u_rmax24 255
#define u_gmax24 255
#define u_bmax24 255
#define u_readcol24(lineptr, x) *((int *) ((char *) (lineptr) + (x) * 3))
#define u_readcol_mask24(lineptr, x) (u_readcol24(lineptr, x) & 0xFFFFFF)
inline void u_writecol24(uint8 *lineptr, int x, int col) {
  *((uint16 *) (lineptr+x*3)) = col & 0xFFFF;
  *((uint8 *) (lineptr+x*3+2)) = col >> 16;
}

#define u_getr32(c) (((c) >> 16) & 0xFF)
#define u_getg32(c) (((c) >>  8) & 0xFF)
#define u_getb32(c) ((c) & 0xFF)
#define u_makecol32(r, g, b) (((r) << 16) | ((g) << 8) | (b))
#define u_rmax32 255
#define u_gmax32 255
#define u_bmax32 255
#define u_readcol32(lineptr, x) (*((uint32 *) ((lineptr) + ((x)<<2))))
#define u_readcol_mask32(lineptr, x) u_readcol32(lineptr, x)
#define u_writecol32(lineptr, x, col) *((uint32 *) ((lineptr) + ((x)<<2))) = (col)
#endif

void blit_additive(BITMAP *src, BITMAP *dest, int xsrc, int ysrc, int xdest, int ydest, int w, int h, int red, int green, int blue, int alpha) {
  if (!src || !dest) { return; }
  int x, y;
  if (xsrc < 0) { w += xsrc; xdest -= xsrc; xsrc = 0; }
  if (ysrc < 0) { h += ysrc; ydest -= ysrc; ysrc = 0; }
  if (xsrc + w > src->w) { w = src->w - xsrc; }
  if (ysrc + h > src->h) { h = src->h - ysrc; }
  if (xdest < dest->cl) { xdest -= dest->cl; w += xdest; xsrc -= xdest; xdest = dest->cl; }
  if (ydest < dest->ct) { ydest -= dest->ct; h += ydest; ysrc -= ydest; ydest = dest->ct; }
  if (xdest + w > dest->cr) { w = dest->cr - xdest; }
  if (ydest + h > dest->cb) { h = dest->cb - ydest; }
  if (w <= 0 || h <= 0) { return; }

  red = red * alpha >> 8;
  green = green * alpha >> 8;
  blue = blue * alpha >> 8;

  if (bitmap_color_depth(src) == 15) {
    if (u_getr15(makecol15(255, 0, 0)) == 0) { swap(red, blue); }
    for (y = 0; y < h; y++) {
      uint8 *srcline = src->line[y + ysrc], *dstline = dest->line[y + ydest];
      for (x = 0; x < w; x++) {
        int c2 = u_readcol15(srcline, x + xsrc);
        int c1 = u_readcol15(dstline, x + xdest);
        int r = u_getr15(c1) + (u_getr15(c2) * red >> 8);
        int g = u_getg15(c1) + (u_getg15(c2) * green >> 8);
        int b = u_getb15(c1) + (u_getb15(c2) * blue >> 8);
        if (r > u_rmax15) { r = u_rmax15; }
        if (g > u_gmax15) { g = u_gmax15; }
        if (b > u_bmax15) { b = u_bmax15; }
        u_writecol15(dstline, x + xdest, u_makecol15(r, g, b));
      }
    }
  } else if (bitmap_color_depth(src) == 16) {
    if (u_getr16(makecol16(255, 0, 0)) == 0) { swap(red, blue); }
    for (y = 0; y < h; y++) {
      uint8 *srcline = src->line[y + ysrc], *dstline = dest->line[y + ydest];
      for (x = 0; x < w; x++) {
        int c2 = u_readcol16(srcline, x + xsrc);
        int c1 = u_readcol16(dstline, x + xdest);
        int r = u_getr16(c1) + (u_getr16(c2) * red >> 8);
        int g = u_getg16(c1) + (u_getg16(c2) * green >> 8);
        int b = u_getb16(c1) + (u_getb16(c2) * blue >> 8);
        if (r > u_rmax16) { r = u_rmax16; }
        if (g > u_gmax16) { g = u_gmax16; }
        if (b > u_bmax16) { b = u_bmax16; }
        u_writecol16(dstline, x + xdest, u_makecol16(r, g, b));
      }
    }
  } else if (bitmap_color_depth(src) == 24) {
    if (u_getr24(makecol24(255, 0, 0)) == 0) { swap(red, blue); }
    for (y = 0; y < h; y++) {
      uint8 *srcline = src->line[y + ysrc], *dstline = dest->line[y + ydest];
      for (x = 0; x < w; x++) {
        int c2 = u_readcol24(srcline, x + xsrc);
        int c1 = u_readcol24(dstline, x + xdest);
        int r = u_getr24(c1) + (u_getr24(c2) * red >> 8);
        int g = u_getg24(c1) + (u_getg24(c2) * green >> 8);
        int b = u_getb24(c1) + (u_getb24(c2) * blue >> 8);
        if (r > u_rmax24) { r = u_rmax24; }
        if (g > u_gmax24) { g = u_gmax24; }
        if (b > u_bmax24) { b = u_bmax24; }
        u_writecol24(dstline, x + xdest, u_makecol24(r, g, b));
      }
    }
  } else if (bitmap_color_depth(src) == 32) {
    if (u_getr32(makecol32(255, 0, 0)) == 0) { swap(red, blue); }
    for (y = 0; y < h; y++) {
      uint8 *srcline = src->line[y + ysrc], *dstline = dest->line[y + ydest];
      for (x = 0; x < w; x++) {
        int c2 = u_readcol32(srcline, x + xsrc);
        int c1 = u_readcol32(dstline, x + xdest);
        int r = u_getr32(c1) + (u_getr32(c2) * red >> 8);
        int g = u_getg32(c1) + (u_getg32(c2) * green >> 8);
        int b = u_getb32(c1) + (u_getb32(c2) * blue >> 8);
        if (r > u_rmax32) { r = u_rmax32; }
        if (g > u_gmax32) { g = u_gmax32; }
        if (b > u_bmax32) { b = u_bmax32; }
        u_writecol32(dstline, x + xdest, u_makecol32(r, g, b));
      }
    }
  }
}

void tint_additive(BITMAP *bmp, int radd, int gadd, int badd) {
  int x, y;
  if (bitmap_color_depth(bmp) == 15) {
    radd = radd * u_rmax15 / 255;
    gadd = gadd * u_gmax15 / 255;
    badd = badd * u_bmax15 / 255;
    if (u_getr15(makecol15(255, 0, 0)) == 0) { swap(radd, badd); }
    for (y = 0; y < bmp->h; y++) {
      uint8 *line = bmp->line[y];
      for (x = 0; x < bmp->w; x++) {
        int c = u_readcol15(line, x);
        int r = u_getr15(c) + radd;
        int g = u_getg15(c) + gadd;
        int b = u_getb15(c) + badd;
        if (r > u_rmax15) { r = u_rmax15; }
        if (g > u_gmax15) { g = u_gmax15; }
        if (b > u_bmax15) { b = u_bmax15; }
        u_writecol15(line, x, u_makecol15(r, g, b));
      }
    }
  } else if (bitmap_color_depth(bmp) == 16) {
    radd = radd * u_rmax16 / 255;
    gadd = gadd * u_gmax16 / 255;
    badd = badd * u_bmax16 / 255;
    if (u_getr16(makecol16(255, 0, 0)) == 0) { swap(radd, badd); }
    for (y = 0; y < bmp->h; y++) {
      uint8 *line = bmp->line[y];
      for (x = 0; x < bmp->w; x++) {
        int c = u_readcol16(line, x);
        int r = u_getr16(c) + radd;
        int g = u_getg16(c) + gadd;
        int b = u_getb16(c) + badd;
        if (r > u_rmax16) { r = u_rmax16; }
        if (g > u_gmax16) { g = u_gmax16; }
        if (b > u_bmax16) { b = u_bmax16; }
        u_writecol16(line, x, u_makecol16(r, g, b));
      }
    }
  } else if (bitmap_color_depth(bmp) == 24) {
    if (u_getr24(makecol24(255, 0, 0)) == 0) { swap(radd, badd); }
    for (y = 0; y < bmp->h; y++) {
      uint8 *line = bmp->line[y];
      for (x = 0; x < bmp->w; x++) {
        int c = u_readcol24(line, x);
        int r = u_getr24(c) + radd;
        int g = u_getg24(c) + gadd;
        int b = u_getb24(c) + badd;
        if (r > u_rmax24) { r = u_rmax24; }
        if (g > u_gmax24) { g = u_gmax24; }
        if (b > u_bmax24) { b = u_bmax24; }
        u_writecol24(line, x, u_makecol24(r, g, b));
      }
    }
  } else if (bitmap_color_depth(bmp) == 32) {
    if (u_getr32(makecol32(255, 0, 0)) == 0) { swap(radd, badd); }
    for (y = 0; y < bmp->h; y++) {
      uint8 *line = bmp->line[y];
      for (x = 0; x < bmp->w; x++) {
        int c = u_readcol32(line, x);
        int r = u_getr32(c) + radd;
        int g = u_getg32(c) + gadd;
        int b = u_getb32(c) + badd;
        if (r > u_rmax32) { r = u_rmax32; }
        if (g > u_gmax32) { g = u_gmax32; }
        if (b > u_bmax32) { b = u_bmax32; }
        u_writecol32(line, x, u_makecol32(r, g, b));
      }
    }
  }
}

class font_type { public:
  BITMAP *glyph[256];
  int height;
  font_type(char *filename, int base = 32) {
    for (int i = 0; i < 256; i++) { glyph[i] = 0; }
    BITMAP *bmp = load_bmp(filename, 0);
    if (!bmp) { die("Couldn't load font resource: %s", filename); }
    int scol = getpixel(bmp, 0, 0);
    int x=1, y=1, xstart, ystart, w, h;
    height = 0;
    while (y < bmp->h) {
      for (;;) {
        for (; x < bmp->w; x++) { if (getpixel(bmp, x, y) != scol) { break; } }
        if (x >= bmp->w) { break; }
        xstart = x;
        ystart = y;
        for (; x < bmp->w; x++) { if (getpixel(bmp, x, y) == scol) { break; } }
        if (x >= bmp->w) { break; }
        w = x - xstart;
        x--;
        for (; y < bmp->h; y++) { if (getpixel(bmp, x, y) == scol) { break; } }
        h = y - ystart;
        if (h > height) { height = h; }
        y = ystart;
        x++;
        glyph[base] = create_bitmap(w, h);
        blit(bmp, glyph[base], xstart, ystart, 0, 0, w, h);
/*
        BITMAP *g = glyph[base];
        int mcolor = bitmap_mask_color(g);
        for (int suby = 0; suby < g->h; suby++) {
          for (int subx = 0; subx < g->w; subx++) {
            if (getpixel(g, subx, suby) == mcolor) {
              putpixel(g, subx, suby, 0);
            }
          }
        }
*/
        base++;
      }
      x = 1;
      y += h;
      for (; y < bmp->h; y++) { if (getpixel(bmp, x, y) != scol) { break; } }
    }
    destroy_bitmap(bmp);
  }
  ~font_type() { for (int i = 0; i < 256; i++) { if (glyph[i]) { destroy_bitmap(glyph[i]); } } }
};

void textout(BITMAP *bmp, font_type *f, char *s, int xbase, int ybase, int r, int g, int b, int a) {
  int x = xbase, y = ybase;
  for (; *s != '\0'; s++) {
    char ch = *s;
    if (ch == '\n') { x = xbase; y += f->height; }
    else {
      BITMAP *glyph = f->glyph[(unsigned char) ch];
      blit_additive(glyph, bmp, 0, 0, x, y, glyph->w, glyph->h, r, g, b, a);
      x += glyph->w;
    }
  }
}

void textprintf(BITMAP *bmp, font_type *f, int xbase, int ybase, int r, int g, int b, int a, char *format, ...) {
  char buf[1024];
  va_list ap;
  va_start(ap, format);
  vsprintf(buf, format, ap);
  va_end(ap);
  textout(bmp, f, buf, xbase, ybase, r, g, b, a);
}

int text_length(font_type *f, char *s) {
  int result = 0;
  for (; *s != '\0' && *s != '\n'; s++) {
    result += f->glyph[(unsigned char) *s]->w;
  }
  return result;
}
int text_height(font_type *f) { return f->height; }

void textout_centre(BITMAP *bmp, font_type *f, char *s, int xbase, int ybase, int r, int g, int b, int a) {
  int x = xbase - text_length(f, s) / 2, y = ybase;
  for (; *s != '\0'; s++) {
    char ch = *s;
    if (ch == '\n') { x = xbase - text_length(f, s + 1) / 2; y += f->height; }
    else {
      BITMAP *glyph = f->glyph[(unsigned char) ch];
      blit_additive(glyph, bmp, 0, 0, x, y, glyph->w, glyph->h, r, g, b, a);
      x += glyph->w;
    }
  }
}

void textprintf_centre(BITMAP *bmp, font_type *f, int xbase, int ybase, int r, int g, int b, int a, char *format, ...) {
  char buf[1024];
  va_list ap;
  va_start(ap, format);
  vsprintf(buf, format, ap);
  va_end(ap);
  textout_centre(bmp, f, buf, xbase, ybase, r, g, b, a);
}

#define pack24(r,g,b) ((r) | ((g) << 8) | ((b) << 16))
void rectfill_gouraud(BITMAP *dest, int x1, int y1, int x2, int y2,
     int r1,int g1,int b1,
     int r2,int g2,int b2,
     int r3,int g3,int b3,
     int r4,int g4,int b4) {
  x2++; y2++;
  V3D_f vtx[4];
  V3D_f *ref_vtx[] = { &vtx[0], &vtx[1], &vtx[2], &vtx[3] };
  vtx[0].x = x1;  vtx[0].y = y1;  vtx[0].z = 0.0;  vtx[0].c = pack24(r1, g1, b1);
  vtx[1].x = x2;  vtx[1].y = y1;  vtx[1].z = 0.0;  vtx[1].c = pack24(r2, g2, b2);
  vtx[2].x = x2;  vtx[2].y = y2;  vtx[2].z = 0.0;  vtx[2].c = pack24(r3, g3, b3);
  vtx[3].x = x1;  vtx[3].y = y2;  vtx[3].z = 0.0;  vtx[3].c = pack24(r4, g4, b4);
  polygon3d_f(dest, POLYTYPE_GRGB, 0, 4, ref_vtx);
}

#define __pivot(src_x, src_y, dest_x, dest_y)                       \
  dest_x = (src_x - cx) * cos_t - (src_y - cy) * sin_t + px;        \
  dest_y = (src_x - cx) * sin_t + (src_y - cy) * cos_t + py;

#define __inv_pivot(src_x, src_y, dest_x, dest_y)                   \
  dest_x = (src_x - px) * mcos_t - (src_y - py) * msin_t + cx;      \
  dest_y = (src_x - px) * msin_t + (src_y - py) * mcos_t + cy;

// ul = (1-x)*(1-y) = 1 - 2x - 2y - xy
// ur = x*(1-y)     =      x      - xy
// ll = (1-x)*y     =           y - xy
// lr = x*y         =               xy
// a * ul + b * ur + c * ll + d * lr =
// a - x(2a) - y(2a) - xy(a) + x(b) - xy(b) + y(c) - xy(c) + xy(d) =
// a + x(b - 2a) + y(c - 2a) + xy(d - c - b - a)

#define __getpixel_bilinear(depth)                                                                \
void getpixel_bilinear##depth##(BITMAP *bmp, real x, real y, int &r, int &g, int &b, int &a, int mask) { \
  int i_x = int(x), i_y = int(y);                                                                 \
  real x_t = x - i_x, y_t = y - i_y;                                                              \
  real ul = (1.0 - x_t) * (1.0 - y_t), ur = x_t * (1.0 - y_t);                                    \
  real ll = (1.0 - x_t) * y_t, lr = x_t * y_t;                                                    \
                                                                                                  \
  uint8 *line1 = bmp->line[i_y], *line2 = bmp->line[i_y + 1];                                     \
  int c1 = u_readcol_mask##depth##(line1, i_x);                                                   \
  int c2 = u_readcol_mask##depth##(line1, i_x + 1);                                               \
  int c3 = u_readcol_mask##depth##(line2, i_x);                                                   \
  int c4 = u_readcol_mask##depth##(line2, i_x + 1);                                               \
                                                                                                  \
  a = 255;                                                                                        \
  if (c1 == mask) { c1 = 0; a -= 64; }                                                            \
  if (c2 == mask) { c2 = 0; a -= 64; }                                                            \
  if (c3 == mask) { c3 = 0; a -= 64; }                                                            \
  if (c4 == mask) { c4 = 0; a -= 64; }                                                            \
  if (a < 0) { a = 0; }                                                                           \
                                                                                                  \
  r = int(u_getr##depth##(c1) * ul + u_getr##depth##(c2) * ur + u_getr##depth##(c3) * ll + u_getr##depth##(c4) * lr);         \
  g = int(u_getg##depth##(c1) * ul + u_getg##depth##(c2) * ur + u_getg##depth##(c3) * ll + u_getg##depth##(c4) * lr);         \
  b = int(u_getb##depth##(c1) * ul + u_getb##depth##(c2) * ur + u_getb##depth##(c3) * ll + u_getb##depth##(c4) * lr);         \
}


__getpixel_bilinear(15)
__getpixel_bilinear(16)
__getpixel_bilinear(24)
__getpixel_bilinear(32)

#define __rotate_depth(depth)                                               \
  int mask = u_makecol##depth##(u_rmax##depth##, 0, u_bmax##depth##);       \
  w--; h--;                                                                 \
  int x, y;                                                                 \
  for (y = ymin; y < ymax; y++) {                                           \
    uint8 *dstline = bmp->line[y];                                          \
    for (x = xmin; x < xmax; x++) {                                         \
      real xsrc, ysrc;                                                      \
      __inv_pivot(x, y, xsrc, ysrc)                                         \
      if (xsrc > 0 && xsrc < w && ysrc > 0 && ysrc < h) {                   \
        int orig = u_readcol##depth##(dstline, x);                          \
        int r, g, b, a;                                                     \
        getpixel_bilinear##depth##(sprite, xsrc, ysrc, r, g, b, a, mask);   \
        r = u_getr##depth##(orig) + ((r - u_getr##depth##(orig)) * a >> 8);     \
        g = u_getg##depth##(orig) + ((g - u_getg##depth##(orig)) * a >> 8);     \
        b = u_getb##depth##(orig) + ((b - u_getb##depth##(orig)) * a >> 8);     \
        u_writecol##depth##(dstline, x, u_makecol##depth##(r, g, b));       \
      }                                                                     \
    }                                                                       \
  }


// draws the sprite rotated around (px, py), with (cx, cy) the pivot pt within
// the sprite.  only works in truecolor modes

void rotate_sprite_aa(BITMAP *bmp, BITMAP *sprite, real px, real py, real cx, real cy, real angle, real scale) {
  real cos_t = cos(angle) * scale;
  real sin_t = sin(angle) * scale;
  real mcos_t = cos(-angle) / scale;
  real msin_t = sin(-angle) / scale;
  // | cos  -sin |
  // | sin  +cos |
  int w = sprite->w, h = sprite->h;
  real view_x[4], view_y[4];
  __pivot(0, 0, view_x[0], view_y[0])
  __pivot(w, 0, view_x[1], view_y[1])
  __pivot(0, h, view_x[2], view_y[2])
  __pivot(w, h, view_x[3], view_y[3])

  int xmin = int(view_x[0]), ymin = int(view_y[0]);
  int xmax = int(view_x[0] + 1.0), ymax = int(view_y[0] + 1.0);

  for (int i = 1; i < 4; i++) {
    if (int(view_x[i]) < xmin) { xmin = int(view_x[i]); }
    if (int(view_y[i]) < ymin) { ymin = int(view_y[i]); }
    if (int(view_x[i] + 1.0) > xmax) { xmax = int(view_x[i] + 1.0); }
    if (int(view_y[i] + 1.0) > ymax) { ymax = int(view_y[i] + 1.0); }
  }
  if (xmin < bmp->cl) { xmin = bmp->cl; }
  if (ymin < bmp->ct) { ymin = bmp->ct; }
  if (xmax > bmp->cr) { xmax = bmp->cr; }
  if (ymax > bmp->cb) { ymax = bmp->cb; }
  if (xmin >= xmax || ymin >= ymax) { return; }

  int depth = bitmap_color_depth(bmp);
  if (depth == 15) { __rotate_depth(15) }
  else if (depth == 16) { __rotate_depth(16) }
  else if (depth == 24) { __rotate_depth(24) }
  else if (depth == 32) { __rotate_depth(32) }
}

#define __process_depth(depth)                                              \
if (u_getr##depth##(makecol##depth##(255, 0, 0)) == 0) { swap(red, blue); } \
for (y = 0; y < bmp->h; y++) {                                              \
  uint8 *scanline = bmp->line[y];                                           \
  for (x = 0; x < bmp->w; x++) {                                            \
    int c = u_readcol##depth##(scanline, x);                                \
    int r = u_getr##depth##(c) + red;                                       \
    int g = u_getg##depth##(c) + green;                                     \
    int b = u_getb##depth##(c) + blue;                                      \
    if (r > u_rmax##depth##) { r = u_rmax##depth##; }                       \
    if (g > u_gmax##depth##) { g = u_gmax##depth##; }                       \
    if (b > u_bmax##depth##) { b = u_bmax##depth##; }                       \
    if (r < 0) { r = 0; }                                                   \
    if (g < 0) { g = 0; }                                                   \
    if (b < 0) { b = 0; }                                                   \
    u_writecol##depth##(scanline, x, u_makecol##depth##(r, g, b));          \
  }                                                                         \
}

void convert_depth(BITMAP *&bmp, int depth) {
  BITMAP *old = bmp;
  bmp = create_bitmap_ex(depth, old->w, old->h);
  if (!bmp) { die("Not enough memory for conversion of bitmap: %dx%dx%d to %dx%dx%d", old->w, old->h, bitmap_color_depth(old), old->w, old->h, depth); }
  blit(old, bmp, 0, 0, 0, 0, old->w, old->h);
  destroy_bitmap(old);
}

#endif
