
// level.cpp: level editor for space.cpp
// shift+click to add enemies/items
// click the icons at the top to change the enemy type

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <allegro.h>

volatile long elapsed_time = 0;
void __inc_elapsed_time(...) { elapsed_time++; }
END_OF_FUNCTION(__inc_elapsed_time);
#define __timer_func() (elapsed_time / 1000.0)

void init_all();

int scr_width = 1024, scr_height = 768;
typedef double real;
int level_range = 2; // can auto-adjust to 2 above last item
BITMAP *icon[400] = { 0 }, *buffer, *cursor;
int do_exit = 0;
char *current_filename = 0;
int current_filenumber = 0;
char *default_pathname = "..\\";
int file_changed = 0;

real xstep = .05, ystep = .05;
real virtualh = 2, virtualw = 3;
real dt, last_frame, frame_time = 0;
void clip(real &value, real min, real max) { if (value > max) { value = max; } else if (value < min) { value = min; } }
int xclick, yclick, click, declick, mouse_button = 0, last_button;
int keyState[128], prevKey[128], keyClick[128];

int foreColor, backColor, hilightColor;

class scrollbar { public:
  int height, barw;
  real value;
  scrollbar() { height = 20; value = .5; barw = 60; }
  void update() {
    int barw = 60;
    if ((mouse_button & 1) && yclick >= buffer->h - height) {
      if (xclick > height && xclick < buffer->w - height) {
        value = real(mouse_x - height - barw / 2) / (buffer->w - height * 2 - barw);
      } else if (xclick <= height) {
        value -= .05 * dt;
      } else {
        value += .05 * dt;
      }
      clip(value, 0, 1);
    }
    rectfill(buffer, 0, buffer->h - 1 - height, buffer->w - 1, buffer->h - 1, backColor);
    rect(buffer, 0, buffer->h - 1 - height, buffer->w - 1, buffer->h - 1, foreColor);
    rectfill(buffer, 2, buffer->h - height + 1, height - 3, buffer->h - 3, foreColor);
    rectfill(buffer, buffer->w + 1 - height, buffer->h - height + 1, buffer->w - 3, buffer->h - 3, foreColor);
    int x = height / 2, y = buffer->h - height / 2;
    triangle(buffer, x - 4, y, x + 1, y - 5, x + 1, y + 5, hilightColor);
    x = buffer->w - 1- height / 2;
    triangle(buffer, x + 4, y, x - 1, y - 5, x - 1, y + 5, hilightColor);
    int xmin = height + barw / 2 - 1;
    int xmax = buffer->w - height - barw / 2 - 1;
    int xleft = int(xmin + value * (xmax - xmin)) - barw / 2;
    rectfill(buffer, xleft, buffer->h - height + 1, xleft + barw, buffer->h - 3, makecol(192, 192, 192));
  }
} scroll;

const int max_message = 1024;
class item_type { public:
  real x, y;
  int type, move;
  char *str;
  int used;
  item_type() { str = 0; used = 0; }
  item_type(real xval, real yval, int t, int m, char *s) :x(xval), y(yval), type(t), move(m) { if (s) { str = new char[max_message]; strcpy(str, s); } else { str = 0; } used = 1; }
  item_type &operator =(const item_type &a) { x=a.x; y=a.y; type=a.type; move=a.move; used = a.used; if (str) { delete[] str; } if (a.str) { str = new char[max_message]; strcpy(str, a.str); } else { str = 0; } }
  ~item_type() { if (str) { delete[] str; } }
};
int item_count = 0;
item_type item[10000];
item_type *selected[1000];
int selected_count = 0;
item_type clipboard[1000];
int clipboard_count = 0;

void update_item(int i) {
  item[i].x = floor(item[i].x / xstep + .5) * xstep;
  item[i].y = floor(item[i].y / ystep + .5) * ystep;
  int old_range = level_range;
  if (item[i].x > level_range - 2) {
    level_range = int(item[i].x + 2);
    scroll.value = scroll.value * old_range / level_range;
    clip(scroll.value, 0, 1);
  }
}
int add_item(const item_type &add) {
  file_changed = 1;
  int i;
  for (i = 0; i < item_count; i++) {
    if (!item[i].used) { break; }
  }
  if (i >= item_count) { item_count++; }
  item[i] = add;
  item[i].used = 1;
  update_item(i);
  return i;
}
void remove_item(int i) { item[i].used = 0; file_changed = 1; }
void update_all() {
  real xmax = 0;
  for (int i = 0; i < item_count; i++) {
    if (item[i].used) {
      if (item[i].x > xmax) { xmax = item[i].x; }
    }
  }
  int old_range = level_range;
  level_range = int(xmax + 2);
  scroll.value = scroll.value * old_range / level_range;
  clip(scroll.value, 0, 1);
}
void remove_all() { item_count = 0; file_changed = 1; }

void copy_selected() {
  if (!selected_count) { return; }
  for (int i = 0; i < selected_count; i++) { clipboard[i] = *selected[i]; }
  clipboard_count = selected_count;
}
void cut_selected() {
  if (!selected_count) { return; }
  for (int i = 0; i < selected_count; i++) {
    clipboard[i] = *selected[i];
    selected[i]->used = 0;
  }
  clipboard_count = selected_count;
  selected_count = 0;
  update_all();
  file_changed = 1;
}
void paste_clipboard() {
  real xmax=0, xmin=1e10, ymax=0, ymin=1e10;
  for (int i = 0; i < clipboard_count; i++) {
    if (clipboard[i].x < xmin) { xmin = clipboard[i].x; }
    if (clipboard[i].x > xmax) { xmax = clipboard[i].x; }
    if (clipboard[i].y < ymin) { ymin = clipboard[i].y; }
    if (clipboard[i].y > ymax) { ymax = clipboard[i].y; }
  }
  real yoffset = -ymin + .5*virtualh - (ymax - ymin) * .5;
  real xoffset = -xmin + .5*virtualw - (ymax - ymin) * .5 + scroll.value * level_range;
  xoffset = floor(xoffset / xstep) * xstep;
  yoffset = floor(yoffset / ystep) * ystep;
  selected_count = 0;
  for (int i = 0; i < clipboard_count; i++) {
    int j = add_item(clipboard[i]);
    item[j].x += xoffset;
    item[j].y += yoffset;
    selected[selected_count++] = &item[j];
    update_item(j);
  }
  file_changed = 1;
}

int strbegins(char *all, char *s) {
  while(*s != '\0' && *all != '\0') {
    if (*s != *all) { return 0; }
    s++; all++;
  }
  return *s == '\0';
}

/*
void load_file(char *filename) {
  item_count = 0;
  level_range = 0;
  char fullname[256];
  sprintf(fullname, "%s%s", default_pathname, filename);
  FILE *f = fopen(fullname, "r");
  char buf[1024], *p;
  real x, y;
  int type, move;
  char *str;
  if (ferror(f)) { fclose(f); return; }
  for (;;) {
    fgets(buf, 1024, f);
    if (feof(f)) { break; }
    x = y = 0;
    type = move = 0;
    str = 0;
    x = atof(buf);
    if (x >= 9999) { break; }
    if (x > 0) {
      if (p = strstr(buf, ",")) {
        y = atof(p + 1);
        if (p = strstr(p + 1, ",")) {
          type = atoi(p + 1);
          p++; while (*p == ' ') { p++; }
          if (strbegins(p, "obj_checkpoint")) { type = 100; }
          else if (strbegins(p, "obj_health")) { type = 102; }
          else if (strbegins(p, "obj_message")) { type = 103; }
          else if (strbegins(p, "obj_boss")) { type = 201; }
          if (p = strstr(p, ",")) {
            move = atoi(p + 1);
            p++; while (*p == ' ') { p++; }
            if (strbegins(p, "move_down")) { move = 1; }
            else if (strbegins(p, "move_up")) { move = 2; }
            else if (strbegins(p, "move_either")) { move = 4; }
            else if (strbegins(p, "move_train_down")) { move = 5; }
            else if (strbegins(p, "move_train_up")) { move = 6; }
            if (type == 103) {
              if (str = strchr(p, '"')) {
                str++;
                char prev = '\0';
                for (char *ch = str;*ch!='\0';ch++) {
                  if (*ch == '"' && prev != '\\') {
                    *ch = '\0'; break;
                  }
                  prev = *ch;
                }
              }
            }
            item[item_count] = item_type(x, y, type, move, str);
            update_item(item_count++);
          }
        }
      }
    }
  }
  fclose(f);
  file_changed = 0;
}
*/

void load_file(char *filename) {
  item_count = 0;
  selected_count = 0;
  level_range = 0;
  char fullname[256];
  sprintf(fullname, "%s%s", default_pathname, filename);
  FILE *f = fopen(fullname, "rb");
  if (!f) { return; }
  char buf[1024], *p;
  real x = 0, y;
  int type, move;
  char *str;
  if (ferror(f)) { fclose(f); return; }
  for (;;) {
    int ch1 = fgetc(f);
    if (feof(f)) { break; }
    int ch2 = fgetc(f);
    if (feof(f)) { break; }
    int ch3 = fgetc(f);
    if (feof(f)) { break; }
    if (ch1 == 255 && ch2 == 255 && ch3 == 255) { break; }
    x += ch1 * .05;
    y = ch2 * .05;
    type = ch3;
    move = 0; str = 0;
    if (type == 103) {
      str = buf;
      char *p = buf;
      for (;;) {
        *p = fgetc(f);
        if (*p == '\0') { break; }
        p++;
      }
    } else {
      move = fgetc(f);
    }
    item[item_count] = item_type(x, y, type, move, str);
    update_item(item_count++);
  }
  fclose(f);
  file_changed = 0;
}


int item_cmp(const void *e1, const void *e2) {
  item_type *i1 = (item_type *) e1, *i2 = (item_type *) e2;
  return i1->x < i2->x ? -1: 1;
}

void save_file(char *filename) {
  char fullname[256];
  sprintf(fullname, "%s%s", default_pathname, filename);
  FILE *f = fopen(fullname, "wb");
  item_type *item_copy = new item_type[item_count];
  for (int i = 0; i < item_count; i++) {
    item_copy[i] = item[i];
  }
  qsort((void *) item_copy, item_count, sizeof(item_type), item_cmp);
  real x = 0;
  for (int i = 0; i < item_count; i++) {
    if (item_copy[i].used) {
      real dx = item_copy[i].x - x;
      x += dx;
      int ch = int(dx / .05 + .5);
      if (ch > 255) { ch = 255; }
      fputc(ch, f);
      fputc(int(item_copy[i].y / .05 + .5), f);
      fputc(item_copy[i].type, f);
      if (item_copy[i].type == 103) {
        fputs(item_copy[i].str, f);
        fputc('\0', f);
      } else {
        fputc(item_copy[i].move, f);
      }
    }
  }
  fputc(255, f); fputc(255, f); fputc(255, f); fputc(255, f);

  fclose(f);
  delete[] item_copy;
  file_changed = 0;
}

int is_selected(int i) {
  for (int j = 0; j < selected_count; j++) {
    if (selected[j] == &item[i]) { return 1; }
  }
  return 0;
}
void add_select(int i) {
  for (int j = 0; j < selected_count; j++) {
    if (selected[j] == &item[i]) { return; }
  }
  selected[selected_count++] = &item[i];
}
void toggle_select(int i) {
  for (int j = 0; j < selected_count; j++) {
    if (selected[j] == &item[i]) {
      selected_count--;
      for (int k = j; k < selected_count; k++) {
        selected[k] = selected[k + 1];
      }
      return;
    }
  }
  selected[selected_count++] = &item[i];
}

#define pack24(r, g, b) (((r) << 16) | ((g) << 8) | (b))
void dialog_box(BITMAP *dest, int x, int y, int w, int h) {
  w--; h--;
  rectfill(dest, x, y, x+w, y+h, backColor);
  rect(dest, x+2,y+2,x+w-4,y+h-4, hilightColor);
  rect(dest, x+4,y+4,x+w-6,y+h-6, foreColor);
}

void update_frame() {
  last_frame = frame_time;
  frame_time = __timer_func();
  dt = frame_time - last_frame;
  last_button = mouse_button;
  mouse_button = mouse_b;
  click = mouse_button & (~last_button);
  declick = last_button & (~mouse_button);
  if (mouse_button && !last_button) { xclick = mouse_x; yclick = mouse_y; }
  for (int i = 0; i < 128; i++) {
    prevKey[i] = keyState[i];
    keyState[i] = key[i];
    keyClick[i] = keyState[i] && !prevKey[i];
  }
}

class button_type { public:
  char *caption; int left, top, width, height, defaultColor, enabled;
  button_type(int xcenter, int ycenter, int w, int h, char *s, int c, int e = 1) { enabled = e; defaultColor = c; caption = s; width = w; height = h; left = xcenter-width/2; top = ycenter-height/2; }
  int update() {
    int inside = mouse_x >= left && mouse_y >= top && mouse_x < left + width && mouse_y < top + height;
    int textColor = defaultColor == backColor ? foreColor: backColor;
    int faceColor = (inside && enabled) ? hilightColor: defaultColor;
    if ((mouse_button & 1) && inside) { textColor = hilightColor; faceColor = backColor; }
    rect(buffer, left, top, left+width-1, top+height-1, foreColor);
    rectfill(buffer, left+2, top+2, left+width-3, top+height-3, faceColor);
    textout_centre(buffer, font, caption, left+width/2, top+height/2-text_height(font)/2, textColor);
    if ((declick & 1) && inside && enabled) {
      return 1;
    }
    return 0;
  }
};
int file_exists(char *fname) {
  char fullname[256];
  sprintf(fullname, "%s%s", default_pathname, fname);
  FILE *f = fopen(fullname, "r"); if (!f) { return 0; }
  if (ferror(f) || feof(f)) { fclose(f); return 0; } fclose(f); return 1;
}
int select_file(int savable) {
  BITMAP *bmp = create_bitmap(buffer->w, buffer->h);
  blit(buffer, bmp, 0, 0, 0, 0, buffer->w, buffer->h);
  int w = buffer->w / 3, h = 3 * buffer->h / 4;
  dialog_box(bmp, (buffer->w-w)/2, (buffer->h-h)/2, w, h);
  char *title = "Save As...";
  if (!savable) { title = "Open..."; }
  textout_centre(bmp, font, title, buffer->w/2+w/4, (buffer->h-text_height(font))/2-h/4, hilightColor);
  int bwidth = 60, bheight = 20;
  const int wcount = 20;
  button_type *wave[wcount];
  char caption[wcount][32];
  for (int i = 0; i < wcount; i++) {
    sprintf(caption[i], "wave%d.dat", i+1);
    int exists = file_exists(caption[i]);
    wave[i] = new button_type((buffer->w-w)/2+w/4, (buffer->h-h)/2+bheight+(h-2*bheight)*i/wcount, 80, bheight, caption[i], exists ? foreColor: backColor, exists || savable);
  }

  button_type cancel((buffer->w-w)/2 + w*3/4, buffer->h/2, bwidth, bheight, "Cancel", foreColor);
  int result = 1, done = 0;
  while (!done) {
    update_frame();
    blit(bmp, buffer, 0, 0, 0, 0, bmp->w, bmp->h);
    if (cancel.update() || keyClick[KEY_C]) { result = 0; break; }
    for (int i = 0; i < wcount; i++) {
      if (wave[i] && wave[i]->update()) {
        delete[] current_filename;
        current_filename = new char[32];
        current_filenumber = i+1;
        sprintf(current_filename, "wave%d.dat", i+1);
        result = 1; done = 1;
      }
    }
    draw_sprite(buffer, cursor, mouse_x, mouse_y);
    blit(buffer, screen, 0, 0, 0, 0, buffer->w, buffer->h);
  }
  for (int i = 0; i < wcount; i++) {
    delete wave[i];
  }
  destroy_bitmap(bmp);
  return result;
}
int prompt_saveas() {
  int result = select_file(1);
  if (result) { save_file(current_filename); }
  return result;
}
int prompt_open() {
  int result = select_file(0);
  if (result) { load_file(current_filename); }
  return result;
}

class textbox_type { public:
  char *text;
  int left, top, width, height;
  textbox_type(int l, int t, int w, int h, char *s) { text=s; left=l; top=t; width=w; height=h; }
  int update() {
    int indent = 5;
    rect(buffer, left, top, left+width-1, top+height-1, foreColor);
    rectfill(buffer, left+1, top+1, left+width-2, top+height-2, makecol(0,0,0));
    textout(buffer, font, text, left+indent, top+indent, hilightColor);
    int x = text_length(font, text) + left + indent + 1;
    if (fmod(frame_time, 1) < .5) {
      rectfill(buffer, x, top+indent, x+text_length(font, " ")-1, top+indent+text_height(font)-1, hilightColor);
    }
    if (keypressed()) {
      char ch = readkey() & 0xFF;
      int length = strlen(text);
      if (ch == '\b') { if (length > 0) { text[length-1] = '\0'; } return 0; }
      if (ch == '\n' || ch == '\r') { return 1; }
      text[length] = ch;
      text[length+1] = '\0';
    }
    return 0;
  }
};

int input_text(char *s) {
  char *backup = new char[max_message];
  strcpy(backup, s);
  BITMAP *bmp = create_bitmap(buffer->w, buffer->h);
  blit(buffer, bmp, 0, 0, 0, 0, buffer->w, buffer->h);
  int w = buffer->w / 3, h = buffer->h / 4;
  dialog_box(bmp, (buffer->w-w)/2, (buffer->h-h)/2, w, h);
  int x1 = (buffer->w-w)/2 + w*1/3;
  int x2 = (buffer->w-w)/2 + w*2/3;
  int y = buffer->h/2 + h/4;
  int bwidth = 60, bheight = 20;
  int indent = 10;
  textout(bmp, font, "Message text:", (buffer->w-w)/2+indent, (buffer->h-h)/2+indent, foreColor);
  button_type ok(x1, y, bwidth, bheight, "OK", foreColor);
  button_type cancel(x2, y, bwidth, bheight, "Cancel", foreColor);
  textbox_type text((buffer->w-w)/2+indent, (buffer->h-h)/2+16+indent, w-2*indent, h/2-2*indent, s);
  int result = 1;
  for (;;) {
    update_frame();
    blit(bmp, buffer, 0, 0, 0, 0, bmp->w, bmp->h);
    if (ok.update() || text.update()) { break; }
    if (cancel.update()) { result = 0; strcpy(s, backup); break; }
    draw_sprite(buffer, cursor, mouse_x, mouse_y);
    blit(buffer, screen, 0, 0, 0, 0, buffer->w, buffer->h);
  }
  delete[] backup;
  destroy_bitmap(bmp);
  return result;
}

int prompt_save() {
  if (!file_changed) { return 1; }
  BITMAP *bmp = create_bitmap(buffer->w, buffer->h);
  blit(buffer, bmp, 0, 0, 0, 0, buffer->w, buffer->h);
  int w = buffer->w / 3, h = buffer->h / 4;
  dialog_box(bmp, (buffer->w-w)/2, (buffer->h-h)/2, w, h);
  textprintf_centre(bmp, font, buffer->w/2, (buffer->h-text_height(font))/2-h/4, hilightColor, "Save changes to '%s'?\n", current_filename ? current_filename: "(Untitled)");
  int x1 = (buffer->w-w)/2 + w*1/4;
  int x2 = (buffer->w-w)/2 + w*2/4;
  int x3 = (buffer->w-w)/2 + w*3/4;
  int y = buffer->h/2 + h/4;
  int bwidth = 60, bheight = 20;
  button_type yes(x1, y, bwidth, bheight, "Yes", foreColor);
  button_type no(x2, y, bwidth, bheight, "No", foreColor);
  button_type cancel(x3, y, bwidth, bheight, "Cancel", foreColor);
  int result = 1;
  for (;;) {
    update_frame();
    blit(bmp, buffer, 0, 0, 0, 0, bmp->w, bmp->h);
    if (yes.update() || keyClick[KEY_Y]) {
      if (!current_filename) { result = prompt_saveas(); break; }
      else { save_file(current_filename); result = 1; break; }
    }
    if (no.update() || keyClick[KEY_N]) { result = 2; break; }
    if (cancel.update() || keyClick[KEY_C]) { result = 0; break; }
    draw_sprite(buffer, cursor, mouse_x, mouse_y);
    blit(buffer, screen, 0, 0, 0, 0, buffer->w, buffer->h);
  }
  destroy_bitmap(bmp);
  return result;
}

int boundary = 19+5;  // last enemy # plus 5
int create_button1=0, create_button2=0;
void create_or_convert(int b1, int b2) {
  file_changed = 1;
  int type = b1+1, mtype = 0;
  char *str = 0;
  if (b2) {
    if (b2 < 4) { mtype = b2 - 1; }
    else { mtype = b2; }
  }
  if (b1 == 5) { type = 6; mtype = 20; }
  else if (b1 == 6) { type = 100; }
  else if (b1 == 7) { type = 102; }
  else if (b1 == boundary-4) {
    type = 104;
  } else if (b1 == boundary-3) {
    type = 105;
  } else if (b1 == boundary-2) {
    type = 103;
    str = new char[max_message];
    str[0] = '\0';
    if (!input_text(str)) { delete[] str; return; }
  } else if (b1 == boundary-1) { type = 201; }
  else if (b1 > 7) { type = b1; }
  for (int i = 0; i < selected_count; i++) {
    if (b2 < 0 && b1 >= 0) {
      if (type != selected[i]->type) {
        selected[i]->move = type == 6 ? 10: 0;
      }
    }
    if (!str) {
      if (selected[i]->str) { delete[] selected[i]->str; }
      selected[i]->str = 0;
    } else {
      if (!selected[i]->str) { selected[i]->str = new char[max_message]; }
      strcpy(selected[i]->str, str);
    }
    if (b1 >= 0) { selected[i]->type = type; }
    if (b2 >= 0) {
      selected[i]->move = mtype;
      if (selected[i]->type == 6) {
        switch (mtype) {
          case 0: selected[i]->move = 10; break;
          case 1: selected[i]->move = 20; break;
          case 2: selected[i]->move = 30; break;
          case 4: selected[i]->move = 40; break;
          case 5: selected[i]->move = 50; break;
          case 6: selected[i]->move = 100; break;
        }
      }
    }
    if (selected[i]->type >= 100) { selected[i]->move = 0; }
  }
  if (str) { delete[] str; }
}

int toolbar_height = 35*2;
class toolbar_type { public:
  int height, row_height, top;
  int button_count;
  BITMAP **button_image;
  int *enabled;
  toolbar_type() {
    row_height = 35;
    height = toolbar_height; top = 20;
    button_image = new (BITMAP *)[button_count = boundary + 15];
    button_image[0] = icon[1];
    button_image[1] = icon[2];
    button_image[2] = icon[3];
    button_image[3] = icon[4];
    button_image[4] = icon[5];
    button_image[5] = icon[7];
    button_image[6] = icon[100];
    button_image[7] = icon[102];
    button_image[8] = icon[8];
    button_image[9] = icon[9];
    button_image[10] = icon[10];
    button_image[11] = icon[11];
    button_image[12] = icon[12];
    button_image[13] = icon[13];
    button_image[14] = icon[14];
    button_image[15] = icon[15];
    button_image[16] = icon[16];
    button_image[17] = icon[17];
    button_image[18] = icon[18];
    button_image[19] = icon[19];
    button_image[boundary-4] = icon[104];
    button_image[boundary-3] = icon[105];
    button_image[boundary-2] = icon[103];
    button_image[boundary-1] = icon[201];
    button_image[boundary] = 0;
    button_image[boundary+1] = icon[300];
    button_image[boundary+2] = icon[301];
    button_image[boundary+3] = icon[302];
    button_image[boundary+4] = icon[304];
    button_image[boundary+5] = icon[305];
    button_image[boundary+6] = icon[306];
    button_image[boundary+7] = icon[307];
    button_image[boundary+8] = icon[308];
    button_image[boundary+9] = icon[309];
    button_image[boundary+10] = icon[310];
    button_image[boundary+11] = icon[311];
    button_image[boundary+12] = icon[312];
    button_image[boundary+13] = icon[313];
    button_image[boundary+14] = icon[314];
// normal, down, up, either, tdown, tup
    enabled = new int[button_count];
    for (int i = 0; i < button_count; i++) {
      enabled[i] = 0;
    }
  }
  ~toolbar_type() {
    delete[] button_image;
  }
  void buttonClick(int n) {
    if (!selected_count) {
      if (n < boundary) { create_button1 = n; for (int i = 0; i < boundary; i++) { enabled[i] = 0; } }
      else { create_button2 = n - boundary; for (int i = boundary; i < button_count; i++) { enabled[i] = 0; } }
      enabled[n] = 1;
    } else if (n < boundary) {
      create_or_convert(n, -1);
    } else {
      create_or_convert(-1, n - boundary);
    }
  }
  void update() {
    int indent = 5;
    rectfill(buffer, 1, top-2, buffer->w - 2, top+height - 2, hilightColor);
    rect(buffer, 0, top-1, buffer->w - 1, top+height - 1, foreColor);
    int x = 0, wrap_y = 0, wrap=0;
    for (int i = 0; i < button_count; i++) {
      x += indent;
      if (!button_image[i]) { x += 30; continue; }
      int y = wrap_y + top+(row_height-button_image[i]->h)/2;
      if (mouse_x >= x && mouse_y >= y && mouse_x < x+button_image[i]->w && mouse_y < y + button_image[i]->h) {
        if (click & 1) { buttonClick(i); }
        draw_sprite(buffer, button_image[i], x, y);
      } else {
        set_trans_blender(160,160,160,160);
        draw_trans_sprite(buffer, button_image[i], x, y);
      }
      if (enabled[i] && !selected_count) { rect(buffer, x-1, y-1, x-1+button_image[i]->w+2,y-1+button_image[i]->h+2, makecol(255,0,0)); }
      x += button_image[i]->w;
      if (i >= boundary - 5 && !wrap) { x = 0; wrap_y += row_height; wrap = 1; }
    }
    if (file_changed) {
      char *txt = "Modified";
      textout(buffer, font, txt, buffer->w - text_length(font, txt) - indent, top+row_height/2, backColor);
    }
  }
};

int iabs(int x) { return x > 0 ? x: -x; }
int region_select = 0;
int drag = 0;
real xdrag, ydrag;
int shift_pressed() { return keyState[KEY_LSHIFT] || keyState[KEY_RSHIFT]; }
int control_pressed() { return keyState[KEY_LCONTROL] || keyState[KEY_RCONTROL]; }
void draw_items() {
  int voffset = 20 + toolbar_height;
  BITMAP *bmp = create_sub_bitmap(buffer, 0, voffset, buffer->w, buffer->h-20-voffset);
  int indent = 15;
  for (int i = 0; i < level_range + 4; i++) {
    int xcenter = int((i - scroll.value * level_range) * (bmp->w - 2 * indent) / virtualw + indent);
    vline(bmp, xcenter, 0, bmp->h - 1, foreColor);
    textprintf_centre(bmp, font, xcenter, 10, hilightColor, "%d", i);
  }
  int inside = 0;
  for (int i = 0; i < item_count; i++) {
    if (item[i].used) {
      BITMAP *sprite = icon[item[i].type];
      if (sprite) {
        int ycenter = int(item[i].y * (bmp->h - 2 * indent) / virtualh + indent);
        int xcenter = int((item[i].x - scroll.value * level_range) * (bmp->w - 2 * indent) / virtualw + indent);
        int x1 = xcenter - sprite->w / 2, y1 = ycenter - sprite->h / 2;
        int x2 = x1 + sprite->w, y2 = y1 + sprite->h;
        if (xclick >= x1 && yclick >= y1+voffset && xclick < x2 && yclick < y2+voffset) {
          if (!region_select && !drag && (mouse_button & 1)) { // && iabs(mouse_x - xclick) > 1 && iabs(mouse_y - yclick) > 1) {
            if (!is_selected(i)) {
              if (shift_pressed()) { add_select(i); }
              else if (control_pressed()) { toggle_select(i); }
              else { selected[0] = &item[i]; selected_count = 1; }
            }
            drag = 1;
            xdrag = real(xclick);
            ydrag = real(yclick);
          }
        }
        if (mouse_x >= x1 && mouse_y >= y1+voffset && mouse_x < x2 && mouse_y < y2+voffset && !inside) {
          if ((declick & 1) && !region_select && !drag) {
            if (shift_pressed()) { add_select(i); }
            else if (control_pressed()) { toggle_select(i); }
            else {
              selected[0] = &item[i];
              selected_count = 1;
            }
          }
          inside = 1;
        }
        draw_sprite(bmp, sprite, xcenter - sprite->w / 2, ycenter - sprite->h / 2);
        if (item[i].type == 6) {
          char info[10];
          sprintf(info, "%d", item[i].move);
          int back = makecol(255,255,255);
          xcenter++;
          textout_centre(bmp, font, info, xcenter+1, ycenter - text_height(font) / 2-1, back);
          textout_centre(bmp, font, info, xcenter-1, ycenter - text_height(font) / 2-1, back);
          textout_centre(bmp, font, info, xcenter+1, ycenter - text_height(font) / 2+1, back);
          textout_centre(bmp, font, info, xcenter-1, ycenter - text_height(font) / 2+1, back);
          textout_centre(bmp, font, info, xcenter, ycenter - text_height(font) / 2, makecol(0,0,255));
        } else if (item[i].move) {
          char *info = "???";
          int front = makecol(255,0,0);
          switch (item[i].move) {
            case 1: info = "Dwn"; break;
            case 2: info = "Up"; break;
            case 4: info = "Any"; break;
            case 5: info = "TD"; front = makecol(0,0,255); break;
            case 6: info = "TU"; front = makecol(0,0,255); break;
            case 7: info = "SD"; front = makecol(0,0,255); break;
            case 8: info = "SU"; front = makecol(0,0,255); break;
            case 9: info = "ZD"; front = makecol(0,0,255); break;
            case 10: info = "ZU"; front = makecol(0,0,255); break;
            case 11: info = "SQ"; front = makecol(0,0,255); break;
            case 12: info = "DD"; front = makecol(0,0,255); break;
            case 13: info = "DU"; front = makecol(0,0,255); break;
            case 14: info = "SQL"; front = makecol(0,0,255); break;
          }
          int back = makecol(255,255,255);
          xcenter++;
          textout_centre(bmp, font, info, xcenter+1, ycenter - text_height(font) / 2-1, back);
          textout_centre(bmp, font, info, xcenter-1, ycenter - text_height(font) / 2-1, back);
          textout_centre(bmp, font, info, xcenter+1, ycenter - text_height(font) / 2+1, back);
          textout_centre(bmp, font, info, xcenter-1, ycenter - text_height(font) / 2+1, back);
          textout_centre(bmp, font, info, xcenter, ycenter - text_height(font) / 2, front);
        }
      }
    }
  }
  if (!inside && click && yclick >= voffset && yclick < voffset + bmp->h && shift_pressed()) {
    real x = (xclick - indent) * virtualw / (buffer->w - 2 * indent) + scroll.value * level_range;
    real y = (yclick-voffset-indent) * virtualh / (buffer->h - voffset - 2 * indent);
    selected_count = 1;
    selected[0] = &item[add_item(item_type(x, y, 1, 0, 0))];
    create_or_convert(create_button1, create_button2);
    update_all();
    click = 0;
  }
  for (int i = 0; i < selected_count; i++) {
    if (selected[i]->used) {
      BITMAP *sprite = icon[selected[i]->type];
      if (sprite) {
        int ycenter = int(selected[i]->y * (bmp->h - 2 * indent) / virtualh + indent);
        int xcenter = int((selected[i]->x - scroll.value * level_range) * (bmp->w - 2 * indent) / virtualw + indent);
        int x1 = xcenter - sprite->w / 2, y1 = ycenter - sprite->h / 2;
        int x2 = x1 + sprite->w, y2 = y1 + sprite->h;
        rect(bmp, x1, y1, x2-1, y2-1, makecol(255,0,0));
      }
    }
  }
  if (drag) {
    real dx = (mouse_x - xdrag) / real(bmp->w - 2 * indent) * virtualw;
    real dy = (mouse_y - ydrag) / real(bmp->h - 2 * indent) * virtualh;
    if (dx > 0) { dx = int(dx / xstep + .5) * xstep; } else { dx = int(-dx / xstep + .5) * -xstep; }
    if (dy > 0) { dy = int(dy / ystep + .5) * ystep; } else { dy = int(-dy / ystep + .5) * -ystep; }
    real xmax = 0, xmin = 1e10;
    for (int i = 0; i < selected_count; i++) {
      selected[i]->x += dx;
      selected[i]->x = floor(selected[i]->x / xstep + .5) * xstep;
      if (selected[i]->x > xmax) { xmax = selected[i]->x; }
      if (selected[i]->x < xmin) { xmin = selected[i]->x; }
      selected[i]->y += dy;
      selected[i]->y = floor(selected[i]->y / ystep + .5) * ystep;
      update_all();
    }
    if (xmin < scroll.value * level_range) {
      real old_value = scroll.value;
      scroll.value -= 3 * dt / level_range;
      clip(scroll.value, 0, 1);
      xdrag -= (scroll.value - old_value) * level_range * (bmp->w - 2 * indent) / virtualw;
    } else if (xmax - xmin < virtualw && xmax > scroll.value * level_range + virtualw) {
      real old_value = scroll.value;
      scroll.value += 3 * dt / level_range;
      clip(scroll.value, 0, 1);
      xdrag -= (scroll.value - old_value) * level_range * (bmp->w - 2 * indent) / virtualw;
    }
    xdrag += dx * real(bmp->w - 2 * indent) / virtualw;
    ydrag += dy * real(bmp->h - 2 * indent) / virtualh;
  }
  if (!(mouse_button & 1) && drag) {
    drag = 0;
  }
  if ((click & 1) && !inside && !drag && yclick >= voffset && yclick < voffset + bmp->h) {
    region_select = 1;
  }
  if (region_select && (mouse_button & 1)) {
    drawing_mode(DRAW_MODE_XOR, 0, 0, 0);
    rect(buffer, xclick, yclick, mouse_x, mouse_y, makecol(255,255,255));
    drawing_mode(DRAW_MODE_SOLID, 0, 0, 0);
  }
  if (!(mouse_button & 1) && region_select) {
    int xmin = xclick, ymin = yclick;
    int xmax = mouse_x, ymax = mouse_y;
    if (xmax < xmin) { int temp = xmin; xmin = xmax; xmax = temp; }
    if (ymax < ymin) { int temp = ymin; ymin = ymax; ymax = temp; }
    if (!shift_pressed()) { selected_count = 0; }
    for (int i = 0; i < item_count; i++) {
      if (item[i].used) {
        BITMAP *sprite = icon[item[i].type];
        if (sprite) {
          int ycenter = int(item[i].y * (bmp->h - 2 * indent) / virtualh + indent);
          int xcenter = int((item[i].x - scroll.value * level_range) * (bmp->w - 2 * indent) / virtualw + indent);
          int x1 = xcenter - sprite->w / 2, y1 = ycenter - sprite->h / 2;
          int x2 = x1 + sprite->w, y2 = y1 + sprite->h;
          if (x1 < xmax && y1+voffset < ymax && xmin < x2 && ymin < y2+voffset) {
            if (!control_pressed()) { add_select(i); }
            else { toggle_select(i); }
          }
        }
      }
    }
    region_select = 0;
  }
  if (mouse_button & 2) {
    for (int i = 0; i < item_count; i++) {
      if (item[i].used) {
        BITMAP *sprite = icon[item[i].type];
        if (sprite) {
          int ycenter = int(item[i].y * (bmp->h - 2 * indent) / virtualh + indent);
          int xcenter = int((item[i].x - scroll.value * level_range) * (bmp->w - 2 * indent) / virtualw + indent);
          int x1 = xcenter - sprite->w / 2, y1 = ycenter - sprite->h / 2;
          int x2 = x1 + sprite->w, y2 = y1 + sprite->h;
          if (mouse_x >= x1 && mouse_y >= y1+voffset && mouse_x < x2 && mouse_y < y2+voffset) {
            char buf[40];
            int dialogw = 190, dialogh = 130;
            char *str, *str2, *str3, *str4;
            switch (item[i].type) {
              case 6: str = "Asteroid cluster"; break;
              case 100: str = "Checkpoint"; break;
              case 102: str = "Health"; break;
              case 103: str = "Message"; break;
              case 201: str = "Boss"; break;
              default: str = "Enemy"; break;
            }
            switch (item[i].move) {
              case 1: str2 = "moveType: Downward"; str3 = "Moves diagonally"; str4 = "downward."; break;
              case 2: str2 = "moveType: Upward"; str3 = "Moves diagonally"; str4 = "upward."; break;
              case 4: str2 = "moveType: Either"; str3 = "Moves diagonally"; str4 = "downward or upward."; break;
              case 5: str2 = "moveType: Train Down"; str3 = "Moves in 'S' shape"; str4 = "across screen."; break;
              case 6: str2 = "moveType: Train Up"; str3 = "Moves in '2' shape"; str4 = "across screen."; break;
              case 7:
              case 8: str2 = "moveType: Sinewave"; str3 = "Moves vertically in"; str4 = "a sinewave function."; break;
              case 9: str2 = "moveType: Zigzag Upward"; str3 = "Moves in 'Z' shape"; str4 = "upward."; break;
              case 10: str2 = "moveType: Zigzag Downward"; str3 = "Moves in 'Z' shape"; str4 = "downward."; break;
              case 11: str2 = "moveType: Squarewave"; str3 = "Moves in a squarewave"; str4 = "shape."; break;
              case 12: str2 = "moveType: Dart Down"; str3 = "Moves forward normally"; str4 = "then darts down."; break;
              case 13: str2 = "moveType: Dart Up"; str3 = "Moves forward normally"; str4 = "then darts up."; break;
              default: str2 = "moveType: Standard"; str3 = "Moves horizontally"; str4 = "at default speed."; break;
            }
            if (item[i].type == 100) {
              str2 = "Saves player's"; str3 = "progress"; str4 = "";
            } else if (item[i].type == 201) {
              str2 = "End-level boss"; str3 = "determined by"; str4 = "current level number";
            } else if (item[i].type == 103) {
              str2 = "Contents:"; str3 = item[i].str; str4 = "";
              int messagew = 2 * indent + text_length(font, item[i].str);
              if (messagew > dialogw) { dialogw = messagew; }
            } else if (item[i].type == 6) {
              sprintf(buf, "Density: %d", item[i].move); str2 = buf; str3 = ""; str4 = "";
            }
            dialog_box(bmp, xcenter, ycenter, dialogw, dialogh);
            textprintf(bmp, font, xcenter+indent, ycenter+indent, hilightColor, "(%0.2f, %0.2f)", item[i].x, item[i].y);
            textprintf(bmp, font, xcenter+indent, ycenter+indent+16, hilightColor, str);
            textprintf(bmp, font, xcenter+indent, ycenter+indent+16*3, hilightColor, str2);
            textprintf(bmp, font, xcenter+indent, ycenter+indent+16*4, hilightColor, str3);
            textprintf(bmp, font, xcenter+indent, ycenter+indent+16*5, hilightColor, str4);
            break;
          }
        }
      }
    }
  }
  destroy_bitmap(bmp);
}

int alt_key(int k) { return keyState[KEY_ALT] && keyClick[k]; }
int control_key(int k) { return control_pressed() & keyClick[k]; }
class menubar_type { public:
  char **menu;
  int menu_count, height, indent;
  menubar_type() {
    height = 20; indent = 10;
    menu = new (char *)[menu_count = 10];
    menu[0] = "&New";
    menu[1] = "&Open";
    menu[2] = "&Save";
    menu[3] = "Save &As";
    menu[4] = "&Delete";
    menu[5] = "Cu&t";
    menu[6] = "&Copy";
    menu[7] = "&Paste";
    menu[8] = "&Run";
    menu[9] = "E&xit";
  }
  void menuClick(int n) {
    if (n == 0) {
      if (prompt_save()) {
        item_count = 0;
        selected_count = 0;
        current_filename = 0;
        current_filenumber = 0;
        update_all();
      }
    }
    if (n == 1) {
      if (prompt_save()) {
        prompt_open();
      }
    }
    if (n == 2) {
      if (current_filename) {
        save_file(current_filename);
      } else { prompt_saveas(); }
    }
    if (n == 3) {
      prompt_saveas();
    }
    if (n == 4) {
      if (selected_count) {
        for (int i = 0; i < selected_count; i++) {
          selected[i]->used = 0;
        }
        selected_count = 0;
      }
    }
    if (n == 5) { cut_selected(); }
    if (n == 6) { copy_selected(); }
    if (n == 7) { paste_clipboard(); }
    if (n == 8) {
      if ((prompt_save() == 1) && current_filename) {
        char cmdline[256];
        sprintf(cmdline, "space -warp %d", current_filenumber);
        allegro_exit();
        system("cd ..");
        system(cmdline);
        system("cd edit");
        init_all();
      }
    }
    if (n == 9) { if (prompt_save()) { do_exit = 1; } }
  }
  void update() {
    rectfill(buffer, 0, 0, buffer->w - 1, height - 1, backColor);
    rect(buffer, 0, 0, buffer->w - 1, height - 1, foreColor);
    int x = 0, y = (height - text_height(font)) / 2;
    int offset[menu_count];
    for (int i = 0; i < menu_count; i++) {
      int lastx = x;
      x += indent;
      char *dest = new char[strlen(menu[i])+2];
      char *src = menu[i];
      for (char *ch = dest; ; ) {
        if ((*ch = *src) == '&') {
          ch[1] = '\0';
          int xend = x + text_length(font, dest) - 1;
          int xstart = xend - text_length(font, ch) + 1;
          hline(buffer, xstart, y + text_height(font) + 1, xend, foreColor);
        } else { ch++; }
        if (*src == '\0') { break; }
        src++;
      }
      x += text_length(font, dest) + indent;
      offset[i] = x;
      int color = (mouse_y < height && mouse_x < x && mouse_x >= lastx) ? hilightColor: foreColor;
      if (key[KEY_ALT]) { color = hilightColor; }
      textout(buffer, font, dest, lastx + indent, y, color);
      delete[] dest;
      vline(buffer, x, 0, height - 1, foreColor);
    }
    char buf[128];
    sprintf(buf, "%d item(s) selected", selected_count);
    textout(buffer, font, buf, buffer->w - text_length(font, buf) - indent, y, selected_count ? hilightColor: foreColor);
    if ((click & 1) && yclick < height) {
      for (int i = 0; i < menu_count; i++) {
        if (mouse_x < offset[i]) {
          menuClick(i);
          break;
        }
      }
    }
    if (alt_key(KEY_N)) { menuClick(0); }
    if (alt_key(KEY_O)) { menuClick(1); }
    if (alt_key(KEY_S)) { menuClick(2); }
    if (alt_key(KEY_A)) { menuClick(3); }
    if (alt_key(KEY_D) || keyClick[KEY_DEL]) { menuClick(4); }
    if (control_key(KEY_X) || alt_key(KEY_T)) { menuClick(5); }
    if (control_key(KEY_C) || alt_key(KEY_C)) { menuClick(6); }
    if (control_key(KEY_V) || alt_key(KEY_P)) { menuClick(7); }
    if (alt_key(KEY_R)) { menuClick(8); }
    if (keyClick[KEY_ESC] || alt_key(KEY_F4) || alt_key(KEY_X)) { menuClick(9); }
  }
} menu;

BITMAP *load_image(char *filename) {
  BITMAP *result = load_bitmap(filename, 0);
  if (!result) {
    set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
    printf("Couldn't load resource: %s\n", filename);
    exit(0);
  }
  return result;
}

int main() {
  init_all();

  foreColor = makecol(128, 128, 128);
  backColor = makecol(64, 64, 64);
  hilightColor = makecol(192, 192, 192);

  buffer = create_bitmap(scr_width, scr_height);
  icon[1] = load_image("item1.bmp");
  icon[2] = load_image("item2.bmp");
  icon[3] = load_image("item3.bmp");
  icon[4] = load_image("item4.bmp");
  icon[5] = load_image("item5.bmp");
  icon[6] = load_image("item6.bmp");
  icon[7] = load_image("item7.bmp");
  icon[8] = load_image("item8.bmp");
  icon[9] = load_image("item9.bmp");
  icon[10] = load_image("item10.bmp");
  icon[11] = load_image("item11.bmp");
  icon[12] = load_image("item12.bmp");
  icon[13] = load_image("item13.bmp");
  icon[14] = load_image("item14.bmp");
  icon[15] = load_image("item15.bmp");
  icon[16] = load_image("item16.bmp");
  icon[17] = load_image("item17.bmp");
  icon[18] = load_image("item18.bmp");
  icon[19] = load_image("item19.bmp");
  icon[100] = load_image("item100.bmp");
  icon[102] = load_image("item102.bmp");
  icon[103] = load_image("item103.bmp");
  icon[104] = load_image("item104.bmp");
  icon[105] = load_image("item105.bmp");
  icon[201] = load_image("item201.bmp");
  icon[300] = load_image("item300.bmp");
  icon[301] = load_image("item301.bmp");
  icon[302] = load_image("item302.bmp");
  icon[304] = load_image("item304.bmp");
  icon[305] = load_image("item305.bmp");
  icon[306] = load_image("item306.bmp");
  icon[307] = load_image("item307.bmp");
  icon[308] = load_image("item308.bmp");
  icon[309] = load_image("item309.bmp");
  icon[310] = load_image("item310.bmp");
  icon[311] = load_image("item311.bmp");
  icon[312] = load_image("item312.bmp");
  icon[313] = load_image("item313.bmp");
  icon[314] = load_image("item314.bmp");

  cursor = load_image("cursor.bmp");

  toolbar_type toolbar;

  while (!do_exit) {
    update_frame();
    clear(buffer);
    scroll.update();
    draw_items();
    toolbar.update();
    menu.update();
    draw_sprite(buffer, cursor, mouse_x, mouse_y);
    blit(buffer, screen, 0, 0, 0, 0, buffer->w, buffer->h);
  }
}

int depth_select(int depth) { set_color_depth(depth); return !set_gfx_mode(GFX_AUTODETECT, scr_width, scr_height, 0, 0); }

void init_all() {
  allegro_init();

  install_keyboard();
  install_mouse();
  install_timer();

  LOCK_VARIABLE(elapsed_time);
  LOCK_FUNCTION(__inc_elapsed_time);
  install_int_ex(__inc_elapsed_time, BPS_TO_TIMER(1000));

  depth_select(16) || depth_select(15) || depth_select(32) || depth_select(24);

  text_mode(-1);
  set_mouse_speed(1, 1);
}
