/* gen-normal.c,
 *
 * Generate tile normals, used after object-tile collision.
 *
 *
 * Controls:
 *
 *  comma, period	- select normal to edit (right/top/left/bottom)
 *  keypad		- point normal in keypad direction
 *			  combine keys for 30 degree angles
 *  keypad +/-		- rotate normal by +/- 1 degree
 *
 *  space		- next tile
 *  backspace		- previous tile
 *
 *  escape/q		- quit and export to ',normal.txt'
 */

#include <allegro.h>
#include <assert.h>
#include <math.h>
#include <stdbool.h>
#include <stdio.h>


#define MAX_TILES	1024
#define TILE_W		32
#define TILE_H		32

#define NORMAL_E	(0)
#define NORMAL_ENE	(30*M_PI/180)
#define NORMAL_NE	(M_PI_4)
#define NORMAL_NNE	(60*M_PI/180)
#define NORMAL_N	(M_PI_2)
#define NORMAL_NNW	(120*M_PI/180)
#define NORMAL_NW	(3*M_PI_4)
#define NORMAL_WNW	(160*M_PI/180)
#define NORMAL_W	(M_PI)
#define NORMAL_WSW	-NORMAL_WNW
#define NORMAL_SW	-NORMAL_NW
#define NORMAL_SSW	-NORMAL_NNW
#define NORMAL_S	-NORMAL_N
#define NORMAL_SSE	-NORMAL_NNE
#define NORMAL_SE	-NORMAL_NE
#define NORMAL_ESE	-NORMAL_ENE

enum EDGE {
    EDGE_RIGHT,
    EDGE_TOP,
    EDGE_LEFT,
    EDGE_BOTTOM
};

static const char *edge_name[4] = {
    "right", "top", "left", "bottom"
};

static int green;
static int pink;
static int red, dred;
static int white;
static float guess_normal[4];
static BITMAP *dbuf;

static struct {
    float right;
    float top;
    float left;
    float bottom;
} keep[MAX_TILES];

/*--------------------------------------------------------------*/

static void draw_pointy(int cx, int cy, int col_normal, int col_tangent,
			float normal)
{
    line(dbuf, cx, cy, cx+48*cos(normal), cy-48*sin(normal), col_normal);

    line(dbuf, cx, cy,
	 cx+48*cos(normal+M_PI_2), cy-48*sin(normal+M_PI_2), col_tangent);
    line(dbuf, cx, cy,
	 cx+48*cos(normal-M_PI_2), cy-48*sin(normal-M_PI_2), col_tangent);
}


static void draw(BITMAP *bmp, int x, int y, int e, float normal)
{
    int cx, cy;
    clear(dbuf);

    cx = 160;
    cy = 100;

    textprintf_ex(dbuf, font, 50, 10, white, -1, "tile (%d, %d)", x, y);
    textprintf_ex(dbuf, font, 50, 20, white, -1, "edge %s", edge_name[e]);

    stretch_blit(bmp, dbuf,
		 x*TILE_W, y*TILE_H, TILE_W, TILE_H,
		 160-TILE_W, 100-TILE_H, 2*TILE_W, 2*TILE_H);

    draw_pointy(cx+TILE_W, cy, dred, 0, guess_normal[EDGE_RIGHT]);
    draw_pointy(cx, cy-TILE_H, dred, 0, guess_normal[EDGE_TOP]);
    draw_pointy(cx-TILE_W, cy, dred, 0, guess_normal[EDGE_LEFT]);
    draw_pointy(cx, cy+TILE_H, dred, 0, guess_normal[EDGE_BOTTOM]);

    if (e == EDGE_RIGHT)  cx += TILE_W;
    if (e == EDGE_TOP)    cy -= TILE_W;
    if (e == EDGE_LEFT)   cx -= TILE_W;
    if (e == EDGE_BOTTOM) cy += TILE_W;
    draw_pointy(cx, cy, red, green, normal);

    blit(dbuf, screen, 0, 0, 0, 0, dbuf->w, dbuf->h);
}

/*--------------------------------------------------------------*/

#define DICTIONARY_LENGTH	16
#define SKIP			"         "


static struct {
    char pattern[4][9];
    float right, top, left, bottom;
} dictionary[DICTIONARY_LENGTH] = {
    {				/* block */
	{ "..."
	  "..."
	  "...", SKIP, SKIP, SKIP
	},
	NORMAL_E, NORMAL_N, NORMAL_W, NORMAL_S
    },

    {				/* block */
	{ ".. "
	  ".. "
	  "   ",
	  
	  " .."
	  " .."
	  "   ",

	  "   "
	  ".. "
	  ".. ",

	  "   "
	  " .."
	  " .."
	},
	NORMAL_E, NORMAL_N, NORMAL_W, NORMAL_S
    },

    {				/* horizontal */
	{ "---"
	  "   "
	  "   ",

	  "---"
	  "---"
	  "   ",

	  "   "
	  "---"
	  "---",

	  "   "
	  "   "
	  "---"
	},
	NORMAL_E, NORMAL_N, NORMAL_W, NORMAL_S
    },

    {				/* vertical */
	{ "|  "
	  "|  "
	  "|  ",

	  "|| "
	  "|| "
	  "|| ",

	  " ||"
	  " ||"
	  " ||",

	  "  |"
	  "  |"
	  "  |",
	},
	NORMAL_E, NORMAL_N, NORMAL_W, NORMAL_S
    },

    {				/* diagonal */
	{
	  "   "
	  "  /"
	  " //",

	  "  /"
	  " ?/"
	  "///",

	  " //"
	  "///"
	  "///", SKIP
	},
	NORMAL_E, NORMAL_NW, NORMAL_NW, NORMAL_S
    },

    {				/* diagonal */
	{
	  "   "
	  "`  "
	  "`` ",

	  "`  "
	  "`? "
	  "```",

	  "`` "
	  "```"
	  "```", SKIP
	},
	NORMAL_NE, NORMAL_NE, NORMAL_W, NORMAL_S
    },

    {				/* diagonal */
	{
	  " ``"
	  "  `"
	  "   ",

	  "```"
	  " ?`"
	  "  `",

	  "```"
	  "```"
	  " ``", SKIP
	},
	NORMAL_E, NORMAL_N, NORMAL_SW, NORMAL_SW
    },

    {				/* diagonal */
	{
	  "// "
	  "/  "
	  "   ",

	  "///"
	  "/? "
	  "/  ",

	  "///"
	  "///"
	  "// ", SKIP
	},
	NORMAL_SE, NORMAL_N, NORMAL_W, NORMAL_SE
    },

    {				/* 30-degree */
	{
	  "   "
	  " ?-"
	  "---",

	  "  -"
	  "---"
	  "---", SKIP, SKIP
	},
	NORMAL_E, NORMAL_NNW, NORMAL_NNW, NORMAL_S
    },

    {				/* 30-degree */
	{
	  "   "
	  "-? "
	  "---",

	  "-  "
	  "---"
	  "---", SKIP, SKIP
	},
	NORMAL_NNE, NORMAL_NNE, NORMAL_W, NORMAL_S
    },

    {				/* 30-degree */
	{
	  "---"
	  " ?-"
	  "   ",

	  "---"
	  "---"
	  "  -", SKIP, SKIP
	},
	NORMAL_E, NORMAL_N, NORMAL_SSW, NORMAL_SSW
    },

    {				/* 30-degree */
	{
	  "---"
	  "-? "
	  "   ",

	  "---"
	  "---"
	  "-  ", SKIP, SKIP
	},
	NORMAL_SSE, NORMAL_N, NORMAL_W, NORMAL_SSE
    },

    {				/* 60-degree */
	{
	  "  -"
	  " ?-"
	  " --",

	  " --"
	  " --"
	  "---", SKIP, SKIP
	},
	NORMAL_E, NORMAL_WNW, NORMAL_WNW, NORMAL_S
    },

    {				/* 60-degree */
	{
	  "-  "
	  "-? "
	  "-- ",

	  "-- "
	  "-- "
	  "---", SKIP, SKIP
	},
	NORMAL_ENE, NORMAL_ENE, NORMAL_W, NORMAL_S
    },

    {				/* 60-degree */
	{
	  " --"
	  " ?-"
	  "  -",

	  "---"
	  " --"
	  " --", SKIP, SKIP
	},
	NORMAL_E, NORMAL_N, NORMAL_WSW, NORMAL_WSW
    },

    {				/* 60-degree */
	{
	  "-- "
	  "-? "
	  "-  ",

	  "---"
	  "-- "
	  "-- ", SKIP, SKIP
	},
	NORMAL_ESE, NORMAL_N, NORMAL_W, NORMAL_ESE
    }
};


static bool cmp(bool cs[9], char pattern[9])
{
    int i;

    for (i = 0; i < 9; i++) {
	bool p = (pattern[i] != ' ');

	if (pattern[i] == '?')
	    continue;

	if (cs[i] != p)
	    return false;
    }

    return true;
}


static void guess_normals(BITMAP *bmp, int x, int y, bool guess)
{
    /* cs goes from top-left to bottom-right */
    bool cs[9];
    int xl, xm, xr;
    int yb, ym, yt;
    int i, j;

    if (!guess) {
	i = (y * bmp->w/TILE_W) + x;
	guess_normal[EDGE_RIGHT]  = keep[i].right;
	guess_normal[EDGE_TOP]    = keep[i].top;
	guess_normal[EDGE_LEFT]   = keep[i].left;
	guess_normal[EDGE_BOTTOM] = keep[i].bottom;
	return;
    }

    xl = x*TILE_W;
    xm = xl + TILE_W/2;
    xr = xl + TILE_W-1;

    yt = y*TILE_H;
    ym = yt + TILE_H/2;
    yb = yt + TILE_H-1;

    cs[0] = getpixel(bmp, xl, yt) != pink;
    cs[1] = getpixel(bmp, xm, yt) != pink;
    cs[2] = getpixel(bmp, xr, yt) != pink;

    cs[3] = getpixel(bmp, xl, ym) != pink;
    cs[4] = getpixel(bmp, xm, ym) != pink;
    cs[5] = getpixel(bmp, xr, ym) != pink;

    cs[6] = getpixel(bmp, xl, yb) != pink;
    cs[7] = getpixel(bmp, xm, yb) != pink;
    cs[8] = getpixel(bmp, xr, yb) != pink;

    for (i = 0; i < DICTIONARY_LENGTH; i++) {
	for (j = 0; j < 4; j++) {
	    if (cmp(cs, dictionary[i].pattern[j])) {
		guess_normal[EDGE_RIGHT]  = dictionary[i].right;
		guess_normal[EDGE_TOP]    = dictionary[i].top;
		guess_normal[EDGE_LEFT]   = dictionary[i].left;
		guess_normal[EDGE_BOTTOM] = dictionary[i].bottom;
		return;
	    }
	}
    }
}

/*--------------------------------------------------------------*/

static float get_normal(float normal)
{
    static bool hack;

    if (key[KEY_6_PAD] && key[KEY_9_PAD]) { hack = 1; return NORMAL_ENE; }
    if (key[KEY_8_PAD] && key[KEY_9_PAD]) { hack = 1; return NORMAL_NNE; }
    if (key[KEY_7_PAD] && key[KEY_8_PAD]) { hack = 1; return NORMAL_NNW; }
    if (key[KEY_4_PAD] && key[KEY_7_PAD]) { hack = 1; return NORMAL_WNW; }

    if (key[KEY_1_PAD] && key[KEY_4_PAD]) { hack = 1; return NORMAL_WSW; }
    if (key[KEY_1_PAD] && key[KEY_2_PAD]) { hack = 1; return NORMAL_SSW; }
    if (key[KEY_2_PAD] && key[KEY_3_PAD]) { hack = 1; return NORMAL_SSE; }
    if (key[KEY_3_PAD] && key[KEY_6_PAD]) { hack = 1; return NORMAL_ESE; }

    if (hack > 0) {
	hack--;
	return normal;
    }

    if (key[KEY_6_PAD]) return NORMAL_E;
    if (key[KEY_7_PAD]) return NORMAL_NW;
    if (key[KEY_8_PAD]) return NORMAL_N;
    if (key[KEY_9_PAD]) return NORMAL_NE;
    if (key[KEY_4_PAD]) return NORMAL_W;
    if (key[KEY_1_PAD]) return NORMAL_SW;
    if (key[KEY_2_PAD]) return NORMAL_S;
    if (key[KEY_3_PAD]) return NORMAL_SE;

    if (key[KEY_PLUS_PAD])  return normal - 1*M_PI/180;
    if (key[KEY_MINUS_PAD]) return normal + 1*M_PI/180;

    return normal;
}


static void gen_loop(BITMAP *bmp, bool guess)
{
    bool quit = false;
    bool last_bkspc = false;
    bool last_space = false;
    bool last_comma = false;
    bool last_period = false;
    int x = 0, y = 0, e = EDGE_RIGHT;

    green = makecol(0x00, 0xff, 0x00); 
    pink  = makecol(0xff, 0x00, 0xff);
    red   = makecol(0xff, 0x00, 0x00);
    dred  = makecol(0x3f, 0x3f, 0x3f);
    white = makecol(0xff, 0xff, 0xff);

    guess_normals(bmp, x, y, guess);

    while (!quit) {
	if (key[KEY_ESC] || key[KEY_Q])
	    quit = true;

	/* Prev tile. */
	if (key[KEY_BACKSPACE] && !last_bkspc) {
	    if (x > 0) {
		x--;
	    }
	    else if (y > 0) {
		y--;
		x = bmp->w / TILE_W - 1;
	    }

	    guess_normals(bmp, x, y, guess);
	    e = EDGE_RIGHT;
	}

	/* Next tile. */
	if (key[KEY_SPACE] && !last_space) {
	    int i = y*(bmp->w/TILE_W) + x;
	    assert(i < MAX_TILES);
	    keep[i].right  = guess_normal[EDGE_RIGHT];
	    keep[i].top    = guess_normal[EDGE_TOP];
	    keep[i].left   = guess_normal[EDGE_LEFT];
	    keep[i].bottom = guess_normal[EDGE_BOTTOM];

	    if (++x >= bmp->w / TILE_W) {
		x = 0;

		if (++y >= bmp->h / TILE_H) {
		    quit = true;
		}
	    }

	    if (!quit)
		guess_normals(bmp, x, y, guess);

	    e = EDGE_RIGHT;
	}

	if (key[KEY_COMMA] && !last_comma) {
	    if (e < EDGE_BOTTOM)
		e++;
	}

	if (key[KEY_STOP] && !last_period) {
	    if (e > 0)
		e--;
	}

	last_bkspc = key[KEY_BACKSPACE];
	last_space = key[KEY_SPACE];
	last_comma = key[KEY_COMMA];
	last_period = key[KEY_STOP];

	guess_normal[e] = get_normal(guess_normal[e]);

	/* Draw */
	draw(bmp, x, y, e, guess_normal[e]);
	rest(50);
    }
}

/*--------------------------------------------------------------*/

static void export(BITMAP *bmp)
{
    FILE *fp;
    int i;

    fp = fopen(",normal.txt", "w");
    if (!fp) {
	printf("Error opening ,normal.txt for writing!\n");
	return;
    }

    for (i = 0; i < (bmp->w/TILE_W) * (bmp->h/TILE_H); i++) {
	fprintf(fp, "{ %g, %g, %g, %g }\n",
		keep[i].right, keep[i].top, keep[i].left, keep[i].bottom);
    }

    fclose(fp);
}


static void import(const char *fn)
{
    FILE *fp;
    int i = 0;

    fp = fopen(fn, "r");
    if (!fp) {
	printf("Error opening %s for reading!\n", fn);
	return;
    }

    while (!feof(fp)) {
	fscanf(fp, "{ %g, %g, %g, %g }\n",
	       &keep[i].right, &keep[i].top, &keep[i].left, &keep[i].bottom);
	i++;
    }

    fclose(fp);
}

/*--------------------------------------------------------------*/

int main(int argc, char **argv)
{
    BITMAP *bmp;

    if (argc <= 1) {
	printf("Usage: %s <bitmap> [normals]\n", argv[0]);
	return -1;
    }

    allegro_init();

    set_color_depth(desktop_color_depth());
    set_gfx_mode(GFX_AUTODETECT, 320, 200, 0, 0);

    install_keyboard();
    install_timer();

    bmp = load_bitmap(argv[1], NULL);
    if (!bmp) {
	printf("Error loading '%s'\n", argv[1]);
	return -2;
    }

    if (argc > 2)
	import(argv[2]);

    dbuf = create_bitmap(SCREEN_W, SCREEN_H);
    assert(dbuf);

    gen_loop(bmp, (argc <= 2));
    export(bmp);

    destroy_bitmap(dbuf);
    destroy_bitmap(bmp);

    return 0;
}
