/* weapon.c,
 *
 * Weapon classes and their stats.
 */

#include <assert.h>
#include <math.h>
#include "weapon.h"


#define NOT_EXPLOSIVE		{false, 0.0, 0.0, SMOKE_SMALL}
#define NOT_SMOKEY		{false, SMOKE_SMALL, 0}

const weapon_t weapon[NUM_WEAPON_CLASSES] = {
    /* Missile launcher */
    { 15,  600.0,   1.5,  500, 48.0/2.0, 10.0/2.0,  75, 150.0,
       {true, 200.0, 65.0, EXPLOSION_BIG}, {true, SMOKE_SMALL, 150} },

    /* Micro-missile launcher */
    {  3,  650.0,  10.0, 9000, 44.0/2.0,  8.0/2.0,  0, 75.0,
       {true, 150.0, 30.0, EXPLOSION_SMALL}, NOT_SMOKEY },

    /* Grenade launcher */
    {  5,  675.0,   3.0,    0,  0.0/2.0,  0.0/2.0,  0, -1.0,
       {true, 175.0, 45.0, EXPLOSION_BIG}, {true, SMOKE_SMALL, 200} },

    /* Blaster */
    {  2,  850.0,  25.0, 3000, 40.0/2.0,  3.0/2.0, 15, -1.0,
       NOT_EXPLOSIVE, NOT_SMOKEY },

    /* Shotgun (10 bullets per shot) */
    {  4,  850.0,   2.9,    0, 56.0/2.0,  6.0/2.0, 20, 50.0,
       NOT_EXPLOSIVE, NOT_SMOKEY },

    /* Vulcan */
    {  2, 7500.0,  50.0,    0, 60.0/2.0,  4.0/2.0,  4, 65.0,
       NOT_EXPLOSIVE, NOT_SMOKEY },

    /* Minigun */
    {  2, 1150.0,  60.0, 4000, 56.0/2.0, -7.0/2.0, 30, 25.0,
       NOT_EXPLOSIVE, NOT_SMOKEY }
};

#undef NOT_SMOKEY
#undef NOT_EXPLOSIVE

const weapon_sprite_t weapon_sprite[NUM_WEAPON_CLASSES] = {
    /* Missile launcher */
    {  2,   2, 77, 31, 6,   6, 23.0, 16.5 },

    /* Micro-missile launcher */
    {  2,  37, 58, 20, 4,   6, 8.0,  9.5 },

    /* Grenade launcher */
    {  2,  59, 63, 15, 6,   5, 5.0,  4.5 },

    /* Blaster */
    {  2,  96, 55, 13, 4,   8, 5.0,  5.5 },

    /* Shotgun */
    {  2, 112, 70, 15, 3,  12, 7.0,  8.5 },

    /* Vulcan */
    {  2, 130, 64, 19, 4,   6, 6.0,  9.5 },

    /* Minigun */
    {  2, 152, 68, 24, 6,   4, 6.0,  4.5 }
};

const sprite_t bullet_sprite[NUM_WEAPON_CLASSES] = {
    {  2,  2, 28, 13 },
    {  2, 18, 20, 10 },
    {  2, 31, 17, 10 },
    {  2, 44,  7,  4 },
    { 12, 44,  4,  5 },
    {  2, 52, 10,  4 },
    { 15, 52,  7,  4 }
};

/*--------------------------------------------------------------*/

bool weapon_is_automatic(const enum WEAPON_CLASS w)
{
    assert(w < NUM_WEAPON_CLASSES);

    if ((w == WEAPON_MICRO_MISSILE) ||
	(w == WEAPON_BLASTER) ||
	(w == WEAPON_VULCAN) ||
	(w == WEAPON_MINIGUN)) {
	return true;
    }
    else {
	return false;
    }
}


bool weapon_is_explosive(const enum WEAPON_CLASS w)
{
    assert(w < NUM_WEAPON_CLASSES);
    return weapon[w].explosion.explosive;
}


bool weapon_is_smokey(const enum WEAPON_CLASS w)
{
    assert(w < NUM_WEAPON_CLASSES);
    return weapon[w].smoke.smokey;
}

/*--------------------------------------------------------------*/

/* weapon_barrel_displacement:
 *
 * Return the displacement a bullet of class w should be spawned at
 * for a player aiming with the given mirror/angle pair via rx, ry.
 *
 * The function name is actually somewhat misleading - (rx, ry) is not
 * the vector connecting a player's shoulder to the barrel of the gun.
 *
 * (rx, ry) lies on the circle radius gun_y such that a bullet spawned
 * at (rx, ry) with angle will pass through the barrel of the gun.
 */
void weapon_barrel_displacement(const enum WEAPON_CLASS w,
				const bool mirror, const double angle,
				double *rx, double *ry)
{
    assert(w < NUM_WEAPON_CLASSES);

    if (!mirror) {
	*rx = -weapon[w].gun_y * sin(angle);
	*ry =  weapon[w].gun_y * cos(angle);
    }
    else {
	*rx =  weapon[w].gun_y * sin(angle);
	*ry = -weapon[w].gun_y * cos(angle);
    }
}
