/* sv-textface.c,
 *
 * Text mode interface for server.
 */

#include <assert.h>
#include <curses.h>
#include <nl.h>
#include <sched.h>
#include <string.h>

#include "chat.h"
#include "game.h"
#include "lobby.h"
#include "maxmin.h"
#include "network.h"
#include "packet-input.h"
#include "packet-server.h"
#include "server.h"
#include "strlcpy.h"
#include "sv-commands.h"
#include "textface/sv-prompt.h"
#include "textface/sv-status.h"
#include "textface/sv-textface.h"


#ifdef TARGET_WINDOWS
int __stdcall AllocConsole(void);
int __stdcall FreeConsole(void);
#endif

static NLsocket sock;
static NLenum type = NL_TCP_PACKETS;


int sv_repaint;

/*--------------------------------------------------------------*/

static void sv_chat_repaint(void)
{
    const chat_t *chat;
    int h = LAST_LOG_ROW, y;
    unsigned int i, l;

    l = chat_num_lines();
    if (l == 0)
	return;

    i = maxi(0, l-h-1);

    for (y = 0; (y <= h) && (i < l); y++, i++) {
	move(y, 0);

	chat = chat_get(i);
	assert(chat->str);

	attrset(A_NORMAL);
	addstr(chat->str);

	clrtoeol();
    }
}

/*--------------------------------------------------------------*/

#ifdef SIGWINCH
static void (*old_sigwinch_handler)(int);

static void sigwinch_handler(int num)
{
    (void)num;
    /* pick up new screen size */
    endwin();
    refresh();

    /* refresh_all(); */
}
#endif

/*--------------------------------------------------------------*/

static bool textface_init(const int port);
static void textface_shutdown(void);
static void sv_textface_incoming(game_state_t *state);
static void sv_textface_outgoing(game_state_t *state);


void textface_loop(const int port)
{
    struct packet_input_info packet;
    NLbyte buf[MAX_PACKET_SIZE];
    NLint len;
    game_state_t state;

    textface_init(port);
    game_state_init(&state, game_mode, true);

    strlcpy(packet.name, "Curses FE", sizeof(packet.name));
    packet.type = CLIENT_ADMIN;
    packet.face = 0;
    len = packet_input_info_encode(&packet, COMMAND_INPUT_INFO,
				   buf, sizeof(buf));
    nlWrite(sock, buf, len);

    while (state.context != SERVER_QUIT) {
	sv_textface_incoming(&state);
	sv_textface_outgoing(&state);

	if (chat_new_messages) {
	    chat_new_messages = false;
	    sv_repaint |= SV_REPAINT_CHAT;
	}

	if (sv_repaint) {
	    if (sv_repaint & SV_REPAINT_CHAT) {
		sv_chat_repaint();
	    }

	    if (sv_repaint & SV_REPAINT_STATUS) {
		sv_status_repaint();
	    }

	    if (sv_repaint & SV_REPAINT_PROMPT) {
		sv_prompt_repaint();
	    }

	    move_to_input_cursor();
	    refresh();
	    sv_repaint = 0;
	}

	sched_yield();
    }

    textface_shutdown();
}


static void sv_textface_incoming(game_state_t *state)
{
    NLbyte packet[MAX_PACKET_SIZE];
    NLint len;
    assert(state);

    len = nlRead(sock, &packet, sizeof(packet));
    if (len <= 0)
	return;

    switch (packet[0]) {
      case COMMAND_SERVER_STATE:
	  packet_server_state_decode(state, packet, len);
	  break;
      case COMMAND_INPUT_CHAT: {
	  char buf[MAX_PACKET_SIZE];
	  packet_input_chat_decode(buf, packet, len);
	  chat_add(0, CHAT_BABBLE, buf);
	  break;
      }
      default:
	  break;
    }
}


static void sv_textface_outgoing(game_state_t *state)
{
    sv_command_fe_func_t func;
    const char *str;
    assert(state);

    str = sv_prompt_poll();
    if (!str || strlen(str) == 0)
	return;

    func = sv_command_fe(str);

    if (func) {
	func(str, sock, state);
    }
    else {
	chat_add(0, CHAT_SERVER_ERROR, "Unrecognised command: %s", str);
    }
}

/*--------------------------------------------------------------*/

static void init_curses(void);
static void end_curses(void);


static bool textface_init(const int port)
{
    NLaddress addr;

#ifdef TARGET_WINDOWS
    AllocConsole();
#endif

    if ((sock = nlOpen(0, type)) == NL_INVALID)
	goto error;
	
    if (nlGetAddrFromName("localhost", &addr) != NL_TRUE)
	goto error;
	
    if (nlSetAddrPort(&addr, port) != NL_TRUE)
	goto error;

    if (nlConnect(sock, &addr) != NL_TRUE) 
	goto error;

    init_curses();
    sv_status_init();
    sv_prompt_init();

    sv_repaint = SV_REPAINT_ALL;
    return true;

 error:
    fprintf(stderr, "Error initialising HawkNL");
    return false;
}


static void textface_shutdown(void)
{
    sv_prompt_shutdown();
    sv_status_shutdown();
    end_curses();

    assert(sock != NL_INVALID);
    nlClose(sock);

#ifdef TARGET_WINDOWS
    FreeConsole();
#endif
}

/*--------------------------------------------------------------*/
/* Curses.							*/
/*--------------------------------------------------------------*/

static void init_curses(void)
{
    initscr();
    keypad(stdscr, true);
    nonl();
    nodelay(stdscr, true);
    noecho();

    if (has_colors()) {
	start_color();

	init_pair(PAIR_INPUT, COLOR_CYAN, COLOR_BLACK);
	init_pair(PAIR_STATUS, COLOR_WHITE, COLOR_BLUE);
    }

#ifdef SIGWINCH
    old_sigwinch_handler = signal(SIGWINCH, sigwinch_handler);
#endif
}


static void end_curses(void)
{
#ifdef SIGWINCH
    signal(SIGWINCH, old_sigwinch_handler);
#endif

    endwin();
}
