/* sv-commands.c,
 *
 * Server commands.
 */

#include <assert.h>
#include <ctype.h>
#include <nl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "chat.h"
#include "map-save.h"
#include "network.h"
#include "packet-editor.h"
#include "packet-input.h"
#include "packet-server.h"
#include "robot.h"
#include "server.h"
#include "strlcpy.h"
#include "sv-commands.h"
#include "sv-internal.h"


#define streq(a,b)		(strcmp(a, b) == 0)
#define strneq(a,b)		(strcmp(a, b) != 0)
#define CHATERROR(str)		chat_add(0, CHAT_FRONTEND_ERROR, str)
#define CHATHELP(str)		chat_add(0, CHAT_FRONTEND_LOG, str)

typedef struct {
    const char *macro, *full;

    /* Server side callback function. */
    sv_command_ss_func_t ss_func;
    
    /* Frontend side callback function. */
    sv_command_fe_func_t fe_func;

    /* Help function. */
    sv_command_help_func_t help_func;
} sv_command_t;

/*--------------------------------------------------------------*/

static client_id client_id_from_string(const char *str)
{
    int i;

    if (sscanf(str, "%d", &i) == 1) {
	if (sv_intern_client_from_client_id(i) < 0)
	    return 0;
	else
	    return i;
    }
    else {
	i = sv_intern_client_from_name(str);
	if (i < 0)
	    return 0;
	else
	    return client_data[i].clid;
    }
}

/*--------------------------------------------------------------*/

static void fe_resend(const char *str, const NLsocket sock)
{
    NLchar buf[MAX_PACKET_SIZE];
    NLint len;

    len = packet_server_encode(str, buf, sizeof(buf));
    nlWrite(sock, buf, len);
}


static void fe_relay(const char *str, const NLsocket sock, game_state_t *_)
{
    (void)_;

    fe_resend(str, sock);
}

/*--------------------------------------------------------------*/

static void ss_map(const char *str, const NLsocket sock, game_state_t *state)
{
    char work[MAX_PACKET_SIZE];
    NLchar buf[MAX_PACKET_SIZE];
    NLint len;
    assert(state);

    if (streq(str, "map")) {
	snprintf(work, sizeof(work), "[Server] Next map is '%s'",
		 state->next_map);
	len = packet_input_chat_encode(work, buf, sizeof(buf));
	server_send(sock, buf, len);
	return;
    }

    if (sscanf(str, "map %s", work) != 1) {
	return;
    }

    /* XXX: test file exists + readable. */
    strlcpy(state->next_map, work, sizeof(state->next_map));
    server_announce("[Server] Set next map to '%s'", state->next_map);
}


static void help_save(void)
{
    CHATHELP("Usage: save <filename>");
    CHATHELP("       Save the current map to 'filename' on the server");
    CHATHELP("       Only applies after the game has started");
}


static void ss_save(const char *str, const NLsocket _, game_state_t *state)
{
    char work[MAX_PACKET_SIZE];
    (void)_;
    assert(state);

    if ((state->context != SERVER_IN_GAME) ||
	(sscanf(str, "save %s", work) != 1)) {
	return;
    }
    else {
	server_announce("[Server] Map saved to '%s'", work);
	map_save(work, state);
    }
}


static void fe_save(const char *str, const NLsocket sock, game_state_t *state)
{
    char work[MAX_PACKET_SIZE];
    assert(state);

    if ((state->context != SERVER_IN_GAME) ||
	(sscanf(str, "save %s", work) != 1)) {
	help_save();
    }
    else {
	fe_resend(str, sock);
    }
}

/*--------------------------------------------------------------*/

static void ss_start(const char *_1, const NLsocket _2, game_state_t *state)
{
    NLchar buf[MAX_PACKET_SIZE];
    NLint len;
    (void)_1, (void)_2;
    assert(state);

    if (state->context == SERVER_IN_LOBBY) {
	state->context = SERVER_IN_GAME;

	len = packet_server_state_encode(state, buf, sizeof(buf));
	server_broadcast(buf, len);
    }
}


static void fe_start(const char *str, const NLsocket sock, game_state_t *state)
{
    assert(state);

    if (state->context == SERVER_IN_LOBBY) {
	fe_resend(str, sock);
	chat_add(0, CHAT_SERVER_LOG, "Entering game");
    }
}


static void ss_stop(const char *_1, const NLsocket _2, game_state_t *state)
{
    NLchar buf[MAX_PACKET_SIZE];
    NLint len;
    (void)_1, (void)_2;
    assert(state);

    if (state->context == SERVER_IN_GAME) {
	state->context = SERVER_IN_LOBBY;

	len = packet_server_state_encode(state, buf, sizeof(buf));
	server_broadcast(buf, len);
    }
}


static void fe_stop(const char *str, const NLsocket sock, game_state_t *state)
{
    assert(state);

    if (state->context == SERVER_IN_GAME) {
	fe_resend(str, sock);
	state->context = SERVER_IN_LOBBY;
	chat_add(0, CHAT_SERVER_LOG, "Entering lobby");
    }
}


static void help_gamemode(void)
{
    CHATHELP("Usage: gamemode <mode>");
    CHATHELP("       Change the game mode to:");
    CHATHELP("       0, classic: classic deathmatch");
    CHATHELP("       1, silhouettes: silhouette deathmatch");
}


static void ss_gamemode(const char *str, const NLsocket _, game_state_t *state)
{
    NLchar buf[MAX_PACKET_SIZE];
    NLint len;
    int arg;
    (void)_;
    assert(state);

    if (sscanf(str, "gamemode %d", &arg) != 1)
	return;
	
    if (arg == 0) {
	state->next_game_mode = GAME_MODE_CLASSIC;
	server_announce("Game mode: classic deathmatch");
    }
    else if (arg == 1) {
	state->next_game_mode = GAME_MODE_SILHOUETTES;
	server_announce("Game mode: silhouette deathmatch");
    }
    else {
	return;
    }

    len = packet_server_state_encode(state, buf, sizeof(buf));
    server_broadcast(buf, len);

    if (state->context == SERVER_IN_LOBBY) {
	state->game_mode = state->next_game_mode;
    }
}


static void fe_gamemode(const char *str, const NLsocket sock, game_state_t *_)
{
    char work[MAX_PACKET_SIZE];
    NLchar buf[MAX_PACKET_SIZE];
    NLint len;
    enum GAME_MODE mode;
    (void)_;

    if (sscanf(str, "gamemode %s", work) != 1) {
	help_gamemode();
	return;
    }

    if (streq(work, "0") || streq(work, "classic")) {
	mode = GAME_MODE_CLASSIC;
    }
    else if (streq(work, "1") || streq(work, "silhouettes")) {
	mode = GAME_MODE_SILHOUETTES;
    }
    else {
	help_gamemode();
	return;
    }

    snprintf(work, sizeof(work), "gamemode %d", mode);
    len = packet_server_encode(work, buf, sizeof(buf));
    nlWrite(sock, buf, len);
}


static void ss_edit(const char *_1, const NLsocket _2, game_state_t *state)
{
    struct packet_editor packet;
    NLchar buf[MAX_PACKET_SIZE];
    NLint len;
    (void)_1, (void)_2;
    assert(state);

    if (!(state->editor_enabled)) {
	state->editor_enabled = true;
	server_announce("Map editor mode");

	packet.mode = EDITOR_ENABLE;
	len = packet_editor_encode(&packet, buf, sizeof(buf));
	server_broadcast(buf, len);
    }
}


static void ss_noedit(const char *_1, const NLsocket _2, game_state_t *state)
{
    struct packet_editor packet;
    NLchar buf[MAX_PACKET_SIZE];
    NLint len;
    (void)_1, (void)_2;
    assert(state);

    if (state->editor_enabled) {
	state->editor_enabled = false;
	server_announce("Game mode");

	packet.mode = EDITOR_DISABLE;
	len = packet_editor_encode(&packet, buf, sizeof(buf));
	server_broadcast(buf, len);
    }
}


static void ss_context(const char *str, const NLsocket sock,
		       game_state_t *state)
{
    NLchar buf[MAX_PACKET_SIZE];
    NLint len;
    assert(state);

    switch (state->context) {
      case SERVER_IN_LOBBY:	str = "In the lobby"; break;
      case SERVER_IN_GAME:	str = "In the game"; break;
      default:			str = "Where am I?"; break;
    }

    len = packet_input_chat_encode(str, buf, sizeof(buf));
    server_send(sock, buf, len);
}

/*--------------------------------------------------------------*/

static void ss_list(const char *_1, const NLsocket sock, game_state_t *_2)
{
    char work[MAX_PACKET_SIZE];
    NLchar buf[MAX_PACKET_SIZE];
    NLint len;
    int c, i;
    (void)_1, (void)_2;

    for (c = 0, i = 0; i < MAX_CLIENTS; i++) {
	if (client_data[i].type == CLIENT_UNUSED)
	    continue;

	c++;
	snprintf(work, sizeof(work), "%d %s (%s)",
		 client_data[i].clid, client_data[i].name,
		 client_type_str(client_data[i].type));
	len = packet_input_chat_encode(work, buf, sizeof(buf));
	server_send(sock, buf, len);
    }

    {
	snprintf(work, sizeof(work), "Connected: %d  Max: %d",
		 c, MAX_CLIENTS);
	len = packet_input_chat_encode(work, buf, sizeof(buf));
	server_send(sock, buf, len);
    }
}


static void ss_score(const char *_1, const NLsocket sock, game_state_t *_2)
{
    client_data_t *data;
    char work[MAX_CHAT_LEN];
    NLbyte buf[MAX_PACKET_SIZE];
    NLint len;
    unsigned int c;
    (void)_1, (void)_2;

    for (c = 0; c < MAX_CLIENTS; c++) {
	data = &client_data[c];

	if (data->type == CLIENT_UNUSED)
	    continue;

	snprintf(work, sizeof(work),
		 "(Client %2d) %-16s: %d (%d), %d (%d), %d (%d)",
		 data->clid, data->name,
		 data->sc.total_score, data->sc.session_score,
		 data->sc.total_kills, data->sc.session_kills,
		 data->sc.total_deaths, data->sc.session_deaths);
	len = packet_input_chat_encode(work, buf, sizeof(buf));
	server_send(sock, buf, len);
    }
}

/*--------------------------------------------------------------*/

static void help_robots(void)
{
    CHATHELP("Usage: robots [num]");
    CHATHELP("       Show the number of robot players enabled");
    CHATHELP("       Or set the number robot players");
}


static void ss_add_robots(int num)
{
    while (num > 0) {
	if (!server_maybe_new_robot())
	    break;

	num--;
    }
}


static void ss_del_robots(int num, game_state_t *state)
{
    int i;
    assert(state);

    for (i = MAX_CLIENTS-1; (i >= 0) && (num > 0); i--) {
	if (client_data[i].type == CLIENT_ROBOT) {
	    server_disconnect(&client_data[i], DISCONNECT_REQUEST, state);
	    num--;
	}
    }
}


static void ss_robots(const char *str, const NLsocket sock, game_state_t *state)
{
    char work[MAX_CHAT_LEN];
    NLchar buf[MAX_PACKET_SIZE];
    NLint len;
    unsigned int num;
    assert(state);

    if (streq(str, "robots")) {
	snprintf(work, sizeof(work), "Robot players: %d", num_robots);
	len = packet_input_chat_encode(work, buf, sizeof(buf));
	server_send(sock, buf, len);
	return;
    }

    if (sscanf(str, "robots %d", &num) != 1) {
	return;
    }

    if (num > MAX_ROBOTS)
	num = MAX_ROBOTS;

    if (num_robots < num)
	ss_add_robots(num - num_robots);
    else if (num_robots > num)
	ss_del_robots(num_robots - num, state);

    snprintf(work, sizeof(work), "[Server] Set to %d robots (max %d)",
	     num_robots, MAX_ROBOTS);
    len = packet_input_chat_encode(work, buf, sizeof(buf));
    server_send(sock, buf, len);
}


static void fe_robots(const char *str, const NLsocket sock, game_state_t *_)
{
    int num;
    (void)_;

    if (strneq(str, "robots") &&
	sscanf(str, "robots %d", &num) != 1) {
	help_robots();
    }
    else {
	fe_resend(str, sock);
    }
}


static void help_setadmin(void)
{
    CHATHELP("Usage: setadmin <id|name>");
    CHATHELP("       Change the given client to an administrator");
    CHATHELP("       Administrators can use commands such as kick");
    CHATHELP("       Client id's are shown in the list command");
}


static void ss_setadmin(const char *str, const NLsocket _1, game_state_t *_2)
{
    char cmd[MAX_PACKET_SIZE];
    int clid, idx;
    (void)_1, (void)_2;

    if (sscanf(str, "%s %d", cmd, &clid) != 2)
	return;

    idx = sv_intern_client_from_client_id(clid);
    if (idx < 0)
	return;

    if (sv_intern_set_admin(&client_data[idx]))
	server_announce("New admin: %s", client_data[idx].name);
}


static void fe_setadmin(const char *str, const NLsocket sock, game_state_t *_)
{
    char cmd[MAX_PACKET_SIZE], work[MAX_PACKET_SIZE];
    NLchar buf[MAX_PACKET_SIZE];
    NLint len;
    client_id clid;
    (void)_;

    if (sscanf(str, "%s %s", cmd, work) != 2) {
	help_setadmin();
	return;
    }

    clid = client_id_from_string(work);
    if (clid == 0)
	return;

    snprintf(work, sizeof(work), "sa %d", clid);
    len = packet_server_encode(work, buf, sizeof(buf));
    nlWrite(sock, buf, len);
}


static void help_setgamer(void)
{
    CHATHELP("Usage: setgamer <id|name>");
    CHATHELP("       Change the given client to a game player");
    CHATHELP("       Client id's are shown in the list command");
}


static void ss_setgamer(const char *str, const NLsocket _1, game_state_t *_2)
{
    char cmd[MAX_PACKET_SIZE];
    int clid, idx;
    (void)_1, (void)_2;

    if (sscanf(str, "%s %d", cmd, &clid) != 2)
	return;

    idx = sv_intern_client_from_client_id(clid);
    if (idx < 0)
	return;

    sv_intern_set_gamer(&client_data[idx]);
}


static void fe_setgamer(const char *str, const NLsocket sock, game_state_t *_)
{
    char cmd[MAX_PACKET_SIZE], work[MAX_PACKET_SIZE];
    NLchar buf[MAX_PACKET_SIZE];
    NLint len;
    client_id clid;
    (void)_;

    if (sscanf(str, "%s %s", cmd, work) != 2) {
	help_setgamer();
	return;
    }

    clid = client_id_from_string(work);
    if (clid == 0)
	return;

    snprintf(work, sizeof(work), "sg %d", clid);
    len = packet_server_encode(work, buf, sizeof(buf));
    nlWrite(sock, buf, len);
}

/*--------------------------------------------------------------*/

static void help_kick(void)
{
    CHATHELP("Usage: kick <id|name>");
    CHATHELP("       Disconnect a client by their id or name");
    CHATHELP("       Client id's are shown in the list command");
}


static void ss_kick(const char *str, const NLsocket _1, game_state_t *state)
{
    char work[MAX_CHAT_LEN];
    int clid, idx;
    (void)_1;
    assert(state);

    if (sscanf(str, "kick %s", work) != 1)
	return;

    if (sscanf(work, "%d", &clid) == 1) {
	idx = sv_intern_client_from_client_id(clid);
	if ((idx >= 0) && 
	    (client_data[idx].type == CLIENT_GAMER)) {
	    server_disconnect(&client_data[idx], DISCONNECT_JERK, state);
	    return;
	}
    }

    for (idx = 0; idx < MAX_CLIENTS; idx++) {
	if ((client_data[idx].type != CLIENT_GAMER) &&
	    (streq(client_data[idx].name, work))) {
	    server_disconnect(&client_data[idx], DISCONNECT_JERK, state);
	    return;
	}
    }
}


static void fe_kick(const char *str, const NLsocket sock, game_state_t *_)
{
    char work[MAX_CHAT_LEN];
    (void)_;

    if (sscanf(str, "kick %s", work) != 1) {
	help_kick();
    }
    else {
	fe_resend(str, sock);
    }
}

/*--------------------------------------------------------------*/

static sv_command_help_func_t sv_command_help(const char *str);


static void fe_help(const char *str, const NLsocket _1, game_state_t *_2)
{
    char work[MAX_PACKET_SIZE];
    (void)_1, (void)_2;

    if (sscanf(str, "help %s", work) == 1) {
	sv_command_help_func_t help;

	help = sv_command_help(work);
	if (help) {
	    help();
	    return;
	}
    }

    CHATHELP("Commands:");
    
    CHATHELP("  map  [filename] - set next map");
    CHATHELP("  save <filename> - save current map");

    CHATHELP("  start           - enter game");
    CHATHELP("  stop            - return to the lobby");
    CHATHELP("  gamemode <mode> - set next game mode");
    CHATHELP("  edit            - switch into map editing mode");
    CHATHELP("  noedit          - switch into game mode");
    CHATHELP("  context         - show current context");

    CHATHELP("  list               - list clients");
    CHATHELP("  score              - show current scores");
    CHATHELP("  robots [num]       - set maximum number of robot players");
    CHATHELP("  setadmin <id|name> - set client to admin");
    CHATHELP("  setgamer <id|name> - set client to gamer");
    CHATHELP("  kick <id|name>     - forcefully disconnect a client");

    CHATHELP("  help [command]");
    CHATHELP("  quit");
}

/*--------------------------------------------------------------*/

static void ss_quit(const char *_1, const NLsocket _2, game_state_t *state)
{
    (void)_1, (void)_2;
    assert(state);

    state->context = SERVER_QUIT;
}

/*--------------------------------------------------------------*/

static const sv_command_t sv_commands[] = {
    { NULL,	"map",		ss_map,		fe_relay,	NULL },
    { NULL,	"save",		ss_save,	fe_save,	help_save },
    { NULL,	"start",	ss_start,	fe_start,	NULL },
    { NULL,	"stop",		ss_stop,	fe_stop,	NULL},
    { NULL,	"gamemode",	ss_gamemode,	fe_gamemode,	help_gamemode },
    { NULL,	"edit",		ss_edit,	fe_relay,	NULL },
    { NULL,	"noedit",	ss_noedit,	fe_relay,	NULL },
    { NULL,	"context",	ss_context,	fe_relay,	NULL },
    { "l",	"list",		ss_list,	fe_relay,	NULL },
    { NULL,	"score",	ss_score,	fe_relay,	NULL },
    { NULL,	"robots",	ss_robots,	fe_robots,	help_robots },
    { "sa",	"setadmin",	ss_setadmin,	fe_setadmin,	help_setadmin },
    { "sg",	"setgamer",	ss_setgamer,	fe_setgamer,	help_setgamer },
    { "k",	"kick",		ss_kick,	fe_kick,	help_kick },
    { "?",	"help",		NULL,		fe_help,	NULL },
    { "q",	"quit",		ss_quit,	fe_relay,	NULL },
    { NULL,	NULL,		NULL,		NULL,		NULL }
};

/*--------------------------------------------------------------*/

static bool sv_command_eq(const unsigned int i, const char *str)
{
    if (sv_commands[i].macro &&
	streq(sv_commands[i].macro, str))
	return true;

    if (sv_commands[i].full &&
	streq(sv_commands[i].full, str))
	return true;

    return false;
}


sv_command_ss_func_t sv_command_ss(const char *str)
{
    char command[MAX_PACKET_SIZE];
    unsigned int i;
    assert(str);

    if (sscanf(str, "%s ", command) < 1)
	return NULL;

    for (i = 0; sv_commands[i].full; i++) {
	if (sv_command_eq(i, command))
	    return sv_commands[i].ss_func;
    }

    return NULL;
}


sv_command_fe_func_t sv_command_fe(const char *str)
{
    char command[MAX_PACKET_SIZE];
    unsigned int i;
    assert(str);

    if (sscanf(str, "%s ", command) < 1)
	return NULL;

    for (i = 0; sv_commands[i].full; i++) {
	if (sv_command_eq(i, command))
	    return sv_commands[i].fe_func;
    }

    return NULL;
}


static sv_command_help_func_t sv_command_help(const char *str)
{
    unsigned int i;

    for (i = 0; sv_commands[i].full; i++) {
	if (sv_command_eq(i, str)) {
	    return sv_commands[i].help_func;
	}
    }

    return NULL;
}
