/* particle.c,
 *
 * Sparks and blood particles.
 *
 * See docs/particle.txt for the some detail about the particle
 * system.
 */

#include <alleggl.h>
#include <assert.h>
#include <math.h>

#include "angle.h"
#include "camera.h"
#include "common.h"
#include "container.h"
#include "game.h"
#include "gettime.h"
#include "list.h"
#include "lux.h"
#include "map.h"
#include "maxmin.h"
#include "particle.h"
#include "server.h"
#include "slist.h"
#include "stats-particle.h"
#include "step.h"
#include "universal.h"


#define PARTICLE_SOFT_LIMIT	5000
#define SLIVER_WIDTH		((640.0/0.8)/4)
#define MAX_SLIVERS		256
#define SQ(x)			((x)*(x))


typedef struct particle {
    enum PARTICLE_CLASS class;

    /* Spawn state. */
    int _spawn_time;
    float _spawn_x, _spawn_y;
    float _spawn_vx, _spawn_vy;

    /* Current state. */
    int life, time;
    float x, y;

    bool inactivate;
} particle_t;

typedef struct particle_array {
    struct particle_array *next;
    struct particle_array *prev;

    unsigned char length;
    particle_t particle[PARTICLE_ARRAY_SIZE];
} particle_array_t;

typedef struct {
    list_head_t list;

    /* Number of particles in the list. */
    int num;
} particle_sliver_t;

typedef struct inactive_particle {
    /* Singly linked-list. */
    struct inactive_particle *next;

    enum PARTICLE_CLASS class;
    int life;
    float x, y;
} inactive_particle_t;

/*--------------------------------------------------------------*/

/* Percentage of particles to spawn.  1 <= particle_multiplier.
 * Useful for debugging and slower machines.
 */
static unsigned int particle_multiplier = 100;

static particle_sliver_t particle_sliver[MAX_SLIVERS];	/* Client only */
static list_head_t parta_free_list;			/* Client only */
static slist_head_t parti_list;				/* Client only */
static slist_head_t parti_free_list;			/* Client only */

int parta_total;
int parte_count;
int parti_total;

#ifdef DEBUG_PARTICLE_STATS
# define INC(x)		x++
# define DEC(x)		x--
# define RESET(x)	x=0
int parta_count, parta_freed;
int parte_steps, parte_stepped, parte_nubbed;
int parte_drawn;
int parti_count, parti_freed;
#else
# define INC(x)
# define DEC(x)
# define RESET(x)
#endif

/*--------------------------------------------------------------*/

static int particle_sliver_from_x(const float x)
{
    if (x < 0)
	return 0;
    else if (x/SLIVER_WIDTH < MAX_SLIVERS)
	return x/SLIVER_WIDTH;
    else
	return MAX_SLIVERS-1;
}

/*--------------------------------------------------------------*/

#define STEP_COLLIDE_WITH_CONTAINER	0x10

static inactive_particle_t *
parti_new(const enum PARTICLE_CLASS class,
	  const float x, const float y, const int life);
static void parta_evict(particle_array_t *p);
static void parti_evict(inactive_particle_t *p, inactive_particle_t *prev);
static void particle_inactivate(particle_t *p);
static void parts_update(particle_sliver_t *sliver, timeval_t *tv0);
static void parta_update(particle_array_t *p);
static void parts_update2(particle_sliver_t *sliver);
static void parta_update2(particle_array_t *p);
static void particle_update_unit(particle_t *p);
static void particle_velocity(const void *q, double *vx, double *vy);
static int particle_step(void *q, const int tick, const void *_);
static bool particle_collided(void *q, const int tick, const int collisions);


void particle_update(void)
{
#define ALLOC_TIME		(1000000 * 0.3*SECONDS_PER_TICK)

    timeval_t tv0, tv1;
    inactive_particle_t *prev, *pi, *ni;
    int sliver_l, sliver_r, s, total;

    RESET(parte_steps);
    RESET(parte_stepped);
    RESET(parte_nubbed);

    prev = (void *)&parti_list;
    slist_for_each_next(pi, ni, &parti_list) {
	if (pi->life <= server_time) {
	    parti_evict(pi, prev);
	}
	else {
	    prev = pi;
	}
    }

    sliver_l = particle_sliver_from_x(camera.x);
    sliver_r = particle_sliver_from_x(camera.x + camera.view_width);
    for (s = sliver_l, total = 1; s <= sliver_r; s++)
	total += particle_sliver[s].num;

    sliver_l = particle_sliver_from_x(camera.x + camera.view_width/2);
    sliver_r = sliver_l+1;

    get_time_of_day(&tv0);
    tv1 = tv0;
    increase_time(&tv1, ALLOC_TIME);

    do {
	if (sliver_l >= 0) {
	    increase_time(&tv0,
			  ALLOC_TIME * particle_sliver[sliver_l].num/total);
	    if (time_between(&tv0, &tv1) < 0)
		tv0 = tv1;

	    parts_update(&particle_sliver[sliver_l], &tv0);
	    sliver_l--;
	}

	if (sliver_r < MAX_SLIVERS) {
	    increase_time(&tv0,
			  ALLOC_TIME * particle_sliver[sliver_r].num/total);
	    if (time_between(&tv0, &tv1) < 0)
		tv0 = tv1;

	    parts_update(&particle_sliver[sliver_r], &tv0);
	    sliver_r++;
	}
	else if (sliver_l < 0)
	    break;
    } while (time_between(&tv0, &tv1) > 0);

    /* Go through and remove old particles from the other lists. */
    while (sliver_l >= 0) {
	parts_update2(&particle_sliver[sliver_l]);
	sliver_l--;
    }

    while (sliver_r < MAX_SLIVERS) {
	parts_update2(&particle_sliver[sliver_r]);
	sliver_r++;
    }

#undef ALLOC_TIME
}


/* parts_update: 
 *
 * Update a sliver (list) of particle arrays.  'tv0' is the
 * recommended time we should stop processing.
 *
 * When returning, we update 'tv0' to the current time.
 */
static void parts_update(particle_sliver_t *sliver, timeval_t *tv0)
{
    particle_array_t *p, *n;
    timeval_t tv1;
    bool time_to_spare = true;
    int c = 0;
    assert(sliver);
    assert(tv0);

    sliver->num = 0;
    list_for_each_next(p, n, &(sliver->list)) {
	c++;
	if (c == 20) {
	    c = 0;

	    get_time_of_day(&tv1);
	    if (time_between(&tv1, tv0) < 0) {
		*tv0 = tv1;
		time_to_spare = false;
		break;
	    }
	}

	parta_update(p);
	if (p->length <= 0) {
	    parta_evict(p);
	    p = list_prev(n);
	}
	else {
	    sliver->num += p->length;
	}
    }

    /* If we didn't finish updating the whole list of particles, we
       want to resume from 'nx' in the next loop.  Rearrange:

       sliver -> a -> ... -> p -> n -> ... -> b -> sliver

       To be:

       sliver -> n -> ... -> b -> a -> ... -> p -> sliver
    */

    if (list_neq(p, &(sliver->list)) && list_neq(n, &(sliver->list))) {
	particle_array_t *a = list_next(&(sliver->list));
	particle_array_t *b = list_prev(&(sliver->list));

	a->prev = b;
	b->next = a;

	list_next(p) = &(sliver->list);
	sliver->list.prev = p;

	list_prev(n) = &(sliver->list);
	sliver->list.next = n;
    }
    else {
	get_time_of_day(tv0);
    }
}


/* parta_update:
 *
 * Update an array of particles.
 */
static void parta_update(particle_array_t *p)
{
    int i, j;

    for (i = 0, j = 0; i < p->length; i++) {
	if (parte_count > PARTICLE_SOFT_LIMIT)
	    p->particle[i].life -= 10 * parte_count/PARTICLE_SOFT_LIMIT;

	if (p->particle[i].life <= server_time) {
	    parte_count--;
	    continue;
	}

	particle_update_unit(&(p->particle[i]));
	INC(parte_stepped);

	if (p->particle[i].inactivate) {
	    particle_inactivate(&(p->particle[i]));
	    parte_count--;
	}
	else {
	    if (i != j)
		p->particle[j] = p->particle[i];

	    j++;
	}
    }

    p->length = j;
}


/* parts_update2:
 *
 * Quickly can through a sliver (list) of particle arrays, and evict
 * any old particles.  This helps keep the number of active particles
 * down.
 */
static void parts_update2(particle_sliver_t *sliver)
{
    particle_array_t *p, *n;
    assert(sliver);

    sliver->num = 0;
    list_for_each_next(p, n, &(sliver->list)) {
	parta_update2(p);
	if (p->length <= 0) {
	    parta_evict(p);
	    p = list_prev(n);
	}
	else {
	    sliver->num += p->length;
	}
    }
}


/* parta_update2:
 *
 * Quickly can through a particle arrays, and evict any old particles.
 * This helps keep the number of active particles down.
 */
static void parta_update2(particle_array_t *p)
{
    int i, j;

    for (i = 0, j = 0; i < p->length; i++) {
	if (parte_count > PARTICLE_SOFT_LIMIT)
	    p->particle[i].life -= 10 * parte_count/PARTICLE_SOFT_LIMIT;

	if (p->particle[i].life <= server_time) {
	    parte_count--;
	}
	else {
	    if (i != j)
		p->particle[j] = p->particle[i];

	    j++;
	}
    }

    p->length = j;
}


static void particle_update_unit(particle_t *p)
{
    const walk_time_callbacks_t callbacks = {
	particle_velocity,
	particle_step,
	particle_collided
    };
    int t;

    t = mini(server_time, p->time + 2.0*1000.0*SECONDS_PER_TICK);

    walk_time_crude(p, &callbacks, p->time, t, NULL);
}


static void particle_velocity(const void *q, double *vx, double *vy)
{
    const particle_t *p = q;
    double t;
    assert(p);

    t = (p->time-p->_spawn_time)/1000.0;
    if (t > 0) {
	*vx = p->_spawn_vx;
	*vy = p->_spawn_vy - GRAVITY2*t;
    }
    else {
	*vx = 0.0;
	*vy = 0.0;
    }
}


static void particle_inactivate(particle_t *p)
{
    inactive_particle_t *new;
    inactive_particle_t *prev, *cmp;
    assert(p);

    slist_for_each_prev(cmp, prev, &parti_list) {
	if (cmp->y > p->y + 1.0)
	    break;

	if (fabsf(cmp->y-p->y) < 1.0) {
	    if (fabsf(cmp->x-p->x) < 1.0) {
		/* Nub. */
		if (cmp->life < p->life) {
		    cmp->life = p->life;
		    cmp->class = p->class;
		}
		INC(parte_nubbed);
		return;
	    }

	    if (cmp->x > p->x)
		break;
	}
    }

    /* Insert the new particle between prev and cmp. */
    new = parti_new(p->class, p->x, p->y, p->life);
    if (!new)
	return;

    slist_add_at_pos(prev, new);
    INC(parti_count);
}


static void particle_move(particle_t *p, const int tick)
{
    double t = (tick - p->_spawn_time)/1000.0;

    p->time = tick;
    if (t > 0) {
	p->x = p->_spawn_x + p->_spawn_vx*t;
	p->y = p->_spawn_y + p->_spawn_vy*t - 0.5*GRAVITY2*t*t;
    }
    else {
	p->x = p->_spawn_x;
	p->y = p->_spawn_y;
    }
}


static int particle_step(void *q, const int tick, const void *_)
{
    double vx, vy;
    particle_t *p = q;
    int collisions = 0;
    (void)_;
    assert(p);

    particle_move(p, tick);
    INC(parte_steps);

    /* Check collisions. */
    if (!x_in_map(p->x))
	return STEP_COLLIDE_X;

    if (!y_in_map(p->y))
	return STEP_COLLIDE_Y;

    particle_velocity(p, &vx, &vy);

    if (vy < 0.0 && map_occupies(p->x, p->y))
	collisions |= STEP_COLLIDE_BOTTOM;
    else {
	if (vy > 0.0 && y_in_map(p->y+1) && map_occupies(p->x, p->y+1))
	    collisions |= STEP_COLLIDE_TOP;

	if ((vx < 0.0 && x_in_map(p->x-1) && map_occupies(p->x-1, p->y)) ||
	    (vx > 0.0 && x_in_map(p->x+1) && map_occupies(p->x+1, p->y)))
	    collisions |= STEP_COLLIDE_X;
    }

    if (collisions == 0) {
	if (container_bounding(p->x, p->y, true))
	    collisions = STEP_COLLIDE_XY|STEP_COLLIDE_WITH_CONTAINER;
    }

    return collisions;
}


static bool particle_collided(void *q, const int tick, const int collisions)
{
    particle_t *p = q;
    assert(p);

    particle_move(p, tick);
    p->_spawn_time = tick;
    p->_spawn_x = p->x;
    p->_spawn_y = p->y;

    if (collisions & STEP_COLLIDE_BOTTOM) {
	p->_spawn_vy = -p->_spawn_vy * 0.75;

	if (collisions & STEP_COLLIDE_WITH_CONTAINER) {
	    p->_spawn_vx *= 0.75;
	    p->_spawn_vy = -0.5;
	}
	else if (SQ(p->_spawn_vy) < SQ(0.01)) {
	    p->_spawn_vx = 0.0;
	    p->inactivate = true;
	}
    }
    else {
	if (collisions & STEP_COLLIDE_X)
	    p->_spawn_vx = -p->_spawn_vx * 0.75;

	if (collisions & STEP_COLLIDE_TOP)
	    p->_spawn_vy = -p->_spawn_vy * 0.75;
    }

    return true;
}

/*--------------------------------------------------------------*/

static void parts_draw(const list_head_t *sliver, enum PARTICLE_CLASS *last);
static void parta_draw(const particle_array_t *pa, enum PARTICLE_CLASS *last);
static void particle_draw_unit(const float x, const float y,
			       const enum PARTICLE_CLASS class,
			       enum PARTICLE_CLASS *last_class);


void particle_draw(void)
{
    const GLfloat emi[] = { 0.10, 0.10, 0.10, 0.10 };
    enum PARTICLE_CLASS last_class = PARTICLE_BLOOD;
    inactive_particle_t *pi;
    int i, sliver_l, sliver_r;

    sliver_l = particle_sliver_from_x(camera.x);
    sliver_r = particle_sliver_from_x(camera.x + camera.view_width);

    if (slist_empty(parti_list)) {
	for (i = sliver_l; i <= sliver_r; i++) {
	    if (particle_sliver[i].num > 0)
		goto start_draw;
	}

	return;
    }

 start_draw:
    glColor3fv(particle_stats[last_class].colour);
    glMaterialfv(GL_FRONT, GL_EMISSION, emi);
    glDisable(GL_TEXTURE_2D);
    glBegin(GL_QUADS);

    slist_for_each(pi, &parti_list) {
	if (!point_in_sight(&camera, pi->x, pi->y))
	    continue;

 	particle_draw_unit(pi->x, pi->y, pi->class, &last_class);
    }

    RESET(parte_drawn);
    for (; sliver_l <= sliver_r; sliver_l++) {
	parts_draw(&particle_sliver[sliver_l].list, &last_class);
    }

    glEnd();			/* glBegin(GL_QUADS) */
    glColor3fv(glWhite);
    glMaterialfv(GL_FRONT, GL_EMISSION, glBlack);
    glEnable(GL_TEXTURE_2D);
}


/* parts_draw:
 *
 * Draw a sliver (list) of particle arrays.
 */
static void parts_draw(const list_head_t *sliver, enum PARTICLE_CLASS *last)
{
    const particle_array_t *p;
    assert(sliver);

    list_for_each(p, sliver) {
	parta_draw(p, last);
    }
}


/* parta_draw:
 *
 * Draw an array of particles.
 */
static void parta_draw(const particle_array_t *pa, enum PARTICLE_CLASS *last)
{
    const particle_t *p;
    int i;
    assert(pa);

    for (i = 0; i < pa->length; i++) {
	p = &(pa->particle[i]);

	if (!point_in_sight(&camera, p->x, p->y))
	    continue;

	particle_draw_unit(p->x, p->y, p->class, last);
    }
}


/* particle_draw_unit:
 *
 * Draw a single particle, making colour changes as necessary.
 */
static void particle_draw_unit(const float x, const float y,
			       const enum PARTICLE_CLASS class,
			       enum PARTICLE_CLASS *last_class)
{
    if (*last_class != class) {
	*last_class = class;

	assert(class < NUM_PARTICLE_CLASSES);
	glColor3fv(particle_stats[class].colour);
    }

    glVertex2d(x, y+1.5);
    glVertex2d(x-1.5, y);
    glVertex2d(x, y-1.0);
    glVertex2d(x+1.5, y);

    INC(parte_drawn);
}

/*--------------------------------------------------------------*/

void particle_new(const enum PARTICLE_CATEGORY cat,
		  const float x, const float y, const int tick)
{
    const part_cat_stats_t *cat_stat;
    const particle_stats_t *stat;
    enum PARTICLE_CLASS c;
    unsigned int i, num;
    lux_t *lux;
    assert(cat < NUM_PARTICLE_CATEGORIES);

    cat_stat = &particle_category[cat];
    num = cat_stat->num * particle_multiplier / 100;

    for (i = 0; i < num; i++) {
	double angle = deg2rad(rand() % 360);
	double v;

	c = particle_class_from_category(cat);
	stat = &particle_stats[c];

	v = stat->v_min + (rand() % stat->v_rand);
	particle_new_unit(c, x, y, v*cos(angle), v*sin(angle),
			  tick + (rand() % stat->time_rand));
    }

    lux = lux_new(x, y, cat_stat->lux_life, cat_stat->lux_colour);
    lux->r = cat_stat->lux_r;
    lux_automate(lux);
}


/* particle_new_unit:
 *
 * Add a new particle with the given parameters into the particle
 * array pa if possible.  Return the particle array the particle was
 * added into.
 */
void particle_new_unit(const enum PARTICLE_CLASS class,
		       const float x, const float y,
		       const float vx, const float vy, const int tick)
{
    const particle_stats_t *stat;
    int sliver;
    particle_array_t *pa = NULL;
    particle_t *p;
    assert(class < NUM_PARTICLE_CLASSES);

    /* Find the correct sliver to insert into. */
    sliver = particle_sliver_from_x(x);

    /* Is there a particle array in the sliver with unused
       particles? */
    if (!(list_empty(particle_sliver[sliver].list)))
	pa = list_next(&(particle_sliver[sliver].list));

    /* We need to add a new particle array into the sliver. */
    if ((pa == NULL) || (pa->length >= PARTICLE_ARRAY_SIZE)) {
	if (list_empty(parta_free_list)) {
	    pa = calloc(1, sizeof(*pa));
	    if (!pa)
		return;

	    parta_total++;
	}
	else {
	    pa = list_next(&parta_free_list);
	    list_remove(pa);
	    DEC(parta_freed);

	    pa->length = 0;
	}

	list_add(particle_sliver[sliver].list, pa);
	INC(parta_count);
    }

    stat = &particle_stats[class];
    p = &(pa->particle[pa->length]);
    pa->length = (pa->length + 1);

    p->class = class;

    /* Spawn state. */
    p->_spawn_time = tick;
    p->_spawn_x = x;
    p->_spawn_y = y;
    p->_spawn_vx = vx;
    p->_spawn_vy = vy;

    /* Current state. */
    p->life = p->_spawn_time + stat->life_min + (rand() % stat->life_rand);
    p->time = p->_spawn_time;
    p->x = p->_spawn_x;
    p->y = p->_spawn_y;
    p->inactivate = false;

    parte_count++;
}


static inactive_particle_t *
parti_new(const enum PARTICLE_CLASS class,
	  const float x, const float y, const int life)
{
    inactive_particle_t *p;

    if (slist_empty(parti_free_list)) {
	p = malloc(sizeof(*p));
	if (!p)
	    return NULL;

	p->next = NULL;
	parti_total++;
    }
    else {
	p = slist_next(&parti_free_list);
	slist_remove(p, &parti_free_list);
	DEC(parti_freed);
    }

    p->class = class;
    p->life = life;
    p->x = x;
    p->y = y;
    return p;
}


static void parta_evict(particle_array_t *p)
{
    list_remove(p);
    list_add(parta_free_list, p);

    DEC(parta_count);
    INC(parta_freed);
}


static void parta_free(particle_array_t *p)
{
    free(p);
    parta_total--;
}


static void parti_evict(inactive_particle_t *p, inactive_particle_t *prev)
{
    slist_remove(p, prev);
    slist_add(parti_free_list, p);

    DEC(parti_count);
    INC(parti_freed);
}


static void parti_free(inactive_particle_t *p)
{
    free(p);
    parti_total--;
}

/*--------------------------------------------------------------*/

void particle_set_multiplier(const int x)
{
    particle_multiplier = MID(1, x, 1000);
}


int particle_get_multiplier(void)
{
    return particle_multiplier;
}

/*--------------------------------------------------------------*/

void particle_start(void)
{
    int s;

    for (s = 0; s < MAX_SLIVERS; s++) {
	list_init(particle_sliver[s].list);
	particle_sliver[s].num = 0;
    }

    list_init(parta_free_list);
    slist_init(parti_list);
    slist_init(parti_free_list);

    parta_total = 0;
    parte_count = 0;
    parti_total = 0;

    RESET(parta_count);
    RESET(parta_freed);
    RESET(parti_count);
    RESET(parti_freed);
}


void particle_stop(void)
{
    int s;

    for (s = 0; s < MAX_SLIVERS; s++) {
	list_free(particle_sliver[s].list, parta_free);
    }

    list_free(parta_free_list, parta_free);
    slist_free(parti_list, parti_free);
    slist_free(parti_free_list, parti_free);

    assert(parta_total == 0);
    assert(parti_total == 0);
}


void particle_init(void)
{
}


void particle_shutdown(void)
{
}
