/* packet-player.c,
 *
 * Convert player command packets to/from a buffer, to be sent across
 * the network.
 */

#include <assert.h>
#include <stdio.h>
#include "angle.h"
#include "network.h"
#include "packet-player.h"


/*--------------------------------------------------------------*/

NLint packet_player_new_begin_encode(const int num,
				     NLbyte *buf, const NLint sz)
{
    NLint count = 0;
    assert(num > 0);
    assert(buf);

    writeByte(buf, count, COMMAND_PLAYER_NEW);
    writeByte(buf, count, num);

    assert(count <= sz);
    return count;
}


NLint packet_player_new_begin_decode(int *num,
				     const NLbyte *buf, const NLint sz)
{
    NLint count = 1;
    int n;
    assert(num);
    assert(buf);

    readByte(buf, count, n); *num = n;

    assert(count <= sz);
    return count;
}


NLint packet_player_new_encode(const struct packet_player_new *packet,
			       NLbyte *buf, const NLint size, NLint count)
{
    int xx = packet->x + zero_x;
    int yy = packet->y + zero_y;
    assert(packet);
    assert(buf);

    writeClientID(buf, count, packet->clid);
    writePlayerID(buf, count, packet->id);
    writeByte (buf, count, packet->alive);

    if (packet->alive) {
	writeShort(buf, count, xx);
	writeShort(buf, count, yy);
    }

    assert(count <= size);
    return count;
}


NLint packet_player_new_decode(struct packet_player_new *packet,
			       const NLbyte *buf, const NLint size,
			       NLint count)
{
    int xx, yy;
    assert(packet);
    assert(buf);

    readClientID(buf, count, packet->clid);
    readPlayerID(buf, count, packet->id);
    readByte(buf, count, packet->alive);

    if (packet->alive) {
	readShort(buf, count, xx); packet->x = xx - zero_x;
	readShort(buf, count, yy); packet->y = yy - zero_y;
    }
    else {
	packet->x = 0.0;
	packet->y = 0.0;
    }

    assert(count <= size);
    return count;
}

/*--------------------------------------------------------------*/

NLint packet_player_attach_encode(const struct packet_player_attach *packet,
				  NLbyte *buf, NLint size)
{
    NLint count = 0;

    writeByte(buf, count, COMMAND_PLAYER_ATTACH);
    writePlayerID(buf, count, packet->id);

    assert(count <= size);
    return count;
}


void packet_player_attach_decode(struct packet_player_attach *packet,
				 const NLbyte *buf, const NLint size)
{
    NLint count = 1;

    readPlayerID(buf, count, packet->id);

    assert(count == size);
}

/*--------------------------------------------------------------*/

NLint packet_player_update_encode(const struct packet_player_update *packet,
				  const bool local,
				  NLbyte *buf, NLint size)
{
    NLint count = 0;
    char command;
    float xx = packet->x + zero_x;
    float yy = packet->y + zero_y;

    if (local) {
	command = ((packet->size == 0)
		   ? COMMAND_PLAYER_UPDATEL
		   : COMMAND_PLAYER_UPDATEL2);
    }
    else {
	command = ((packet->size == 0)
		   ? COMMAND_PLAYER_UPDATER
		   : COMMAND_PLAYER_UPDATER2);
    }

    writeByte(buf, count, command);
    writeLong(buf, count, packet->time);

    if (!local)
	writePlayerID(buf, count, packet->id);

    writeFloat(buf, count, xx);
    writeFloat(buf, count, yy);

    if (packet->size == 0)
	goto end;

    /* Optional: */
    writeByte(buf, count, packet->size);

    if (packet->size & PACKET_INCLUDE_WEAPON)
	writeWeaponClass(buf, count, packet->weapon);

    if (packet->size & PACKET_INCLUDE_ANGLE) {
	char a = charize_angle(packet->angle);
	writeByte(buf, count, packet->mirror);
	writeByte(buf, count, a);
    }

    if (packet->size & PACKET_INCLUDE_AMMO)
	writeShort(buf, count, packet->ammo);

    if (packet->size & PACKET_INCLUDE_HEALTH)
	writeByte(buf, count, packet->health);

    if (packet->size & PACKET_INCLUDE_TRACKER)
	writeFloat(buf, count, packet->tracker);

 end:
    assert(count <= size);
    return count;
}


void packet_player_update_decode(struct packet_player_update *packet,
				 const bool local, const bool large,
				 const NLbyte *buf, const NLint size)
{
    NLint count = 1;

    readLong(buf, count, packet->time);

    if (!local)
	readPlayerID(buf, count, packet->id);
    else
	packet->id = 0;

    readFloat(buf, count, packet->x); packet->x -= zero_x;
    readFloat(buf, count, packet->y); packet->y -= zero_y;

    if (!large) {
	packet->size = 0;
	goto end;
    }

    /* Large packets may include: */
    readByte(buf, count, packet->size);

    if (packet->size & PACKET_INCLUDE_WEAPON)
	readWeaponClass(buf, count, packet->weapon);

    if (packet->size & PACKET_INCLUDE_ANGLE) {
	char a;
	readByte(buf, count, packet->mirror);
	readByte(buf, count, a);
	packet->angle = uncharize_angle(a);
    }

    if (packet->size & PACKET_INCLUDE_AMMO)
	readShort(buf, count, packet->ammo);

    if (packet->size & PACKET_INCLUDE_HEALTH)
	readByte(buf, count, packet->health);

    if (packet->size & PACKET_INCLUDE_TRACKER)
	readFloat(buf, count, packet->tracker);

 end:
    assert(count == size);
}

/*--------------------------------------------------------------*/

NLint packet_player_respawn_encode(const struct packet_player_respawn *packet,
				   NLbyte *buf, NLint size)
{
    NLint count = 0;

    writeByte(buf, count, COMMAND_PLAYER_RESPAWN);
    writePlayerID(buf, count, packet->id);
    writeStartID(buf, count, packet->loc);

    assert(count <= size);
    return count;
}


void packet_player_respawn_decode(struct packet_player_respawn *packet,
				  const NLbyte *buf, const NLint size)
{
    NLint count = 1;

    readPlayerID(buf, count, packet->id);
    readStartID(buf, count, packet->loc);

    assert(count == size);
}

/*--------------------------------------------------------------*/

NLint packet_player_die_encode(const struct packet_player_die *packet,
			       NLbyte *buf, NLint size)
{
    NLint count = 0;
    float xx = packet->x + zero_x;
    float yy = packet->y + zero_y;

    writeByte (buf, count, COMMAND_PLAYER_DIE);
    writeLong (buf, count, packet->time);
    writePlayerID(buf, count, packet->id);
    writeByte (buf, count, packet->why);
    writeFloat(buf, count, xx);
    writeFloat(buf, count, yy);
    writeFloat(buf, count, packet->vx);
    writeFloat(buf, count, packet->vy);
    writeShort(buf, count, packet->bp);
    writePlayerID(buf, count, packet->murderer);

    assert(count <= size);
    return count;
}


void packet_player_die_decode(struct packet_player_die *packet,
			      const NLbyte *buf, const NLint size)
{
    NLint count = 1;

    readLong (buf, count, packet->time);
    readPlayerID(buf, count, packet->id);
    readByte (buf, count, packet->why);
    readFloat(buf, count, packet->x); packet->x -= zero_x;
    readFloat(buf, count, packet->y); packet->y -= zero_y;
    readFloat(buf, count, packet->vx);
    readFloat(buf, count, packet->vy);   
    readShort(buf, count, packet->bp);
    readPlayerID(buf, count, packet->murderer);

    assert(count == size);
}

/*--------------------------------------------------------------*/

NLint packet_disconnect_encode(const struct packet_disconnect *packet,
			       NLbyte *buf, NLint size)
{
    NLint count = 0;

    writeByte(buf, count, COMMAND_DISCONNECT);
    writeClientID(buf, count, packet->clid);
    writeByte(buf, count, packet->why);

    assert(count <= size);
    return count;
}


void packet_disconnect_decode(struct packet_disconnect *packet,
			      const NLbyte *buf, const NLint size)
{
    NLint count = 1;

    readClientID(buf, count, packet->clid);
    readByte(buf, count, packet->why);

    assert(count == size);
}
