/* packet-candela.c,
 *
 * Convert pickup command packets to/from a buffer, to be sent across
 * the network.
 */

#include <assert.h>
#include "angle.h"
#include "network.h"
#include "packet-candela.h"


/*--------------------------------------------------------------*/

NLint packet_candela_new_begin_encode(const int num,
				      NLbyte *buf, const NLint sz)
{
    NLint count = 0;
    assert(buf);
    
    writeByte(buf, count, COMMAND_CANDELA_NEW);
    writeByte(buf, count, num);

    assert(count <= sz);
    return count;
}


NLint packet_candela_new_begin_decode(int *num,
				      const NLbyte *buf, const NLint sz)
{
    NLint count = 1;
    int n;
    assert(num);
    assert(buf);

    readByte(buf, count, n); *num = n;

    assert(count <= sz);
    return count;
}


NLint packet_candela_new_encode(const struct packet_candela_new *packet,
				NLbyte *buf, const NLint sz, NLint count)
{
    int i;
    int xx = packet->posi[0] + zero_x;
    int yy = packet->posi[1] + zero_y;
    unsigned char x;
    assert(packet);
    assert(buf);

    writeCandelaID(buf, count, packet->id);
    writeByte(buf, count, packet->size);

    if (packet->size & CANDELA_INCLUDE_POSITION) {
	writeShort(buf, count, xx);
	writeShort(buf, count, yy);
    }

    if (packet->size & CANDELA_INCLUDE_AMBIENT) {
	for (i = 0; i < 3; i++) {
	    x = packet->ambi[i]*255.0;
	    writeByte(buf, count, x);
	}
    }

    if (packet->size & CANDELA_INCLUDE_CUTOFF) {
	x = charize_angle(packet->cutoff/90.0);
	writeByte(buf, count, x);
    }

    if (packet->size & CANDELA_INCLUDE_EXPONENT)
	writeByte(buf, count, packet->exponent);

    assert(count <= sz);
    return count;
}


NLint packet_candela_new_decode(struct packet_candela_new *packet,
				const NLbyte *buf, const NLint sz, NLint count)
{
    int i, xx, yy;
    unsigned char x;
    assert(packet);
    assert(buf);

    readCandelaID(buf, count, packet->id);
    readByte(buf, count, packet->size);

    if (packet->size & CANDELA_INCLUDE_POSITION) {
	readShort(buf, count, xx); packet->posi[0] = xx - zero_x;
	readShort(buf, count, yy); packet->posi[1] = yy - zero_y;
    }

    if (packet->size & CANDELA_INCLUDE_AMBIENT) {
	for (i = 0; i < 3; i++) {
	    readByte(buf, count, x);
	    packet->ambi[i] = x/255.0;
	}
    }

    if (packet->size & CANDELA_INCLUDE_CUTOFF) {
	readByte(buf, count, x);
	packet->cutoff = uncharize_angle(x)*90.0;
    }

    if (packet->size & CANDELA_INCLUDE_EXPONENT)
	readByte(buf, count, packet->exponent);

    assert(count <= sz);
    return count;
}

/*--------------------------------------------------------------*/
/* Map editor only.						*/
/*--------------------------------------------------------------*/

NLint packet_candela_delete_encode(const struct packet_candela_del *packet,
				   NLbyte *buf, const NLint buf_size)
{
    NLint count = 0;
    assert(packet);

    writeCandelaID(buf, count, packet->id);

    assert(count <= buf_size);
    return count;
}


void packet_candela_delete_decode(struct packet_candela_del *packet,
				  const NLbyte *buf, const NLint buf_size)
{
    NLint count = 0;
    assert(packet);

    readCandelaID(buf, count, packet->id);

    assert(count == buf_size);
}
