/* packet-backpack.c,
 *
 * Convert backpack command packets to/from a buffer, to be sent
 * across the network.
 */

#include <assert.h>
#include "network.h"
#include "packet-backpack.h"


/*--------------------------------------------------------------*/

NLint packet_backpack_new_begin_encode(const int num,
				       NLbyte *buf, const NLint sz)
{
    NLint count = 0;
    assert(num > 0);
    assert(buf);

    writeByte(buf, count, COMMAND_BACKPACK_NEW);
    writeByte(buf, count, num);

    assert(count <= sz);
    return count;
}


NLint packet_backpack_new_begin_decode(int *num,
				       const NLbyte *buf, const NLint sz)
{
    NLint count = 1;
    int n;
    assert(num);
    assert(buf);

    readByte(buf, count, n); *num = n;

    assert(count <= sz);
    return count;
}


NLint packet_backpack_new_encode(const struct packet_backpack_new *packet,
				 NLbyte *buf, const NLint sz, NLint count)
{
    int xx, yy;
    assert(packet);
    assert(buf);

    xx = packet->x + zero_x;
    yy = packet->y + zero_y;

    writeLong (buf, count, packet->time);
    writeBackpackID(buf, count, packet->id);
    writeShort(buf, count, xx);
    writeShort(buf, count, yy);
    writeFloat(buf, count, packet->vx);
    writeFloat(buf, count, packet->vy);

    assert(count <= sz);
    return count;
}


NLint packet_backpack_new_decode(struct packet_backpack_new *packet,
				 const NLbyte *buf, const NLint sz,
				 NLint count)
{
    int xx, yy;
    assert(packet);
    assert(buf);

    readLong (buf, count, packet->time);
    readBackpackID(buf, count, packet->id);
    readShort(buf, count, xx); packet->x = xx - zero_x;
    readShort(buf, count, yy); packet->y = yy - zero_y;
    readFloat(buf, count, packet->vx);
    readFloat(buf, count, packet->vy);

    assert(count <= sz);
    return count;
}

/*--------------------------------------------------------------*/

NLint packet_backpack_taken_encode(const struct packet_backpack_taken *packet,
				   const bool local,
				   NLbyte *buf, NLint size)
{
    NLint count = 0;
    char command;
    enum WEAPON_CLASS w;

    if (local)
	command = COMMAND_BACKPACK_TAKE;
    else
	command = COMMAND_BACKPACK_TAKEN;

    writeByte (buf, count, command);
    writeLong (buf, count, packet->time);
    writeBackpackID(buf, count, packet->id);

    if (!local)
	goto end;

    /* Skip blaster. */
    for (w = WEAPON_FIRST; w < NUM_WEAPON_CLASSES; w++) {
	if (w == WEAPON_BLASTER)
	    continue;

	writeShort(buf, count, packet->ammo[w]);
    }

 end:
    assert(count <= size);
    return count;
}


void packet_backpack_taken_decode(struct packet_backpack_taken *packet,
				  const bool local,
				  const NLbyte *buf, const NLint size)
{
    NLint count = 1;
    enum WEAPON_CLASS w;

    readLong (buf, count, packet->time);
    readBackpackID(buf, count, packet->id);

    if (!local)
	goto end;

    packet->ammo[WEAPON_BLASTER] = 0;
    for (w = WEAPON_FIRST; w < NUM_WEAPON_CLASSES; w++) {
	if (w == WEAPON_BLASTER)
	    continue;

	readShort(buf, count, packet->ammo[w]);
    }

 end:
    assert(count == size);
}
