/* widget.c,
 */

#include <alleggl.h>
#include <allegro.h>
#include <assert.h>
#include <nl.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>

#include "chat.h"
#include "client.h"
#include "game-state.h"
#include "maxmin.h"
#include "menu/menu-runner.h"
#include "menu/widget.h"
#include "player.h"
#include "server.h"
#include "strlcpy.h"
#include "sv-commands.h"
#include "sv-internal.h"

/*--------------------------------------------------------------*/

/* d_aalg_push_proc from AGUP. */
int d_aalg_push_proc(int msg, DIALOG *d, int c)
{
   int ret = D_O_K;

   d->flags |= D_EXIT;

   ret |= d_button_proc(msg, d, c);

   if (ret & D_CLOSE) {
      ret &= ~D_CLOSE;
      ret |= D_REDRAWME;

      if (d->dp3)
	  ret |= ((int (*)(DIALOG *))d->dp3)(d);
   }

   return ret;
}

/*--------------------------------------------------------------*/

/* d_algl_hack_proc:
 *
 * Some widgets block when you hold down a mouse button, so AllegroGL
 * doesn't get a chance to render to screen.  Fix it.
 */
int d_algl_hack_proc(int msg, DIALOG *_1, int c)
{
    (void)_1;

    if ((msg == MSG_IDLE) && (c == 0) && (gui_mouse_b())) {
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	broadcast_dialog_message(MSG_DRAW, 0);
	algl_draw_mouse();
	allegro_gl_flip();
    }

    return D_O_K;
}

/*--------------------------------------------------------------*/

static void d_chat_proc_draw(const DIALOG *d);


/* d_chat_proc:
 *
 * dp: dialog to redirect keyboard focus to.
 */
int d_chat_proc(int msg, DIALOG *d, int c)
{
    DIALOG *redirect = d->dp;
    assert(redirect);

    switch (msg) {
      case MSG_DRAW:
	  d_box_proc(msg, d, c);
	  d_chat_proc_draw(d);
	  return D_O_K;
      case MSG_GOTFOCUS:
	  d->flags &=~D_GOTFOCUS;
	  redirect->flags |= D_GOTFOCUS;
	  return D_O_K;
      case MSG_LOSTFOCUS:
	  redirect->flags &=~D_GOTFOCUS;
	  return D_O_K;
      default:
	  return object_message(redirect, msg, c);
    }
}

/*--------------------------------------------------------------*/

static void d_chat_proc_draw(const DIALOG *d)
{
    const chat_t *message;
    int X1, Y1, X2, Y2;
    int i = 0;
    int y = d->y+8;

    get_clip_rect(screen, &X1, &Y1, &X2, &Y2);
    set_clip_rect(screen, d->x+1, d->y+1, d->x+d->w-2, d->y+d->h-2);

    while ((i < 32) && (message = chat_get(i))) {
	textout_ex(screen, font, message->str, d->x+8, y,
		   chat_colour_from_class(message->class), d->bg);

	y += 10;
	i++;
    }

    set_clip_rect(screen, X1, Y1, X2, Y2);
}

/*--------------------------------------------------------------*/

typedef struct {
    char name[512];		/* Same as al_fflk. */
} d_lister_proc_element_t;

static void d_lister_proc_start(DIALOG *d);
static void d_lister_proc_end(DIALOG *d);
static void d_lister_proc_draw(DIALOG *d);
static int d_lister_proc_char(DIALOG *d, const int c);
static int d_lister_proc_click(DIALOG *d);
static void d_lister_proc_clamp(DIALOG *d);
static int d_lister_proc_num_files(const DIALOG *d);


/* d_lister_proc:
 *
 * Map file lister that goes over the top of the stats frame.  We
 * can't just use file_select_ex because it is blocking and doesn't
 * get rendered.
 *
 * d1:  Selected item.
 * d2:  Scrolled lines.
 * dp1: File list, managed by the widget itself.
 * dp2: Game state.
 * dp3: Callback function.  int func(const char *)
 *
 * MSG_USER: Use this to load (c=0) or free (otherwise) the file list.
 */
int d_lister_proc(int msg, DIALOG *d, int c)
{
    switch (msg) {
      case MSG_USER:
	  if (c == 0)
	      d_lister_proc_start(d);
	  else
	      d_lister_proc_end(d);
	  break;
      case MSG_WANTFOCUS:
	  return D_WANTFOCUS;
      case MSG_CHAR:
	  return d_lister_proc_char(d, c);
      case MSG_CLICK:
	  if (c == 1) {		/* lmb */
	      d->d1 = d_lister_proc_click(d);
	  }
	  return D_O_K;
      case MSG_DCLICK:
	  /* 'c' is NOT the mouse button clicked. */
	  if (gui_mouse_b() == 1) {
	      d->d1 = d_lister_proc_click(d);
	      d_lister_proc_char(d, KEY_ENTER<<8);
	  }
	  return D_O_K;
      case MSG_WHEEL:
	  d->d1 -= c;
	  d_lister_proc_clamp(d);
	  break;
      case MSG_DRAW:
	  d_box_proc(msg, d, c);
	  d_lister_proc_draw(d);
	  break;
    }

    return D_O_K;
}


static int d_lister_proc_sorter(const void *aa, const void *bb)
{
    const d_lister_proc_element_t *a = aa;
    const d_lister_proc_element_t *b = bb;

    return strcmp(a->name, b->name);
}


static void d_lister_proc_start(DIALOG *d)
{
#define MAP_PATTERN	"*.map"
#define MAP_ATTRIB	(FA_RDONLY|FA_ARCH)

    d_lister_proc_element_t *filelist = NULL;
    struct al_ffblk info;
    int num = 0;

    if (d->dp) {
	fprintf(stderr, "[Menu.Widget] d_lister_proc->dp is non-NULL\n");
	return;
    }

    if (al_findfirst(MAP_PATTERN, &info, MAP_ATTRIB) != 0)
	return;

    do {
	void *p;

	/* +1 for new element, +1 for sentinel. */
	p = realloc(filelist, sizeof(*filelist) * (num+1+1));
	assert(p);

	filelist = p;
	strlcpy(filelist[num].name, info.name, sizeof(filelist[num].name));
	num++;
    } while (al_findnext(&info) == 0);

    al_findclose(&info);

    filelist[num].name[0] = '\0';

    /* Sort the file list, but not the sentinel. */
    qsort(filelist, num, sizeof(*filelist), d_lister_proc_sorter);

    d->dp = filelist;

#undef MAP_ATTRIB
#undef MAP_PATTERN
}


static void d_lister_proc_end(DIALOG *d)
{
    if (d->dp) {
	free(d->dp);
	d->dp = NULL;
    }
}


static int d_lister_proc_char(DIALOG *d, const int c)
{
    int sc = (c >> 8);

    if (sc == KEY_UP) {
	d->d1--;
    }
    else if (sc == KEY_DOWN) {
	d->d1++;
    }
    else if (sc == KEY_PGUP) {
	d->d1 -= 10;
    }
    else if (sc == KEY_PGDN) {
	d->d1 += 10;
    }
    else if (sc == KEY_HOME) {
	d->d1 = 0;
    }
    else if (sc == KEY_END) {
	d->d1 = d_lister_proc_num_files(d)-1;
    }
    else if (sc == KEY_ENTER) {
	int (*proc)(DIALOG *, const char *) = d->dp3;
	int ret = D_O_K;
	
	if (proc) {
	    d_lister_proc_element_t *filelist = d->dp;

	    ret = proc(d, filelist[d->d1].name);
	}

	return ret;
    }
    else {
	return D_O_K;
    }

    d_lister_proc_clamp(d);
    return D_USED_CHAR;
}


static int d_lister_proc_click(DIALOG *d)
{
    int num = d_lister_proc_num_files(d);
    int my = gui_mouse_y() - d->y - 8;

    return MID(0, my/12 + d->d2, num-1);
}


static void d_lister_proc_draw(DIALOG *d)
{
    d_lister_proc_element_t *filelist = d->dp;
    int X1, Y1, X2, Y2;
    int i, y;

    if (!filelist)
	return;

    get_clip_rect(screen, &X1, &Y1, &X2, &Y2);
    set_clip_rect(screen, d->x+1, d->y+1, d->x+d->w-2, d->y+d->h-2);

    y = d->y+8 - (d->d2)*12;
    for (i = 0; filelist[i].name[0] != '\0'; i++) {
	/* Selected */
	if (i == d->d1) {
	    rectfill(screen, d->x+5, y-1, d->x+d->w-5, y+9,
		     makecol(0x00, 0x00, 0x40));
	}

	textout_ex(screen, font, filelist[i].name, d->x+8, y, d->fg, -1);

	y += 12;
	if (y >= d->y+d->h)
	    break;
    }

    set_clip_rect(screen, X1, Y1, X2, Y2);
}


static void d_lister_proc_clamp(DIALOG *d)
{
    int num = d_lister_proc_num_files(d);
    int rows_per_screen = (d->h-16) / 12;

    d->d1 = MID(0, d->d1, num-1);
    if (d->d1 < d->d2+2) {
	d->d2 = maxi(0, d->d1-2);
    }
    if (d->d1 > d->d2+rows_per_screen-2) {
	d->d2 = mini(d->d1-rows_per_screen+2, num-1 - rows_per_screen);
    }
}


static int d_lister_proc_num_files(const DIALOG *d)
{
    d_lister_proc_element_t *filelist = d->dp;
    int i = 0;

    if (!filelist)
	return 0;

    for (i = 0; filelist[i].name[0] != '\0'; i++);

    return i;
}

/*--------------------------------------------------------------*/

static bool d_prompt_proc_char(DIALOG *d, int c);
static bool d_prompt_proc_char_forward(DIALOG *d);


/* d_prompt_proc:
 *
 * The same as d_edit_proc, but pressing ENTER sends and clears the
 * buffer.
 *
 * d1:  Buffer length.
 * d2:  Cursor position.
 * dp1: Buffer.
 * dp2: Game state.
 */
int d_prompt_proc(int msg, DIALOG *d, int c)
{
    if (msg == MSG_START) {
	char *str = d->dp;
	str[0] = '\0';
	d->d2 = 0;
	return D_O_K;
    }
    else if ((msg == MSG_CHAR) && 
	     (d_prompt_proc_char(d, c))) {
	return D_USED_CHAR;
    }
    else {
	return d_edit_proc(msg, d, c);
    }
}


static bool d_prompt_proc_char(DIALOG *d, int c)
{
#define CTRL(ac)		((ac) - 'a' + 1)
#define META(sc)		((sc) << 8)
#define SIMULATE_KEY(sc)	(d_edit_proc(MSG_CHAR, d, (sc)<<8))

    char *str = d->dp;
    int ac = (c & 0xff);
    int sc = (c >> 8);
    assert(str);

    if (sc == KEY_ENTER) {
	if (str[0] == '/') {
	    sv_command_ss_func_t func;
	    game_state_t *state = d->dp2;

	    func = sv_command_fe(str+1);
	    if (func) {
		func(str+1, client_sock(), state);
	    }
	    else {
		chat_add(0, CHAT_FRONTEND_ERROR,
			 "Unrecognised command: %s", str);
	    }
	}
	else {
	    client_send_chat(str);
	}

	goto clear_and_return;
    }

    if (ac == CTRL('a'))
	return SIMULATE_KEY(KEY_HOME);

    if (ac == CTRL('b')) {
	if (d->d2 > 0)
	    d->d2--;
	return true;
    }

    if (ac == CTRL('d'))
	return SIMULATE_KEY(KEY_DEL);

    if (c == META(KEY_D)) {
	while (str[d->d2] == ' ')
	    SIMULATE_KEY(KEY_DEL);

	while ((str[d->d2] != '\0') &&
	       (str[d->d2] != ' ')) {
	    SIMULATE_KEY(KEY_DEL);
	}

	return true;
    }

    if (ac == CTRL('e'))
	return SIMULATE_KEY(KEY_END);

    if (ac == CTRL('f'))
	return d_prompt_proc_char_forward(d);

    if (ac == CTRL('k')) {
	str[d->d2] = '\0';
	return true;
    }

    if (ac == CTRL('t')) {
	char t;
	int i;

	if (str[d->d2] == '\0') {
	    if (d->d2 <= 1)
		return true;
	    i = d->d2-1;
	}
	else {
	    if (d->d2 <= 0)
		return true;
	    i = d->d2;
	}

	t = str[i];
	str[i] = str[i-1];
	str[i-1] = t;
	return d_prompt_proc_char_forward(d);
    }

    if (ac == CTRL('u'))
	goto clear_and_return;

    return false;

 clear_and_return:

    str[0] = '\0';
    d->d2 = 0;
    return true;

#undef SIMULATE_KEY
#undef META
#undef CTRL
}


static bool d_prompt_proc_char_forward(DIALOG *d)
{
    char *str = d->dp;

    if (!d->dp)
	return false;

    if ((d->d2+1 <= d->d1) &&
	(str[d->d2] != '\0')) {
	d->d2++;
    }

    return true;
}

/*--------------------------------------------------------------*/

static const DIALOG *d_stats_proc_find_tab(const DIALOG *dialog);
static int d_stats_proc_char(DIALOG *d, const int c);
static void d_stats_proc_lmb_player_info(DIALOG *d, const int x, const int y);
static void d_stats_proc_draw_server_info(const DIALOG *d);
static void d_stats_proc_draw_player_info(const DIALOG *d);
static void d_stats_proc_draw_avatar(const unsigned char face_num,
				     const int x, const int y);
static void d_stats_proc_clamp(DIALOG *d);
static void d_stats_client_proc_open(DIALOG *d, const int idx);


/* d_stats_proc:
 *
 * d2:  Scrolled amount (pixels)
 * dp1: Pointer to the entire dialog.
 * dp2: Game state.
 *
 * D_USER: client-server lobby mode (extra buttons)
 */
int d_stats_proc(int msg, DIALOG *d, int c)
{
    const DIALOG *tab;
    bool server;
    assert(d->dp);

    tab = d_stats_proc_find_tab((DIALOG *)(d->dp));
    server = d->flags & D_USER;

    switch (msg) {
      case MSG_WANTFOCUS:
	  return D_WANTFOCUS;

      case MSG_CHAR:
	  return d_stats_proc_char(d, c);

      case MSG_LRELEASE:
	  if (!tab)
	      break;
	  else if (tab->d2 == WIDGET_STATS_PLAYER_INFO)
	      d_stats_proc_lmb_player_info(d, gui_mouse_x(), gui_mouse_y());
	  break;

      case MSG_WHEEL:
	  d->d2 -= c*10;
	  d_stats_proc_clamp(d);
	  break;

      case MSG_DRAW:
	  d_box_proc(msg, d, c);
	  if (!tab)
	      break;
	  else if (tab->d2 == WIDGET_STATS_SERVER_INFO)
	      d_stats_proc_draw_server_info(d);
	  else if (tab->d2 == WIDGET_STATS_PLAYER_INFO)
	      d_stats_proc_draw_player_info(d);

	  break;
    }

    return D_O_K;
}


static const DIALOG *d_stats_proc_find_tab(const DIALOG *dialog)
{
    int i;

    for (i = 0; dialog[i].proc; i++) {
	if ((dialog[i].proc == d_tab_proc) &&
	    (dialog[i].flags & D_SELECTED))
	    return &(dialog[i]);
    }

    return NULL;
}


static int d_stats_proc_char(DIALOG *d, const int c)
{
    int sc = c >> 8;

    if (sc == KEY_UP) {
	d->d2 -= 5;
    }
    else if (sc == KEY_DOWN) {
	d->d2 += 5;
    }
    else if (sc == KEY_PGUP) {
	d->d2 -= 50;
    }
    else if (sc == KEY_PGDN) {
	d->d2 += 50;
    }
    else if (sc == KEY_HOME) {
	d->d2 = 0;
    }
    else if (sc == KEY_END) {
	d->d2 = 50*MAX_CLIENTS;
    }
    else {
	return D_O_K;
    }

    d_stats_proc_clamp(d);
    return D_USED_CHAR;
}


static void d_stats_proc_lmb_player_info(DIALOG *d, const int x, const int y)
{
    game_state_t *state = d->dp2;
    bool server = d->flags & D_USER;
    int j = ((y - (d->y - d->d2))/50);
    int mx = x - d->x;
    int my = y - (d->y - d->d2) - j*50;
    int i;

    for (i = 0; i < MAX_CLIENTS; i++) {
	if (client_data[i].clid != 0) {
	    j--;
	    if (j < 0)
		break;
	}
    }

    if (i == MAX_CLIENTS) {
	/* Add robot button */
	if ((j == 0) &&
	    (server_may_add_robot())) {
	    char buf[MAX_PACKET_SIZE];

	    snprintf(buf, sizeof(buf), "robots %d", num_robots+1);
	    client_send_command(buf);
	}

	return;
    }

    /* We clicked on client[i]'s tab. */

    if ((server) &&
	(d->w-39 < mx) && (mx < d->w-8) && (9 < my) && (my < 42)) {
	/* Kick button */
	server_disconnect(&client_data[i], DISCONNECT_JERK, state);
    }
    else {
	d_stats_client_proc_open(d, i);
    }
}


static void d_stats_proc_draw_server_info(const DIALOG *d)
{
    const game_state_t *state = d->dp2;
    int x = d->x+8;
    int y = d->y+8;

    if (!state)
	return;

    textout_ex(screen, font, "Game mode:", x, y, d->fg, d->bg);
    textout_ex(screen, font, 
	       game_mode_str_from_mode(state->game_mode),
	       x+100, y, d->fg, d->bg);

    textout_ex(screen, font, "Editor:", x, y+15, d->fg, d->bg);
    textout_ex(screen, font,
	       state->editor_enabled ? "Enabled" : "Disabled",
	       x+100, y+15, d->fg, d->bg);
}


static void d_stats_proc_draw_player_info(const DIALOG *d)
{
    int X1, Y1, X2, Y2;
    int c;
    int y = d->y - d->d2;
    bool server = d->flags & D_USER;

    get_clip_rect(screen, &X1, &Y1, &X2, &Y2);
    set_clip_rect(screen, d->x+1, d->y+1, d->x+d->w-2, d->y+d->h-2);

    for (c = 0; c < MAX_CLIENTS; c++) {
	const client_data_t *data;

	if (y >= d->y+d->h-1)
	    break;

	data = &client_data[c];
	if (data->clid == 0)
	    continue;

	d_stats_proc_draw_avatar(data->face_num, d->x, y);

	textout_ex(screen, font, data->name, d->x+55, y+8, d->fg, d->bg);
	textprintf_ex(screen, font, d->x+65, y+25, d->fg, d->bg,
		      "Score: %d (%d)",
		      data->sc.total_score, data->sc.session_score);

	if ((server) && (data->type != CLIENT_ADMIN)) {
	    /* Boot */
	    masked_blit(menu_data, screen, 0, 0, d->x+d->w-39, y+9, 32, 32);
	}

	/* Separator */
	hline(screen, d->x, y+50, d->x+d->w-1, d->fg);
	y += 50;
    }

    /* One extra for add robot. */
    if (server && server_may_add_robot()) {
	textout_centre_ex(screen, font, "Add Robot",
			  d->x + d->w/2, y+24, d->fg, d->bg);

	/* Separator */
	hline(screen, d->x, y+50, d->x+d->w-1, d->fg);
	y += 50;
    }

    set_clip_rect(screen, X1, Y1, X2, Y2);
}


static void d_stats_proc_draw_avatar(const unsigned char face_num,
				     const int x, const int y)
{
    assert(face_num < NUM_FACES);

    /* Background. */
    rectfill(screen, x+4, y+4, x+47, y+47, makecol(0xc0, 0xc0, 0xc0));
    rect    (screen, x+3, y+3, x+47, y+47, makecol(0xff, 0xff, 0xff));
    rect    (screen, x+7, y+7, x+43, y+43, makecol(0xff, 0xff, 0xff));
    rectfill(screen, x+8, y+8, x+43, y+43, makecol(0xa0, 0x80, 0xff));

    /* Face. */
    player_draw_avatar(face_num, x+46/2 - 10, y+46/2 - 8);
}


static void d_stats_proc_clamp(DIALOG *d)
{
    bool server = d->flags & D_USER;
    int i, yy;

    if (d->d2 < 0) {
	d->d2 = 0;
    }
    else {
	for (i = 0, yy = -d->h; i < MAX_CLIENTS; i++) {
	    if (client_data[i].clid != 0)
		yy += 50;
	}

	if (server && server_may_add_robot())
	    yy += 50;

	if (d->d2 > yy)
	    d->d2 = maxi(0, yy);
    }
}

/*--------------------------------------------------------------*/

static int d_stats_client_proc_lmb(DIALOG *d);
static int d_stats_client_proc_close(DIALOG *d);
static void d_stats_client_proc_draw(DIALOG *d);


/* d_stats_client_proc:
 *
 * Draw the stats of a single client.
 *
 * d1:  client id (NOT the index in client_data)
 * dp1: Pointer to the entire dialog.
 */
int d_stats_client_proc(int msg, DIALOG *d, int c)
{
    switch (msg) {
      case MSG_WANTFOCUS:
	  return D_WANTFOCUS;

      case MSG_LRELEASE:
	  return d_stats_client_proc_lmb(d);

      case MSG_RRELEASE:
	  return d_stats_client_proc_close(d);

      case MSG_DRAW:
	  d_box_proc(msg, d, c);
	  d_stats_client_proc_draw(d);
	  break;
    }

    return D_O_K;
}


static void d_stats_client_proc_select(DIALOG *d, const int dx)
{
    int idx;
    assert(dx != 0);

    idx = sv_intern_client_from_client_id(d->d1);
    if (idx < 0)
	return;

    for (idx += dx; (0 <= idx) && (idx <= MAX_CLIENTS); idx += dx) {
	if (client_data[idx].clid != 0) {
	    d->d1 = client_data[idx].clid;
	    return;
	}
    }
}


static int d_stats_client_proc_lmb(DIALOG *d)
{
    int mx = gui_mouse_x() - d->x;
    int my = gui_mouse_y() - d->y;

    /* Prev/Next button */
    if ((8 <= my) && (my <= 8+31)) {
	if ((d->w-72 <= mx) && (mx <= d->w-72+31)) {
	    d_stats_client_proc_select(d, -1);
	    return D_O_K;
	}

	if ((d->w-40 <= mx) && (mx <= d->w-40+31)) {
	    d_stats_client_proc_select(d, +1);
	    return D_O_K;
	}
    }

    /* Close button */
    if ((d->w/2 - 50 <= mx) && (mx <= d->w/2 + 50) &&
	(d->h - 58 <= my) && (my <= d->h-18))
	return d_stats_client_proc_close(d);

    return D_O_K;
}


static void d_stats_client_proc_draw(DIALOG *d)
{
    client_data_t *data;
    int idx;
    int x, y;
    int rank;
    char rank_c;

    idx = sv_intern_client_from_client_id(d->d1);
    if (idx < 0)
	return;

    data = &client_data[idx];
    
    d_stats_proc_draw_avatar(data->face_num, d->x+8, d->y+8);
    textout_ex(screen, font, data->name, d->x+80, d->y+20, d->fg, d->bg);

    rank = score_get_rank(idx, &rank_c);
    textprintf_ex(screen, font, d->x+80, d->y+40, d->fg, d->bg,
		  "Rank: %c%d", rank_c, rank);

    /* Total Stats */
    y = d->y + 80;
    textprintf_centre_ex(screen, font, d->x+d->w/2, y, d->fg, d->bg,
			 "Score");

    textprintf_ex(screen, font, d->x+20, y+20, d->fg, d->bg,
		  "Score: %d", data->sc.total_score);
    textprintf_ex(screen, font, d->x+20, y+35, d->fg, d->bg,
		  "Kills: %d", data->sc.total_kills);
    textprintf_ex(screen, font, d->x+20, y+50, d->fg, d->bg,
		  "Deaths: %d", data->sc.total_deaths);

    if (data->sc.total_deaths == 0) {
	textprintf_ex(screen, font, d->x+20, y+70, d->fg, d->bg,
		      "Kills per life: N/A");
    }
    else {
	textprintf_ex(screen, font, d->x+20, y+70, d->fg, d->bg,
		      "Kills per life: %5.3f",
		      (double)data->sc.total_score / data->sc.total_deaths);
    }

    /* Session Stats */
    y += 100;
    textprintf_centre_ex(screen, font, d->x+d->w/2, y, d->fg, d->bg,
			 "Last Session");

    textprintf_ex(screen, font, d->x+20, y+20, d->fg, d->bg,
		  "Score: %d", data->sc.session_score);
    textprintf_ex(screen, font, d->x+20, y+35, d->fg, d->bg,
		  "Kills: %d", data->sc.session_kills);
    textprintf_ex(screen, font, d->x+20, y+50, d->fg, d->bg,
		  "Deaths: %d", data->sc.session_deaths);

    /* Prev/Next button */
    x = d->x + d->w - 72;
    y = d->y + 8;
    blit(menu_data, screen, 33, 0, x, y, 32, 32);
    blit(menu_data, screen, 65, 0, x+32, y, 32, 32);

    /* Close button */
    x = d->x + d->w/2;
    y = d->y + d->h - 58;
    rect(screen, x-50, y, x+50, y+40, d->fg);
    textout_centre_ex(screen, font, "Close", x, y+17, d->fg, -1);
}


/* d_stats_client_proc_open:
 *
 * 'd' is actually a d_stats_proc.  'idx' is the client we want to
 * view in closer detail.
 */
static void d_stats_client_proc_open(DIALOG *d, const int idx)
{
    DIALOG *dd = d->dp;

    for (; dd->proc; dd++) {
	if ((dd->proc == d_tab_proc) ||
	    (dd->proc == d_stats_proc)) {
	    dd->flags |= (D_DISABLED | D_HIDDEN);
	}
	else if (dd->proc == d_stats_client_proc) {
	    dd->d1 = client_data[idx].clid;
	    dd->flags &=~(D_DISABLED | D_HIDDEN);
	}
    }
}


static int d_stats_client_proc_close(DIALOG *d)
{
    DIALOG *dd = d->dp;

    for (; dd->proc; dd++) {
	if ((dd->proc == d_tab_proc) ||
	    (dd->proc == d_stats_proc)) {
	    dd->flags &=~(D_DISABLED | D_HIDDEN);
	}
	else if (dd->proc == d_stats_client_proc) {
	    dd->flags |= (D_DISABLED | D_HIDDEN);
	}
    }

    d->flags |= (D_DISABLED | D_HIDDEN);
    return D_O_K;
}

/*--------------------------------------------------------------*/

/* d_tab_proc:
 *
 * d1:  radio button group number.
 * d2:  tab contents.
 * dp1: tab text.
 */
int d_tab_proc(int msg, DIALOG *d, int c)
{
    if (msg == MSG_DRAW) {
	return d_button_proc(msg, d, c);
    }
    else {
	return d_radio_proc(msg, d, c);
    }
}

/*--------------------------------------------------------------*/

/* d_toggle_proc:
 *
 * d_button_proc with a callback when widget toggles selection.
 *
 * dp1: Button text.
 * dp3: Callback.
 */
int d_toggle_proc(int msg, DIALOG *d, int c)
{
    int oldsel = d->flags & D_SELECTED;
    int ret = d_button_proc(msg, d, c);

    if ((d->flags & D_SELECTED) != oldsel) {
	if (d->dp3)
	    ret |= ((int (*)(DIALOG *))d->dp3)(d);
    }

    return ret;
}

/*--------------------------------------------------------------*/

/* algl_do_dialog_nested:
 *
 * The same as algl_do_dialog, but don't call set/unset- allegro_mode
 * because we already are in Allegro mode.
 */
int algl_do_dialog_nested(DIALOG *dialog, int focus_obj)
{
    DIALOG_PLAYER *play;

    play = init_dialog(dialog, focus_obj);

    while (update_dialog(play)) {
	/* Redraw the GUI every frame */
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	broadcast_dialog_message(MSG_DRAW, 0);

	/* Draw the mouse cursor */
	algl_draw_mouse();

	/* Flip buffers */
	allegro_gl_flip();
    }

    return shutdown_dialog(play);
}
