/* main.c,
 *
 * Red Pixel, a platform deathmatch game.
 */

#include <alleggl.h>
#include <allegro.h>
#include <assert.h>
#include <getopt.h>
#include <stdio.h>
#include <string.h>

#include "chat.h"
#include "client.h"
#include "common.h"
#include "config.h"
#include "coro.h"
#include "game.h"
#include "gettime.h"
#include "lobby.h"
#include "menu/lobby-client.h"
#include "menu/lobby-server.h"
#include "menu/menu-runner.h"
#include "network.h"
#include "particle.h"
#include "server.h"
#include "sound.h"
#ifdef WITH_CURSES_INTERFACE
# include "textface/sv-textface.h"
#endif


#define streq(a,b)	(strcmp(a, b) == 0)

/*--------------------------------------------------------------*/

enum MAIN_EXIT_STATUS {
    EXIT_SUCCESSFUL,
    EXIT_VIDEO_ERROR,
    EXIT_SERVER_ERROR, 
    EXIT_CLIENT_ERROR,
    EXIT_MENU_ERROR
};

static bool main_init_window(const bool windowed,
			     const int screen_width, const int screen_height);
static enum MAIN_EXIT_STATUS main_begin_server(const NLushort port);
static enum MAIN_EXIT_STATUS main_begin_client(const char *host);
static enum MAIN_EXIT_STATUS main_begin_client_server(const NLushort port);
static enum MAIN_EXIT_STATUS main_begin_menu(void);


static enum MAIN_EXIT_STATUS
main_body(const bool windowed,
	  const int screen_width, const int screen_height,
	  const char *host, const NLushort port)
{
    bool text_only = (client_server_mode == I_AM_SERVER_ONLY);

    if (!text_only) {
	if (!main_init_window(windowed, screen_width, screen_height))
	    return EXIT_VIDEO_ERROR;
    }

    switch (client_server_mode) {
      case I_AM_SERVER_ONLY:
	  return main_begin_server(port);
      case I_AM_CLIENT_ONLY:
	  return main_begin_client(host);
      case I_AM_CLIENT_SERVER:
	  return main_begin_client_server(port);
      case I_AM_MENU:
	  return main_begin_menu();
      default:
	  assert(false);
	  return -1;
    }
}


#ifdef WITH_CURSES_INTERFACE
static void *main_begin_server2(void *arg)
{
    lobby_server_loop(arg);
    return NULL;
}


static enum MAIN_EXIT_STATUS main_begin_server(const NLushort port)
{
    NLthreadID thread_id;

    if (!server_init(port))
	return EXIT_SERVER_ERROR;

    set_color_conversion(COLORCONV_NONE);
    game_init(game_mode);
    thread_id = nlThreadCreate(main_begin_server2, NULL, NL_TRUE);
    textface_loop(port);
    nlThreadJoin(thread_id, NULL);

    server_shutdown();
    game_shutdown();
    return EXIT_SUCCESS;
}
#else
static enum MAIN_EXIT_STATUS main_begin_server(const NLushort _)
{
    (void)_;

    fprintf(stdout,
	    "This version of Red Pixel was not compiled\n"
	    "with the curses interface.\n");

    return EXIT_SUCCESS;
}
#endif


static enum MAIN_EXIT_STATUS main_begin_client(const char *host)
{
    char *name = getenv("USER");

    game_init(game_mode);
    if (!client_init(name, host))
	return EXIT_CLIENT_ERROR;

    menu_init();
    lobby_loop(NULL, &clobby_fe);
    menu_shutdown();
    client_shutdown();
    game_shutdown();
    return EXIT_SUCCESS;
}


static enum MAIN_EXIT_STATUS main_begin_client_server(const NLushort port)
{
    enum MAIN_EXIT_STATUS ret;
    char localhost_port[32];
    coro_t tid_server;
    coro_t tid_client;
    char *name = getenv("USER");

    if (!server_init(port))
	return EXIT_SERVER_ERROR;

    game_init(game_mode);
    tid_client = coro_self();
    tid_server = coro_spawn(lobby_server_loop, tid_client);

    if (tid_server == NULL) {
	fprintf(stderr, "[Main] Could not spawn server.\n");
	server_shutdown();
	game_shutdown();
	return EXIT_SERVER_ERROR;
    }

    snprintf(localhost_port, sizeof(localhost_port), "localhost:%d", port);
    if (client_init(name, localhost_port)) {
	menu_init();
	lobby_loop(tid_server, &slobby_fe);
	menu_shutdown();
	client_shutdown();

	ret = EXIT_SUCCESS;
    }
    else {
	ret = EXIT_CLIENT_ERROR;
    }

    coro_join(tid_server);
    tid_server = NULL;
    server_shutdown();

    game_shutdown();
    return ret;
}


static enum MAIN_EXIT_STATUS main_begin_menu(void)
{
    enum MAIN_EXIT_STATUS ret;

    game_init(game_mode);

    if (menu_loop()) {
	ret = EXIT_SUCCESS;
    }
    else {
	ret = EXIT_MENU_ERROR;
    }

    game_shutdown();
    return ret;
}

/*--------------------------------------------------------------*/

static bool main_init(int *screen_width, int *screen_height)
{
    srand(time(0));

    if (allegro_init() != 0) {
	fprintf(stderr, "Error initialising Allegro: %s\n", allegro_error);
	return false;
    }

    set_window_title("Red Pixel III");

    install_allegro_gl();
    config_set_file();
    config_load(screen_width, screen_height);

    gettime_start();

    return true;
}


static bool main_init_window(const bool windowed,
			     const int screen_width, const int screen_height)
{
    allegro_gl_clear_settings();
    allegro_gl_set(AGL_COLOR_DEPTH, 16);
    allegro_gl_set(AGL_DOUBLEBUFFER, 1);
    allegro_gl_set(AGL_WINDOWED, windowed);
    allegro_gl_set(AGL_RENDERMETHOD, 1);
    allegro_gl_set(AGL_SUGGEST,
                   AGL_COLOR_DEPTH | AGL_DOUBLEBUFFER |
                   AGL_WINDOWED | AGL_RENDERMETHOD);

    set_color_depth(16);
    if (set_gfx_mode(GFX_OPENGL, screen_width, screen_height, 0, 0) != 0) {
        fprintf(stderr, "Error setting video mode: %s\n", allegro_error);
        return false;
    }

    install_timer();
    install_mouse();
    install_keyboard();

    chat_init_colours();
    sound_init();

    al_font = 
	allegro_gl_convert_allegro_font(font,
					AGL_FONT_TYPE_TEXTURED, 0.75);
    assert(al_font);

    /* OpenGL stuff. */
    glEnable(GL_TEXTURE_2D);
    glPolygonMode(GL_FRONT, GL_FILL);

    glEnable(GL_ALPHA_TEST);
    glAlphaFunc(GL_GREATER, 0.5);

    glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE);
    glEnable(GL_COLOR_MATERIAL);
    glEnable(GL_LIGHT0);
    glEnable(GL_LIGHT1);
    glEnable(GL_CULL_FACE);

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(0.0, 640.0, 0.0, 480.0, -1.0, 1.0);

    glMatrixMode(GL_MODELVIEW);

    return true;
}


static void main_shutdown(void)
{
    if (al_font) {
	allegro_gl_destroy_font(al_font);
	al_font = NULL;
    }
}

/*--------------------------------------------------------------*/

static int port_from_optarg(const char *arg)
{
    if (arg) {
	int port = atoi(optarg);

	if (0 < port)
	    return port;
    }

    return 25000;
}


static enum GAME_MODE game_mode_from_string(const char *str)
{
    if (streq(str, "classic")) {
	return GAME_MODE_CLASSIC;
    }

    if (streq(str, "silhouettes")) {
	return GAME_MODE_SILHOUETTES;
    }

    fprintf(stdout, "Unknown game mode: %s\nSet to classic deathmatch\n", str);
    return GAME_MODE_CLASSIC;
}


int main(int argc, char *argv[])
{
    char *host = "localhost";
    int port = 25000;
    int screen_width = 640, screen_height = 480;
    int windowed = true;
    int disable_reverse_subtract = false;
    char c;

    const struct option long_opts[] = {
	{ "server",	optional_argument,	0, 's' },
	{ "connect",	required_argument,	0, 'c' },
	{ "mode",	required_argument,	0, 'm' },
	{ "fullscreen",	no_argument,		0, 'f' },
	{ "windowed",	no_argument,	&windowed, true },
	{ "width",	required_argument,	0, 'w' },
	{ "height",	required_argument,	0, 'h' },
	{ "particles",	required_argument,	0, 'P' },
	{ "disable-reverse-subtract",
			no_argument, &disable_reverse_subtract, true }
    };

    if (!main_init(&screen_width, &screen_height))
	return -1;

    /* Command line options. */
    while ((c = getopt_long(argc, argv, "s::S::c:m:fw:h:P:", long_opts, NULL))
	   != -1) {
	switch (c) {
	  case 's':
	      client_server_mode = I_AM_SERVER_ONLY;
	      port = port_from_optarg(optarg);
	      break;
	  case 'S':
	      client_server_mode = I_AM_CLIENT_SERVER;
	      port = port_from_optarg(optarg);
	      break;
	  case 'c':
	      client_server_mode = I_AM_CLIENT_ONLY;
	      host = optarg;
	      break;

	  case 'm':
	      game_mode = game_mode_from_string(optarg);
	      break;

	  case 'f':
	      windowed = false;
	      break;
	  case 'w':
	      screen_width = atoi(optarg);
	      break;
	  case 'h':
	      screen_height = atoi(optarg);
	      break;

	  case 'P':
	      particle_set_multiplier(atoi(optarg));
	      break;
	}
    }

    if (disable_reverse_subtract)
	gl_reverse_subtract = false;

    if ((client_server_mode == I_AM_CLIENT_SERVER) ||
	(client_server_mode == I_AM_MENU)) {
	if (!coro_init())
	    return -1;
    }

    if (!network_init())
	return -1;

    chat_init();
    main_body(windowed, screen_width, screen_height, host, port);
    sound_shutdown();
    chat_shutdown();
    network_shutdown();
    main_shutdown();

    return 0;
}
END_OF_MAIN()
