/* lobby.c,
 *
 * Let's all go to the lobby...
 */

#include <alleggl.h>
#include <allegro.h>
#include <assert.h>
#include <stdio.h>

#include "client.h"
#include "common.h"
#include "coro.h"
#include "game.h"
#include "lobby.h"
#include "map-save.h"
#include "network.h"
#include "packet-server.h"
#include "robot.h"
#include "server.h"
#include "sv-internal.h"


/*--------------------------------------------------------------*/

static void lobby_init(void);
static void lobby_shutdown(void);

/*--------------------------------------------------------------*/

static void lobby_server_make_players(game_state_t *state);


static void lobby_server_feed_clients(game_state_t *state)
{
    NLsocket client[MAX_CLIENTS];
    NLint i, n;
    assert(state);

    n = nlPollGroup(group_all_clients, NL_WRITE_STATUS, client, MAX_CLIENTS, 0);
    assert(n != NL_INVALID);

    map_load(state->next_map, state);

    for (i = 0; i < n; i++) {
	server_feed_map(client[i], state);
	server_feed_candela(client[i]);
	server_feed_starts(client[i]);
	server_feed_containers(client[i]);
	server_feed_gizmos(client[i]);
	server_feed_pickup(client[i]);
    }

    lobby_server_make_players(state);
}


static void lobby_server_make_players(game_state_t *state)
{
    unsigned int i, r;
    assert(state);

    for (i = 0, r = 0; i < MAX_CLIENTS; i++) {
	if (client_data[i].type == CLIENT_ROBOT) {
	    if (r < MAX_ROBOTS) {
		server_join_robot(&robot[r], &client_data[i], state);
		r++;
	    }
	}
	else if ((client_data[i].sock != NL_INVALID) &&
		 (client_data[i].type == CLIENT_GAMER)) {
	    server_join_player(&client_data[i], state);
	}
    }
}

/*--------------------------------------------------------------*/

static void lobby_server_update(game_state_t *state, const coro_t tid_client);
static void lobby_server_quit(game_state_t *state);


/* lobby_server_loop:
 *
 * When in client-server mode, tid_client is the thread of the client.
 * Otherwise, it is NULL.
 */
void __stdcall lobby_server_loop(void *tid_client)
{
    game_state_t state;

    game_state_init(&state, game_mode, true);

    while (state.context != SERVER_QUIT) {
	lobby_server_update(&state, tid_client);

	if (client_server_mode == I_AM_CLIENT_SERVER &&
	    state.context != SERVER_QUIT) {

	    if (!coro_yield(tid_client)) {
		fprintf(stderr, "[Server] Client thread died!\n");
		state.context = SERVER_QUIT;
	    }
	}
	else {
	    rest(0);
	}
    }

    lobby_server_quit(&state);

    if (client_server_mode == I_AM_CLIENT_SERVER)
	coro_yield(tid_client);
}


static void lobby_server_update(game_state_t *state, const coro_t tid_client)
{
    assert(state);

    server_maybe_new_connection();
    server_incoming(state);

    if (state->context == SERVER_IN_GAME) {
	game_start(state);
	lobby_server_feed_clients(state);
	game_loop(state, tid_client);
	server_end_session();
	game_stop(state);
    }
}


static void lobby_server_quit(game_state_t *state)
{
    NLbyte buf[MAX_PACKET_SIZE];
    NLint len;
    assert(state);

    len = packet_server_state_encode(state, buf, sizeof(buf));
    server_broadcast(buf, len);
}

/*--------------------------------------------------------------*/

static void lobby_client_update(game_state_t *state, const coro_t tid_server)
{
    assert(state);

    client_incoming(state);

    if (state->context == SERVER_IN_GAME) {
	allegro_gl_unset_allegro_mode();

	game_start(state);
	game_loop(state, tid_server);
	game_stop(state);

	allegro_gl_set_allegro_mode();
	sv_intern_sort_by_score();
    }
}

/*--------------------------------------------------------------*/

/* lobby_loop:
 *
 * When in client-server mode, tid_server is the thread of the server.
 * Otherwise, it is NULL.
 */
void lobby_loop(const coro_t tid_server, const lobby_client_fe_t *fe)
{
    game_state_t state;
    void *a;
    assert(fe);

    game_state_init(&state, game_mode, false);
    lobby_init();
    a = fe->initialise(&state);

    while (state.context != SERVER_QUIT) {
	if (!fe->update(a))
	    state.context = SERVER_QUIT;

	lobby_client_update(&state, tid_server);
	fe->redraw();

	if (client_server_mode == I_AM_CLIENT_SERVER) {
	    if (!coro_yield(tid_server)) {
		fprintf(stderr, "[Client] Server thread died!\n");
		state.context = SERVER_QUIT;
	    }
	}
	else {
	    rest(0);
	}
    }

    fe->shutdown(a);
    lobby_shutdown();
}

/*--------------------------------------------------------------*/

static void lobby_init(void)
{
}


static void lobby_shutdown(void)
{
}
