/* game.c,
 *
 * The main game loop.
 */

#include <alleggl.h>
#include <allegro.h>
#include <assert.h>
#include <math.h>
#include <stdbool.h>
#include <stdio.h>

#include "angle.h"
#include "background.h"
#include "backpack.h"
#include "bullet.h"
#include "camera.h"
#include "candela.h"
#include "chat.h"
#include "client.h"
#include "common.h"
#include "container.h"
#include "coro.h"
#include "cursor.h"
#include "editor.h"
#include "explosion.h"
#include "fps.h"
#include "game.h"
#include "gizmo.h"
#include "hud.h"
#include "input.h"
#include "lux.h"
#include "map.h"
#include "meat.h"
#include "network.h"
#include "particle.h"
#include "pickup.h"
#include "player.h"
#include "robot.h"
#include "robots/01dwarf.h"
#include "server.h"
#include "smoke.h"
#include "sound.h"
#include "start-loc.h"
#include "sv-internal.h"
#include "universal.h"


enum GAME_MENU_MODE {
    /* Playing game, editting map. */
    GAME_MENU_OFF,

    /* Return to lobby? (server only) */
    GAME_MENU_LOBBY,

    /* Disconnect? (client only) */
    GAME_MENU_DISCONNECT
};


/* Keep track of the final enabled light in the previous tick. */
static GLenum last_light_end;

/*--------------------------------------------------------------*/

static volatile int counter;
static void ticker_timer(void)
{
    counter++;
}
END_OF_FUNCTION(ticker_timer);

/*--------------------------------------------------------------*/

static bool lighting_enabled(const game_state_t *state)
{
    assert(state);

    if (state->editor_enabled) {
	return editor_lighting_enabled;
    }
    else {
	return true;
    }
}

/*--------------------------------------------------------------*/

static void game_input(game_state_t *state, enum GAME_MENU_MODE *menu);
static void game_input_menu(game_state_t *state, enum GAME_MENU_MODE *menu,
			    const int sc);
static void game_camera_logic(camera_t *cam, const int mx, const int my,
			      const bool editor_enabled);


static void game_logic(game_state_t *state, enum GAME_MENU_MODE *menu)
{
    assert(state);

    if (client_server_mode != I_AM_CLIENT_SERVER || state->as_server)
	server_time += 1000.0/GAME_SPEED;

    if (state->as_server) {
	server_maybe_new_connection();
	server_incoming(state);

	robot_logic(state);
	player_update(state);
	pickup_update(state);
	backpack_update(state);
	bullet_update(state);
	explosion_update(state);
	container_update(state);

	server_outgoing(state);
    }
    else {
	game_input(state, menu);

	client_incoming(state);
	player_predict_move(state);

	pickup_update(state);
	backpack_update(state);
	bullet_update(state);
	explosion_update(state);
	meat_update();
	smoke_update();
	particle_update();
	lux_update();
	sound_poll();
    }

    chat_scroll(server_time);
}


static void game_input(game_state_t *state, enum GAME_MENU_MODE *menu)
{
    enum WEAPON_CLASS select_weapon;
    static int last_mz;
    double mx, my;
    int mz = mouse_z;
    int impy;
    assert(state);

    if (!player)
	return;

    select_weapon = player->weapon;
    cursor_coordinates(mouse_x, mouse_y, camera.scale, &mx, &my);

    /* Read navigation/mouse. */
    if (state->editor_enabled) {
	impy = editor_get_impies(camera.x+mx, camera.y+my, mz-last_mz);
    }
    else {
	impy = input_get_impies(camera.x+mx, camera.y+my, mz-last_mz,
				&select_weapon);
    }

    if (input_enabled) {
	/* Discard all keyboard impies if chat enabled. */
	impy = impy & INPUT_FIRE;
    }

    if (*menu != GAME_MENU_OFF) {
	impy &=~INPUT_RESPAWN;
    }

    /* Read chat/menu. */
    while (keypressed()) {
	const char *chat;
	int c, sc;

	c = ureadkey(&sc);

	/* Chat. */
	chat = chat_input(c, sc);
	if (chat) {
	    if (chat[0] == '/')
		client_send_command(chat+1);
	    else
		client_send_chat(chat);
	}

	/* Menu. */
	game_input_menu(state, menu, sc);
    }

    last_mz = mz;
    client_outgoing(player, impy, select_weapon);
    game_camera_logic(&camera, mx, my, state->editor_enabled);
}


static void game_input_menu(game_state_t *state,
			    enum GAME_MENU_MODE *menu, const int sc)
{
    assert(menu);

    if (sc == KEY_F12)
	toggle_fps();

    switch (*menu) {
      case GAME_MENU_OFF:
	  if (sc == KEY_ESC) {
	      if (client_server_mode == I_AM_CLIENT_SERVER) {
		  *menu = GAME_MENU_LOBBY;
	      }
	      else {
		  *menu = GAME_MENU_DISCONNECT;
	      }
	  }
	  break;

      case GAME_MENU_LOBBY:
	  if (sc == KEY_Y) {
	      assert(client_server_mode == I_AM_CLIENT_SERVER);
	      client_send_command("stop");
	  }
	  else if ((sc == KEY_N) ||
		   (sc == KEY_ESC)) {
	      *menu = GAME_MENU_OFF;
	  }
	  break;

      case GAME_MENU_DISCONNECT:
	  if (sc == KEY_Y) {
	      assert(client_server_mode == I_AM_CLIENT_ONLY);
	      client_request_disconnect();
	      state->context = SERVER_QUIT;
	  }
	  else if ((sc == KEY_N) ||
		   (sc == KEY_ESC)) {
	      *menu = GAME_MENU_OFF;
	  }
	  break;

      default:
	  assert(false);
	  break;
    }
}


static void game_camera_logic(camera_t *cam, const int mx, const int my,
			      const bool editor_enabled)
{
    if (!editor_enabled) {
	if (player->alive) {
	    double dx = mx*640.0/cam->view_width-320.0;
	    double dy = my*480.0/cam->view_height-240.0;
	    double r = dx*dx + dy*dy;

	    if (r < 240.0*240.0) {
		camera.desired_scale = 1.2 - 0.4*sqrt(r)/240.0;
	    }
	    else {
		camera.desired_scale = 0.8;
	    }
	}
	else {
	    player->x += player->vx;
	    player->y += player->vy;
	    player->vx *= 0.85;
	    player->vy *= 0.85;
	}
    }

    camera_zoom(&camera);
    cam->view_width = 640.0/cam->scale;
    cam->view_height = 480.0/cam->scale;

    camera_track_point_with_mouse(cam, player->x, player->y+19.0, mx, my);
}

/*--------------------------------------------------------------*/

static void game_position_lights_classic(const game_state_t *state);
static void game_position_lights_silhouettes(const game_state_t *state);


static void game_position_lights(const game_state_t *state)
{
    assert(state);

    switch (state->game_mode) {
      case GAME_MODE_CLASSIC:
	  game_position_lights_classic(state);
	  break;
      case GAME_MODE_SILHOUETTES:
	  game_position_lights_silhouettes(state);
	  break;
      default:
	  assert(false);
    }
}


static void game_position_player_light(void)
{
    GLfloat posi[] = { player->x, player->y+19.0, 120.0, 1.0 };

    if (player->powerup_duration[POWERUP_LIGHT_AMP] > server_time) {
	const GLfloat ambi[] = { 0.4, 0.4, 0.6, 0.5 };
	const GLfloat diff[] = { 0.6, 0.6, 1.0, 0.8 };

	glLightfv(GL_LIGHT0, GL_AMBIENT, ambi);
	glLightfv(GL_LIGHT0, GL_DIFFUSE, diff);
	posi[2] = 200.0;
    }
    else {
	const GLfloat ambi[] = { 0.2, 0.2, 0.2, 0.2 };
	const GLfloat diff[] = { 0.8, 0.8, 0.8, 0.8 };

	glLightfv(GL_LIGHT0, GL_AMBIENT, ambi);
	glLightfv(GL_LIGHT0, GL_DIFFUSE, diff);
    }

    glLightfv(GL_LIGHT0, GL_SPECULAR, glBlack);
    glLightfv(GL_LIGHT0, GL_POSITION, posi);
    glLightf(GL_LIGHT0, GL_SPOT_CUTOFF, 75);
    glLightf(GL_LIGHT0, GL_SPOT_EXPONENT, 5.0);
}


static void game_position_player_torch(void)
{
    const GLfloat ambi[] = { 0.25, 0.25, 0.25, 0.25 };
    const GLfloat diff[] = { 0.6, 0.6, 0.6, 0.8 };
    const GLfloat spec[] = { 1.0, 1.0, 1.0, 1.0 };
    const GLfloat posi[] = { 0.0, 0.0, 50.0, 1.0 };
    const GLfloat dire[] = { M_SQRT2*2, 0, -M_SQRT2 };

    glPushMatrix();

    glTranslatef(player->x, player->y+19, 0);
    glRotated(rad2deg(player->angle), 0,0,1);
    glTranslatef(-115.0, 0.0, 0);

    glLightfv(GL_LIGHT1, GL_AMBIENT, ambi);
    glLightfv(GL_LIGHT1, GL_DIFFUSE, diff);
    glLightfv(GL_LIGHT1, GL_SPECULAR, spec);
    glLightfv(GL_LIGHT1, GL_POSITION, posi);
    glLightfv(GL_LIGHT1, GL_SPOT_DIRECTION, dire);
    glLightf(GL_LIGHT1, GL_SPOT_CUTOFF, 25.0);
    glLightf(GL_LIGHT1, GL_SPOT_EXPONENT, 5.0);

    glPopMatrix();
}


static void game_position_enemy_light(const player_t *p, const GLenum light)
{
    const GLfloat ambi[] = { 0.0, 0.0, 0.0, 0.0 };
    const GLfloat diff[] = { 0.5, 0.5, 0.5, 0.5 };
    const GLfloat spec[] = { 0.0, 0.0, 0.0, 0.0 };
    const GLfloat posi[] = { p->x, p->y+19.0, 80.0, 1.0 };

    glLightfv(light, GL_AMBIENT, ambi);
    glLightfv(light, GL_DIFFUSE, diff);
    glLightfv(light, GL_SPECULAR, spec);
    glLightfv(light, GL_POSITION, posi);
    glLightf(light, GL_SPOT_CUTOFF, 35);
    glLightf(light, GL_SPOT_EXPONENT, 15);

    glEnable(light);
}


static void game_position_lights_classic(const game_state_t *state)
{
    player_t *enemy;
    GLenum light, ll;
    assert(state);
    assert(!(state->as_server));

    if (!player)
        return;

    game_position_player_light();
    game_position_player_torch();

    light = GL_LIGHT2;
    list_for_each(enemy, &state->player_list) {
        if (!(enemy->alive) || (enemy == player))
	    continue;

	game_position_enemy_light(enemy, light);
        light++;
	if (light >= GL_LIGHT0+GL_MAX_LIGHTS)
	    break;
    }

    light = candela_position(light);
    for (ll = light; ll < last_light_end; ll++)
	glDisable(ll);
    last_light_end = light;
}


static void game_position_lights_silhouettes(const game_state_t *_)
{
    const GLfloat iLikeDarker[] = { -0.1, -0.1, -0.1, 0.0 };
    const GLfloat posi[] = { 0.0, 0.0, 1.0, 0.0 };
    (void)_;

    glLightfv(GL_LIGHT0, GL_AMBIENT,  iLikeDarker);
    glLightfv(GL_LIGHT0, GL_DIFFUSE,  glBlack);
    glLightfv(GL_LIGHT0, GL_SPECULAR, glBlack);
    glLightfv(GL_LIGHT0, GL_POSITION, posi);
    glLightfv(GL_LIGHT1, GL_AMBIENT,  glBlack);
    glLightfv(GL_LIGHT1, GL_DIFFUSE,  glBlack);
    glLightfv(GL_LIGHT1, GL_SPECULAR, glBlack);
    glLightfv(GL_LIGHT1, GL_POSITION, posi);
    last_light_end = GL_LIGHT2;
}


static void game_darkening_magic(void)
{
    glDisable(GL_ALPHA_TEST);
    glDisable(GL_TEXTURE_2D);
    glDisable(GL_LIGHTING);
    glEnable(GL_BLEND);

    glBlendEquation(GL_FUNC_REVERSE_SUBTRACT);
    glBlendFunc(GL_SRC_COLOR, GL_ONE);

    glPushMatrix();
    glLoadIdentity();
    glColor4f(0.33, 0.33, 0.33, 0.33);
    glRectf(0.0, 0.0, 640.0, 480.0);
    glPopMatrix();

    glDisable(GL_BLEND);
    glEnable(GL_TEXTURE_2D);
    glEnable(GL_LIGHTING);
    glEnable(GL_ALPHA_TEST);
}

/*--------------------------------------------------------------*/

static void game_draw_background(const game_state_t *state);

#ifdef DEBUG_PARTICLE_STATS
static void game_draw_particle_stats(const float y);
#else
# define game_draw_particle_stats(a)
#endif


static void game_draw(game_state_t *state, const enum GAME_MENU_MODE menu)
{
    assert(state);

    fps_counter++;

    glLoadIdentity();
    glClear(GL_COLOR_BUFFER_BIT);

    game_draw_background(state);

    glScalef(camera.scale, camera.scale, camera.scale);
    glTranslatef(-rint(camera.x), -rint(camera.y), 0.0);

    if (lighting_enabled(state))
	game_position_lights(state);

    map_draw();

    if (state->editor_enabled)
	start_loc_draw();

    gizmo_draw(state->editor_enabled);
    backpack_draw(state);
    pickup_draw(state->editor_enabled);
    container_draw(state->editor_enabled);
    player_draw(state);
    robot_draw();
    bullet_draw(state);
    meat_draw();
    smoke_draw();
    particle_draw();
    lux_draw();

    if (state->editor_enabled)
	candela_draw();

    if ((gl_reverse_subtract) && (lighting_enabled(state)))
	game_darkening_magic();

    glDisable(GL_LIGHTING);
    hud_draw_special(state);
    glLoadIdentity();

    /* FPS. */
    fps_draw();
    game_draw_particle_stats(60.0);

    hud_draw_chat();

    if (state->editor_enabled)
	editor_draw();
    else
	hud_draw(menu == GAME_MENU_OFF);
}


/* game_draw_background:
 *
 * Draw the background either after enabling lighting (normally), or
 * before enabling lighting (silhouettes mode).
 */
static void game_draw_background(const game_state_t *state)
{
    bool lighting;
    assert(state);

    lighting = lighting_enabled(state);
    if ((state->game_mode != GAME_MODE_SILHOUETTES) && lighting)
	glEnable(GL_LIGHTING);

    background_draw(&camera);

    if ((state->game_mode == GAME_MODE_SILHOUETTES) && lighting)
	glEnable(GL_LIGHTING);
}


#ifdef DEBUG_PARTICLE_STATS
static void game_draw_particle_stats(const float y)
{
    glColor3fv(glWhite);

    allegro_gl_printf_ex(al_font, 5.0, y+20.0, 0.0,
			 "Particles:  %5d I:%5d/%5d/%5d  Draw:%5d",
			 parte_count,
			 parti_count, parti_freed, parti_total,
			 parte_drawn);

    allegro_gl_printf_ex(al_font, 5.0, y+10.0, 0.0,
			 "Array: %5d +%5d=%5d  (%4.2f)",
			 parta_count, parta_freed, parta_total,
			 parta_count ? (float)
			 parte_count/(parta_count*PARTICLE_ARRAY_SIZE) : 0);

    allegro_gl_printf_ex(al_font, 5.0, y, 0.0,
			 "Steps: %5d /%5d=%5.2f  Nub:%2d",
			 parte_steps, parte_stepped,
			 parte_stepped ? (float)parte_steps/parte_stepped : 0,
			 parte_nubbed);
}
#endif

/*--------------------------------------------------------------*/

static void game_draw_menu(const enum GAME_MENU_MODE mode)
{
    switch (mode) {
      case GAME_MENU_OFF:
	  break;

      case GAME_MENU_LOBBY:
	  algl_textout_centre(al_font, "Return to lobby?",
			      320.0, 300.0, 0.0);
	  algl_textout_centre(al_font, "Press Y or N", 320.0, 260.0, 0.0);
	  break;

      case GAME_MENU_DISCONNECT:
	  algl_textout_centre(al_font, "Disconnect from server?",
			      320.0, 300.0, 0.0);
	  algl_textout_centre(al_font, "Press Y or N", 320.0, 260.0, 0.0);
	  break;

      default:
	  assert(false);
	  break;
    }
}

/*--------------------------------------------------------------*/

static void game_screenshot(void)
{
    static int shot_num = 0;
    char filename[32];
    BITMAP *bmp = NULL;
    unsigned char *pixels = NULL;
    unsigned char *px;
    int x, y;

    pixels = malloc(sizeof(*pixels) * SCREEN_W * SCREEN_H * 3);
    if (!pixels)
	goto exit;

    bmp = create_bitmap(SCREEN_W, SCREEN_H);
    if (!bmp)
	goto exit;

    glReadBuffer(GL_FRONT);
    glReadPixels(0, 0, SCREEN_W, SCREEN_H, GL_RGB, GL_UNSIGNED_BYTE, pixels);

    px = &pixels[0];
    for (y = 0; y < SCREEN_H; y++) {
	for (x = 0; x < SCREEN_W; x++) {
	    int r = *(px+0);
	    int g = *(px+1);
	    int b = *(px+2);

	    putpixel(bmp, x, SCREEN_H-y-1, makecol(r, g, b));
	    px += 3;
	}
    }

    snprintf(filename, sizeof(filename), "shot%04d.bmp", shot_num);
    save_bitmap(filename, bmp, NULL);
    shot_num++;

 exit:

    if (bmp) {
	destroy_bitmap(bmp);
	bmp = NULL;
    }

    if (pixels) {
	free(pixels);
	pixels = NULL;
    }
}

/*--------------------------------------------------------------*/

void game_loop(game_state_t *state, coro_t tid)
{
    enum GAME_MENU_MODE menu = GAME_MENU_OFF;
    bool redraw = true;
    assert(state);

    if (client_server_mode == I_AM_CLIENT_SERVER) {
	coro_yield(tid);
    }

    while (state->context == SERVER_IN_GAME) {
	while ((state->context == SERVER_IN_GAME) && (counter > 0)) {
	    redraw = true;

	    game_logic(state, &menu);

	    if (client_server_mode == I_AM_CLIENT_SERVER) {
		if (state->as_server)
		    counter--;

		coro_yield(tid);
	    }
	    else {
		counter--;
	    }
	}

        if (redraw && !state->as_server) {
	    redraw = false;
	    game_draw(state, menu);
	    game_draw_menu(menu);
	    cursor_draw();

	    if (key[KEY_F10])
		game_screenshot();

	    allegro_gl_flip();
        }

	rest(0);
    }

    if (client_server_mode == I_AM_CLIENT_SERVER) {
	coro_yield(tid);
    }
}

/*--------------------------------------------------------------*/

void game_start(game_state_t *state)
{
    assert(state);

    background_init(state->game_mode);

    if (client_server_mode != I_AM_CLIENT_SERVER || state->as_server) {
	score_begin_session();

	container_start();
	gizmo_start();
	pickup_start();
	candela_start();
	start_loc_start();
    }

    if (!state->as_server) {
	sound_start();

	particle_start();
	smoke_start();
	meat_start();

	lux_start();
    }

    explosion_start(state);
    bullet_start(state);
    backpack_start(state);
    player_start(state);

    counter = 0;
    server_time = 0;
    last_light_end = GL_LIGHT2;

    background_offx = 0;
    background_offy = 0;

    camera.x = 0;
    camera.y = 0;
    camera.view_width  = 640;
    camera.view_height = 480;
    camera.pushable = false;
    camera.max_dist = CAMERA_DEFAULT_MAX_DIST;
    camera.scale = 1.0;
    camera.desired_scale = 1.0;

    zero_tx = 0.0;
    zero_ty = 0.0;
    zero_x = 0.0;
    zero_y = 0.0;
}


void game_stop(game_state_t *state)
{
    int i;
    GLenum ll;
    assert(state);

    for (i = 0; i < MAX_CLIENTS; i++) {
	client_data[i].id = 0;
    }

    player_stop(state);
    backpack_stop(state);
    bullet_stop(state);
    explosion_stop(state);

    if ((client_server_mode != I_AM_CLIENT_SERVER) ||
	!(state->as_server)) {
	/* Disable all but the local player's lights. */
	for (ll = GL_LIGHT2; ll < last_light_end; ll++)
	    glDisable(ll);
	last_light_end = GL_LIGHT2;
    }

    if (client_server_mode != I_AM_CLIENT_SERVER || state->as_server) {
	start_loc_stop();
	candela_stop();
	pickup_stop();
	gizmo_stop();
	container_stop();
    }

    if (!state->as_server) {
	lux_stop();

	meat_stop();
	smoke_stop();
	particle_stop();

	sound_stop();
    }

    /* Ready next game. */
    state->game_mode = state->next_game_mode;
}


void game_init(const enum GAME_MODE mode)
{
    LOCK_VARIABLE(counter);
    LOCK_FUNCTION(ticker_timer);
    install_int_ex(ticker_timer, BPS_TO_TIMER(GAME_SPEED));
    counter = 0;

    fps_init();
    hud_init();
    cursor_init();
    editor_init();

    background_init(mode);
    map_init();
    container_init();
    gizmo_init();
    pickup_init();
    start_loc_init();

    particle_init();
    smoke_init();
    meat_init();

    candela_init();
    explosion_init();
    bullet_init();
    backpack_init();
    player_init();
}


void game_shutdown(void)
{
    player_shutdown();
    backpack_shutdown();
    bullet_shutdown();
    explosion_shutdown();
    candela_shutdown();

    meat_shutdown();
    smoke_shutdown();
    particle_shutdown();

    start_loc_shutdown();
    pickup_shutdown();
    gizmo_shutdown();
    container_shutdown();
    map_shutdown();
    background_shutdown();

    editor_shutdown();
    cursor_shutdown();
    hud_shutdown();
    fps_shutdown();
}
