/* editor.c,
 *
 * Extra map-editor related stuff.
 */

#include <alleggl.h>
#include <allegro.h>
#include <assert.h>
#include <math.h>
#include <stdio.h>

#include "angle.h"
#include "brush.h"
#include "camera.h"
#include "candela.h"
#include "chat.h"
#include "client.h"
#include "common.h"
#include "container.h"
#include "cursor.h"
#include "gizmo.h"
#include "input.h"
#include "editor.h"
#include "map.h"
#include "maxmin.h"
#include "network.h"
#include "pickup.h"
#include "region.h"
#include "start-loc.h"


#define SNAP_TO_NOTHING		0x00
#define SNAP_TO_TILE		0x01
#define SNAP_TO_GRID		0x02
#define SNAP_TO_ALL		(SNAP_TO_TILE|SNAP_TO_GRID)


enum EDITOR_MODE editor_mode;
bool editor_lighting_enabled;

static enum {
    EDITOR_NORMAL,
    EDITOR_CANDELA_HUE,
    EDITOR_CANDELA_RESIZE,
    EDITOR_CANDELA_EXPONENT
} editor_minor_mode;

static char grid_snap;
static unsigned int selected_tile;
static enum CONTAINER_CLASS selected_container;
static enum GIZMO_CLASS selected_gizmo;
static enum PICKUP_CLASS selected_pickup;
static unsigned char tile_brush_size;
static container_id hover_container, work_container;
static gizmo_id hover_gizmo, work_gizmo;
static pickup_id hover_pickup, work_pickup;
static start_loc_id hover_start, work_start;
static candela_id hover_candela, work_candela;
static unsigned int hover_region;

/* This is a copy of the last tile we set to prevent resending a
 * tile_set command while we are waiting confirmation of the tile
 * change from the server.
 *
 * Bug: If we just extended the map, the position of the tile set is
 * not updated.
 */
static struct packet_tile_set last_tile_set;

/*--------------------------------------------------------------*/

static bool packet_tile_set_eq(const struct packet_tile_set *a,
			       const struct packet_tile_set *b)
{
    return ((a->tile == b->tile) &&
	    (a->brush_size == b->brush_size) &&
	    (a->x == b->x) &&
	    (a->y == b->y));
}

/*--------------------------------------------------------------*/

static void editor_keyboard_control(void);
static void editor_select_mode(void);
static void editor_do_hover(const int mx, const int my);
static void editor_do_lmb(const int mx, const int my, const int last_mb);
static void editor_do_candela_packet(struct packet_candela_new *packet,
				     const int mx, const int my);
static void editor_do_lmb_off(void);
static void editor_do_rmb(const int mx, const int my, const int last_mb);
static void editor_do_mmb_off(const int mx, const int my, const int last_mb);
static void editor_do_selection(const int mz);
static bool editor_move_container(struct packet_editor *packet,
				  const int mx, const int my);
static bool editor_move_gizmo(struct packet_editor *packet,
			      const int mx, const int my);


int editor_get_impies(const int mx, const int my, const int mz)
{
    static int last_mouse_b;
    int impies;

    impies = input_get_move_impies(mx, my);
    editor_select_mode();

    if (player && !player->alive) {
	impies |= INPUT_RESPAWN;
    }

    editor_do_hover(mx, my);

    if (mouse_b & 1) {
	editor_do_lmb(mx, my, last_mouse_b);
    }
    else {
	editor_do_lmb_off();
    }

    if (mouse_b & 2) {
	editor_do_rmb(mx, my, last_mouse_b);
    }

    if (!(mouse_b & 4)) {
	editor_do_mmb_off(mx, my, last_mouse_b);
    }

    if (mz) {
	editor_do_selection(mz);
    }

    if (!input_enabled) {
	editor_keyboard_control();
    }

    last_mouse_b = mouse_b;
    return impies;
}


static void editor_keyboard_control(void)
{
    static char last_keys;

    if ((key[KEY_OPENBRACE]) && !((last_keys >> 0) & 0x01)) {
	if (grid_snap > SNAP_TO_NOTHING)
	    grid_snap--;
    }

    if ((key[KEY_CLOSEBRACE]) && !((last_keys >> 1) & 0x01)) {
	if (grid_snap < SNAP_TO_ALL)
	    grid_snap++;
    }

    if ((key[KEY_SPACE]) && !((last_keys >> 2) & 0x01)) {
	editor_lighting_enabled = !editor_lighting_enabled;
    }

    if (key[KEY_HOME])
	camera.desired_scale = 1.0;
    if (key[KEY_PGUP])
	camera.desired_scale = MIN(camera.desired_scale+0.01, 2.0);
    if (key[KEY_PGDN])
	camera.desired_scale = MAX(camera.desired_scale-0.01, 0.50);

    last_keys = ((key[KEY_OPENBRACE]	<< 0) |
		 (key[KEY_CLOSEBRACE]	<< 1) |
		 (key[KEY_SPACE]	<< 2));
}


static void editor_select_mode(void)
{
    const int ks[NUM_EDITOR_MODES] = {
	KEY_1, KEY_2, KEY_3, KEY_4, KEY_5, KEY_0, KEY_EQUALS
    };
    enum EDITOR_MODE i;

    for (i = EDIT_TILES; i < NUM_EDITOR_MODES; i++) {
	if (key[ks[i]]) {
	    editor_mode = i;
	    return;
	}
    }
}


static void editor_do_hover(const int mx, const int my)
{
    switch (editor_mode) {
      case EDIT_TILES:
	  break;
      case EDIT_CONTAINERS: {
	  static container_id last_hover;
	  container_t *c;

	  hover_container = container_bounding(mx, my, false);

	  if ((last_hover) && (c = container_from_id(last_hover))) {
	      c->highlight = false;
	  }

	  if ((work_container) && (c = container_from_id(work_container))) {
	      hover_container = work_container;
	      c->highlight = true;
	  }
	  else if (hover_container) {
	      c = container_from_id(hover_container);
	      c->highlight = true;
	  }

	  last_hover = hover_container;
	  break;
      }
      case EDIT_GIZMOS: {
	  static gizmo_id last_hover;
	  gizmo_t *g;

	  hover_gizmo = gizmo_bounding(mx, my);

	  if ((last_hover) && (g = gizmo_from_id(last_hover))) {
	      g->highlight = false;
	  }

	  if ((work_gizmo) && (g = gizmo_from_id(work_gizmo))) {
	      hover_gizmo = work_gizmo;
	      g->highlight = true;
	  }
	  else if (hover_gizmo) {
	      g = gizmo_from_id(hover_gizmo);
	      g->highlight = true;
	  }

	  last_hover = hover_gizmo;
	  break;
      }
      case EDIT_PICKUPS: {
	  static pickup_id last_hover;
	  pickup_t *p;

	  hover_pickup = pickup_bounding(mx, my);

	  if ((last_hover) && (p = pickup_from_id(last_hover))) {
	      p->highlight = false;
	  }

	  if ((work_pickup) && (p = pickup_from_id(work_pickup))) {
	      hover_pickup = work_pickup;
	      p->highlight = true;
	  }
	  else if (hover_pickup) {
	      p = pickup_from_id(hover_pickup);
	      p->highlight = true;
	  }

	  last_hover = hover_pickup;
	  break;
      }
      case EDIT_STARTS: {
	  static start_loc_id last_hover;

	  hover_start = start_loc_get_bounding(mx, my);

	  if (last_hover < MAX_START_LOCS) {
	      starts[last_hover].highlight = false;
	  }

	  if (work_start < MAX_START_LOCS) {
	      hover_start = work_start;
	      starts[work_start].highlight = true;
	  }
	  else if (hover_start < MAX_START_LOCS) {
	      starts[hover_start].highlight = true;
	  }

	  last_hover = hover_start;
	  break;
      }
      case EDIT_CANDELA: {
	  static candela_id last_hover;
	  candela_t *c;

	  hover_candela = candela_bounding(mx, my);

	  if ((last_hover) && (c = candela_from_id(last_hover))) {
	      c->highlight = false;
	  }

	  if ((work_candela) && (c = candela_from_id(work_candela))) {
	      hover_candela = work_candela;
	      c->highlight = true;
	  }
	  else if (hover_candela) {
	      c = candela_from_id(hover_candela);
	      c->highlight = true;
	  }

	  last_hover = hover_candela;
	  break;
      }
      case EDIT_REGIONS:
	  if (x_in_map(mx) && y_in_map(my))
	      hover_region = map.tile[my/TILE_H][mx/TILE_W].region;
	  break;
    }
}


static void editor_do_lmb(const int mx, const int my, const int last_mb)
{
#define CLICK_LMB		(!(last_mb & 1))

    struct packet_editor packet;
    NLbyte buf[MAX_PACKET_SIZE];
    NLint len;
    int mmx, mmy;

    packet.editor_mode = editor_mode;
    packet.action = EDIT_ADD;

    if (grid_snap & SNAP_TO_GRID) {
	mmx = (mx / TILE_W) * TILE_W + TILE_W/2;
	mmy = (my / TILE_H) * TILE_H;
    }
    else {
	mmx = mx;
	mmy = my;
    }

    switch (editor_mode) {
      case EDIT_TILES:
	  if (key_shifts & KB_SHIFT_FLAG) {
	      if (x_in_map(mx) && y_in_map(my))
		  selected_tile = map.tile[my/TILE_H][mx/TILE_W].shape;
	      return;
	  }

	  if (x_in_map(mx) && y_in_map(my)) { 
	      if (!map_is_different(mx, my, selected_tile, tile_brush_size))
		  return;
	  }

	  packet.new.tile.brush_size = tile_brush_size;
	  packet.new.tile.tile = selected_tile;
	  packet.new.tile.x = mx/TILE_W;
	  packet.new.tile.y = my/TILE_H;

	  if (mx < 0)
	      packet.new.tile.x--;
	  if (my < 0)
	      packet.new.tile.y--;

	  if (packet_tile_set_eq(&packet.new.tile, &last_tile_set))
	      return;

	  last_tile_set = packet.new.tile;
	  break;
      case EDIT_CONTAINERS:
	  if (CLICK_LMB)
	      work_container = hover_container;

	  if (key_shifts & KB_SHIFT_FLAG) {
	      if (work_container && CLICK_LMB) {
		  container_t *c = container_from_id(work_container);
		  assert(c);
		  selected_container = c->class;
	      }
	      return;
	  }
	  else if (work_container) {
	      if (!editor_move_container(&packet, mmx, mmy))
		  return;
	  }
	  else if (CLICK_LMB) {
	      packet.new.container_new.id = 0;
	      packet.new.container_new.class = selected_container;
	      packet.new.container_new.x = mmx;
	      packet.new.container_new.y = mmy;
	      packet.new.container_new.hidden = false;
	  }
	  else
	      return;

	  break;
      case EDIT_GIZMOS:
	  if (CLICK_LMB)
	      work_gizmo = hover_gizmo;

	  if (key_shifts & KB_SHIFT_FLAG) {
	      if (work_gizmo && CLICK_LMB) {
		  gizmo_t *g = gizmo_from_id(work_gizmo);
		  assert(g);
		  selected_gizmo = g->class;
	      }
	      return;
	  }
	  else if (work_gizmo) {
	      if (!editor_move_gizmo(&packet, mmx, mmy))
		  return;
	  }
	  else if (!(last_mb & 1)) {
	      packet.new.gizmo_new.id = 0;
	      packet.new.gizmo_new.class = selected_gizmo;
	      packet.new.gizmo_new.x = mmx;
	      packet.new.gizmo_new.y = mmy;
	  }
	  else
	      return;

	  break;
      case EDIT_PICKUPS:
	  if (CLICK_LMB)
	      work_pickup = hover_pickup;

	  if (key_shifts & KB_SHIFT_FLAG) {
	      if (work_pickup && CLICK_LMB) {
		  pickup_t *p = pickup_from_id(work_pickup);
		  assert(p);
		  selected_pickup = p->class;
	      }
	      return;
	  }
	  else if (work_pickup) {
	      packet.action = EDIT_MOVE;
	      packet.new.pickup_mov.id = work_pickup;
	      packet.new.pickup_mov.x = mmx;
	      packet.new.pickup_mov.y = mmy;
	  }
	  else if (CLICK_LMB) {
	      packet.new.pickup_new.time = 0;
	      packet.new.pickup_new.id = 0;
	      packet.new.pickup_new.class = selected_pickup;
	      packet.new.pickup_new.x = mmx;
	      packet.new.pickup_new.y = mmy;
	  }
	  else
	      return;

	  break;
      case EDIT_STARTS: {
	  if (CLICK_LMB)
	      work_start = hover_start;

	  if ((work_start >= MAX_START_LOCS) && (last_mb & 1))
	      return;

	  packet.new.start_new.id = work_start;
	  packet.new.start_new.x = mmx;
	  packet.new.start_new.y = mmy;
	  break;
      }
      case EDIT_CANDELA: {
	  packet.new.candela_new.size = 0;

	  if (CLICK_LMB) {
	      if (hover_candela) {
		  work_candela = hover_candela;
		  if (key_shifts & KB_SHIFT_FLAG)
		      editor_minor_mode = EDITOR_CANDELA_EXPONENT;
		  else if (key_shifts & KB_CTRL_FLAG)
		      editor_minor_mode = EDITOR_CANDELA_HUE;
	      }
	      else {
		  /* Try to resize a light. */
		  work_candela = candela_radius_bounding(mmx, mmy);
		  if (work_candela)
		      editor_minor_mode = EDITOR_CANDELA_RESIZE;
	      }
	  }
	  else if (!work_candela) {
	      return;
	  }

	  editor_do_candela_packet(&packet.new.candela_new, mmx, mmy);
	  if (packet.new.candela_new.size == 0)
	      return;
	  break;
      }
      case EDIT_REGIONS: {
	  region_id region = 0;
	  int xx = mx/TILE_W;
	  int yy = my/TILE_H;

	  if (!x_in_map(mx) || !y_in_map(my))
	      return;

	  if (map.tile[yy][xx].region)
	      return;

	  if (!region && (xx > 0))
	      region = map.tile[yy][xx-1].region;
	  if (!region && (xx < map.w-1))
	      region = map.tile[yy][xx+1].region;
	  if (!region && (yy > 0))
	      region = map.tile[yy-1][xx].region;
	  if (!region && (yy < map.h-1))
	      region = map.tile[yy+1][xx].region;
	  if (!region)
	      region = unused_region();

	  packet.new.region.brush_size = tile_brush_size;
	  packet.new.region.region = region;
	  packet.new.region.x = xx;
	  packet.new.region.y = yy;
	  break;
      }
    }

    packet.mode = to_mode_action(packet.editor_mode, packet.action);
    len = packet_editor_encode(&packet, buf, sizeof(buf));
    client_write(buf, len);

#undef CLICK_LMB
}


static void editor_do_candela_packet(struct packet_candela_new *packet,
				     const int mx, const int my)
{
    candela_t *c;
    double dx, dy, r;
    packet->id = work_candela;

    if (editor_minor_mode == EDITOR_NORMAL) {
	packet->size |= CANDELA_INCLUDE_POSITION;
	packet->posi[0] = mx;
	packet->posi[1] = my;
	packet->posi[2] = 100;
	return;
    }

    if (!work_candela)
	return;

    c = candela_from_id(work_candela);
    dx = mx - c->posi[0];
    dy = my - c->posi[1];
    r = sqrt(dx*dx + dy*dy);

    if (editor_minor_mode == EDITOR_CANDELA_HUE) {
	float h, s;
	int rr, gg, bb;

	packet->size |= CANDELA_INCLUDE_AMBIENT;

	h = atan2(dy, dx);
	if (h < 0)
	    h += 2*M_PI;

	s = MID(0.0, r/100.0, 1.0);

	hsv_to_rgb(rad2deg(h), s, 1.0, &rr, &gg, &bb);
	packet->ambi[0] = (float)rr/255.0;
	packet->ambi[1] = (float)gg/255.0;
	packet->ambi[2] = (float)bb/255.0;
    }
    else if (editor_minor_mode == EDITOR_CANDELA_RESIZE) {
	packet->size |= CANDELA_INCLUDE_CUTOFF;
	packet->cutoff = rad2deg(atan(r/c->posi[2]));
	packet->cutoff = MID(0.0, packet->cutoff, 255.0);
    }
    else if (editor_minor_mode == EDITOR_CANDELA_EXPONENT) {
	packet->size |= CANDELA_INCLUDE_EXPONENT;
	packet->exponent = MID(0, 128 - r*128/256, 128);
    }
}


static void editor_do_lmb_off(void)
{
    work_container = 0;
    work_gizmo = 0;
    work_pickup = 0;
    work_start = MAX_START_LOCS;
    work_candela = 0;
    editor_minor_mode = EDITOR_NORMAL;
}


static void editor_do_rmb(const int mx, const int my, const int last_mb)
{
#define HELD_RMB		(last_mb & 2)

    struct packet_editor packet;
    NLbyte buf[MAX_PACKET_SIZE];
    NLint len;

    packet.editor_mode = editor_mode;
    packet.action = EDIT_DELETE;

    switch (editor_mode) {
      case EDIT_TILES:
	  if (!map_is_different(mx, my, 0, tile_brush_size))
	      return;

	  packet.action = EDIT_ADD;
	  packet.new.tile.brush_size = tile_brush_size;
	  packet.new.tile.tile = 0;
	  packet.new.tile.x = mx/TILE_W;
	  packet.new.tile.y = my/TILE_H;

	  if (packet_tile_set_eq(&packet.new.tile, &last_tile_set))
	      return;

	  last_tile_set = packet.new.tile;
	  break;
      case EDIT_CONTAINERS:
	  if (HELD_RMB || !(hover_container))
	      return;

	  packet.new.container_del.id = hover_container;
	  work_container = 0;
	  break;
      case EDIT_GIZMOS:
	  if (HELD_RMB || !(hover_gizmo))
	      return;

	  packet.new.gizmo_del.id = hover_gizmo;
	  work_gizmo = 0;
	  break;
      case EDIT_PICKUPS:
	  if (HELD_RMB || !(hover_pickup))
	      return;

	  packet.new.pickup_del.id = hover_pickup;
	  work_pickup = 0;
	  break;
      case EDIT_STARTS:
	  if (HELD_RMB || !(hover_start < MAX_START_LOCS))
	      return;

	  packet.new.start_del.id = hover_start;
	  work_start = MAX_START_LOCS;
	  break;
      case EDIT_CANDELA:
	  if (HELD_RMB || !(hover_candela))
	      return;

	  packet.new.candela_del.id = hover_candela;
	  work_candela = 0;
	  break;
      case EDIT_REGIONS:
	  packet.action = EDIT_ADD;
	  packet.new.region.brush_size = tile_brush_size;
	  packet.new.region.region = 0;
	  packet.new.region.x = mx/TILE_W;
	  packet.new.region.y = my/TILE_H;
	  break;
    }

    packet.mode = to_mode_action(packet.editor_mode, packet.action);
    len = packet_editor_encode(&packet, buf, sizeof(buf));
    client_write(buf, len);

#undef HELD_RMB
}


static void editor_do_mmb_off(const int mx, const int my, const int last_mb)
{
    struct packet_editor packet;
    NLbyte buf[MAX_PACKET_SIZE];
    NLint len;

    if (!(last_mb & 4))
	return;

    packet.mode = EDIT_PLAYER_TELEPORT;
    packet.new.player_tp.id = player->id;
    packet.new.player_tp.x = mx;
    packet.new.player_tp.y = my;

    len = packet_editor_encode(&packet, buf, sizeof(buf));
    client_write(buf, len);
}


static void editor_do_selection(const int mz)
{
    switch (editor_mode) {
      case EDIT_TILES:
      case EDIT_REGIONS:
	  if (key_shifts & KB_SHIFT_FLAG)
	      tile_brush_size = MID(0, tile_brush_size+mz, NUM_BRUSH_SIZES-1);
	  else
	      selected_tile = map_select_tile(selected_tile-mz);
	  break;
      case EDIT_CONTAINERS:
	  selected_container = container_select(selected_container-mz);
	  break;
      case EDIT_GIZMOS:
	  selected_gizmo = gizmo_select(selected_gizmo-mz);
	  break;
      case EDIT_PICKUPS:
	  selected_pickup = pickup_select(selected_pickup-mz);
	  break;
      case EDIT_STARTS:
      case EDIT_CANDELA:
	  break;
    }
}

/*--------------------------------------------------------------*/

typedef struct {
    bool (*check)(const void *q, const int xx, const int yy);
    void (*move)(struct packet_editor *packet, const int xx, const int yy);
} do_move_func_t;


static bool editor_do_move_general(struct packet_editor *packet,
				   const void *q, const int sx, const int sy,
				   const int mx, const int my,
				   const do_move_func_t *func)
{
    int xx, yy;
    assert(packet);
    assert(q);

    /* 1) Try to go directly to the mouse. */
    if (!(grid_snap & SNAP_TO_TILE) ||
	func->check(q, mx, my)) {
	func->move(packet, mx, my);
	return true;
    }

    /* 2) Try to step towards the mouse. */
    xx = sx;
    yy = sy;

    if (fabs(sx - mx) >= 1.0) {
	int dx = (sx < mx) ? 1 : -1;

	while (func->check(q, xx+dx, yy)) {
	    xx += dx;

	    if (((dx > 0) && (xx >= mx)) ||
		((dx < 0) && (xx <= mx))) {
		xx = mx;
		break;
	    }
	}
    }

    if (fabs(sy - my) >= 1.0) {
	int dy = (sy < my) ? 1 : -1;

	while (func->check(q, xx, yy+dy)) {
	    yy += dy;

	    if (((dy > 0) && (yy >= my)) ||
		((dy < 0) && (yy <= my))) {
		yy = my;
		break;
	    }
	}
    }

    if ((xx == sx) && (yy == sy))
	return false;

    func->move(packet, xx, yy);
    return true;
}


static bool editor_do_check_container(const void *q,
				      const int xx, const int yy)
{
    const container_t *c = q;
    assert(c);

    return !container_in_tile(c->class, xx, yy);
}


static void editor_do_move_container(struct packet_editor *packet,
				     const int xx, const int yy)
{
    assert(packet);

    packet->action = EDIT_MOVE;
    packet->new.container_mov.id = work_container;
    packet->new.container_mov.x = xx;
    packet->new.container_mov.y = yy;
}


static bool editor_move_container(struct packet_editor *packet,
				  const int mx, const int my)
{
    const do_move_func_t func = {
	editor_do_check_container,
	editor_do_move_container
    };
    const container_t *c = container_from_id(work_container);
    assert(packet);

    if (!c)
	return false;

    return editor_do_move_general(packet, c, c->x, c->y, mx, my, &func);
}


static bool editor_do_check_gizmo(const void *q,
				  const int xx, const int yy)
{
    const gizmo_t *g = q;
    assert(g);

    return !gizmo_in_tile(g->class, xx, yy);
}


static void editor_do_move_gizmo(struct packet_editor *packet,
				 const int xx, const int yy)
{
    assert(packet);

    packet->action = EDIT_MOVE;
    packet->new.gizmo_mov.id = work_gizmo;
    packet->new.gizmo_mov.x = xx;
    packet->new.gizmo_mov.y = yy;
}


static bool editor_move_gizmo(struct packet_editor *packet,
			      const int mx, const int my)
{
    const do_move_func_t func = {
	editor_do_check_gizmo,
	editor_do_move_gizmo
    };
    const gizmo_t *g = gizmo_from_id(work_gizmo);
    assert(packet);

    if (!g)
	return false;

    return editor_do_move_general(packet, g, g->x, g->y, mx, my, &func);
}

/*--------------------------------------------------------------*/

static void draw_map_palette(void);
static void draw_tile_brush(const int mx, const int my);
static void draw_container_palette(void);
static void draw_gizmo_palette(void);
static void draw_pickup_palette(void);
static void draw_start_palette(void);
static void draw_candela_palette(void);
static void draw_region(const unsigned int id);
static void draw_snapping_modes(void);


void editor_draw(void)
{
    double mx, my;

    cursor_coordinates(mouse_x, mouse_y, camera.scale, &mx, &my);

    switch (editor_mode) {
      case EDIT_TILES:
	  draw_map_palette();
	  draw_tile_brush(mx, my);
	  break;
      case EDIT_CONTAINERS:
	  draw_container_palette();
	  break;
      case EDIT_GIZMOS:
	  draw_gizmo_palette();
	  break;
      case EDIT_PICKUPS:
	  draw_pickup_palette();
	  break;
      case EDIT_STARTS:
	  draw_start_palette();
	  break;
      case EDIT_CANDELA:
	  draw_candela_palette();
	  break;
      case EDIT_REGIONS:
	  draw_region(hover_region);
	  draw_tile_brush(mx, my);
	  break;
    }

    if (mouse_b & 4) {
	glPushMatrix();
	glScalef(camera.scale, camera.scale, camera.scale);

	start_loc_bind_texture();
	glBegin(GL_QUADS);
	start_loc_draw_unit(START_LOC_TELEPORT, mx-16.0, my);
	glEnd();		/* glBegin(GL_QUADS) */

	glPopMatrix();
    }

    draw_snapping_modes();
}


static void draw_map_palette(void)
{
    unsigned int tile, end;
    int x, y;

    tile = MAX(1, (int)((selected_tile-1)/4-1)*4+1);
    end = map_select_tile(tile+16);

    map_bind_tiles();
    glBegin(GL_QUADS);

    for (y = -1; (tile <= end) && (y >= -4); y--) {
	for (x = -4; (tile <= end) && (x < 0); tile++, x++) {
	    if (tile == selected_tile)
		glColor3fv(glWhite);
	    else
		glColor3fv(glGray50);

	    map_draw_tile(tile, 640/TILE_W+x, 480/TILE_H+y);
	}
    }

    glEnd();			/* glBegin(GL_QUADS) */
    glColor3fv(glWhite);
}


static void draw_tile_brush(const int mx, const int my)
{
    int xx, yy;
    int mmx = ((int)(camera.x+mx))/TILE_W*TILE_W;
    int mmy = ((int)(camera.y+my))/TILE_H*TILE_H;

    glPushMatrix();
    glDisable(GL_TEXTURE_2D);
    glLoadIdentity();
    glScalef(camera.scale, camera.scale, camera.scale);
    glTranslatef(rint(-camera.x), rint(-camera.y), 0.0);
    glBegin(GL_LINES);
    glColor3f(0.0, 1.0, 1.0);

    for (yy = 0; yy < MAX_BRUSH_H; yy++) {
	for (xx = 0; xx < MAX_BRUSH_W; xx++) {
	    int X0, Y0, X1, Y1;

	    if (brushes[yy][xx] > tile_brush_size)
		continue;

	    X0 = mmx+(xx-BRUSH_W_2)*TILE_W;
	    Y0 = mmy+(yy-BRUSH_H_2)*TILE_H;
	    X1 = X0+TILE_W;
	    Y1 = Y0+TILE_H;

	    if ((xx == 0) || (brushes[yy][xx-1] > tile_brush_size)) {
		glVertex2f(X0, Y0);
		glVertex2f(X0, Y1);
	    }

	    if ((xx == MAX_BRUSH_W-1)||(brushes[yy][xx+1] > tile_brush_size)) {
		glVertex2f(X1, Y0);
		glVertex2f(X1, Y1);
	    }

	    if ((yy == 0) || (brushes[yy-1][xx] > tile_brush_size)) {
		glVertex2f(X0, Y0);
		glVertex2f(X1, Y0);
	    }

	    if ((yy == MAX_BRUSH_H-1)||(brushes[yy+1][xx] > tile_brush_size)) {
		glVertex2f(X0, Y1);
		glVertex2f(X1, Y1);
	    }
	}
    }

    glColor3fv(glWhite);
    glEnd();			/* glBegin(GL_LINES); */
    glEnable(GL_TEXTURE_2D);
    glPopMatrix();
}


static void draw_container_palette(void)
{
    enum CONTAINER_CLASS c, end;
    int y;

    c = container_select(selected_container-3);
    end = container_select(c+7);

    container_bind_texture();
    glBegin(GL_QUADS);

    for (y = 480-48; c <= end; c++, y -= 32) {
	if (c == selected_container)
	    glColor3fv(glWhite);
	else
	    glColor3fv(glGray50);

	container_draw_unit(c, 640-64, y, false);
    }

    glEnd();			/* glBegin(GL_QUADS) */
    glColor3fv(glWhite);
}


static void draw_gizmo_palette(void)
{
    enum GIZMO_CLASS g, end;
    int y;

    g = gizmo_select(selected_gizmo-3);
    end = gizmo_select(g+7);

    gizmo_bind_texture();
    glBegin(GL_QUADS);

    for (y = 480-48; g <= end; g++, y -= 32) {
	if (g == selected_gizmo)
	    glColor3fv(glWhite);
	else
	    glColor3fv(glGray50);

	gizmo_draw_unit(g, 640-64, y, false);
    }

    glEnd();			/* glBegin(GL_QUADS) */
    glColor3fv(glWhite);
}


static void draw_pickup_palette(void)
{
    enum PICKUP_CLASS p, end;
    int y;

    p = pickup_select(selected_pickup-3);
    end = pickup_select(p+7);

    pickup_bind_texture();
    glBegin(GL_QUADS);

    for (y = 480-32; p <= end; p++, y -= 32) {
	if (p == selected_pickup)
	    glColor3fv(glWhite);
	else
	    glColor3fv(glGray50);

	pickup_draw_unit(p, 640-64, y, true);
    }

    glEnd();			/* glBegin(GL_QUADS) */
    glColor3fv(glWhite);
}


static void draw_start_palette(void)
{
    start_loc_bind_texture();
    glBegin(GL_QUADS);

    start_loc_draw_unit(0, 640-64, 480-48);

    glEnd();			/* glBegin(GL_QUADS) */
}


static void draw_candela_palette(void)
{
    candela_draw_bulb(640-64, 480-48);
}


static void draw_region(const unsigned int id)
{
    const GLfloat col_hover[]  = { 0.10, 0.10, 0.25, 0.51 };
    const GLfloat col_normal[] = { 0.05, 0.05, 0.10, 0.51 };
    int x, y;
    int X0 = MID(0, camera.x/TILE_W, map.w-1);
    int Y0 = MID(0, camera.y/TILE_H, map.h-1);
    int xend = mini((camera.x + camera.view_width)/TILE_W,  map.w-1);
    int yend = mini((camera.y + camera.view_height)/TILE_H, map.h-1);

    glPushMatrix();
    glLoadIdentity();
    glScalef(camera.scale, camera.scale, camera.scale);
    glTranslatef(rint(-camera.x), rint(-camera.y), 0.0);

    glDisable(GL_TEXTURE_2D);
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_DST_ALPHA);
    glBlendEquation(GL_FUNC_ADD);
    glBegin(GL_QUADS);

    for (y = Y0; y <= yend; y++) {
	for (x = X0; x <= xend; x++) {
	    if (map.tile[y][x].region == 0)
		continue;

	    if (map.tile[y][x].region == id)
		glColor4fv(col_hover);
	    else
		glColor4fv(col_normal);

	    glVertex2f(x*TILE_W, y*TILE_H);
	    glVertex2f(x*TILE_W+TILE_W, y*TILE_H);
	    glVertex2f(x*TILE_W+TILE_W, y*TILE_H+TILE_H);
	    glVertex2f(x*TILE_W, y*TILE_H+TILE_H);
	}
    }

    glEnd();			/* glBegin(GL_QUADS) */
    glDisable(GL_BLEND);
    glEnable(GL_TEXTURE_2D);
    glPopMatrix();
}


static void draw_snapping_modes(void)
{
    int c = makecol(0xff, 0xff, 0xff);

    if (grid_snap == SNAP_TO_NOTHING) {
	allegro_gl_printf(al_font, 5.0, 15.0, 0.0, c, "Snap: Off");
	return;
    }

    allegro_gl_printf(al_font, 5.0, 15.0, 0.0, c,
		      "Snap: %s%s",
		      (grid_snap & SNAP_TO_TILE) ? "Tiles " : "",
		      (grid_snap & SNAP_TO_GRID) ? "Grid" : "");
}

/*--------------------------------------------------------------*/
     
void editor_init(void)
{
    editor_mode = EDIT_TILES;
    editor_lighting_enabled = true;
    editor_minor_mode = EDITOR_NORMAL;

    grid_snap = SNAP_TO_NOTHING;
    selected_tile = 1;
    selected_container = CONTAINER_BARREL;
    selected_gizmo = GIZMO_LADDER;
    selected_pickup = PICKUP_WEAPON_BLASTER;

    tile_brush_size = 0;

    hover_container = work_container = 0;
    hover_gizmo = work_gizmo = 0;
    hover_pickup = work_pickup = 0;
    hover_start = work_start = MAX_START_LOCS;
    hover_candela = work_candela = 0;
}


void editor_shutdown(void)
{
}
