/* client.c,
 *
 * Client code related to processing incoming commands from server,
 * and sending input packages off to the server.
 */

#include <assert.h>
#include <stdio.h>
#include <nl.h>

#include "backpack.h"
#include "bullet.h"
#include "camera.h"
#include "candela.h"
#include "chat.h"
#include "client.h"
#include "common.h"
#include "container.h"
#include "explosion.h"
#include "gizmo.h"
#include "maxmin.h"
#include "meat.h"
#include "network.h"
#include "particle.h"
#include "packet-backpack.h"
#include "packet-candy.h"
#include "packet-editor.h"
#include "packet-input.h"
#include "packet-map.h"
#include "packet-server.h"
#include "pickup.h"
#include "player.h"
#include "score.h"
#include "server.h"
#include "smoke.h"
#include "sound.h"
#include "start-loc.h"
#include "stats-container.h"
#include "strlcpy.h"
#include "universal.h"
#include "weapon-pick.h"
#include "weapon.h"


static NLenum type = NL_TCP_PACKETS;
static NLsocket sock;

/*--------------------------------------------------------------*/

#define client_print_error()	network_print_error("Client", __LINE__)


static const char *client_name_from_id(const player_id id)
{
    int idx;

    idx = sv_intern_client_from_id(id);
    if (idx < 0) {
	return "Player ";
    }
    else {
	return client_data[idx].name;
    }
}

/*--------------------------------------------------------------*/

static void client_send_buffer(const char command, const char *buf);


NLsocket client_sock(void)
{
    return sock;
}


bool client_write(const NLvoid *buf, NLint nbytes)
{
    NLint len;

    len = nlWrite(sock, buf, nbytes);

    if (len != NL_INVALID) {
	return true;
    }
    else {
	client_print_error();
	return false;
    }
}


/* client_send_chat:
 *
 * Clients sending input line to the server, as chat.
 */
void client_send_chat(const char *buf)
{
    client_send_buffer(COMMAND_INPUT_CHAT, buf);
}


/* client_send_command:
 *
 * Clients sending input line to the server, as command.
 * Note: do NOT pass the '/' from the chat prompt.
 */
void client_send_command(const char *buf)
{
    client_send_buffer(COMMAND_SERVER, buf);
}


static void client_send_buffer(const char command, const char *buf)
{
    char buf2[MAX_PACKET_SIZE];
    int len;
    assert(buf);

    len = snprintf(buf2, sizeof(buf2), "%c%s", command, buf);
    if (len > 1)
	client_write(buf2, mini(len+1, sizeof(buf2)));
}

/*--------------------------------------------------------------*/
/* Commands in game and lobby.					*/
/*--------------------------------------------------------------*/

enum COMMAND_PROCESS_DONE {
    PROCESSED,
    PROCESSED_AND_EXIT,
    UNKNOWN_COMMAND
};


static void client_input_info(const NLbyte *buf, const NLint len);
static void client_input_chat(const NLbyte *buf, const NLint len);
static void client_disconnect		(const NLbyte *buf, const NLint len,
					 game_state_t *state);


static enum COMMAND_PROCESS_DONE
client_process_common_commands(const NLbyte *buf, const NLint len,
			       game_state_t *state)
{
#define SKIP(mode)	if (client_server_mode == mode) return PROCESSED

    assert(state);

    switch (buf[0]) {
      case COMMAND_SERVER_STATE:
	  packet_server_state_decode(state, buf, len);
	  return PROCESSED_AND_EXIT;
      case COMMAND_INPUT_INFO2:
	  SKIP(I_AM_CLIENT_SERVER);
	  client_input_info(buf, len);
	  return PROCESSED;
      case COMMAND_INPUT_CHAT:
	  client_input_chat(buf, len);
	  return PROCESSED;
      case COMMAND_INPUT_EDITOR: {
	  struct packet_editor packet;

	  packet_editor_decode(&packet, buf, len);

	  if (packet.mode == EDITOR_ENABLE) {
	      state->editor_enabled = true;
	      return PROCESSED;
	  }

	  if (packet.mode == EDITOR_DISABLE) {
	      state->editor_enabled = false;
	      return PROCESSED;
	  }

	  return UNKNOWN_COMMAND;
      }
      case COMMAND_DISCONNECT:
	  SKIP(I_AM_CLIENT_SERVER);
	  client_disconnect(buf, len, state);
	  return PROCESSED;
      default:
	  return UNKNOWN_COMMAND;
    }

#undef SKIP
}


static void client_input_info(const NLbyte *buf, const NLint len)
{
    struct packet_input_info packet;
    client_data_t *data;
    int idx;
 
    packet_input_info_decode(&packet, buf[0], buf, len);
 
    idx = sv_intern_unused_client();
    if (idx < 0) {
	chat_add(server_time, CHAT_CLIENT_ERROR,
		 "Couldn't add %s to client list!", packet.name);
	chat_add(server_time, CHAT_CLIENT_ERROR,
		 "Things will probably go horribly wrong.");
	return;
    }
 
    data = &client_data[idx];
    data->clid = packet.clid;
    strlcpy(data->name, packet.name, sizeof(data->name));
    data->type = packet.type;
    data->face_num = packet.face;
    data->sc = packet.sc;

    chat_add(server_time, CHAT_SERVER_LOG,
	     "%s (client %d) enters the game",
	     packet.name, packet.clid);

    sv_intern_sort_by_score();
}


static void client_input_chat(const NLbyte *buf, const NLint len)
{
    char chat[MAX_CHAT_LEN];

    packet_input_chat_decode(chat, buf, len);
    chat_add(server_time, CHAT_BABBLE, chat);
}


static void client_disconnect(const NLbyte *buf, const NLint len,
			      game_state_t *state)
{
    struct packet_disconnect packet;
    client_data_t *data;
    player_t *p;
    char *reason;
    int idx;
    assert(state);

    packet_disconnect_decode(&packet, buf, len);

    idx = sv_intern_client_from_client_id(packet.clid);
    if (idx >= 0) {
	data = &client_data[idx];
    }
    else {
	chat_add(server_time, CHAT_CLIENT_ERROR,
		 "Client %d not found.  Disconnect aborted");
	return;
    }

    p = player_from_id(data->id, state);
    if (p) {
	player_free(p);
	p = NULL;
    }

    data->sock = NL_INVALID;
    data->clid = 0;
    data->id = 0;
    data->type = CLIENT_UNUSED;

    switch (packet.why) {
      case DISCONNECT_REQUEST:
	  reason = "has quit";
	  break;
      case DISCONNECT_LAG:
	  reason = "was disconnected due to lag";
	  break;
      case DISCONNECT_JERK:
	  reason = "was booted";
	  break;
      default:
	  reason = "was disconnected.  Unknown reason";
	  break;
    }

    chat_add(server_time, CHAT_SERVER_LOG, "%s %s",
	     data->name, reason);
}

/*--------------------------------------------------------------*/
/* Commands in game only.					*/
/*--------------------------------------------------------------*/

static void client_set_start_loc	(const NLbyte *buf, const NLint len);
static void client_set_clock		(const NLbyte *buf, const NLint len);
static void client_player_new		(const NLbyte *buf, const NLint len,
					 game_state_t *state);
static void client_player_attach	(const NLbyte *buf, const NLint len,
					 game_state_t *state);
static void client_player_update	(const NLbyte *buf, const NLint len,
					 game_state_t *state);
static void client_player_respawn	(const NLbyte *buf, const NLint len,
					 game_state_t *state);
static void client_player_die		(const NLbyte *buf, const NLint len,
					 game_state_t *state);
static void client_bullet_new		(const NLbyte *buf, const NLint len,
					 game_state_t *state);
static void client_bullet_die		(const NLbyte *buf, const NLint len,
					 game_state_t *state);
static void client_pickup_new		(const NLbyte *buf, const NLint len);
static void client_pickup_quaff		(const NLbyte *buf, const NLint len,
					 game_state_t *state);
static void client_backpack_new		(const NLbyte *buf, const NLint len,
					 game_state_t *state);
static void client_backpack_taken	(const NLbyte *buf, const NLint len,
					 game_state_t *state);
static void client_explosion_new	(const NLbyte *buf, const NLint len,
					 game_state_t *state);
static void client_explosion_die	(const NLbyte *buf, const NLint len,
					 game_state_t *state);
static void client_container_new	(const NLbyte *buf, const NLint len);
static void client_container_show	(const NLbyte *buf, const NLint len);
static void client_container_hide	(const NLbyte *buf, const NLint len);
static void client_gizmo_new		(const NLbyte *buf, const NLint len);
static void client_candela_new		(const NLbyte *buf, const NLint len);
static void client_bleeding		(const NLbyte *buf, const NLint len);
static void client_editor		(const NLbyte *buf, const NLint len,
					 game_state_t *state);


static bool client_process_game_commands(const NLbyte *buf, const NLint len,
					 game_state_t *state)
{
#define SKIP(mode)	if (client_server_mode == mode) return true
#define TRY(x)		if (state->context == SERVER_IN_GAME) x

    assert(state);

    switch (buf[0]) {
      case COMMAND_SET_CLOCK:
	  TRY(client_set_clock(buf, len));
	  return true;
      case COMMAND_MAP:
	  SKIP(I_AM_CLIENT_SERVER);
	  TRY(packet_map_decode(&map, buf, len, state));
	  return true;
      case COMMAND_SET_ZERO:
	  SKIP(I_AM_CLIENT_SERVER);
	  TRY(packet_set_zero_decode(buf, len));
	  return true;
      case COMMAND_START_LOC:
	  SKIP(I_AM_CLIENT_SERVER);
	  TRY(client_set_start_loc(buf, len));
	  return true;
      case COMMAND_START_LOCS:
	  SKIP(I_AM_CLIENT_SERVER);
	  TRY(packet_start_locs_decode(starts, buf, len));
	  return true;

      case COMMAND_PLAYER_NEW:
	  TRY(client_player_new(buf, len, state));
	  return true;
      case COMMAND_PLAYER_ATTACH:
	  TRY(client_player_attach(buf, len, state));
	  return true;
      case COMMAND_PLAYER_UPDATEL:
      case COMMAND_PLAYER_UPDATEL2:
      case COMMAND_PLAYER_UPDATER:
      case COMMAND_PLAYER_UPDATER2:
	  TRY(client_player_update(buf, len, state));
	  return true;
      case COMMAND_PLAYER_RESPAWN:
	  TRY(client_player_respawn(buf, len, state));
	  return true;
      case COMMAND_PLAYER_DIE:
	  TRY(client_player_die(buf, len, state));
	  return true;

      case COMMAND_BULLET_NEW:
	  TRY(client_bullet_new(buf, len, state));
	  return true;
      case COMMAND_BULLET_DIE:
	  TRY(client_bullet_die(buf, len, state));
	  return true;

      case COMMAND_PICKUP_NEW:
	  SKIP(I_AM_CLIENT_SERVER);
	  TRY(client_pickup_new(buf, len));
	  return true;
      case COMMAND_PICKUP_QUAFF:
	  TRY(client_pickup_quaff(buf, len, state));
	  return true;

      case COMMAND_BACKPACK_NEW:
	  TRY(client_backpack_new(buf, len, state));
	  return true;
      case COMMAND_BACKPACK_TAKE:
      case COMMAND_BACKPACK_TAKEN:
	  TRY(client_backpack_taken(buf, len, state));
	  return true;

      case COMMAND_EXPLOSION_NEW:
	  TRY(client_explosion_new(buf, len, state));
	  return true;
      case COMMAND_EXPLOSION_DIE:
	  TRY(client_explosion_die(buf, len, state));
	  return true;

      case COMMAND_CONTAINER_NEW:
	  SKIP(I_AM_CLIENT_SERVER);
	  TRY(client_container_new(buf, len));
	  return true;
      case COMMAND_CONTAINER_SHOW:
	  SKIP(I_AM_CLIENT_SERVER);
	  TRY(client_container_show(buf, len));
	  return true;
      case COMMAND_CONTAINER_HIDE:
	  /* Do not skip on client-server since we want debris. */
	  TRY(client_container_hide(buf, len));
	  return true;

      case COMMAND_GIZMO_NEW:
	  SKIP(I_AM_CLIENT_SERVER);
	  TRY(client_gizmo_new(buf, len));
	  return true;

      case COMMAND_CANDELA_NEW:
	  SKIP(I_AM_CLIENT_SERVER);
	  TRY(client_candela_new(buf, len));
	  return true;

      case COMMAND_BLEED:
	  TRY(client_bleeding(buf, len));
	  return true;

      case COMMAND_INPUT_EDITOR:
	  TRY(client_editor(buf, len, state));
	  return true;

      default:
	  return false;
    }

#undef TRY
#undef SKIP
}


static void client_set_clock(const NLbyte *buf, const NLint len)
{
    struct packet_set_clock packet;

    packet_set_clock_decode(&packet, buf, len);
    server_time = packet.time;
}


static void client_set_start_loc(const NLbyte *buf, const NLint len)
{
    struct packet_start_loc packet;

    packet_start_loc_decode(&packet, buf, len);
    start_loc_set(&packet);
}


static void client_player_new(const NLbyte *buf, const NLint len,
			      game_state_t *state)
{
    struct packet_player_new packet;
    NLint count;
    int num, idx;
    assert(state);

    count = packet_player_new_begin_decode(&num, buf, len);
    while (num > 0) {
	count = packet_player_new_decode(&packet, buf, len, count);

	idx = sv_intern_client_from_client_id(packet.clid);
	if (idx < 0) {
	    chat_add(server_time, CHAT_CLIENT_ERROR,
		     "Client %d not found!  Player %d not created",
		     packet.clid, packet.id);
	    return;
	}

	client_data[idx].id = packet.id;
	player_new(&packet, state);
	num--;
    }
}


static void client_player_attach(const NLbyte *buf, const NLint len,
				 game_state_t *state)
{
    struct packet_player_attach packet;
    player_t *p;
    assert(state);

    packet_player_attach_decode(&packet, buf, len);
    p = player_from_id(packet.id, state);

    if (!p) {
	chat_add(server_time, CHAT_CLIENT_ERROR,
		 "Player %d not found!  Attach aborted",
		 packet.id);
	return;
    }

    player = p;
    camera.x = player->x-320.0;
    camera.y = player->y-240.0;

    chat_add(server_time, CHAT_GAME_MESSAGE,
	     "You are player %d",
	     player->id);
}


static void client_player_update(const NLbyte *buf, const NLint len,
				 game_state_t *state)
{
    struct packet_player_update packet;
    bool local = (buf[0] == COMMAND_PLAYER_UPDATEL ||
		  buf[0] == COMMAND_PLAYER_UPDATEL2);
    bool large = (buf[0] == COMMAND_PLAYER_UPDATEL2 ||
		  buf[0] == COMMAND_PLAYER_UPDATER2);
    player_t *p;
    assert(state);

    packet_player_update_decode(&packet, local, large, buf, len);

    if (local) {
	p = player;
	assert(p);
    }
    else {
	p = player_from_id(packet.id, state);
	if (!p) {
	    chat_add(server_time, CHAT_CLIENT_ERROR,
		     "Player %d not found.  Update aborted",
		     packet.id);
	    return;
	}
    }

    player_save_history(p, &packet);
    p->x = packet.x;
    p->y = packet.y;
    p->packet_size = packet.size;

    if (!large)
	return;

    if (packet.size & PACKET_INCLUDE_WEAPON) {
	p->weapon = packet.weapon;
	p->frame = 0;
	p->frame_delay = 0;
    }

    if (packet.size & PACKET_INCLUDE_ANGLE) {
	p->mirror = packet.mirror;
	p->angle = packet.angle;
    }

    if (player->weapon != WEAPON_BLASTER) {
	if (packet.size & PACKET_INCLUDE_AMMO)
	    player->weapon_ammo[player->weapon] = packet.ammo;
	else if (packet.size & PACKET_INCLUDE_FIRING)
	    player->weapon_ammo[player->weapon]--;
    }

    if (packet.size & PACKET_INCLUDE_HEALTH)
	player->health = packet.health;

    if (packet.size & PACKET_INCLUDE_TRACKER)
	player->tracker_angle = packet.tracker;

    if (packet.size & PACKET_INCLUDE_BLEED) {
	packet.y += 19.0;
	particle_new(PARTICLE_BLOOD, packet.x, packet.y, packet.time);
	particle_new(PARTICLE_BLOOD, packet.x, packet.y, packet.time+50);
	particle_new(PARTICLE_BLOOD, packet.x, packet.y, packet.time+100);
    }
}


static void client_player_respawn(const NLbyte *buf, const NLint len,
				  game_state_t *state)
{
    struct packet_player_respawn packet;
    player_t *p;
    assert(state);

    packet_player_respawn_decode(&packet, buf, len);

    p = player_from_id(packet.id, state);
    if (!p) {
	struct packet_player_new new_packet;
	new_packet.id = packet.id;
	start_loc_get(packet.loc, &new_packet.x, &new_packet.y);
	player_new(&new_packet, state);
	return;
    }

    if (p->alive) {
	chat_add(server_time, CHAT_CLIENT_ERROR,
		 "Player %d (%s) is alive!  Respawn aborted",
		 packet.id, client_name_from_id(packet.id));
    }

    start_loc_get(packet.loc, &p->x, &p->y);
    player_new_life(p);
    particle_new(PARTICLE_RESPAWNING, p->x, p->y+19.0, server_time);

    if (p == player) {
	camera.max_dist = CAMERA_DEFAULT_MAX_DIST;
    }
}


static void client_player_die(const NLbyte *buf, const NLint len,
			      game_state_t *state)
{
    struct packet_player_die packet;
    player_t *p;
    assert(state);

    packet_player_die_decode(&packet, buf, len);

    p = player_from_id(packet.id, state);
    if (!p) {
	chat_add(server_time, CHAT_CLIENT_ERROR,
		 "Player %d not found!  Die aborted",
		 packet.id);
	return;
    }

    p->alive = false;
    p->health = 0;
    p->x = packet.x;
    p->y = packet.y;
    meat_new(MEAT_FROM_PLAYER, p->x, p->y+19.0,
	     packet.vx, packet.vy, packet.time);

    if (packet.bp)
	backpack_new(&packet, state);

    if (p == player) {
	camera.max_dist = CAMERA_DEFAULT_MAX_DIST/2;
    }

    if (client_server_mode != I_AM_CLIENT_SERVER) {
	score(packet.murderer, packet.id);
    }

    if ((player) && (player->alive) && (packet.murderer == player->id)) {
	player->kills_this_life++;
    }

    switch (packet.why) {
      case DEATH_BY_FALL:
	  chat_add(packet.time, CHAT_GAME_MESSAGE,
		   "%s fell into a bottomless pit",
		   client_name_from_id(packet.id));
	  break;
      case DEATH_BY_SUICIDE:
	  chat_add(packet.time, CHAT_GAME_MESSAGE,
		   "%s committed suicide",
		   client_name_from_id(packet.id));
	  break;
      case DEATH_BY_OPPONENT:
	  chat_add(packet.time, CHAT_GAME_MESSAGE,
		   "%s killed %s",
		   client_name_from_id(packet.murderer),
		   client_name_from_id(packet.id));
	  break;
      default:
	  chat_add(packet.time, CHAT_GAME_MESSAGE,
		   "%s died.  Cause unknown",
		   client_name_from_id(packet.id));
	  break;
    }
}


static void client_bullet_new(const NLbyte *buf, const NLint len,
			      game_state_t *state)
{
    struct packet_bullet_new packet;
    player_t *p;
    NLint count;
    int num;
    assert(state);

    count = packet_bullet_new_begin_decode(&num, buf, len);

    while (num > 0) {
	count = packet_bullet_new_decode(&packet, buf, len, count);
	bullet_new(&packet, state);
	play_weapon_fire(packet.class, packet.parent, packet.x, packet.y);

	p = player_from_id(packet.parent, state);
	if (p) {
	    if ((p->frame == 0) ||
		(!weapon_is_automatic(packet.class)))
		p->frame = 1;
	}
	num--;
    }
}


static void client_bullet_die(const NLbyte *buf, const NLint len,
			      game_state_t *state)
{
    struct packet_bullet_die packet;
    NLint count;
    int num;
    bullet_t *b;

    count = packet_bullet_die_begin_decode(&num, buf, len);
    while (num > 0) {
	count = packet_bullet_die_decode(&packet, buf, len, count);

	b = bullet_from_id(packet.id, state);
	if (!b) {
	    chat_add(server_time, CHAT_CLIENT_ERROR,
		     "Bullet %d not found!  Die aborted",
		     packet.id);
	    num--;
	    continue;
	}

	bullet_move(b, packet.time);
	particle_new(packet.particles, b->x, b->y, b->time);

	if (packet.particles == PARTICLE_BLOOD) {
	    float dx = b->x - (camera.x+640/2);
	    float dy = b->y - (camera.y+480/2);
	    play_player_hurt(dx*dx + dy*dy, dx);
	}

	if (weapon_is_explosive(b->class))
	    smoke_new(weapon[b->class].explosion.boom, b->x, b->y,
		      0.0, 0.0, b->time);

	if (b->sound_source_n > 0) {
	    stop_missile_flight(b->sound_source_n);
	}

	bullet_free(b);
	num--;
    }

    assert(count == len);
}


static void client_pickup_new(const NLbyte *buf, const NLint len)
{
    struct packet_pickup_new packet;
    NLint count;
    int num;

    count = packet_pickup_new_begin_decode(&num, buf, len);
    while (num > 0) {
	count = packet_pickup_new_decode(&packet, buf, len, count);

	pickup_new(&packet);
	num--;
    }

    assert(count == len);
}


static void client_pickup_quaff(const NLbyte *buf, const NLint len,
				game_state_t *state)
{
    struct packet_pickup_quaff packet;
    pickup_t *p;
    player_t *q;
    assert(state);

    packet_pickup_quaff_decode(&packet, buf, len);

    p = pickup_from_id(packet.id);
    if (!p) {
	chat_add(server_time, CHAT_CLIENT_ERROR,
		 "Pickup %d not found!  Quaff aborted",
		 packet.id);
	return;
    }

    q = player_from_id(packet.quaffer, state);
    if (!q) {
	chat_add(server_time, CHAT_CLIENT_ERROR,
		 "Player %d not found!  Pickup %d not quaffed",
		 packet.quaffer, packet.id);
	return;
    }

    pickup_quaff_by(p, q, packet.time);
}


static void client_backpack_new(const NLbyte *buf, const NLint len,
				game_state_t *state)
{
    struct packet_backpack_new packet;
    struct packet_player_die dummy;
    NLint count;
    int num;
    assert(state);

    count = packet_backpack_new_begin_decode(&num, buf, len);

    while (num > 0) {
	count = packet_backpack_new_decode(&packet, buf, len, count);

	dummy.time = packet.time;
	dummy.id = 0;
	dummy.why = DEATH_UNKNOWN_CAUSE;
	dummy.x = packet.x;
	dummy.y = packet.y;
	dummy.vx = packet.vx;
	dummy.vy = packet.vy;
	dummy.bp = packet.id;
	dummy.murderer = 0;

	backpack_new(&dummy, state);
	num--;
    }

    assert(count == len);
}


static void client_backpack_taken(const NLbyte *buf, const NLint len,
				  game_state_t *state)
{
    struct packet_backpack_taken packet;
    bool local = (buf[0] == COMMAND_BACKPACK_TAKE);
    backpack_t *b;
    assert(state);

    packet_backpack_taken_decode(&packet, local, buf, len);

    b = backpack_from_id(packet.id, state);
    if (!b) {
	chat_add(server_time, CHAT_CLIENT_ERROR,
		 "Backpack %d not found!  %s aborted",
		 packet.id, (local ? "Take" : "Theft"));
	return;
    }

    if (local) {
	enum WEAPON_CLASS w;
	assert(player);

	for (w = WEAPON_FIRST; w < NUM_WEAPON_CLASSES; w++) {
	    if (player->weapon_ammo[w] == 0)
		is_new_ammo[w] = true;

	    player->weapon_ammo[w] += packet.ammo[w];
	}

	if (b->parent == player->id)
	    chat_add(packet.time, CHAT_GAME_MESSAGE,
		     "You got your backpack back");
	else
	    chat_add(packet.time, CHAT_GAME_MESSAGE,
		     "You stole a backpack");
    }

    backpack_free(b);
}


static void client_explosion_new(const NLbyte *buf, const NLint len,
				 game_state_t *state)
{
    struct packet_explosion_new packet;
    assert(state);

    packet_explosion_new_decode(&packet, buf, len);
    explosion_new(&packet, 0, state);
    play_explosion(packet.x, packet.y);
}


static void client_explosion_die(const NLbyte *buf, const NLint len,
				 game_state_t *state)
{
    struct packet_explosion_die packet;
    explosion_t *e;
    assert(state);

    packet_explosion_die_decode(&packet, buf, len);

    e = explosion_from_id(packet.id, state);
    if (!e) {
	chat_add(server_time, CHAT_CLIENT_ERROR,
		 "Explosion %d not found!  Die aborted",
		 packet.id);
	return;
    }

    explosion_free(e);
}


static void client_container_new(const NLbyte *buf, const NLint len)
{
    struct packet_container_new packet;
    NLint count;
    int num;

    count = packet_container_new_begin_decode(&num, buf, len);

    while (num > 0) {
	count = packet_container_new_decode(&packet, buf, len, count);

	container_new(&packet);
	num--;
    }

    assert(count == len);
}


static void client_container_show(const NLbyte *buf, const NLint len)
{
    struct packet_container_show packet;
    container_t *c;

    packet_container_show_decode(&packet, buf, len);
    c = container_from_id(packet.id);
    assert(c);

    c->hidden = false;
    c->respawn_time = 0;
}


static void client_container_hide(const NLbyte *buf, const NLint len)
{
    struct packet_container_hide packet;
    const container_stats_t *stats;
    container_t *c;
    int y;

    packet_container_hide_decode(&packet, buf, len);
    c = container_from_id(packet.id);
    assert(c);
    assert(c->class < NUM_CONTAINER_CLASSES);

    stats = &container_stats[c->class];

    c->hidden = true;
    c->respawn_time = packet.time + stats->respawn_time;

    y = c->y + container_sprite[c->class].h/2;
    meat_new(stats->meat, c->x, y, 0.0, 0.0, packet.time);
    smoke_new(stats->boom, c->x, y, 0.0, 0.0, packet.time);
}


static void client_gizmo_new(const NLbyte *buf, const NLint len)
{
    struct packet_gizmo_new packet;
    NLint count;
    int num;

    count = packet_gizmo_new_begin_decode(&num, buf, len);

    while (num > 0) {
	count = packet_gizmo_new_decode(&packet, buf, len, count);

	gizmo_new(&packet);
	num--;
    }

    assert(count == len);
}


static void client_candela_new(const NLbyte *buf, const NLint len)
{
    struct packet_candela_new packet;
    candela_t *c;
    NLint count;
    int num;

    count = packet_candela_new_begin_decode(&num, buf, len);

    while (num > 0) {
	count = packet_candela_new_decode(&packet, buf, len, count);

	c = candela_from_id(packet.id);
	if (c) {
	    candela_set_parameters(c, &packet);
	}
	else {
	    candela_new(&packet);
	}

	num--;
    }

    assert(count == len);
}


static void client_bleeding(const NLbyte *buf, const NLint len)
{
    struct packet_bleeding packet;

    packet_bleeding_decode(&packet, buf, len);

    particle_new(PARTICLE_BLOOD, packet.x, packet.y, packet.time);
    particle_new(PARTICLE_BLOOD, packet.x, packet.y, packet.time+50);
    particle_new(PARTICLE_BLOOD, packet.x, packet.y, packet.time+100);
}


static void client_editor(const NLbyte *buf, const NLint len,
			  game_state_t *state)
{
    struct packet_editor packet;
    assert(state);

    packet_editor_decode(&packet, buf, len);

    if (client_server_mode == I_AM_CLIENT_SERVER) {
	/* All commands below are handled by the server. */
	return;
    }

    switch (packet.mode) {
      case EDIT_TILE_SET:
	  map_set_tiles(packet.new.tile.x, packet.new.tile.y,
			packet.new.tile.tile, packet.new.tile.brush_size,
			state);
	  break;
      case EDIT_CONTAINER_MOVE:
	  container_move(&packet.new.container_mov);
	  break;
      case EDIT_CONTAINER_DELETE:
	  container_delete(&packet.new.container_del);
	  break;
      case EDIT_GIZMO_MOVE:
	  gizmo_move(&packet.new.gizmo_mov);
	  break;
      case EDIT_GIZMO_DELETE:
	  gizmo_delete(&packet.new.gizmo_del);
	  break;
      case EDIT_PICKUP_MOVE:
	  pickup_move(&packet.new.pickup_mov);
	  break;
      case EDIT_PICKUP_DELETE:
	  pickup_delete(&packet.new.pickup_del);
	  break;
      case EDIT_START_DELETE:
	  start_loc_unset(&packet.new.start_del);
	  break;
      case EDIT_CANDELA_DELETE:
	  candela_delete(packet.new.candela_del.id);
	  break;
      case EDIT_REGION_SET:
	  map_set_regions(packet.new.region.x, packet.new.region.y,
			  packet.new.region.region,
			  packet.new.region.brush_size);
	  break;
      case EDITOR_ENABLE:
      case EDITOR_DISABLE:
      case EDIT_CONTAINER_ADD:
      case EDIT_GIZMO_ADD:
      case EDIT_PICKUP_ADD:
      case EDIT_START_ADD:
      case EDIT_CANDELA_ADD:
      case EDIT_PLAYER_TELEPORT:
      case EDIT_UNKNOWN:
      default:
	  chat_add(server_time, CHAT_CLIENT_ERROR,
		   "Unknown editor mode/action: %d",
		   packet.mode);
    }
}

/*--------------------------------------------------------------*/

void client_incoming(game_state_t *state)
{
    NLbyte packet[MAX_PACKET_SIZE];
    NLint len;
    assert(state);

    while ((len = nlRead(sock, &packet, sizeof(packet))) > 0) {
	enum COMMAND_PROCESS_DONE done;

	done = client_process_common_commands(packet, len, state);
	if (done == PROCESSED)
	    continue;

	if (done == PROCESSED_AND_EXIT)
	    break;

	if (client_process_game_commands(packet, len, state))
	    continue;

	chat_add(server_time, CHAT_CLIENT_ERROR,
		 "Unrecognised command: %c",
		 packet[0]);
    }

    if (len == NL_INVALID) {
	client_print_error();
	state->context = SERVER_QUIT;
    }
}

/*--------------------------------------------------------------*/

void client_outgoing(const player_t *p, const int impy,
		     const enum WEAPON_CLASS select_weapon)
{
    const struct packet_input_move packet = {
	p->id, impy, p->mirror, p->angle, select_weapon
    };
    NLbyte buf[MAX_PACKET_SIZE];
    NLint len;
    char command;

    if (p->weapon == select_weapon) {
	command = COMMAND_INPUT_MOVE;
    }
    else {
	command = COMMAND_INPUT_WIELD;
    }

    len = packet_input_move_encode(&packet, command, buf, sizeof(buf));
    client_write(buf, len);
}


void client_request_disconnect(void)
{
    const char command = COMMAND_INPUT_DISCONNECT;

    client_write(&command, sizeof(command));
}

/*--------------------------------------------------------------*/

/* client_init:
 *
 * Try to connect to addr.  The port is optional, and if not given, is
 * assumed to be 25000.
 */
bool client_init(const char *name, const char *addr_str)
{
    struct packet_input_info packet;
    NLaddress addr;
    NLchar addr_buf[NL_MAX_STRING_LENGTH];
    NLbyte buf[MAX_PACKET_SIZE];
    NLint len;
    assert(addr_str);

    if ((sock = nlOpen(0, type)) == NL_INVALID)
	goto error;
    
    if (nlGetAddrFromName(addr_str, &addr) != NL_TRUE)
	goto error;

    if (nlGetPortFromAddr(&addr) == 0) {
	if (nlSetAddrPort(&addr, 25000) != NL_TRUE)
	    goto error;
    }

    if (nlConnect(sock, &addr) != NL_TRUE)
	goto error;

    fprintf(stdout, "[Client] Connected to %s\n",
	    nlAddrToString(&addr, addr_buf));

    /* Send a packet about us. */
    strlcpy(packet.name, name, sizeof(packet.name));
    packet.type = CLIENT_GAMER;
    packet.face = rand() % NUM_HUMAN_FACES;
    len = packet_input_info_encode(&packet, COMMAND_INPUT_INFO,
				   buf, sizeof(buf));
    
    if (client_write(buf, len) == false)
	goto error;

    if (client_server_mode == I_AM_CLIENT_ONLY)
	sv_intern_init();

    return true;

 error:
    if (sock != NL_INVALID) {
	nlClose(sock);
	sock = NL_INVALID;
    }

    client_print_error();
    return false;
}


void client_shutdown(void)
{
    int i;

    assert(sock != NL_INVALID);
    if (nlClose(sock) == NL_TRUE) {
	sock = NL_INVALID;
    }
    else {
	client_print_error();
    }

    if (client_server_mode == I_AM_CLIENT_ONLY) {
	for (i = 0; i < MAX_CLIENTS; i++) {
	    client_data[i].sock = NL_INVALID;
	    client_data[i].type = CLIENT_UNUSED;
	}
    }
}
