/* candela.c,
 *
 * Light sources.  Candela are GL_LIGHTS unlike luxes which are not.
 */

#include <alleggl.h>
#include <allegro.h>
#include <assert.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include "angle.h"
#include "camera.h"
#include "candela.h"
#include "common.h"
#include "sprite.h"
#include "texdraw.h"


static BITMAP *bmp_candela;
static GLuint tex_candela;

static int candela_count;
list_head_t candela_list;	/* Shared */

/*--------------------------------------------------------------*/

static void candela_position_unit(const candela_t *c, const GLenum light);


GLenum candela_position(GLenum light)
{
    candela_t *c;

    if (light >= GL_LIGHT0+GL_MAX_LIGHTS) {
	return light;
    }

    list_for_each(c, &candela_list) {
	if (!circle_in_sight(&camera, c->posi[0], c->posi[1], c->r))
	    continue;
	    
	candela_position_unit(c, light);

	light++;
	if (light >= GL_LIGHT0+GL_MAX_LIGHTS)
	    break;
    }

    return light;
}


static void candela_position_unit(const candela_t *c, const GLenum light)
{
    assert(c);

    glLightiv(light, GL_POSITION, c->posi);
    glLightfv(light, GL_AMBIENT, c->ambi);
    glLightfv(light, GL_DIFFUSE, c->ambi);
    glLightfv(light, GL_SPECULAR, glBlack);
    glLightf(light, GL_SPOT_CUTOFF, c->cutoff);
    glLighti(light, GL_SPOT_EXPONENT, c->exponent);

    glEnable(light);
}

/*--------------------------------------------------------------*/

static void candela_draw_unit(const int x, const int y);
static void candela_draw_colour_wheel(const int x, const int y);


void candela_draw(void)
{
    const GLfloat *last = glWhite;
    candela_t *c;
    int i, col;

    if (list_empty(candela_list))
	return;

    col = makecol(0xa0, 0xa0, 0xa0);
    glBindTexture(GL_TEXTURE_2D, tex_candela);
    glBegin(GL_QUADS);

    list_for_each(c, &candela_list) {
	sprite_tint_col(c->highlight, false, &last);
	candela_draw_unit(c->posi[0]-CANDELA_W_2, c->posi[1]-CANDELA_H_2);

	if (c->highlight) {
	    glEnd();		/* glBegin(GL_QUADS) */
	    candela_draw_colour_wheel(c->posi[0], c->posi[1]);
	    glColor3fv(glWhite);
	    last = glWhite;
	    glBegin(GL_QUADS);
	}
    }

    glEnd();			/* glBegin(GL_QUADS) */

    list_for_each(c, &candela_list) {
	allegro_gl_printf(al_font, c->posi[0]-10.0, c->posi[1]-10.0, 0.0, col,
			  "%d", c->exponent);
    }

    glDisable(GL_TEXTURE_2D);
    glBegin(GL_LINES);
    glColor3fv(glWhite);

    list_for_each(c, &candela_list) {
	for (i = 0; i < 360; i += 10) {
	    glVertex2f(c->posi[0] + c->r*cos(deg2rad(i)),
		       c->posi[1] + c->r*sin(deg2rad(i)));
	}
    }

    glEnd();			/* glBegin(GL_LINES) */
    glEnable(GL_TEXTURE_2D);
}


void candela_draw_bulb(const int x, const int y)
{
    glBindTexture(GL_TEXTURE_2D, tex_candela);
    glBegin(GL_QUADS);
    candela_draw_unit(x, y);
    glEnd();			/* glBegin(GL_QUADS) */
}


static void candela_draw_unit(const int x, const int y)
{
    const texcoord2d_t coord = {
	29.0, 30.0,
	1.0/32.0, 31.0/32.0, 1.0/32.0, 30.0/32.0
    };

    gl_draw_sprite_2d(&coord, x, y);
}


static void candela_draw_colour_wheel(const int x, const int y)
{
    GLfloat X0 = 105.0;
    GLfloat X1 = 105.0/2.0;
    GLfloat Y0 = 0;
    GLfloat Y1 = 105*M_SQRT3_2;

    glDisable(GL_TEXTURE_2D);
    glDisable(GL_LIGHTING);

    glColor3fv(glRed);
    glRectf(x+X0-5.0, y+Y0-5.0, x+X0+5.0, y+Y0+5.0);

    glColor3fv(glYellow);
    glRectf(x+X1-5.0, y+Y1-5.0, x+X1+5.0, y+Y1+5.0);

    glColor3fv(glGreen);
    glRectf(x-X1-5.0, y+Y1-5.0, x-X1+5.0, y+Y1+5.0);

    glColor3fv(glCyan);
    glRectf(x-X0-5.0, y+Y0-5.0, x-X0+5.0, y+Y0+5.0);

    glColor3fv(glBlue);
    glRectf(x-X1-5.0, y-Y1-5.0, x-X1+5.0, y-Y1+5.0);

    glColor3fv(glMagenta);
    glRectf(x+X1-5.0, y-Y1-5.0, x+X1+5.0, y-Y1+5.0);

    glEnable(GL_LIGHTING);
    glEnable(GL_TEXTURE_2D);
}

/*--------------------------------------------------------------*/

candela_t *candela_from_id(const candela_id id)
{
    candela_t *c;

    list_for_each(c, &candela_list) {
	if (c->id == id)
	    return c;
    }

    return NULL;
}


candela_id candela_bounding(const int x, const int y)
{
    candela_t *c;

    list_for_each(c, &candela_list) {
	if ((c->posi[0]-CANDELA_W_2 <= x) && (x <= c->posi[0]+CANDELA_W_2) &&
	    (c->posi[1]-CANDELA_H_2 <= y) && (y <= c->posi[1]+CANDELA_H_2))
	    return c->id;
    }

    return 0;
}


/* candela_radius_bounding:
 *
 * Find the candela whose radius (with thickness DELTA_R) contains the
 * point (x, y).
 */
candela_id candela_radius_bounding(const int x, const int y)
{
#define DELTA_R	10.0

    candela_t *c;

    list_for_each(c, &candela_list) {
	double dx, dy, rr, r1, r2;

	dx = x - c->posi[0];
	dy = y - c->posi[1];
	rr = dx*dx + dy*dy;

	r1 = c->r-DELTA_R/2;
	r2 = c->r+DELTA_R/2;

	if ((r1*r1 <= rr) && (rr <= r2*r2))
	    return c->id;
    }

    return 0;

#undef DELTA_R
}

/*--------------------------------------------------------------*/

candela_id candela_generate_unique_id(void)
{
    static candela_id id;
    id++;
    return id;
}


int candela_total(void)
{
    return candela_count;
}


void candela_new(const struct packet_candela_new *packet)
{
    candela_t *c = malloc(sizeof(*c));
    assert(c);
    assert(packet);

    c->id = packet->id;
    c->posi[0] = 0.0; c->posi[1] = 0.0; c->posi[2] = 100.0; c->posi[3] = 1.0;
    c->ambi[0] = 1.0; c->ambi[1] = 1.0; c->ambi[2] = 1.0; c->ambi[3] = 1.0;
    c->cutoff = 45.0;
    c->exponent = 0;
    c->highlight = false;
    candela_set_parameters(c, packet);

    list_add(candela_list, c);
    candela_count++;
}


void candela_set_parameters(candela_t *c,
			    const struct packet_candela_new *packet)
{
    int i;
    assert(c);
    assert(packet);

    if (packet->size & CANDELA_INCLUDE_POSITION) {
	for (i = 0; i < 2; i++)
	    c->posi[i] = packet->posi[i];
    }

    if (packet->size & CANDELA_INCLUDE_AMBIENT) {
	for (i = 0; i < 3; i++)
	    c->ambi[i] = packet->ambi[i];
    }

    if (packet->size & CANDELA_INCLUDE_CUTOFF)
	c->cutoff = packet->cutoff;

    if (packet->size & CANDELA_INCLUDE_EXPONENT)
	c->exponent = packet->exponent;

    if (c->cutoff >= 90.0) {
	c->r = 640.0;
    }
    else {
	/* Make the circle radius slightly larger than it actually is. */
	c->r = c->posi[2]*tan(deg2rad(c->cutoff)) + 8.0;
    }
}


void candela_shift(const GLint offx, const GLint offy)
{
    candela_t *c;

    list_for_each(c, &candela_list) {
	c->posi[0] += offx;
	c->posi[1] += offy;
    }
}


static void candela_free(candela_t *c)
{
    free(c);
    candela_count--;
}


void candela_delete(const candela_id id)
{
    candela_t *c = candela_from_id(id);

    c = candela_from_id(id);
    if (!c) {
	fprintf(stderr, "[Candela] Candela %d not found!  Delete failed\n",
		id);
	return;
    }

    list_remove(c);
    candela_free(c);
}

/*--------------------------------------------------------------*/

void candela_start(void)
{
    list_init(candela_list);
    candela_count = 0;
}


void candela_stop(void)
{
    list_free(candela_list, candela_free);
    assert(candela_count == 0);
}


void candela_init(void)
{
    bmp_candela = load_bitmap("data/bulb.tga", NULL);
    assert(bmp_candela);

    tex_candela = allegro_gl_make_texture_ex(AGL_TEXTURE_MASKED,
					     bmp_candela, -1);
}


void candela_shutdown(void)
{
    if (bmp_candela) {
	destroy_bitmap(bmp_candela);
	bmp_candela = NULL;
    }
}
