/* camera.c,
 *
 * Camera code, from Red Pixel II.
 */

#include <assert.h>
#include <math.h>
#include "camera.h"


#define MIN(x,y)	(((x) < (y)) ? (x) : (y))
#define MAX(x,y)	(((x) > (y)) ? (x) : (y))
#define SQ(x)		((x) * (x))

#define PUSH_MARGIN	15
#define STAY_MARGIN	50
#define REVERT_MARGIN	80
#define PUSH_SPEED	2.5
#define PUSH_DECAY	0.98


camera_t camera;

/*--------------------------------------------------------------*/

bool camera_can_see(const camera_t *cam, const int x, const int y,
		    const int w, const int h)
{
    if ((x+w/2 < cam->x) || (x-w/2 > cam->x+cam->view_width) ||
	(y+h/2 < cam->y) || (y-h/2 > cam->y+cam->view_height))
	return false;
    else
	return true;
}


bool camera_can_see2(const camera_t *cam, const int x, const int y,
		     const int w, const int h)
{
    if ((x+w/2 < cam->x) || (x-w/2 > cam->x+cam->view_width) ||
	(y+h < cam->y) || (y > cam->y+cam->view_height))
	return false;
    else
	return true;
}


bool point_in_sight(const camera_t *cam, const int x, const int y)
{
    if ((cam->x < x) && (x < cam->x+cam->view_width) &&
	(cam->y < y) && (y < cam->y+cam->view_height))
	return true;
    else
	return false;
}


bool circle_in_sight(const camera_t *cam, const int x, const int y,
		     const double r)
{
    int X1 = cam->x - x;
    int X2 = cam->x + cam->view_width - x;
    int Y1 = cam->y - y;
    int Y2 = cam->y + cam->view_height - y;
    int dx, dy;

    if (X1 > 0)			/* rect is right of circle */
	dx = X1;
    else if (X2 < 0)		/* rect is left of circle */
	dx = X2;
    else
	dx = 0;

    if (Y1 > 0)			/* rect is above circle */
	dy = Y1;
    else if (Y2 < 0)		/* rect is below circle */
	dy = Y2;
    else
	dy = 0;

    return (dx*dx + dy*dy < r*r);
}

/*--------------------------------------------------------------*/

void camera_zoom(camera_t *cam)
{
    if (cam->scale < cam->desired_scale)
	cam->scale = (0.90*cam->scale + 0.10*cam->desired_scale);
    else
	cam->scale = (0.80*cam->scale + 0.20*cam->desired_scale);
}

/*--------------------------------------------------------------*/

static void handle_push(camera_t *cam, const int mx, const int my);
static void move_camera_closer_to_target(camera_t *cam);


/* Camera tracks the point between a point and the mouse.  */
void camera_track_point_with_mouse(camera_t *cam,
				   const int px, const int py,
				   const int mx, const int my)
{
    int cx, cy;
    int dx, dy;

    cx = cam->view_width / 2;
    cy = cam->view_height / 2;
    dx = mx - cx;
    dy = my - cy;

    if (cam->pushable)
	handle_push(cam, mx, my);
    else {
	cam->push_x *= PUSH_DECAY;
	cam->push_y *= PUSH_DECAY;
    }

    dx += cam->push_x;
    dy += cam->push_y;

    if (SQ(dx) + SQ(dy) > SQ(cam->max_dist)) {
	double angle = atan2(dy, dx);
	dx = cam->max_dist * cos(angle);
	dy = cam->max_dist * sin(angle);
    }


    cam->target_x = (px + cam->x - cx + dx) / 2;
    cam->target_y = (py + cam->y - cy + dy) / 2;

    move_camera_closer_to_target(cam);
}


static void handle_push(camera_t *cam, const int mx, const int my)
{
    /* maybe this could be coded better? */
	
    /* horizontal */
    if (mx < PUSH_MARGIN)
	cam->push_x -= PUSH_SPEED;
    else if (mx < STAY_MARGIN)
	;
    else if (mx < REVERT_MARGIN)
	cam->push_x = MIN(cam->push_x + PUSH_SPEED, 0);
    else if (mx >= cam->view_width - PUSH_MARGIN)
	cam->push_x += PUSH_SPEED;
    else if (mx >= cam->view_width - STAY_MARGIN)
	;
    else if (mx >= cam->view_width - REVERT_MARGIN)
	cam->push_x = MAX(cam->push_x - PUSH_SPEED, 0);
    else
	cam->push_x *= PUSH_DECAY;

    /* vertical */
    if (my < PUSH_MARGIN)
	cam->push_y -= PUSH_SPEED;
    else if (my < STAY_MARGIN)
	;
    else if (my < REVERT_MARGIN)
	cam->push_y = MIN(cam->push_y + PUSH_SPEED, 0);
    else if (my >= cam->view_height - PUSH_MARGIN)
	cam->push_y += PUSH_SPEED;
    else if (my >= cam->view_height - STAY_MARGIN)
	;
    else if (my >= cam->view_height - REVERT_MARGIN)
	cam->push_y = MAX(cam->push_y - PUSH_SPEED, 0);
    else
	cam->push_y *= PUSH_DECAY;
}


static void move_camera_closer_to_target(camera_t *cam)
{
    float dx, dy;

    dx = cam->target_x - cam->x;
    dy = cam->target_y - cam->y;
    
    cam->xv += dx * 0.11;	/* [0.1] */
    cam->yv += dy * 0.11;	/* [0.1] */

    cam->x += cam->xv;
    cam->y += cam->yv;

    cam->xv *= 0.35;		/* [0.9] */
    cam->yv *= 0.35;		/* [0.9] */

    /* For a fun effect, replace the dampening values with the
     * ones in [brackets].  */
}
