/* bullet.c,
 *
 * Projectile related routines.
 */

#include <alleggl.h>
#include <allegro.h>
#include <assert.h>
#include <math.h>
#include "angle.h"
#include "bullet.h"
#include "camera.h"
#include "common.h"
#include "container.h"
#include "explosion.h"
#include "list.h"
#include "map.h"
#include "maxmin.h"
#include "network.h"
#include "player.h"
#include "server.h"
#include "smoke.h"
#include "sound.h"
#include "step.h"
#include "texdraw.h"
#include "universal.h"
#include "weapon.h"


#define SQ(x)                    ((x)*(x))


static BITMAP *bmp_bullets;
static GLuint tex_bullets;

static list_head_t bullet_new_list;	/* Server only */
static list_head_t bullet_delete_list;	/* Server only */

static int bullet_new_count;
static int bullet_delete_count;

/*--------------------------------------------------------------*/

#define STEP_BOUNCE_OFF_WALL		0x20

static void bullet_velocity(const void *q, double *vx, double *vy);
static int bullet_step(void *q, const int tick, const void *state);
static bool bullet_collided(void *q, const int tick, const int collisions);
static void bullet_maybe_explode(const bullet_t *b, game_state_t *state);
static void bullet_maybe_smokey(bullet_t *b);


void bullet_update(game_state_t *state)
{
    bullet_t *b, *nx;
    assert(state);

    list_for_each_next(b, nx, &state->bullet_list) {
	if (bullet_update_unit(b, server_time, state)) {
	    if (!state->as_server)
		bullet_maybe_smokey(b);

	    if (b->lux) {
		b->lux->_spawn_x = b->x;
		b->lux->_spawn_y = b->y;
	    }

	    continue;
	}

	/* Servers move this bullet to the delete list.  Tell clients
	   about it later. */
	if (state->as_server) {
	    if (b->hurtee) {
		player_t *p = player_from_id(b->hurtee, state);
		assert(p);

		player_hurt(p, weapon[b->class].damage*b->modifier,
			    b->_spawn_vx*weapon[b->class].mass/100,
			    b->_spawn_vy*weapon[b->class].mass/100,
			    b->parent, state);
	    }
	    else if (b->hurtee_container) {
		container_hurt(b->hurtee_container,
			       weapon[b->class].damage*b->modifier, b->parent,
			       b->time, state);
	    }

	    bullet_maybe_explode(b, state);

	    b->prev->next = nx;
	    nx->prev = b->prev;
	    list_add(bullet_delete_list, b);
	    bullet_delete_count++;
	}
    }
}


/* bullet_update_unit:
 *
 * Update a single bullet.  Returns true if the bullet is still alive.
 */
bool bullet_update_unit(bullet_t *b, const int tick, const game_state_t *state)
{
    const walk_time_callbacks_t callbacks = {
	bullet_velocity,
	bullet_step,
	bullet_collided
    };

    if (b->class != WEAPON_GRENADE_LAUNCHER) {
	b->spin = b->_spawn_angle;
    }
    else {
	double t = (server_time-b->_spawn_time)/1000.0;
	b->spin = b->_spawn_angle + deg2rad(360.0)*t;
    }

    return walk_time(b, &callbacks, b->time, tick, state);
}


static void bullet_velocity(const void *q, double *vx, double *vy)
{
    const bullet_t *b = q;
    assert(b);
    assert(vx);
    assert(vy);

    if (b->class == WEAPON_GRENADE_LAUNCHER) {
	double t = (b->time - b->_spawn_time)/1000.0;
	*vx = b->_spawn_vx;
	*vy = b->_spawn_vy - GRAVITY2*t;
    }
    else {    
	*vx = b->_spawn_vx;
	*vy = b->_spawn_vy;
    }
}


void bullet_move(bullet_t *b, const int tick)
{
    double t;
    assert(b);

    b->time = tick;
    t = (tick-b->_spawn_time)/1000.0;

    if (b->class == WEAPON_GRENADE_LAUNCHER) {
	b->x = b->_spawn_x + b->_spawn_vx*t;
	b->y = b->_spawn_y + b->_spawn_vy*t - 0.5*GRAVITY2*t*t;
    }
    else {
	b->x = b->_spawn_x + b->_spawn_vx*t;
	b->y = b->_spawn_y + b->_spawn_vy*t;
    }
}


static int bullet_step(void *q, const int tick, const void *state0)
{
    bullet_t *b = q;
    player_t *p;
    game_state_t *state = (game_state_t *)state0; /* may be NULL */
    container_id c;
    assert(b);

    bullet_move(b, tick);

    if (!x_in_map(b->x))
	return STEP_COLLIDE_X;

    if (!y_in_map(b->y))
	return STEP_COLLIDE_Y;

    if (map_occupies(b->x, b->y)) {
	if (b->class == WEAPON_GRENADE_LAUNCHER) {
	    if (!map_occupies(b->x+1, b->y))
		return (STEP_BOUNCE_OFF_WALL|STEP_COLLIDE_RIGHT);

	    if (!map_occupies(b->x-1, b->y))
		return (STEP_BOUNCE_OFF_WALL|STEP_COLLIDE_LEFT);

	    if (!map_occupies(b->x, b->y+1))
		return (STEP_BOUNCE_OFF_WALL|STEP_COLLIDE_TOP);

	    if (!map_occupies(b->x, b->y-1))
		return (STEP_BOUNCE_OFF_WALL|STEP_COLLIDE_BOTTOM);

	    return (STEP_BOUNCE_OFF_WALL|STEP_COLLIDE_XY);
	}
	else {
	    return STEP_COLLIDE_XY;
	}
    }

    c = container_bounding(b->x, b->y, true);
    if (c) {
	b->hurtee_container = c;
	return STEP_COLLIDE_XY;
    }

    if (state && state->as_server) {
	p = player_touching_point(b->x, b->y, state);
	if ((p) && (p->id != b->parent)) {
	    b->hurtee = p->id;
	    return STEP_COLLIDE_XY;
	}
    }

    return STEP_NO_COLLIDE;
}


static enum TILE_FACE
tile_face_from_collisions(const int collisions,
			  const double vx, const double vy)
{
    assert(collisions & STEP_BOUNCE_OFF_WALL);

    if (collisions == (STEP_BOUNCE_OFF_WALL|STEP_COLLIDE_XY)) {
	if (ABS(vx) > ABS(vy)) {
	    if (vx < 0.0)
		return TILE_FACE_RIGHT;
	    else
		return TILE_FACE_LEFT;
	}
	else {
	    if (vy < 0.0)
		return TILE_FACE_TOP;
	    else
		return TILE_FACE_BOTTOM;
	}
    }
    else {
	if (collisions & STEP_COLLIDE_RIGHT)
	    return TILE_FACE_RIGHT;
	else if (collisions & STEP_COLLIDE_TOP)
	    return TILE_FACE_TOP;
	else if (collisions & STEP_COLLIDE_LEFT)
	    return TILE_FACE_LEFT;
	else
	    return TILE_FACE_BOTTOM;
    }
}


static bool bullet_collided(void *q, const int tick, const int collisions)
{
    bullet_t *b = q;
    double hitx, hity;
    int hitt;
    assert(b);

    if (b->_spawn_time > tick)
	return false;

    hitt = b->time;
    hitx = b->x;
    hity = b->y;

    bullet_move(b, tick);
    b->_spawn_x = b->x;
    b->_spawn_y = b->y;

    if (collisions & STEP_BOUNCE_OFF_WALL) {
	double speed, angle;
	double vx, vy;
	float n;

	assert(b->class == WEAPON_GRENADE_LAUNCHER);
	bullet_velocity(b, &vx, &vy);
	n = normal_angle(hitx, hity,
			 tile_face_from_collisions(collisions, vx, vy));

	speed = 0.5 * sqrt(vx*vx + vy*vy);
	angle = atan2(-vy, -vx);	/* angle of incidence */
	angle = 2*n - angle;		/* angle of reflection */

	b->_spawn_vx = speed*cos(angle);
	b->_spawn_vy = speed*sin(angle);

	if (speed < 10.0) {
	    b->grenade_life -= (hitt - b->time);

	    if (b->grenade_life <= 0)
		return true;
	}

	b->_spawn_time = b->time;
	return false;
    }

    b->_spawn_time = b->time;
    return true;
}


static void bullet_maybe_explode(const bullet_t *b, game_state_t *state)
{
    struct packet_explosion_new packet;
    NLbyte buf[MAX_PACKET_SIZE];
    NLint len;
    assert(b);
    assert(state);

    if (!weapon_is_explosive(b->class))
	return;

    packet.time = b->time;
    packet.id = explosion_generate_unique_id();
    packet.x = b->x;
    packet.y = b->y;
    packet.max_r = weapon[b->class].explosion.size;
    packet.damage = weapon[b->class].explosion.damage;
    packet.modifier = b->modifier;

    len = packet_explosion_new_encode(&packet, buf, sizeof(buf));

    server_broadcast(buf, len);

    explosion_new(&packet, b->parent, state);
}


static void bullet_maybe_smokey(bullet_t *b)
{
    double vx, vy;

    if (!weapon_is_smokey(b->class))
	return;

    if (b->next_smoke_time > server_time)
	return;

    bullet_velocity(b, &vx, &vy);
    smoke_new(weapon[b->class].smoke.class, b->x, b->y,
	      -vx/10.0, -vy/10.0, b->next_smoke_time);
    b->next_smoke_time = server_time+weapon[b->class].smoke.delay;
}

/*--------------------------------------------------------------*/

static void bullet_draw_unit(const bullet_t *b);
static void bullet_draw_trail(const double len);


void bullet_draw(const game_state_t *state)
{
    bullet_t *b;
    assert(state);

    glDisable(GL_ALPHA_TEST);
    glBlendFunc(GL_SRC_ALPHA, GL_DST_ALPHA);
    glBlendEquation(GL_FUNC_ADD);

    glBindTexture(GL_TEXTURE_2D, tex_bullets);
    glMaterialfv(GL_FRONT, GL_EMISSION, glGray25);

    list_for_each(b, &state->bullet_list)
	bullet_draw_unit(b);

    glMaterialfv(GL_FRONT, GL_EMISSION, glBlack);
    glEnable(GL_ALPHA_TEST);
    glAlphaFunc(GL_GREATER, 0.5);
}


static void bullet_draw_unit(const bullet_t *b)
{
    const sprite_t *sprite;
    texcoord2d_t coord;
    assert(b->class < NUM_WEAPON_CLASSES);

    sprite = &bullet_sprite[b->class];

    if (!camera_can_see(&camera, b->x, b->y, sprite->w, sprite->h))
	return;

    texcoord_from_sprite(&coord, sprite, bmp_bullets->w, bmp_bullets->h);

    glPushMatrix();
    glTranslated(rint(b->x), rint(b->y), 0.0);
    glRotatef(rad2deg(b->spin), 0.0, 0.0, 1.0);
    glTranslated(-sprite->w/4.0, -sprite->h/4.0, 0.0);

    /* Bullet trails for non-grenades. */
    if (weapon[b->class].trail_len > 0.0) {
	double len = SQ(b->x - b->_spawn_x) + SQ(b->y - b->_spawn_y);

	if (len > SQ(weapon[b->class].trail_len))
	    len = weapon[b->class].trail_len;
	else
	    len = sqrt(len);
	bullet_draw_trail(len);
    }

    glBegin(GL_QUADS);
    gl_draw_sprite_2d(&coord, 0.0, 0.0);
    glEnd();			/* glBegin(GL_QUADS) */

    glPopMatrix();
}


static void bullet_draw_trail(const double len)
{
    const GLfloat col[4] = { 1.0, 1.0, 1.0, 0.18 };

    glDisable(GL_TEXTURE_2D);
    glEnable(GL_BLEND);
    /* 'glBlendFunc' and 'glBlendEquation' set by bullet_draw. */

    glColor4fv(col);
    glBegin(GL_TRIANGLES);
    glVertex2d(-len, 1.0);
    glVertex2d( 0.0, 0.0);
    glVertex2d( 0.0, 2.0);
    glEnd();			/* glBegin(GL_TRIANGLES) */

    glDisable(GL_BLEND);
    glEnable(GL_TEXTURE_2D);
    glColor3fv(glWhite);
}

/*--------------------------------------------------------------*/

bullet_t *bullet_from_id(const bullet_id id, const game_state_t *state)
{
    bullet_t *b;
    assert(state);

    list_for_each(b, &state->bullet_list) {
	if (b->id == id)
	    return b;
    }

    return NULL;
}

/*--------------------------------------------------------------*/

/* bullet_encode_and_spawn:
 *
 * Pack as many bullets in the 'bullet_new_list' as possible into buf
 * and move the bullets into 'bullet_list' afterwards.
 */
NLint bullet_encode_and_spawn(NLbyte *buf, const NLint sz, game_state_t *state)
{
    struct packet_bullet_new packet;
    bullet_t *b, *nx;
    NLint count;
    int n;
    assert(state);
    assert(state->as_server);

    if (bullet_new_count <= 0)
	return 0;

    n = mini(bullet_new_count,
	     (sz-sizeof_packet_bullet_new_begin)/sizeof_packet_bullet_new);

    count = packet_bullet_new_begin_encode(n, buf, sz);

    list_for_each_next(b, nx, &bullet_new_list) {
	packet.time = b->_spawn_time;
	packet.id = b->id;
	packet.class = b->class;
	packet.parent = b->parent;
	packet.x = b->_spawn_x;
	packet.y = b->_spawn_y;
	packet.angle = b->_spawn_angle;
	packet.modifier = b->modifier;

	count = packet_bullet_new_encode(&packet, buf, sz, count);

	list_remove(b);
	list_add(state->bullet_list, b);
	bullet_new_count--;
	n--;
	if (n <= 0)
	    break;
    }

    return count;
}


/* bullet_encode_and_delete:
 *
 * Pack as many bullets in the 'bullet_delete_list' as possible into
 * buf.  The bullets are then freed.
 */
NLint bullet_encode_and_delete(NLbyte *buf, const NLint sz)
{
    struct packet_bullet_die packet;
    bullet_t *b, *nx;
    NLint count;
    int n;
    assert(client_server_mode != I_AM_CLIENT_ONLY);

    if (bullet_delete_count == 0)
	return 0;

    n = mini(bullet_delete_count,
	     (sz-sizeof_packet_bullet_die_begin)/sizeof_packet_bullet_die);

    count = packet_bullet_die_begin_encode(n, buf, sz);

    list_for_each_next(b, nx, &bullet_delete_list) {
	packet.time = b->time;
	packet.id = b->id;

	if (b->hurtee)
	    packet.particles = PARTICLE_BLOOD;
	else
	    packet.particles = PARTICLE_SPARK;

	count = packet_bullet_die_encode(&packet, buf, sz, count);

	bullet_free(b);
	bullet_delete_count--;
	n--;
	if (n <= 0)
	    break;
    }

    return count;
}

/*--------------------------------------------------------------*/

static void bullet_new_unit(const struct packet_bullet_new *packet,
			    const bool new_list, game_state_t *state);


void bullet_new(const struct packet_bullet_new *packet, game_state_t *state)
{
    bool new_list;
    assert(packet);
    assert(state);

    new_list = state->as_server;

    if (packet->class == WEAPON_SHOTGUN) {
	struct packet_bullet_new p;
	int dt[10] = { 0,  5, 10, 15, 20, 25, 30, 35, 40, 50 };
	int da[10] = { 0, -1,  3, -3,  1, -5,  3, -3,  5,  0 };
	int i;

	memcpy(&p, packet, sizeof(p));
	for (i = 0; i < 10; i++) {
	    p.id = packet->id + i;
	    p.time = packet->time + dt[i];
	    p.angle = packet->angle + deg2rad(da[i]);
	    bullet_new_unit(&p, (new_list) && (i == 0), state);
	}
    }
    else {
	bullet_new_unit(packet, new_list, state);
    }
}


static void bullet_new_unit(const struct packet_bullet_new *packet,
			    const bool new_list, game_state_t *state)
{
    const GLfloat col[4] = { 0.8, 0.8, 0.6, 0.10 };
    bullet_t *b = malloc(sizeof(*b));
    assert(b);
    assert(packet);
    assert(packet->class <= NUM_WEAPON_CLASSES);
    assert(state);

    b->id = packet->id;
    b->class = packet->class;
    b->parent = packet->parent;

    /* Spawn state. */
    b->_spawn_time = packet->time;
    b->_spawn_x = packet->x;
    b->_spawn_y = packet->y;
    b->_spawn_angle = packet->angle;
    b->_spawn_vx = weapon[b->class].speed * cos(b->_spawn_angle);
    b->_spawn_vy = weapon[b->class].speed * sin(b->_spawn_angle);

    /* Current state. */
    b->time = b->_spawn_time;
    b->x = b->_spawn_x;
    b->y = b->_spawn_y;
    b->spin = 0.0;
    b->next_smoke_time = b->_spawn_time + 50 + rand()%100;
    b->hurtee_container = 0;
    b->hurtee = 0;
    b->modifier = packet->modifier;
    b->grenade_life = 100;

    if (state->as_server) {
	b->lux = NULL;
    }
    else {
	b->lux = lux_new(b->x, b->y, 100, col);
	b->lux->r = 4.0;
    }

    if (new_list) {
	list_add(bullet_new_list, b);
	bullet_new_count++;
    }
    else {
	list_add(state->bullet_list, b);
    }

    if ((!state->as_server) && 
	(b->class == WEAPON_MISSILE_LAUNCHER)) {
	b->sound_source_n = play_missile_flight(b->_spawn_x, b->_spawn_y,
						b->_spawn_vx, b->_spawn_vy);
    }
    else {
	b->sound_source_n = -1;
    }
}


void bullet_free(bullet_t *b)
{
    assert(b);

    if (b->lux) {
	lux_automate(b->lux);
	b->lux = NULL;
    }

    list_remove(b);
    free(b);
}

/*--------------------------------------------------------------*/

void bullet_start(game_state_t *state)
{
    assert(state);

    list_init(state->bullet_list);

    if (state->as_server) {
	list_init(bullet_new_list);
	list_init(bullet_delete_list);

	bullet_new_count = 0;
	bullet_delete_count = 0;
    }
}


void bullet_stop(game_state_t *state)
{
    assert(state);

    list_free(state->bullet_list, bullet_free);

    if (state->as_server) {
	list_free(bullet_new_list, bullet_free);
	list_free(bullet_delete_list, bullet_free);

	bullet_new_count = 0;
	bullet_delete_count = 0;
    }
}


void bullet_init(void)
{
    bmp_bullets = load_bitmap("data/bullets.tga", NULL);
    assert(bmp_bullets);

    tex_bullets = allegro_gl_make_texture_ex(AGL_TEXTURE_MASKED,
					     bmp_bullets, -1);

    weapon_sound_load();
}


void bullet_shutdown(void)
{
    weapon_sound_free();

    if (bmp_bullets) {
	destroy_bitmap(bmp_bullets);
	bmp_bullets = NULL;
    }
}
