/* background.c,
 *
 * Parallax background.
 */

#include <alleggl.h>
#include <allegro.h>
#include <assert.h>

#include "background.h"
#include "texdraw.h"


static enum GAME_MODE background_loaded;
static BITMAP *bmp_background;
static GLuint tex_background;

/* Prevent a 'popping' effect when we extend the map. */
int background_offx, background_offy;

/*--------------------------------------------------------------*/

static int mod(int a, const int b)
{
    if (a > 0)
	return a % b;
    else
	return b - ((-a) % b);
}

/*--------------------------------------------------------------*/

void background_draw(const camera_t *cam)
{
    texcoord2d_t coord;
    double scale;
    int x, y, cx, cy, xx, yy;
    assert(cam);

    if (!bmp_background)
	return;

    scale = 0.75+0.25*cam->scale;
    cx = (cam->x - background_offx) / 10.0;
    cy = (cam->y - background_offy) / 10.0;
    x = mod(cx, bmp_background->h);
    y = mod(cy, bmp_background->w);

    coord.w = bmp_background->w;
    coord.h = bmp_background->h;
    coord.top = 0.0;
    coord.bottom = 1.0;
    coord.left = 0.0;
    coord.right = 1.0;

    glPushMatrix();
    glScalef(scale, scale, scale);
    glTranslatef(-x, -y, 0.0);

    glBindTexture(GL_TEXTURE_2D, tex_background);
    glBegin(GL_QUADS);

    for (yy = 0; yy-y < 480/scale; yy += bmp_background->h) {
	for (xx = 0; xx-x < 640/scale; xx += bmp_background->w)
	    gl_draw_sprite_2d(&coord, xx, yy);
    }

    glEnd();			/* glBegin(GL_QUADS) */
    glPopMatrix();
}

/*--------------------------------------------------------------*/

void background_init(const enum GAME_MODE game_mode)
{
    const char *fn;

    if ((bmp_background) && (game_mode == background_loaded))
	return;

    if (bmp_background) {
	glDeleteTextures(1, &tex_background);
	tex_background = 0;

	destroy_bitmap(bmp_background);
	bmp_background = NULL;
    }

    fn = ((game_mode == GAME_MODE_SILHOUETTES) ? "data/bg-silhouette.tga" :
	  "data/bg.tga");

    bmp_background = load_bitmap(fn, NULL);

    if (bmp_background) {
	background_loaded = game_mode;
	tex_background = allegro_gl_make_texture_ex(AGL_TEXTURE_MASKED,
						    bmp_background, -1);
    }
}


void background_shutdown(void)
{
    if (bmp_background) {
	destroy_bitmap(bmp_background);
	bmp_background = NULL;
    }
}
