/* sv-prompt.c,
 *
 * The command prompt for the text interface server.
 */

#include <assert.h>
#include <ctype.h>
#include <curses.h>
#include <string.h>

#include "maxmin.h"
#include "strlcpy.h"
#include "textface/sv-prompt.h"
#include "textface/sv-textface.h"


#define MAX_INPUT_LEN		1024
#define CTRL(k)			(k - 'A' + 1)


static char save_input[MAX_INPUT_LEN];
static char input[MAX_INPUT_LEN];
static int left, pos;

/*--------------------------------------------------------------*/

void move_to_input_cursor(void)
{
    move(INPUT_ROW, 2 + pos - left);
}


void sv_prompt_repaint(void)
{
    int width = COLS-2;
    int margin = (width > 10) ? 10 : 0;

    if (pos < left) 
	left = maxi(0, pos - margin);
    else if (pos > left + width)
	left = maxi(0, pos - width + margin);

    attrset(A_NORMAL);
    mvaddstr(INPUT_ROW, 0, (left == 0) ? "> " : "< ");

    attrset(A_BOLD | COLOR_PAIR(PAIR_INPUT));
    addstr(input + left);
    clrtoeol();
}

/*--------------------------------------------------------------*/

const char *sv_prompt_poll(void)
{
    int c = getch();

    switch (c) {
      case ERR:
	  break;

      case CTRL('L'):
	  sv_repaint = SV_REPAINT_ALL;
	  break;

      case '\r':
	  while ((pos > 0) && isspace(input[pos-1]))
	      pos--;
	  input[pos] = '\0';
	  strlcpy(save_input, input, sizeof(input));
	  pos = 0;
	  input[0] = input[1] = '\0';
	  sv_repaint = SV_REPAINT_ALL;
	  return save_input;

      case CTRL('B'):
      case KEY_LEFT:
	  pos = maxi(0, pos-1);
	  sv_repaint |= SV_REPAINT_PROMPT;
	  break;

      case CTRL('F'):
      case KEY_RIGHT:
	  pos = mini(strlen(input), pos+1);
	  sv_repaint |= SV_REPAINT_PROMPT;
	  break;
	    
      case CTRL('A'):
      case KEY_HOME:
	  pos = 0;
	  sv_repaint |= SV_REPAINT_PROMPT;
	  break;
	    
      case CTRL('E'):
      case KEY_END:
	  pos = strlen(input);
	  sv_repaint |= SV_REPAINT_PROMPT;
	  break;
	    
      case CTRL('K'):
	  input[pos] = input[pos+1] = '\0';
	  sv_repaint |= SV_REPAINT_PROMPT;
	  break;

      case CTRL('U'):
	  pos = 0;
	  input[0] = input[1] = '\0';
	  sv_repaint |= SV_REPAINT_PROMPT;
	  break;
	    
      case CTRL('D'):
      case KEY_DC:
	  memmove(input+pos, input+pos+1, strlen(input+pos+1) + 1);
	  sv_repaint |= SV_REPAINT_PROMPT;
	  break;

      case CTRL('H'):
      case KEY_BACKSPACE:
      case 0x7F:
	  if (pos > 0) {
	      memmove(input+pos-1, input+pos, strlen(input+pos) + 1);
	      pos--;
	      sv_repaint |= SV_REPAINT_PROMPT;
	  }
	  break;
	    
      default:
	  if (isprint(c) && (strlen(input) + 1 < sizeof(input))) {
	      memmove(input+pos+1, input+pos, strlen(input+pos) + 1);
	      input[pos] = c;
	      pos++;
	      sv_repaint |= SV_REPAINT_PROMPT;
	  }
	  break;
    }

    return NULL;
}

/*--------------------------------------------------------------*/

void sv_prompt_init(void)
{
    memset(input, 0, sizeof(input));
    left = 0;
    pos = 0;    
}


void sv_prompt_shutdown(void)
{
}
