/* start-loc.c,
 *
 * Starting location stuff.  Unlike most things, a starting location's
 * unique id begins at 0.
 */

#include <alleggl.h>
#include <allegro.h>
#include <assert.h>
#include "common.h"
#include "sprite.h"
#include "start-loc.h"
#include "texdraw.h"


static BITMAP *bmp_starts;
static GLuint tex_starts;

start_loc_t starts[MAX_START_LOCS];	/* Shared */
static unsigned int num_starts;		/* Shared */

/*--------------------------------------------------------------*/

void start_loc_set(const struct packet_start_loc *packet)
{
    assert(packet);
    assert(packet->id < MAX_START_LOCS);

    if (!starts[packet->id].used) {
	starts[packet->id].used = true;
	num_starts++;
    }

    starts[packet->id].x = packet->x;
    starts[packet->id].y = packet->y;
}


void start_loc_unset(const struct packet_start_del *packet)
{
    assert(packet);
    assert(packet->id < MAX_START_LOCS);

    if (starts[packet->id].used) {
	starts[packet->id].used = false;
	num_starts--;
    }
}


void start_loc_get(const start_loc_id id, double *x, double *y)
{
    assert(id < MAX_START_LOCS);
    assert(starts[id].used);

    *x = starts[id].x;
    *y = starts[id].y;
}


start_loc_id start_loc_get_bounding(const int x, const int y)
{
    unsigned int s;

    for (s = 0; s < MAX_START_LOCS; s++) {
	if (!starts[s].used)
	    continue;

	if ((starts[s].x-16.0 <= x) && (x <= starts[s].x+16.0) &&
	    (starts[s].y <= y) && (y < starts[s].y+32.0))
	    return s;
    }

    return MAX_START_LOCS;
}


start_loc_id start_loc_rand(void)
{
    unsigned int n;
    start_loc_id s;
    assert(num_starts > 0);

    n = rand() % num_starts;

    for (s = 0; s < MAX_START_LOCS; s++) {
	if (!starts[s].used)
	    continue;

	if (n == 0)
	    return s;
	else
	    n--;
    }

    return 0;
}

/*--------------------------------------------------------------*/

void start_loc_bind_texture(void)
{
    glBindTexture(GL_TEXTURE_2D, tex_starts);
}


void start_loc_draw(void)
{
    const GLfloat *last = glWhite;
    unsigned int s;

    start_loc_bind_texture();
    glBegin(GL_QUADS);

    for (s = 0; s < MAX_START_LOCS; s++) {
	sprite_tint_col(starts[s].highlight, false, &last);

	if (starts[s].used)
	    start_loc_draw_unit(0, starts[s].x-16.0, starts[s].y);
    }

    if (last != glWhite)
	glColor3fv(glWhite);

    glEnd();			/* glBegin(GL_QUADS) */
}


void start_loc_draw_unit(const int nth, const int x, const int y)
{
#define BITMAP_W	64.0
#define BITMAP_H	32.0

    const texcoord2d_t coord = {
	32.0, 32.0,
	0.0, 32.0/BITMAP_H, (32.0*nth)/BITMAP_W, (32.0*nth+32.0)/BITMAP_W
    };

    gl_draw_sprite_2d(&coord, x, y);

#undef BITMAP_H
#undef BITMAP_W
}

/*--------------------------------------------------------------*/

unsigned int start_loc_total(void)
{
    return num_starts;
}


start_loc_id start_loc_find_unused(void)
{
    start_loc_id s;

    for (s = 0; s < MAX_START_LOCS; s++) {
	if (!starts[s].used)
	    return s;
    }

    return MAX_START_LOCS-1;
}


void start_loc_shift(const int offx, const int offy)
{
    start_loc_id s;

    for (s = 0; s < MAX_START_LOCS; s++) {
	starts[s].x += offx;
	starts[s].y += offy;
    }
}

/*--------------------------------------------------------------*/

void start_loc_start(void)
{
    num_starts = 0;
}


void start_loc_stop(void)
{
    unsigned int s;

    for (s = 0; s < MAX_START_LOCS; s++) {
	starts[s].used = false;
    }
    num_starts = 0;
}


void start_loc_init(void)
{
    bmp_starts = load_bitmap("data/starts.tga", NULL);
    assert(bmp_starts);

    tex_starts = allegro_gl_make_texture_ex(AGL_TEXTURE_MASKED,
					    bmp_starts, -1);
}


void start_loc_shutdown(void)
{
    if (bmp_starts) {
	destroy_bitmap(bmp_starts);
	bmp_starts = NULL;
    }
}
