/* server.c,
 *
 * Code to broadcast commands to clients.
 */

#include <assert.h>
#include <math.h>
#include <nl.h>
#include <stdarg.h>
#include <stdio.h>

#include "angle.h"
#include "backpack.h"
#include "bullet.h"
#include "candela.h"
#include "container.h"
#include "explosion.h"
#include "gizmo.h"
#include "map-save.h"
#include "map.h"
#include "maxmin.h"
#include "packet-backpack.h"
#include "packet-candy.h"
#include "packet-editor.h"
#include "packet-input.h"
#include "packet-map.h"
#include "packet-server.h"
#include "pickup.h"
#include "player.h"
#include "network.h"
#include "server.h"
#include "start-loc.h"
#include "strlcpy.h"
#include "sv-commands.h"
#include "sv-internal.h"
#include "universal.h"


static NLsocket sock;
static NLenum type = NL_TCP_PACKETS;
static int num_clients;

static int last_sync_time;
int server_time;

/* This is only used before creating the server.  Afterwards, we use
   the game mode stored in the game state. */
enum GAME_MODE game_mode = GAME_MODE_CLASSIC;

/*--------------------------------------------------------------*/

#define server_print_error()	network_print_error("Server", __LINE__)

/*--------------------------------------------------------------*/

/* server_announce:
 *
 * Broadcast using the chat log.
 */
void server_announce(const char *format, ...)
{
    char cat[1024];
    NLbyte buf[MAX_PACKET_SIZE];
    NLint len;
    va_list ap;

    va_start(ap, format);
    vsnprintf(cat, sizeof(cat), format, ap);
    va_end(ap);

    len = packet_input_chat_encode(cat, buf, sizeof(buf));
    server_broadcast(buf, len);
}


/* server_broadcast:
 *
 * Write the buffer, size nbytes, to all connected clients.
 */
bool server_broadcast(const NLvoid *buf, NLint nbytes)
{
    NLint len;

    len = nlWrite(group_all_clients, buf, nbytes);
    if (len == NL_INVALID) {
	server_print_error();
	return false;
    }

    if (len < nbytes) {
	fprintf(stderr, "[Server] Only wrote %d / %d bytes", len, nbytes);
	return false;
    }

    return true;
}


/* server_broadcast2:
 *
 * Broadcast the buffer rbuf, size rlen, to all connected clients
 * except the client who is controlling player id.  For that player,
 * send lbuf instead.
 */
void server_broadcast2(const player_id id,
		       const NLvoid *lbuf, const NLint llen,
		       const NLvoid *rbuf, const NLint rlen)
{
    unsigned int c;

    for (c = 0; c < MAX_CLIENTS; c++) {
	if (client_data[c].sock == NL_INVALID)
	    continue;

	if (id == client_data[c].id)
	    server_send(client_data[c].sock, lbuf, llen);
	else
	    server_send(client_data[c].sock, rbuf, rlen);
    }
}


/* server_send:
 *
 * Write the buffer, size nbytes, to one client.
 */
bool server_send(const NLsocket client, const NLvoid *buf, const NLint nbytes)
{
    NLint len;

    len = nlWrite(client, buf, nbytes);
    if (len == NL_INVALID) {
	server_print_error();
	return false;
    }

    if (len < nbytes) {
	fprintf(stderr, "[Server] Only wrote %d / %d bytes to client %d",
		len, nbytes, (int)client);
	return false;
    }

    return true;
}

/*--------------------------------------------------------------*/

typedef struct {
    int header_len, element_len;

    NLint (*write_header)(const int num, NLbyte *buf, const NLint sz);
    NLint (*add_element)(const void *q, NLbyte *buf, const NLint sz,
			 NLint count);
} feed_proc_t;


static void server_feed_general(const NLsocket client, const feed_proc_t *proc,
				const list_head_t *list)
{
    void *q, *nx;
    NLbyte buf[MAX_PACKET_SIZE];
    NLint count;
    int n, m;
    assert(proc);
    assert(proc->write_header);
    assert(proc->add_element);
    assert(list);

    nx = list_next(list);

    /* The maximum number of elements in the packet. */
    m = (sizeof(buf) - proc->header_len) / (proc->element_len);

    while (list_neq(nx, list)) {
	count = proc->write_header(m, buf, sizeof(buf));

	for (n = 0, q = nx; (n < m) && (list_neq(q, list));
	     n++, q = list_next(q)) {
	    count = proc->add_element(q, buf, sizeof(buf), count);
	}

	if (n < m) {
	    /* If we have less elements in the packet than the
	       maximum, fix up the start. */
	    proc->write_header(n, buf, sizeof(buf));
	}

	server_send(client, buf, count);
	nx = q;
    }
}

/*--------------------------------------------------------------*/

static void server_feed_player	(const NLsocket, const game_state_t *);
static void server_feed_bullet	(const NLsocket, const game_state_t *);
static void server_feed_backpack(const NLsocket, const game_state_t *);


static void server_feed_client(const NLsocket client, game_state_t *state)
{
    NLbyte buf[MAX_PACKET_SIZE];
    NLint len;
    int idx;
    assert(state);

    idx = sv_intern_client_from_sock(client);
    assert(idx >= 0);

    /* Go. */
    len = packet_server_state_encode(state, buf, sizeof(buf));
    server_send(client, buf, len);

    server_feed_map(client, state);
    server_feed_candela(client);
    server_feed_player(client, state);
    server_feed_starts(client);
    server_feed_containers(client);
    server_feed_gizmos(client);
    server_feed_pickup(client);
    server_feed_bullet(client, state);
    server_feed_backpack(client, state);
    server_join_player(&client_data[idx], state);
}


void server_feed_map(const NLsocket client, game_state_t *state)
{
    NLbyte buf[MAX_PACKET_SIZE];
    NLint len;
    int x, y;

    map_crop(state);

    for (y = 0; y < map.h; y += MAP_BLOCK_H) {
	for (x = 0; x < map.w; x += MAP_BLOCK_W) {
	    len = packet_map_encode(&map, x, y, buf, sizeof(buf));
	    server_send(client, buf, len);
	}
    }
}


void server_feed_starts(const NLsocket client)
{
    NLbyte buf[MAX_PACKET_SIZE];
    NLint len;

    len = packet_start_locs_encode(starts, buf, sizeof(buf));
    server_send(client, buf, len);
}


static NLint svfeed_container_add(const void *q, NLbyte *buf, const NLint sz,
				  NLint count)
{
    const container_t *c = q;
    assert(c);
    assert(buf);

    const struct packet_container_new packet = {
	c->id, c->class,
	c->x, c->y,
	c->hidden
    };

    return packet_container_new_encode(&packet, buf, sz, count);
}


void server_feed_containers(NLsocket client)
{
    const feed_proc_t proc = {
	sizeof_packet_container_new_begin,
	sizeof_packet_container_new,
	packet_container_new_begin_encode,
	svfeed_container_add
    };

    server_feed_general(client, &proc, &container_list);
}


static NLint svfeed_gizmo_add(const void *q, NLbyte *buf, const NLint sz,
			      NLint count)
{
    const gizmo_t *g = q;
    assert(g);
    assert(buf);

    const struct packet_gizmo_new packet = {
	g->id, g->class,
	g->x, g->y
    };

    return packet_gizmo_new_encode(&packet, buf, sz, count);
}


void server_feed_gizmos(NLsocket client)
{
    const feed_proc_t proc = {
	sizeof_packet_gizmo_new_begin,
	sizeof_packet_gizmo_new,
	packet_gizmo_new_begin_encode,
	svfeed_gizmo_add
    };

    server_feed_general(client, &proc, &gizmo_list);
}


static NLint svfeed_candela_add(const void *q, NLbyte *buf, const NLint sz,
				NLint count)
{
    const candela_t *c = q;
    assert(c);
    assert(buf);

    const struct packet_candela_new packet = {
	c->id, CANDELA_INCLUDE_ALL,
	{ c->posi[0], c->posi[1] },
	{ c->ambi[0], c->ambi[1], c->ambi[2] },
	c->cutoff, c->exponent
    };

    return packet_candela_new_encode(&packet, buf, sz, count);
}


void server_feed_candela(const NLsocket client)
{
    const feed_proc_t proc = {
	sizeof_packet_candela_new_begin,
	sizeof_packet_candela_new,
	packet_candela_new_begin_encode,
	svfeed_candela_add
    };

    server_feed_general(client, &proc, &candela_list);
}


void server_join_robot(robot_t *rob, client_data_t *data, game_state_t *state)
{
    player_id id = player_generate_unique_id();
    NLbyte buf[MAX_PACKET_SIZE];
    NLint len;
    assert(data);
    assert(data->type == CLIENT_ROBOT);
    assert(state);

    data->id = id;

    /* Tell everyone about us. */
    {
	struct packet_player_new packet;
	start_loc_id loc;

	loc = start_loc_rand();
	packet.clid = data->clid;
	packet.id = data->id;
	packet.alive = true;
	start_loc_get(loc, &packet.x, &packet.y);

	len = packet_player_new_begin_encode(1, buf, sizeof(buf));
	len = packet_player_new_encode(&packet, buf, sizeof(buf), len);
	server_broadcast(buf, len);
    
	player_new(&packet, state);
    }

    rob->clid = data->clid;
    robot_attach(rob, id, state);
}


static NLint svfeed_player_add(const void *q, NLbyte *buf, const NLint sz,
			       NLint count)
{
    const player_t *p = q;
    int idx;
    assert(p);
    assert(buf);

    struct packet_player_new packet = {
	0, p->id, p->alive, p->x, p->y
    };

    idx = sv_intern_client_from_id(p->id);
    assert(idx >= 0);

    packet.clid = client_data[idx].clid;

    return packet_player_new_encode(&packet, buf, sz, count);
}


static void server_feed_player(const NLsocket client, const game_state_t *state)
{
    const feed_proc_t proc = {
	sizeof_packet_player_new_begin,
	sizeof_packet_player_new,
	packet_player_new_begin_encode,
	svfeed_player_add
    };
    assert(state);

    server_feed_general(client, &proc, &state->player_list);
}


void server_join_player(client_data_t *data, game_state_t *state)
{
    player_id id = player_generate_unique_id();
    NLbyte buf[MAX_PACKET_SIZE];
    NLint len;
    assert(data);
    assert(data->type == CLIENT_GAMER);
    assert(state);

    data->id = id;

    /* Tell everyone about us. */
    {
	struct packet_player_new packet;
	start_loc_id loc;

	loc = start_loc_rand();
	packet.clid = data->clid;
	packet.id = id;
	packet.alive = true;
	start_loc_get(loc, &packet.x, &packet.y);

	len = packet_player_new_begin_encode(1, buf, sizeof(buf));
	len = packet_player_new_encode(&packet, buf, sizeof(buf), len);
	server_broadcast(buf, len);
    
	player_new(&packet, state);
    }

    /* And then attach the new client to this player. */
    {
	struct packet_player_attach packet = { id };

	len = packet_player_attach_encode(&packet, buf, sizeof(buf));
	server_send(data->sock, buf, len);
    }
}


static NLint svfeed_bullet_add(const void *q, NLbyte *buf, const NLint sz,
			       NLint count)
{
    const bullet_t *b = q;
    assert(b);
    assert(buf);

    const struct packet_bullet_new packet = {
	b->_spawn_time,
	b->id, b->class, b->parent,
	b->_spawn_x, b->_spawn_y,
	b->_spawn_angle,
	0.0
    };

    return packet_bullet_new_encode(&packet, buf, sz, count);
}


static void server_feed_bullet(const NLsocket client, const game_state_t *state)
{
    const feed_proc_t proc = {
	sizeof_packet_bullet_new_begin,
	sizeof_packet_bullet_new,
	packet_bullet_new_begin_encode,
	svfeed_bullet_add
    };
    assert(state);

    server_feed_general(client, &proc, &state->bullet_list);
}


static NLint svfeed_pickup_add(const void *q, NLbyte *buf, const NLint sz,
			       NLint count)
{
    const pickup_t *p = q;
    assert(p);
    assert(buf);

    const struct packet_pickup_new packet = {
	p->respawn_time,
	p->id, p->class,
	p->x, p->y
    };

    return packet_pickup_new_encode(&packet, buf, sz, count);
}


void server_feed_pickup(NLsocket client)
{
    const feed_proc_t proc = {
	sizeof_packet_pickup_new_begin,
	sizeof_packet_pickup_new,
	packet_pickup_new_begin_encode,
	svfeed_pickup_add
    };

    server_feed_general(client, &proc, &pickup_list);
}


static NLint svfeed_backpack_add(const void *q, NLbyte *buf, const NLint sz,
				 NLint count)
{
    const backpack_t *b = q;
    assert(b);
    assert(buf);

    const struct packet_backpack_new packet = {
	b->_spawn_time,
	b->id,
	b->_spawn_x, b->_spawn_y,
	b->_spawn_vx, b->_spawn_vy
    };

    return packet_backpack_new_encode(&packet, buf, sz, count);
}


static void server_feed_backpack(const NLsocket client,
				 const game_state_t *state)
{
    const feed_proc_t proc = {
	sizeof_packet_backpack_new_begin,
	sizeof_packet_backpack_new,
	packet_backpack_new_begin_encode,
	svfeed_backpack_add
    };
    assert(state);

    server_feed_general(client, &proc, &state->backpack_list);
}

/*--------------------------------------------------------------*/

static client_id generate_unique_client_id(void)
{
    static client_id id;
    id++;
    return id;
}


bool server_maybe_new_connection(void)
{
    NLsocket newsock;
    NLaddress addr;
    NLchar addr_buf[NL_MAX_STRING_LENGTH];
    int idx;

    newsock = nlAcceptConnection(sock);
    if (newsock == NL_INVALID) {
	return false;
    }

    if (nlGetRemoteAddr(newsock, &addr) != NL_TRUE) {
	server_print_error();
	return false;
    }

    server_announce("[Server] Incoming connection from %s",
		    nlAddrToString(&addr, addr_buf));

    idx = sv_intern_open(newsock);
    if (idx < 0) {
	server_announce("[Server] Connection refused: too many clients");
	nlClose(newsock);
	return false;
    }

    client_data[idx].clid = generate_unique_client_id();
    client_data[idx].last_packet_received = server_time;

    num_clients++;

    server_announce("[Server] New client (%d / %d)",
		    num_clients, MAX_CLIENTS);
    return true;
}


bool server_may_add_robot(void)
{
    if (num_clients >= MAX_CLIENTS)
	return false;

    if (num_robots >= MAX_ROBOTS)
	return false;

    return true;
}


bool server_maybe_new_robot(void)
{
    struct packet_input_info packet;
    NLbyte buf[MAX_PACKET_SIZE];
    NLint len;
    client_data_t *data;
    int idx, name;

    idx = sv_intern_unused_client();
    if (idx < 0)
	return false;

    data = &client_data[idx];
    name = idx % NUM_ROBOT_NAMES;

    data->clid = generate_unique_client_id();
    data->sock = NL_INVALID;
    strlcpy(data->name, robot_name[name], sizeof(data->name));
    data->type = CLIENT_ROBOT;
    data->last_packet_received = 0;
    data->id = 0;
    data->face_num = NUM_HUMAN_FACES + rand() % NUM_ROBOT_FACES;
    score_reset(&data->sc);

    strlcpy(packet.name, data->name, sizeof(packet.name));
    packet.type = data->type;
    packet.face = data->face_num;
    packet.clid = data->clid;
    packet.sc = data->sc;

    len = packet_input_info_encode(&packet, COMMAND_INPUT_INFO2,
				   buf, sizeof(buf));
    server_broadcast(buf, len);
    num_clients++;
    num_robots++;

    sv_intern_sort_by_score();
    return true;
}

/*--------------------------------------------------------------*/

void server_disconnect(client_data_t *data, const enum DISCONNECT_REASON why,
		       game_state_t *state)
{
    struct packet_disconnect packet;
    NLaddress addr;
    NLchar addr_buf[NL_MAX_STRING_LENGTH];
    NLbyte buf[MAX_PACKET_SIZE];
    NLint len;
    assert(data);
    assert(state);

    if (data->sock != NL_INVALID) {
	nlGetRemoteAddr(data->sock, &addr);
	nlAddrToString(&addr, addr_buf);
	nlGroupDeleteSocket(group_all_clients, data->sock);
	nlClose(data->sock);
	data->sock = NL_INVALID;
    }
    else {
	strlcpy(addr_buf, "localhost", sizeof(addr_buf));
    }

    packet.clid = data->clid;
    packet.why = why;
    len = packet_disconnect_encode(&packet, buf, sizeof(buf));
    server_broadcast(buf, len);

    if (data->type == CLIENT_ROBOT)
	num_robots--;

    if (data->id) {
	player_t *p = player_from_id(data->id, state);

	if (p)
	    player_free(p);
    }

    server_announce("[Server] %s (client %d - %s) disconnected",
		    data->name, data->clid, addr_buf);

    data->clid = 0;
    data->id = 0;
    data->type = CLIENT_UNUSED;
    num_clients--;

    sv_intern_sort_by_score();
}

/*--------------------------------------------------------------*/
/* Commands in game and lobby.					*/
/*--------------------------------------------------------------*/

static void server_read_client_info(const NLsocket client,
				    const NLbyte *buf, const NLint len,
				    game_state_t *state);
static void server_read_client_chat(const NLsocket client,
				    const NLbyte *buf, const NLint len);
static void server_do_request(const NLsocket client, 
			      const NLbyte *buf, const NLint len,
			      game_state_t *state);


static bool server_process_common_commands(const NLsocket client,
					   const NLbyte *buf, const NLint len,
					   game_state_t *state)
{
    assert(state);

    switch (buf[0]) {
      case COMMAND_INPUT_INFO:
	  server_read_client_info(client, buf, len, state);
	  return true;
      case COMMAND_INPUT_CHAT:
	  server_read_client_chat(client, buf, len);
	  return true;
      case COMMAND_INPUT_DISCONNECT:
	  server_disconnect(&client_data[sv_intern_client_from_sock(client)],
			    DISCONNECT_REQUEST, state);
	  return true;
      case COMMAND_SERVER:
	  server_do_request(client, buf, len, state);
	  return true;
      default:
	  return false;
    }
}


static void server_read_client_info(const NLsocket client,
				    const NLbyte *buf, const NLint len,
				    game_state_t *state)
{
    struct packet_input_info packet;
    client_data_t *data;
    NLbyte nbuf[MAX_PACKET_SIZE];
    NLint nlen;
    int idx;
    assert(state);

    idx = sv_intern_client_from_sock(client);

    if (idx < 0) {
	fprintf(stderr, "[Server] Client %d was not found!\n", (int)client);
	return;
    }

    data = &client_data[idx];
    if (data->type != CLIENT_CONNECTING)
	return;

    packet_input_info_decode(&packet, buf[0], buf, len);

    if ((packet.type == CLIENT_CONNECTING) ||
	(packet.type >= NUM_CLIENT_TYPES))
	packet.type = CLIENT_GAMER;

    strlcpy(data->name, packet.name, sizeof(data->name));
    data->type = packet.type;
    data->face_num = packet.face;
    score_reset(&data->sc);

    /* Tell existing clients about new client. */
    packet.clid = data->clid;
    packet.sc = data->sc;
    nlen = packet_input_info_encode(&packet, COMMAND_INPUT_INFO2,
				    nbuf, sizeof(nbuf));
    server_broadcast(nbuf, nlen);

    /* Tell new client about existing clients. */
    for (idx = 0; idx < MAX_CLIENTS; idx++) {
	data = &client_data[idx];

	if ((data->type == CLIENT_UNUSED) ||
	    (data->sock == client))
	    continue;

	strlcpy(packet.name, data->name, sizeof(packet.name));
	packet.clid = data->clid;
	packet.type = data->type;
	packet.face = data->face_num;
	packet.sc = data->sc;

	nlen = packet_input_info_encode(&packet, COMMAND_INPUT_INFO2,
					nbuf, sizeof(nbuf));
	server_send(client, nbuf, nlen);
    }

    /* Now feed the client. */
    if (state->context == SERVER_IN_GAME) {
	server_feed_client(client, state);
    }
}


static void server_read_client_chat(const NLsocket client,
				    const NLbyte *buf, const NLint len)
{
    char chat[MAX_CHAT_LEN];
    char chat2[MAX_CHAT_LEN];
    int idx;
    int len2;

    idx = sv_intern_client_from_sock(client);
    if (idx < 0)
	return;

    packet_input_chat_decode(chat, buf, len);
    len2 = snprintf(chat2, sizeof(chat2), "%c<%s> %s",
		    COMMAND_INPUT_CHAT, client_data[idx].name, chat);

    server_broadcast(chat2, mini(len2+1, sizeof(chat2)));
}


static void server_do_request(const NLsocket client, 
			      const NLbyte *buf, const NLint len,
			      game_state_t *state)
{
    NLbyte packet[MAX_PACKET_SIZE];
    sv_command_ss_func_t func;
    assert(state);

    packet_server_decode(packet, buf, len);
    func = sv_command_ss(packet);

    if (func) {
	func(packet, client, state);
    }
}

/*--------------------------------------------------------------*/
/* Commands in game only.					*/
/*--------------------------------------------------------------*/

static void server_read_client_move(const NLbyte *buf, const NLint len,
				    game_state_t *state);
static void server_read_client_edit(const NLbyte *buf, const NLint len,
				    game_state_t *state);


static bool server_process_game_commands(const NLsocket client,
					 const NLbyte *buf, const NLint len,
					 game_state_t *state)
{
#define TRY(x)		if (state->context == SERVER_IN_GAME) x

    (void)client;
    assert(state);

    switch (buf[0]) {
      case COMMAND_INPUT_MOVE:
      case COMMAND_INPUT_WIELD:
	  TRY(server_read_client_move(buf, len, state));
	  return true;
      case COMMAND_INPUT_EDITOR:
	  TRY(server_read_client_edit(buf, len, state));
	  return true;
      default:
	  return false;
    }

#undef TRY
}


static void server_read_client_move(const NLbyte *buf, const NLint len,
				    game_state_t *state)
{
    struct packet_input_move packet;
    player_t *p;
    assert(state);

    packet_input_move_decode(&packet, buf[0], buf, len);
    
    p = player_from_id(packet.id, state);
    if (p) {
	p->mirror = packet.mirror;
	p->angle = packet.angle;
	p->last_impy = packet.impy;

	if (buf[0] == COMMAND_INPUT_MOVE)
	    return;

	if (packet.weapon >= NUM_WEAPON_CLASSES) {
	    fprintf(stderr, "[Server] Player %d selected unknown weapon: %d\n",
		    packet.id, packet.weapon);
	}
	else if (!(p->have_weapon[packet.weapon])) {
	    fprintf(stderr, "[Server] Player %d can't select weapon %d\n",
		    packet.id, packet.weapon);
	}
	else {
	    p->weapon = packet.weapon;
	    p->packet_size |= PACKET_INCLUDE_WEAPON;
	}
    }
    else {
	fprintf(stderr, "[Server] Could not find player %d!  Move failed\n",
		packet.id);
    }
}


static void server_read_client_edit(const NLbyte *buf, const NLint len,
				    game_state_t *state)
{
    struct packet_editor packet;
    NLbyte nbuf[MAX_PACKET_SIZE];
    NLint nlen;
    assert(state);

    packet_editor_decode(&packet, buf, len);

    switch (packet.mode) {
      case EDIT_TILE_SET:
	  /* XXX: Bit of a hack?  Relay the command before we set the
	     tile. */
	  server_broadcast(buf, len);
	  map_set_tiles(packet.new.tile.x, packet.new.tile.y,
			packet.new.tile.tile, packet.new.tile.brush_size,
			state);
	  return;

      case EDIT_CONTAINER_ADD: {
	  packet.new.container_new.id = container_generate_unique_id();
	  nlen = packet_container_new_begin_encode(1, nbuf, sizeof(nbuf));
	  nlen = packet_container_new_encode(&packet.new.container_new,
					     nbuf, sizeof(nbuf), nlen);
	  server_broadcast(nbuf, nlen);
	  container_new(&packet.new.container_new);
	  return;
      }
      case EDIT_CONTAINER_MOVE:
	  container_move(&packet.new.container_mov);
	  break;
      case EDIT_CONTAINER_DELETE:
	  container_delete(&packet.new.container_del);
	  break;

      case EDIT_GIZMO_ADD:
	  packet.new.gizmo_new.id = gizmo_generate_unique_id();
	  nlen = packet_gizmo_new_begin_encode(1, nbuf, sizeof(nbuf));
	  nlen = packet_gizmo_new_encode(&packet.new.gizmo_new,
					  nbuf, sizeof(nbuf), nlen);
	  server_broadcast(nbuf, nlen);
	  gizmo_new(&packet.new.gizmo_new);
	  return;
      case EDIT_GIZMO_MOVE:
	  gizmo_move(&packet.new.gizmo_mov);
	  break;
      case EDIT_GIZMO_DELETE:
	  gizmo_delete(&packet.new.gizmo_del);
	  break;

      case EDIT_PICKUP_ADD:
	  packet.new.pickup_new.id = pickup_generate_unique_id();
	  nlen = packet_pickup_new_begin_encode(1, nbuf, sizeof(nbuf));
	  nlen = packet_pickup_new_encode(&packet.new.pickup_new,
					  nbuf, sizeof(nbuf), nlen);
	  server_broadcast(nbuf, nlen);
	  pickup_new(&packet.new.pickup_new);
	  return;
      case EDIT_PICKUP_MOVE:
	  pickup_move(&packet.new.pickup_mov);
	  break;
      case EDIT_PICKUP_DELETE:
	  pickup_delete(&packet.new.pickup_del);
	  break;

      case EDIT_START_ADD:
	  if (packet.new.start_new.id >= MAX_START_LOCS)
	      packet.new.start_new.id = start_loc_find_unused();

	  nlen = packet_start_loc_encode(&packet.new.start_new, 
					 nbuf, sizeof(nbuf));
	  server_broadcast(nbuf, nlen);
	  start_loc_set(&packet.new.start_new);
	  return;
      case EDIT_START_DELETE:
	  start_loc_unset(&packet.new.start_del);
	  break;

      case EDIT_CANDELA_ADD: {
	  candela_t *c;
	  bool spawn = (packet.new.candela_new.id == 0);

	  if (spawn)
	      packet.new.candela_new.id = candela_generate_unique_id();

	  nlen = packet_candela_new_begin_encode(1, nbuf, sizeof(nbuf));
	  nlen = packet_candela_new_encode(&packet.new.candela_new,
					   nbuf, sizeof(nbuf), nlen);
	  server_broadcast(nbuf, nlen);

	  if (spawn) {
	      candela_new(&packet.new.candela_new);
	  }
	  else {
	      c = candela_from_id(packet.new.candela_new.id);
	      assert(c);
	      candela_set_parameters(c, &packet.new.candela_new);
	  }
	  return;
      }
      case EDIT_CANDELA_DELETE:
	  candela_delete(packet.new.candela_del.id);
	  break;

      case EDIT_PLAYER_TELEPORT: {
	  player_t *p;

	  p = player_from_id(packet.new.player_tp.id, state);
	  if (!p) {
	      return;
	  }

	  p->x = packet.new.player_tp.x;
	  p->y = packet.new.player_tp.y;
	  p->vx = 0.0;
	  p->vy = 0.0;
	  return;
      }

      case EDIT_REGION_SET:
	  map_set_regions(packet.new.region.x, packet.new.region.y,
			  packet.new.region.region,
			  packet.new.region.brush_size);
	  break;

      case EDIT_UNKNOWN:
      default:
	  fprintf(stderr, "[Server] Unknown editor mode/action: %d\n",
		  packet.mode);
	  return;
    }

    /* Just spit out whatever we received to the clients. */
    server_broadcast(buf, len);
}

/*--------------------------------------------------------------*/
/* Commands in lobby only.					*/
/*--------------------------------------------------------------*/

static bool server_process_lobby_commands(const NLsocket client,
					  const NLbyte *buf, const NLint len,
					  game_state_t *state)
{
    (void)client, (void)buf, (void)len, (void)state;
    return false;
}


/*--------------------------------------------------------------*/

static void server_read_client_packet(const NLsocket client,
				      game_state_t *state);


void server_incoming(game_state_t *state)
{
    NLsocket client[MAX_CLIENTS];
    NLint i, n;
    assert(state);

    n = nlPollGroup(group_all_clients, NL_READ_STATUS, client, MAX_CLIENTS, 0);
    if (n == NL_INVALID)
	server_print_error();

    for (i = 0; i < n; i++)
	server_read_client_packet(client[i], state);

    if (state->context != SERVER_IN_GAME)
	return;

    /* Check who has been snoozing for a long long time. */
    for (i = 0; i < MAX_CLIENTS; i++) {
	if ((client_data[i].sock == NL_INVALID) ||
	    (client_data[i].type == CLIENT_ADMIN))
	    continue;

	if (server_time - client_data[i].last_packet_received > 10000) {
	    fprintf(stderr, "[Server] Client %d is not responding!\n",
		    (int)client_data[i].sock);
	    server_disconnect(&client_data[i], DISCONNECT_LAG, state);
	}
    }
}


static void server_read_client_packet(const NLsocket client,
				      game_state_t *state)
{
    NLbyte packet[MAX_PACKET_SIZE];
    NLint len;
    assert(state);

    len = nlRead(client, &packet, sizeof(packet));
    if (len != NL_INVALID) {
	int idx;

	idx = sv_intern_client_from_sock(client);
	if (idx >= 0)
	    client_data[idx].last_packet_received = server_time;
	else
	    fprintf(stderr, "[Server] Client %d not found", (int)client);
    }

    while (len > 0) {
	if (server_process_common_commands(client, packet, len, state))
	    goto next;

	if (server_process_game_commands(client, packet, len, state))
	    goto next;

	if (server_process_lobby_commands(client, packet, len, state))
	    goto next;

	fprintf(stderr, "[Server] Unrecognised command: %c\n", packet[0]);

    next:
	len = nlRead(client, &packet, sizeof(packet));
    }

    if (len == NL_INVALID) {
	NLenum err = nlGetError();

	if (err == NL_MESSAGE_END || err == NL_SOCK_DISCONNECT)
	    server_disconnect(&client_data[sv_intern_client_from_sock(client)],
			      DISCONNECT_LAG, state);
    }
}

/*--------------------------------------------------------------*/

static void server_player_update(game_state_t *state);
static void server_bullet_outgoing(game_state_t *state);
static void server_explosion_delete(void);
static void server_maybe_sync_time(void);


void server_outgoing(game_state_t *state)
{
    assert(state);

    server_player_update(state);
    server_bullet_outgoing(state);
    server_explosion_delete();
    server_maybe_sync_time();
}


static NLint server_create_lpacket(const player_t *p, NLbyte *buf, NLint size)
{
    struct packet_player_update packet;

    packet.time = server_time;
    packet.x = p->x;
    packet.y = p->y;
    packet.size = (p->packet_size & PACKET_LOCAL_INCLUDES);

    if (packet.size & PACKET_INCLUDE_WEAPON) {
	packet.size |= PACKET_INCLUDE_AMMO;
	packet.weapon = p->weapon;
    }

    if (p->weapon == WEAPON_BLASTER)
	packet.size &=~PACKET_INCLUDE_AMMO;

    if (packet.size & PACKET_INCLUDE_AMMO) {
	packet.size &=~PACKET_INCLUDE_FIRING;
	packet.ammo = p->weapon_ammo[p->weapon];
    }

    if (packet.size & PACKET_INCLUDE_HEALTH)
	packet.health = p->health;

    if (packet.size & PACKET_INCLUDE_TRACKER)
	packet.tracker = p->tracker_angle;

    return packet_player_update_encode(&packet, true, buf, size);
}


static NLint server_create_rpacket(const player_t *p, NLbyte *buf, NLint size)
{
    struct packet_player_update packet;

    packet.time = server_time;
    packet.id = p->id;
    packet.x = p->x;
    packet.y = p->y;
    packet.size = (p->packet_size & PACKET_REMOTE_INCLUDES);

    if (packet.size & PACKET_INCLUDE_WEAPON)
	packet.weapon = p->weapon;

    if (packet.size & PACKET_INCLUDE_ANGLE) {
	packet.mirror = p->mirror;
	packet.angle = p->angle;
    }

    return packet_player_update_encode(&packet, false, buf, size);
}


static void server_player_update(game_state_t *state)
{
    player_t *p;
    assert(state);

    /* Update. */
    list_for_each(p, &state->player_list) {
	if (!p->alive)
	    continue;

	p->fire_delay -= SECONDS_PER_TICK;

	if (fabs(p->sent_angle - p->angle) > deg2rad(10.0)) {
	    p->packet_size |= PACKET_INCLUDE_ANGLE;
	    p->sent_angle = p->angle;
	}

	if ((p->health <= 25) && (server_time >= p->next_bleed_time)) {
	    p->packet_size |= PACKET_INCLUDE_BLEED;
	    p->next_bleed_time = server_time + 1000;
	}
    }

    /* Send. */
    list_for_each(p, &state->player_list) {
	NLbyte lbuf[MAX_PACKET_SIZE], rbuf[MAX_PACKET_SIZE];
	NLint llen, rlen;
	int i;

	if (!p->alive)
	    continue;

	llen = server_create_lpacket(p, lbuf, sizeof(lbuf));
	rlen = server_create_rpacket(p, rbuf, sizeof(rbuf));

	/* Only send out the packet to players who can see you. */
	for (i = 0; i < MAX_CLIENTS; i++) {
	    if ((client_data[i].type == CLIENT_UNUSED) ||
		(client_data[i].sock == NL_INVALID))
		continue;

	    if (client_data[i].id == p->id) {
		server_send(client_data[i].sock, lbuf, llen);
	    }
	    else {
		player_t *q;
		double dx, dy;

		q = player_from_id(client_data[i].id, state);
		if (!q)
		    continue;

		dx = p->x - q->x;
		dy = p->y - q->y;

		if ((dx*dx < 640.0*640.0) && (dy*dy < 480.0*480.0))
		    server_send(client_data[i].sock, rbuf, rlen);
	    }
	}

	p->packet_size = 0;
    }
}


static void server_bullet_outgoing(game_state_t *state)
{
    NLbyte buf[MAX_PACKET_SIZE];
    NLint len;
    assert(state);

    len = bullet_encode_and_spawn(buf, sizeof(buf), state);
    if (len > 0)
	server_broadcast(buf, len);

    len = bullet_encode_and_delete(buf, sizeof(buf));
    if (len > 0)
	server_broadcast(buf, len);
}


static void server_explosion_delete(void)
{
    explosion_t *e;

    list_for_each(e, &explosion_delete_list) {
	const struct packet_explosion_die packet = {
	    e->id
	};
	NLbyte buf[MAX_PACKET_SIZE];
	NLint len;

	len = packet_explosion_die_encode(&packet, buf, sizeof(buf));
	if (!server_broadcast(buf, len))
	    return;
    }

    list_free(explosion_delete_list, explosion_free);
    list_init(explosion_delete_list);
}


static void server_maybe_sync_time(void)
{
    const struct packet_set_clock packet = {
	server_time
    };
    NLbyte buf[MAX_PACKET_SIZE];
    NLint len;

    if ((server_time - last_sync_time) < 1000)
	return;

    len = packet_set_clock_encode(&packet, buf, sizeof(buf));
    server_broadcast(buf, len);

    last_sync_time = server_time;
}

/*--------------------------------------------------------------*/

void server_end_session(void)
{
    unsigned int i;

    for (i = 0; i < MAX_ROBOTS; i++) {
	robot[i].player = NULL;
    }

    last_sync_time = 0;
}

/*--------------------------------------------------------------*/

bool server_init(const NLushort port)
{
    sv_intern_init();

    if ((sock = nlOpen(port, type)) == NL_INVALID)
	goto error;

    group_all_clients = nlGroupCreate();
    if (group_all_clients == NL_INVALID)
	goto error;

    if (nlListen(sock) != NL_TRUE)
	goto error;

    num_clients = 0;
    num_robots = 0;

    return true;

 error:

    if (sock != NL_INVALID) {
	nlClose(sock);
	sock = NL_INVALID;
    }
	
    server_print_error();
    return false;
}


void server_shutdown(void)
{
    NLsocket client[MAX_CLIENTS];
    int n = MAX_CLIENTS;

    nlGroupGetSockets(group_all_clients, client, &n);
    for (; n > 0; n--) {
	if (nlGroupDeleteSocket(group_all_clients, client[n-1]) != NL_TRUE)
	    server_print_error();

	if (nlClose(client[n-1]) != NL_TRUE)
	    server_print_error();
    }

    if (nlGroupDestroy(group_all_clients) != NL_TRUE)
	fprintf(stderr, "[Server] %s", nlGetErrorStr(nlGetError()));

    group_all_clients = NL_INVALID;

    assert(sock != NL_INVALID);
    if (nlClose(sock) != NL_TRUE)
	server_print_error();

    sock = NL_INVALID;
}
