/* 00grunt.c,
 *
 * A dumb robot.  Behaviour:
 *
 *  - Chases the closest enemy horizontally
 *  - Jumps at random
 *  - Constantly fires at the closest enemy
 *  - Selects weapons (except grenade launcher)
 */

#include <assert.h>
#include <math.h>
#include <stdlib.h>

#include "bullet.h"
#include "input.h"
#include "robots/00grunt.h"
#include "weapon-pick.h"
#include "weapon.h"


#define SQ(x)	((x)*(x))

/*--------------------------------------------------------------*/

int grunt_move(const robot_t *rob, const player_t *enemy)
{
    int impy = 0;
    assert(rob);
    assert(rob->player);
    assert(enemy);

    if (enemy->x < rob->player->x-128)
	impy |= INPUT_MOVE_LEFT;
    if (enemy->x > rob->player->x+128)
	impy |= INPUT_MOVE_RIGHT;
    if (rand() % 10 < 8)
	impy |= INPUT_MOVE_UP;

    return impy;
}

/*--------------------------------------------------------------*/

static void grunt_pick_weapon(robot_t *rob, const player_t *enemy);
static bool grunt_try_weapon(robot_t *rob, const enum WEAPON_CLASS class);


int grunt_fire(robot_t *rob, const player_t *enemy)
{
    enum WEAPON_CLASS old_weapon;
    assert(rob);
    assert(rob->player);
    assert(enemy);

    if (rob->player->fire_delay > 0.0)
	return 0;

    old_weapon = rob->player->weapon;
    grunt_pick_weapon(rob, enemy);

    if (rob->player->weapon != old_weapon)
	rob->player->packet_size |= PACKET_INCLUDE_WEAPON;

    return INPUT_FIRE;
}


static void grunt_pick_weapon(robot_t *rob, const player_t *enemy)
{
    double dx, dy;
    assert(rob);
    assert(rob->player);
    assert(enemy);

    dx = enemy->x - rob->player->x;
    dy = enemy->y-rob->player->y;
    rob->player->angle = atan2(dy, dx);
    rob->player->mirror =
	((M_PI_2 < rob->player->angle) || (rob->player->angle < -M_PI_2));

    if ((rob->player->have_weapon[WEAPON_MISSILE_LAUNCHER]) &&
	(rob->player->weapon_ammo[WEAPON_MISSILE_LAUNCHER] > 0)) {
	if (grunt_try_weapon(rob, WEAPON_MISSILE_LAUNCHER)) {
	    rob->player->weapon = WEAPON_MISSILE_LAUNCHER;
	    return;
	}
    }

    if ((rob->player->have_weapon[WEAPON_MICRO_MISSILE]) &&
	(rob->player->weapon_ammo[WEAPON_MICRO_MISSILE] > 0)) {
	if (grunt_try_weapon(rob, WEAPON_MICRO_MISSILE)) {
	    rob->player->weapon = WEAPON_MICRO_MISSILE;
	    return;
	}
    }

    /* After explosives are considered, pick one of minigun, vulcan
       and blaster. */
    rob->player->weapon = weapon_pick_no_ammo(rob->player);
}


static bool grunt_try_weapon(robot_t *rob, const enum WEAPON_CLASS class)
{
    bullet_t dummy;
    double dx, dy;
    assert(class < NUM_WEAPON_CLASSES);

    dummy.class = class;
    dummy.parent = rob->player->id;
    dummy._spawn_time = 0;
    dummy._spawn_x = rob->player->x;
    dummy._spawn_y = rob->player->y + 19.0;
    dummy._spawn_vx = weapon[class].speed*cos(rob->player->angle);
    dummy._spawn_vy = weapon[class].speed*sin(rob->player->angle);
    dummy.time = dummy._spawn_time;
    dummy.x = dummy._spawn_x;
    dummy.y = dummy._spawn_y;

    if (bullet_update_unit(&dummy, 2000, NULL)) {
	return true;
    }
    else {
	dx = dummy.x - rob->player->x;
	dy = dummy.y - rob->player->y;
	if (SQ(dx) + SQ(dy) > SQ(1.25*weapon[class].explosion.size))
	    return true;
	else
	    return false;
    }
}

/*--------------------------------------------------------------*/

int grunt_get_impies(robot_t *rob, const game_state_t *state)
{
    player_t *enemy;
    double min_dist_sq;
    int impy = 0;

    if (!rob->player->alive)
	return INPUT_RESPAWN;

    enemy = player_closest(rob->player, &min_dist_sq, state);
    if (!enemy)
	return 0;

    impy  = grunt_move(rob, enemy);
    impy |= grunt_fire(rob, enemy);

    return impy;
}
