/* pickup.c,
 *
 * Weapons, health, etc.  Pickups only collide with players.
 * Backpacks are NOT pickups.
 */

#include <alleggl.h>
#include <allegro.h>
#include <assert.h>
#include <stdio.h>
#include "angle.h"
#include "camera.h"
#include "common.h"
#include "editor.h"
#include "list.h"
#include "network.h"
#include "particle.h"
#include "pickup.h"
#include "stats-pickup.h"
#include "player.h"
#include "server.h"
#include "texdraw.h"
#include "universal.h"
#include "weapon-pick.h"


static BITMAP *bmp_pickups;
static GLuint tex_pickups;

list_head_t pickup_list;
static int pickup_count;

/*--------------------------------------------------------------*/

static void pickup_update_unit(pickup_t *p, const game_state_t *state);
static void pickup_update_hidden(pickup_t *p);
static void pickup_respawn_effects(pickup_t *p);


void pickup_update(game_state_t *state)
{
    pickup_t *p;
    assert(state);

    list_for_each(p, &pickup_list) {
	if (p->hidden) {
	    pickup_update_hidden(p);

	    if (!state->as_server) {
		pickup_respawn_effects(p);
	    }
	}
	else {
	    if (state->as_server)
		pickup_update_unit(p, state);
	}
    }
}


static void pickup_update_unit(pickup_t *p, const game_state_t *state)
{
    const pickup_stats_t *pickup;
    const sprite_t *sprite;
    struct packet_pickup_quaff packet;
    player_t *q;
    NLbyte buf[MAX_PACKET_SIZE];
    NLint len;
    assert(p->class < NUM_PICKUP_CLASSES);
    assert(state);

    pickup = &pickup_stats[p->class];
    sprite = &pickup_sprite[p->class];

    q = player_touching_box(p->x, p->y, sprite->w/2.0, sprite->h/2.0, state);
    if (!q)
	return;

    /* Don't take health powerups unnecessarily. */
    if ((pickup->give_health > 0) && (q->health >= 100))
	return;

    packet.time = server_time;
    packet.id = p->id;
    packet.quaffer = q->id;

    len = packet_pickup_quaff_encode(&packet, buf, sizeof(buf));
    server_broadcast(buf, len);

    pickup_quaff_by(p, q, packet.time);
}


static void pickup_update_hidden(pickup_t *p)
{
    if (server_time >= p->respawn_time) {
	p->hidden = false;
    }
}


static void pickup_respawn_effects(pickup_t *p)
{
    enum PARTICLE_CLASS class;
    double v;
    assert(p);

    if ((p->respawn_time <= server_time) ||
	(p->respawn_time >= server_time + 400))
	return;

    class = PARTICLE_RESPAWNING1 + rand()%2;
    v = particle_stats[class].v_min + rand()%particle_stats[class].v_rand;
    particle_new_unit(class, p->x+16, p->y, v*COS_100, v*SIN_100, server_time);

    class = PARTICLE_RESPAWNING1 + rand()%2;
    v = particle_stats[class].v_min + rand()%particle_stats[class].v_rand;
    particle_new_unit(class, p->x-16, p->y, v*COS_80, v*SIN_80, server_time);
}

/*--------------------------------------------------------------*/

void pickup_quaff_by(pickup_t *p, player_t *q, const int tick)
{
    const pickup_stats_t *pickup;
    assert(p);
    assert(q);
    assert(p->class < NUM_PICKUP_CLASSES);

    pickup = &pickup_stats[p->class];

    p->hidden = true;
    p->respawn_time = tick + pickup->respawn_time;
    q->health = MIN(q->health+pickup->give_health, 100);

    switch (pickup->type) {
      case PICKUP_WEAPON: {
	  enum WEAPON_CLASS w = pickup->give.weapon.weapon;

	  if ((q == player) && !(q->have_weapon[w]))
	      is_new_weapon[w] = true;

	  q->have_weapon[w] = true;
	  /* Fall through. */
      }
      case PICKUP_AMMO: {
	  enum WEAPON_CLASS w = pickup->give.weapon.weapon;

	  if ((q == player) && (q->weapon_ammo[w] == 0))
	      is_new_ammo[w] = true;
	  
	  q->weapon_ammo[w] += pickup->give.weapon.ammo;
	  break;
      }
      case PICKUP_HEALTH:
	  break;
      case PICKUP_POWERUP: {
	  enum POWERUP_CLASS w = pickup->give.powerup.powerup;
	  q->powerup_duration[w] = tick + pickup->give.powerup.duration;
	  break;
      }
    }
}

/*--------------------------------------------------------------*/

void pickup_bind_texture(void)
{
    glBindTexture(GL_TEXTURE_2D, tex_pickups);
}


void pickup_draw(const bool editor_enabled)
{
    const GLfloat *last = glWhite;
    pickup_t *p;

    pickup_bind_texture();
    glBegin(GL_QUADS);

    list_for_each(p, &pickup_list) {
	if (editor_enabled)
	    sprite_tint_col(p->highlight, p->hidden, &last);
	else if (p->hidden)
	    continue;

	pickup_draw_unit(p->class, p->x, p->y, false);

    }

    if (last != glWhite)
	glColor3fv(glWhite);

    glEnd();			/* glBegin(GL_QUADS) */
}


/* pickup_draw_unit:
 *
 * Draw the pickup 'class' at (x, y).  If for_hud is true, the
 * coordinates designate the lower-left cornet of the sprite.
 */
void pickup_draw_unit(enum PICKUP_CLASS class, const int x, const int y,
		      const bool for_hud)
{
    const sprite_t *sprite;
    texcoord2d_t coord;
    assert(class < NUM_PICKUP_CLASSES);

    sprite = &pickup_sprite[class];

    if (!for_hud &&
	!camera_can_see2(&camera, x, y, sprite->w/2.0, sprite->h/2.0))
	return;

    texcoord_from_sprite(&coord, sprite, bmp_pickups->w, bmp_pickups->h);

    if (for_hud) {
	coord.w *= 1.5;
	coord.h *= 1.5;
	gl_draw_sprite_2d(&coord, x, y);
    }
    else
	gl_draw_sprite_2d(&coord, x-(sprite->w/2.0)/2.0, y);
}

/*--------------------------------------------------------------*/

pickup_t *pickup_from_id(const pickup_id id)
{
    pickup_t *p;

    list_for_each(p, &pickup_list) {
	if (p->id == id)
	    return p;
    }

    return NULL;
}


pickup_id pickup_bounding(const int x, const int y)
{
    pickup_t *p;
    
    list_for_each(p, &pickup_list) {
	const sprite_t *sprite;
	sprite = &pickup_sprite[p->class];

	if ((p->x-sprite->w/2.0/2.0 <= x) && (x <= p->x+sprite->w/2.0/2.0) &&
	    (p->y <= y) && (y < p->y+sprite->h/2.0))
	    return p->id;
    }

    return 0;
}


enum PICKUP_CLASS pickup_select(const int n)
{
    return MID(PICKUP_WEAPON_BLASTER, n, NUM_PICKUP_CLASSES-1);
}

/*--------------------------------------------------------------*/

pickup_id pickup_generate_unique_id(void)
{
    static pickup_id id;
    id++;
    return id;
}


int pickup_total(void)
{
    return pickup_count;
}


void pickup_new(const struct packet_pickup_new *packet)
{
    pickup_t *p = malloc(sizeof(*p));
    assert(p);
    assert(packet);

    p->id = packet->id;
    p->class = packet->class;
    p->x = packet->x;
    p->y = packet->y;
    p->respawn_time = packet->time;
    p->hidden = (p->respawn_time >= server_time) ? true : false;
    p->highlight = false;

    list_add(pickup_list, p);
    pickup_count++;
}


void pickup_move(const struct packet_pickup_mov *packet)
{
    pickup_t *p;
    assert(packet);

    p = pickup_from_id(packet->id);
    if (!p) {
	fprintf(stderr, "[Pickup] Pickup %d not found!  Move failed",
		packet->id);
	return;
    }

    p->x = packet->x;
    p->y = packet->y;
}


void pickup_shift(const double offx, const double offy)
{
    pickup_t *p;

    list_for_each(p, &pickup_list) {
	p->x += offx;
	p->y += offy;
    }
}


static void pickup_free(pickup_t *p)
{
    free(p);
    pickup_count--;
}


void pickup_delete(const struct packet_pickup_del *packet)
{
    pickup_t *p;
    assert(packet);

    p = pickup_from_id(packet->id);
    if (!p) {
	fprintf(stderr, "[Pickup] Pickup %d not found!  Delete failed\n",
		packet->id);
	return;
    }

    list_remove(p);
    pickup_free(p);
}

/*--------------------------------------------------------------*/

void pickup_start(void)
{
    list_init(pickup_list);
    pickup_count = 0;
}


void pickup_stop(void)
{
    list_free(pickup_list, pickup_free);
    assert(pickup_count == 0);
}


void pickup_init(void)
{
    bmp_pickups = load_bitmap("data/pickups.tga", NULL);
    assert(bmp_pickups);

    tex_pickups = allegro_gl_make_texture_ex(AGL_TEXTURE_MASKED,
					     bmp_pickups, -1);
}


void pickup_shutdown(void)
{
    if (bmp_pickups) {
	destroy_bitmap(bmp_pickups);
	bmp_pickups = NULL;
    }
}
