/* packet-input.c,
 *
 * Convert (client) input packets to/from a buffer, to be across the
 * network.
 */

#include <assert.h>
#include "network.h"
#include "packet-input.h"


/*--------------------------------------------------------------*/

NLint packet_input_info_encode(const struct packet_input_info *packet,
			       const char command,
			       NLbyte *buf, NLint buf_size)
{
    NLint count = 0;
    assert((command == COMMAND_INPUT_INFO) ||
	   (command == COMMAND_INPUT_INFO2));

    writeByte(buf, count, command);
    writeString(buf, count, packet->name);
    writeByte(buf, count, packet->type);
    writeByte(buf, count, packet->face);

    if (command == COMMAND_INPUT_INFO2) {
	writeClientID(buf, count, packet->clid);
	writeShort(buf, count, packet->sc.total_score);
	writeShort(buf, count, packet->sc.session_score);
	writeShort(buf, count, packet->sc.total_kills);
	writeShort(buf, count, packet->sc.session_kills);
	writeShort(buf, count, packet->sc.total_deaths);
	writeShort(buf, count, packet->sc.session_deaths);
    }

    assert(count <= buf_size);
    return count;
}


void packet_input_info_decode(struct packet_input_info *packet,
			      const char command,
			      const NLbyte *buf, const NLint buf_size)
{
    NLint count = 1;
    assert((command == COMMAND_INPUT_INFO) ||
	   (command == COMMAND_INPUT_INFO2));

    readString(buf, count, packet->name);
    readByte(buf, count, packet->type);
    readByte(buf, count, packet->face);

    if (command == COMMAND_INPUT_INFO2) {
	readClientID(buf, count, packet->clid);
	readShort(buf, count, packet->sc.total_score);
	readShort(buf, count, packet->sc.session_score);
	readShort(buf, count, packet->sc.total_kills);
	readShort(buf, count, packet->sc.session_kills);
	readShort(buf, count, packet->sc.total_deaths);
	readShort(buf, count, packet->sc.session_deaths);
    }

    assert(count == buf_size);
}

/*--------------------------------------------------------------*/

NLint packet_input_move_encode(const struct packet_input_move *packet,
			       const char command,
			       NLbyte *buf, NLint buf_size)
{
    NLint count = 0;
    assert((command == COMMAND_INPUT_MOVE) ||
	   (command == COMMAND_INPUT_WIELD));

    writeByte (buf, count, command);
    writeShort(buf, count, packet->id);
    writeByte (buf, count, packet->impy);
    writeByte (buf, count, packet->mirror);
    writeFloat(buf, count, packet->angle);

    if (command == COMMAND_INPUT_WIELD)
	writeByte(buf, count, packet->weapon);

    assert(count <= buf_size);
    return count;
}


void packet_input_move_decode(struct packet_input_move *packet,
			      const char command,
			      const NLbyte *buf, const NLint buf_size)
{
    NLint count = 1;		/* Skip the command. */

    readShort(buf, count, packet->id);
    readByte (buf, count, packet->impy);
    readByte (buf, count, packet->mirror);
    readFloat(buf, count, packet->angle);

    if (command == COMMAND_INPUT_WIELD)
	readByte(buf, count, packet->weapon);

    assert(count == buf_size);
}

/*--------------------------------------------------------------*/

NLint packet_input_chat_encode(const NLbyte *string,
			       NLbyte *buf, NLint buf_size)
{
    NLint count = 0;

    writeByte(buf, count, COMMAND_INPUT_CHAT);
    writeString(buf, count, string);

    assert(count <= buf_size);
    return count;
}


void packet_input_chat_decode(NLbyte *string,
			      const NLbyte *buf, const NLint buf_size)
{
    NLint count = 1;

    readString(buf, count, string);

    assert(count == buf_size);
}
