/* packet-container.c,
 *
 * Convert container command packets to/from a buffer, to be sent
 * across the network.
 */

#include <assert.h>
#include "network.h"
#include "packet-container.h"
#include "stats-container.h"


/*--------------------------------------------------------------*/

NLint packet_container_new_begin_encode(const int num,
					NLbyte *buf, const NLint sz)
{
    NLint count = 0;
    assert(num > 0);
    assert(buf);

    writeByte(buf, count, COMMAND_CONTAINER_NEW);
    writeByte(buf, count, num);

    assert(count <= sz);
    return count;
}


NLint packet_container_new_begin_decode(int *num,
					const NLbyte *buf, const NLint sz)
{
    NLint count = 1;
    int n;
    assert(num);
    assert(buf);

    readByte(buf, count, n); *num = n;

    assert(count <= sz);
    return count;
}


NLint packet_container_new_encode(const struct packet_container_new *packet,
				  NLbyte *buf, const NLint sz, NLint count)
{
    int xx = packet->x + zero_x;
    int yy = packet->y + zero_y;
    char symbol = container_class_to_symbol(packet->class);
    assert(packet);
    assert(buf);

    writeContainerID(buf, count, packet->id);
    writeContainerClass(buf, count, symbol);
    writeShort(buf, count, xx);
    writeShort(buf, count, yy);
    writeByte (buf, count, packet->hidden);

    assert(count <= sz);
    return count;
}


NLint packet_container_new_decode(struct packet_container_new *packet,
				  const NLbyte *buf, const NLint sz,
				  NLint count)
{
    int xx, yy;
    char symbol;
    assert(packet);
    assert(buf);

    readContainerID(buf, count, packet->id);
    readContainerClass(buf, count, symbol);
    packet->class = container_class_from_symbol(symbol);
    readShort(buf, count, xx); packet->x = xx - zero_x;
    readShort(buf, count, yy); packet->y = yy - zero_y;
    readByte (buf, count, packet->hidden);

    assert(count <= sz);
    return count;
}

/*--------------------------------------------------------------*/

NLint packet_container_show_encode(const struct packet_container_show *packet,
				   NLbyte *buf, NLint size)
{
    NLint count = 0;

    writeByte(buf, count, COMMAND_CONTAINER_SHOW);
    writeContainerID(buf, count, packet->id);

    assert(count <= size);
    return count;
}


void packet_container_show_decode(struct packet_container_show *packet,
				  const NLbyte *buf, const NLint size)
{
    NLint count = 1;

    readContainerID(buf, count, packet->id);

    assert(count == size);
}

/*--------------------------------------------------------------*/

NLint packet_container_hide_encode(const struct packet_container_hide *packet,
				   NLbyte *buf, NLint size)
{
    NLint count = 0;

    writeByte (buf, count, COMMAND_CONTAINER_HIDE);
    writeLong (buf, count, packet->time);
    writeContainerID(buf, count, packet->id);

    assert(count <= size);
    return count;
}


void packet_container_hide_decode(struct packet_container_hide *packet,
				  const NLbyte *buf, const NLint size)
{
    NLint count = 1;

    readLong (buf, count, packet->time);
    readContainerID(buf, count, packet->id);

    assert(count == size);
}

/*--------------------------------------------------------------*/

NLint packet_container_mov_encode(const struct packet_container_mov *packet,
				  NLbyte *buf, NLint size)
{
    NLint count = 0;
    int xx = packet->x + zero_x;
    int yy = packet->y + zero_y;

    writeContainerID(buf, count, packet->id);
    writeShort(buf, count, xx);
    writeShort(buf, count, yy);

    assert(count <= size);
    return count;
}


void packet_container_mov_decode(struct packet_container_mov *packet,
				 const NLbyte *buf, const NLint size)
{
    NLint count = 0;
    int xx, yy;

    readContainerID(buf, count, packet->id);
    readShort(buf, count, xx); packet->x = xx - zero_x;
    readShort(buf, count, yy); packet->y = yy - zero_y;

    assert(count == size);
}

/*--------------------------------------------------------------*/

NLint packet_container_del_encode(const struct packet_container_del *packet,
				  NLbyte *buf, NLint size)
{
    NLint count = 0;

    writeContainerID(buf, count, packet->id);

    assert(count <= size);
    return count;
}


void packet_container_del_decode(struct packet_container_del *packet,
				 const NLbyte *buf, const NLint size)
{
    NLint count = 0;

    readContainerID(buf, count, packet->id);

    assert(count == size);
}
